#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (c) 2016, Intel Corporation. All rights reserved.
# This file is licensed under the GPLv2 license.
# For the full content of this license, see the LICENSE.txt
# file at the top level of this source tree.

import subprocess
import json
import sys
import os
from tools import logging_helper, network_ops, shell_ops, sysinfo_ops
import manage_config
import manage_package
from manage_auth import require


sys.path.append('..')


def configure_default_repo():
    """  Reads the configuration file for the corresponding architecture dedicated repository.

    Configure the default repository if it has not already been configured.

    Returns:
        dict: the dict of system info
    """
    log_helper = logging_helper.logging_helper.Logger()
    data_collector = sysinfo_ops.DataCollect()
    result = data_collector.getDataSet()

    config = manage_config.read_config_file()
    developer_hub_url = config.get('DefaultRepo', 'base_repo')
    architecture, rcpl_version = data_collector.platform_details()
    developer_hub_url = developer_hub_url + rcpl_version + '/' + architecture

    # Check if repository already configured
    try:
        cmd = "smart channel --show | grep " + developer_hub_url
        subprocess.check_output(cmd, shell=True)
        log_helper.logger.debug("Default repository is configured.")
        update_channels()
    except subprocess.CalledProcessError:
        # Verify entry does not exist with different URL. Then configure default repository.
        try:
            cmd = "smart channel --show | grep Intel_Repository"
            subprocess.check_output(cmd, shell=True)
            remove_repo("Intel_Repository")
        except:
            pass
        log_helper.logger.debug("Configuring default repository...")
        # Do not need to smart update.
        # When there is no network, smart update takes a long time to timeout.
        # Also, we want to add the default repo no matter what.
        add_repo(developer_hub_url, "None", "None", "Intel_Repository", do_update=False)
    return result


def list_repos():
    """ List all repositories from the 'smart channel --list' command.

    Returns:
        list: list of String of channel names
    """
    if manage_config.use_new_list_repos:
        result = list_repos_new()
    else:
        result = list_repos_original()
    return result


def list_repos_original():
    """ List all repositories from the 'smart channel --list' command.

    Returns:
        list: list of String of channel names
    """
    log_helper = logging_helper.logging_helper.Logger()
    response = []
    channel_list = shell_ops.run_command('smart channel --list').split("\n")
    for item in channel_list:
        if item != "" and 'rpmsys' not in item:
            response.append(item)
    log_helper.logger.debug("List of repositories: '%s'" % str(response))
    return response


def list_repos_new():
    """ List all repositories from smart's cache /var/lib/smart/channels.

    The requirements of using this is to make sure that smart update is called to update cache,
    when add/remove repo is done.

    Returns:
        list: list of String of channel names
    """
    log_helper = logging_helper.logging_helper.Logger()
    response = []

    # Smart cache should be updated already upon repo/channel ops.

    # Find out the channels by checking /var/lib/smart/channels folder. (This won’t include rpmsys channel.)
    try:
        for f in os.listdir(manage_config.smart_cache_path):
            if os.path.isfile(manage_config.smart_cache_path + '/' + f):
                repo_name = f.split('%%')[0]
                if not (repo_name in response):
                    response.append(repo_name)
    except Exception as e:
        log_helper.logger.debug(str(e))

    log_helper.logger.debug("List of repositories: '%s'" % str(response))
    return response


def add_repo(url, user_name, password, name, do_update=True):
    """ Add a repository to the channel cache.

    Args:
        url (str): Location of the repository
        user_name (str): Login protected repository username
        password (str): Login protected repository password
        name (str): Desired name of the repository
        do_update (bool): do smart update or not

    Returns:
        str: Json string with key 'status' with values 'success' or 'failure' and 'error' with None or 'Error message'
    """
    log_helper = logging_helper.logging_helper.Logger()
    response = {'status': 'failure', 'error': None}

    if ' ' in url:
        response['error'] = "Failed to add repository: URL cannot contain spaces"
        log_helper.logger.error("Error: URL cannot contain spaces")
        return json.dumps(response)

    if 'http://' in url or 'https://' in url:
        pass
    else:
        response['error'] = "Failed to add repository: URL must contain protocol (example: http://)"
        log_helper.logger.error("Error: URL must contain protocol")
        return json.dumps(response)

    repo_list = list_repos()
    if name in repo_list:  # Confrim repo does not exist
        response['error'] = "Duplicate name"
        log_helper.logger.error("Error: Duplicate repository name '%s'" % name)
        return json.dumps(response)

    if user_name != "None":  # Build URL if username and password are required
        user_name = user_name.replace("@", "%40")
        if password == "None":
            url = url.replace("://", "://%s@" % user_name)
        else:
            url = url.replace("://", "://%s:%s@" % (user_name, password))

    command = "smart channel --add '" + name + "' type=rpm-md baseurl=" + url + " -y"
    result = shell_ops.run_cmd_chk(command)
    if result['returncode']:
        if "error" in result['cmd_output']:
            remove_repo(name)
            error = "Failed to add repository: " + result['cmd_output'][result['cmd_output'].index("error:") + 7:].replace("\n", "")
            response['error'] = error
            log_helper.logger.error("Failed to add repository. Error output: '%s'" % response['error'])
    else:
        if do_update:
            command = "smart update '" + name + "'"
            result = shell_ops.run_cmd_chk(command)  # Attempt to connect to new repo
        else:
            result = {'returncode': None, 'cmd_output': None}
        if result['returncode']:  # If attempt fails determine error and remove repo
            if "error" in result['cmd_output']:
                if "Invalid URL" in result['cmd_output']:
                    response['error'] = "Failed to add repository: Invalid URL."
                elif "Failed to connect" in result['cmd_output'] or 'URL returned error: 503' in result['cmd_output']:
                    response['error'] = "Failed to add repository: Unable to connect to repository."
                elif "Invalid XML" in result['cmd_output']:
                    response['error'] = "Failed to add repository: Repository XML file invalid. "
                else:
                    response['error'] = result['cmd_output'][result['cmd_output'].index("error:") + 7:].replace("\n", "")
            else:
                response['error'] = 'Error adding repository'
            remove_repo(name)
            log_helper.logger.error("Failed to add repository. Error output: '%s'" % response['error'])
        else:  # Update channels and package list
            update_channels()
            response['status'] = "success"
            log_helper.logger.debug("Successfully added repository '%s'" % name)
    return json.dumps(response)
    

def remove_repo(name, UpdateCache=False):
    """ Remove a repository from the channel cache.

    Args:
        name (str): Name of the repository from 'smart channel --list'
        UpdateCache (bool): False if we do not want to do smart update. True if we want to do smart update.

    Returns:
        str: Json string with key 'status' and values 'success' or 'fail'

    """
    log_helper = logging_helper.logging_helper.Logger()
    log_helper.logger.debug("Removing repository named '%s'" + name)

    result = subprocess.check_output(["smart", "channel", "--remove", name, "-y"])
    if "error:" in result:
        response = ({
            'status': "fail",
        })
        log_helper.logger.error("Failed to remove repository named '%s'" % name)
    else:
        if UpdateCache:
            network_checker = network_ops.NetworkCheck()
            net_conn, net_resp = network_checker.test_network_connection(check_http=manage_config.network_check_http)
            if (net_conn['https_conn'] != "False") and (net_conn['http_conn'] != "False"):
                subprocess.check_output(['smart', 'update'])

        response = ({
            'status': "success",
        })
        log_helper.logger.debug("Successfully removed repository named '%s'" % name)

    # package list is updated. Recreate pacakge database
    manage_package.build_package_database()
    return json.dumps(response)


def update_channels(CheckNetworkAgain=True):
    """ Update the channel cache.
    Args:
        CheckNetworkAgain (bool):  True if we want to check network again.
    Returns:
        str: Json string with key 'status' and values 'success' or 'fail'
    """
    log_helper = logging_helper.logging_helper.Logger()
    response = ({
        'status': "fail",
    })

    do_run = False
    if CheckNetworkAgain:
        network_checker = network_ops.NetworkCheck()
        net_conn, net_resp = network_checker.test_network_connection(check_http=manage_config.network_check_http)
        if (net_conn['https_conn'] != "False") and (net_conn['http_conn'] != "False"):
            do_run = True
        else:
            pass
    else:
        do_run = True

    if do_run:
        # We need this "smart update"... Add repo is relying on this.
        shell_ops.run_command("smart update")
        response = ({
                'status': "success",
            })
        log_helper.logger.debug("Successfully updated channel.")
        manage_package.update_package_list()
    return json.dumps(response)


@require()
class Repository(object):
    exposed = True

    def GET(self, **kwargs):
        return json.dumps(list_repos())

    def POST(self, **kwargs):
        return add_repo(kwargs['url'], kwargs['username'], kwargs['password'], kwargs['name'])

    def PUT(self):
        return update_channels()

    def DELETE(self, **kwargs):
        return remove_repo(kwargs['name'], UpdateCache=True)
