#!/usr/bin/env python

# Copyright (c) 2016, Intel Corporation. All rights reserved.
# This file is licensed under the GPLv2 license.
# For the full content of this license, see the LICENSE.txt
# file at the top level of this source tree.

import cherrypy
import os
import mimetypes
from tools import logging_helper, rsyslog_update
import manage_config
import manage_package
import manage_repo
import manage_proxy
import manage_auth
import manage_service
import manage_usb
import manage_os_controls
import manage_os_update
import manage_service
import manage_usb
import manage_hac
import manage_pro_upgrade
import manage_security
import manage_mec
import manage_self_update
import manage_nginx_userDB


'''
Create a server using CherryPy and expose the /action/ directory.
These actions can be routed to @action_select and distributed by
calling different functions based on specified arguments.

Usage:
    /action/install_package?package=gedit
        action is "install_package"
        arguments are "package (key), gedit (value)"

    /action/add_repo?name=intel_repo&url=http://myurl
        action is "add_repo"
        arguments are "name (key), intel_repo (value)",
         "url (key), http://myurl (value)"
'''


class Server(object):
    authenticated = False
    proxy_override = False

    @cherrypy.expose
    def action(self, action, **kwargs):
        arg_list = []
        for key, value in kwargs.iteritems():
            arg_list.append([key, value])
        return action_select(action, arg_list)

    @cherrypy.expose
    def index(self):
        index_location = os.path.abspath(os.path.join(os.path.dirname(__file__), '..'))
        return open(index_location + '/index.html')


if __name__ == '__main__':
    # Logging config
    my_logger = logging_helper.logging_helper.Logger(logger_name='backend_general')

    sys_info_dict = manage_repo.configure_default_repo()
    manage_package.set_signature_verification_status(True)

    timeout_value = manage_config.cherrypy_response_timeout_other
    timeout_session_value = manage_config.cherrypy_session_timeout_other
    if 'Quark' in sys_info_dict['model']:
        timeout_value = manage_config.cherrypy_response_timeout_quark
        timeout_session_value = manage_config.cherrypy_session_timeout_quark

    my_logger.logger.debug('Cherrpy response timeout (s): ' + str(timeout_value) + ', and session timeout (m): ' + str(timeout_session_value))

    # Server config
    cherrypy.config.update({
        'tools.sessions.on': True,
        'tools.sessions.timeout': timeout_session_value,
        'tools.auth.on': True,
    })

    mimetypes.types_map['.svg'] = 'image/svg+xml'
    mimetypes.types_map['.svgz'] = 'image/svg+xml'

    root_config = {
        '/': {
            'tools.staticdir.on': True,
            'tools.staticdir.dir': '',
            'tools.staticdir.root': os.path.abspath(os.path.join(os.path.dirname(__file__), '..')),
        },
        '/secure': {
            'tools.staticdir.on': True,
            'tools.staticdir.dir': 'partials',
        },
        '/images': {
            'tools.staticdir.on': True,
            'tools.staticdir.dir': 'images',
        },
        '/branding': {
            'tools.staticdir.on': True,
            'tools.staticdir.dir': 'branding',
        },
        '/css': {
            'tools.staticdir.on': True,
            'tools.staticdir.dir': 'css',
        },
        '/js': {
            'tools.staticdir.on': True,
            'tools.staticdir.dir': 'js',
        },
        '/fonts': {
            'tools.staticdir.on': True,
            'tools.staticdir.dir': 'fonts',
        },
        'tools.staticdir.content_types': {'svg': 'image/svg+xml'},
    }
    error_location = os.path.abspath(os.path.join(os.path.dirname(__file__), '..')) + '/404.html'
    cherrypy.config.update({'error_page.404': error_location})

    cherrypy.tools.auth = cherrypy.Tool('before_handler', manage_auth.check_auth)

    # Configure server with directories
    app_config = {
        '/':
            {'request.dispatch': cherrypy.dispatch.MethodDispatcher(),
             'response.timeout': timeout_value}
        }
    cherrypy.tree.mount(manage_package.Packages(), '/api/packages', config=app_config)
    cherrypy.tree.mount(manage_repo.Repository(), '/api/repository', config=app_config)
    cherrypy.tree.mount(manage_proxy.Proxy(), '/api/proxy', config=app_config)
    cherrypy.tree.mount(manage_self_update.SelfUpgrade(), '/api/selfupgrade', config=app_config)
    cherrypy.tree.mount(manage_auth.Auth(), '/api/auth', config=app_config)
    cherrypy.tree.mount(manage_security.SecurtiyAutomation(), '/api/sa', config=app_config)
    cherrypy.tree.mount(manage_service.Service_control(), '/api/sc', config=app_config)
    cherrypy.tree.mount(manage_os_controls.OSControls(), '/api/osc', config=app_config)
    cherrypy.tree.mount(manage_usb.USB_API(), '/api/usb', config=app_config)
    cherrypy.tree.mount(manage_pro_upgrade.EnablePro(), '/api/pro', config=app_config)
    cherrypy.tree.mount(manage_os_update.OSUpdate(), '/api/osup', config=app_config)
    cherrypy.tree.mount(Server(), '/', config=root_config)
    cherrypy.tree.mount(manage_hac.HAC(), '/api/hac', config=app_config)
    cherrypy.tree.mount(manage_package.PackageInfo(), '/api/packageinfo', config=app_config)
    cherrypy.tree.mount(manage_mec.MEC_API(), '/api/mec', config=app_config)
    cherrypy.tree.mount(manage_auth.Session(), '/api/validate_session', config=app_config)
    cherrypy.server.unsubscribe()

    config = manage_config.read_config_file()

    hdc_config = config.get('HDC', 'config')
    if hdc_config == 'False':
        my_logger.logger.debug('setting hdc')
        manage_config.HDCSettings.set_hdc_server_details()
        rsys = rsyslog_update.RSYSLOG()
        rsys.update_rsyslog()

    secure_http = config.get('SecurityAutomation', 'secure_http')
    if secure_http == 'true':
        https_server = cherrypy._cpserver.Server()
        https_server.socket_port = 3092
        https_server._socket_host = "0.0.0.0"
        # Enable SSL
        https_server.ssl_module = 'builtin'
        https_server.ssl_certificate = "/etc/nginx/ssl/ssl-cert-snakeoil.pem"
        https_server.ssl_private_key = "/etc/nginx/ssl/ssl-cert-snakeoil.key"
        https_server.subscribe()

    else:
        http_server = cherrypy._cpserver.Server()
        http_server.socket_port = 80
        http_server._socket_host = '0.0.0.0'
        http_server.subscribe()
     
    cherrypy.engine.start()
    nginx_userDB = manage_nginx_userDB.ManageNginxUserDB()
    nginx_userDB.start_thread()
    cherrypy.engine.block()
    nginx_userDB.stop_thread()

def shutdown_http_server():
    #   import __main__
    #   __main__.http_server.unsubscribe()
    cherrypy.engine.restart()
