#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (c) 2016, Intel Corporation. All rights reserved.
# This file is licensed under the GPLv2 license.
# For the full content of this license, see the LICENSE.txt
# file at the top level of this source tree.


import math
import socket
import platform
import os
from time import ctime
from uptime import uptime
from netifaces import interfaces, ifaddresses, AF_INET
from tools import logging_helper, shell_ops

# global variables used across modules
sysVersion = str("Not Set")
rcpl_version = str("Not Set")
arch = str("Not Set")
display_arch = str("Not Set")


class DataCollect(object):

    def __init__(self):
        self.__ss_id = None
        self.__data_set = {}
        self.__lan_ip_addr = {}
        self.__wan_ip_addr = {}
        up_time_secs = uptime()
        self.__minutes = int(math.floor((up_time_secs % 3600)/60))
        self.__hours = int(math.floor((up_time_secs % 86400)/3600))
        self.__days = int(math.floor((up_time_secs % 2592000)/86400))
        self.__up_time_buf = '%sd  %sh %sm' % (self.__days, self.__hours, self.__minutes)
        self.__disk_usage = {}
        self.__free = 0.0
        self.__total = 0.0
        self.__used = 0.0
        self.__log_helper = logging_helper.logging_helper.Logger()

    def getHostname(self):
        """ Return host name.
        Returns:
            str: host name
        """
        return socket.gethostname()

    def getModel(self):
        """ Return processor model.
        Returns:
            str: processor model
        """
        return platform.processor()

    def getLanIPAddr(self):
        """ Return land and wan ip addresses.
        Returns:
            tuple: 1st is str of lan ip address. 2nd is str of wan ip address.
        """
        iface_name = interfaces()
        lan_adapter_list = [i for i in iface_name if ('eth' in i)]
        wan_adapter_list = [i for i in iface_name if ('wlan' in i)]
        for adapter in lan_adapter_list:
            adapter_config = ifaddresses(adapter).setdefault(AF_INET, [{'addr': 'na'}])
            ip_address = adapter_config[0]['addr']
            self.__lan_ip_addr[adapter] = ip_address
        for adapter in wan_adapter_list:
            adapter_config = ifaddresses(adapter).setdefault(AF_INET, [{'addr': 'na'}])
            ip_address = adapter_config[0]['addr']
            if ip_address == 'na':
                brlan_adapter_list = [i for i in iface_name if ('br-lan' in i)]
                for bridge in brlan_adapter_list:
                    adapter_config = ifaddresses(bridge).setdefault(AF_INET, [{'addr': 'na'}])
                    ip_address = adapter_config[0]['addr']
            self.__wan_ip_addr[adapter] = ip_address
        return self.__lan_ip_addr, self.__wan_ip_addr

    def getWifiSSID(self):
        """ Return WiFi SSID.
        Returns:
            str: WiFi SSID
        """
        self.__ss_id = shell_ops.run_command('uci show wireless.@wifi-iface[0].ssid')
        if 'not found' in self.__ss_id:
            return 'No Wireless'
        self.__ss_id = self.__ss_id.strip('\n').split('=')
        return self.__ss_id[1]

    def getDateTime(self):
        """ Return current date time.
        Returns:
            str: current date time.
        """
        return ctime()

    def getUpTime(self):
        """ Return system up time.
        Returns:
            str: system up time.
        """
        self.__log_helper.logger.debug(str(self.__up_time_buf))
        return self.__up_time_buf

    def getSystemVersion(self):
        """ Return system version.
        Returns:
            str: system version
        """
        global sysVersion
        global rcpl_version
        if rcpl_version == "Not Set":
            version = platform.release()
            version = version.split('_')
            version = version[2].split('-')
            sysVersion = version[1]
            version = version[1].split('.')
            rcpl_version = version[3]
        return sysVersion

    def getCPUType(self):
        """ Return cpu arch.
        Returns:
            str: cpu arch
        """
        global arch
        global display_arch
        if arch == "Not Set":
            self.__log_helper.logger.debug('Setting Arch')
            command = '''rpm -q --queryformat %{ARCH} bash'''
            arch = shell_ops.run_command(command)
            if arch == 'corei7_64':
                arch = "baytrail"
            elif arch == 'quark':
                arch = "quark"
            else:
                arch = "haswell"
        cpu_name = shell_ops.run_cmd_chk("cat /proc/cpuinfo | grep 'model name' | uniq")
        display_arch = cpu_name['cmd_output'].split(':')[1]
        return display_arch

    def getDiskUsage(self, path):
        """ Return disk usage statistics about the given path.
        Args:
            path (str): the target path to get usage info for.
        Returns:
            dict: keys are 'total', 'used' and 'free', which are the amount of total, used and free space, in bytes.
        """
        st = os.statvfs(path)
        self.__free = (st.f_bavail * st.f_frsize)/1000000000.0
        self.__total = (st.f_blocks * st.f_frsize)/1000000000.0
        self.__used = ((st.f_blocks - st.f_bfree) * st.f_frsize)/1000000000.0
        self.__disk_usage['total'] = '%.1f' % round(self.__total, 1)
        self.__disk_usage['used'] = '%.1f' % round(self.__used, 1)
        self.__disk_usage['free'] = '%.1f' % round(self.__free, 1)
        return self.__disk_usage

    def getDataSet(self):
        """ Return system info data.
        Returns:
            dict: keys are 'host_name', 'model', 'time', 'uptime', 'lan', 'wan', 'ssid', 'disk', 'system_version'
        """
        self.__data_set['host_name'] = self.getHostname()
        self.__data_set['model'] = self.getCPUType()
        self.__data_set['time'] = self.getDateTime()
        self.__data_set['uptime'] = self.getUpTime()
        lan, wan = self.getLanIPAddr()
        self.__data_set['lan'] = lan
        self.__data_set['wan'] = wan
        self.__data_set['ssid'] = self.getWifiSSID()
        self.__data_set['disk'] = self.getDiskUsage('/')
        self.__data_set['system_version'] = self.getSystemVersion()
        return self.__data_set

    def platform_details(self):
        """ Return platform architecture and rcpl version.
        Returns:
            tuple: 1st is str of platform architecture. 2nd is str of rcpl version.
        """
        architecture = platform.machine()
        version = platform.release()
        version = version.split('_')
        version = version[2].split('-')
        version = version[1].split('.')
        rcpl_version = 'rcpl' + str(version[3])
        return architecture, rcpl_version


