/******************************************************************************
 *
 *                   INTEL CORPORATION PROPRIETARY INFORMATION
 *       This software is supplied under the terms of a license agreement or
 *       nondisclosure agreement with Intel Corporation and may not be copied
 *       or disclosed except in accordance with the terms of that agreement.
 *
 *            Copyright (c) 2008-2009 Intel Corporation. All Rights Reserved.
 *
 *       All rights reserved.  No part of this program or publication may be
 *       reproduced, transmitted, transcribed, stored in a retrieval system,
 *       or translated into any language or computer language, in any form or
 *       by any means, electronic, mechanical, magnetic, optical, chemical,
 *       manual, or otherwise, without the prior written permission of Intel
 *       Corporation.
 *
 *
 *******************************************************************************/
#ifndef __CR_MAP_H_
#define __CR_MAP_H_

#include <os/include/cr_malloc.h>
#include <string.h>
#include <stdio.h>

#include "cr_default_functors.h"
#include "cr_vector.h"
#include "cr_pair.h"

namespace osal_stl {
template<class Key, class Data, class Compare = cr_less<Key> >
class cr_map {
//public typedefs and member variables.
public:
    typedef Key key_type;
    typedef Data data_type;
    typedef cr_pair<const key_type, data_type> value_type;
    typedef value_type*         pointer;
    typedef value_type&         reference;
    typedef const value_type&     const_reference;
    typedef size_t size_type;
    typedef size_t difference_type;

    typedef Compare key_compare;

    typedef  cr_vector<pointer>                     btree_type;
    typedef typename btree_type::iterator btreeitr_type;

    typedef class iterator {
public:
        typedef cr_map<Key, Data, Compare>              Map_type;
        typedef Map_type*                               MapPtr_type;
        typedef typename    Map_type::value_type value_type;
        typedef typename    Map_type::reference reference;
        typedef typename    Map_type::pointer pointer;
        typedef typename    Map_type::size_type size_type;
        typedef typename    Map_type::btreeitr_type btreeitr_type;


/* default constructor */
        CR_NO_THROW inline iterator()
        {
            //printf("default map::iterator constructor called.\n");
            m_itr = (btreeitr_type) 0;
            m_mapPtr = (MapPtr_type) 0;
        }

/* copy constructor */
        CR_NO_THROW iterator(const iterator& mItr)
        {
            //printf("map iterator copy constructor called.\n");
            m_itr = mItr.m_itr;
            m_mapPtr = mItr.m_mapPtr;
        }

/* constructor */
        CR_NO_THROW inline iterator(btreeitr_type itr, MapPtr_type mptr) : m_itr(itr), m_mapPtr(mptr)
        {
            //printf ("came into map::iterator constructor with parameters.\n");
        }

/* operator* */
        CR_NO_THROW reference operator*() const
        {
            reference i = *(*m_itr);

            return i;
        }

/* operator-> */
        CR_NO_THROW pointer operator->() const
        {
            return &(operator*());
        }

/* ++operator */
        CR_NO_THROW inline iterator& operator++(void)
        {
            m_itr++;
            return *this;
        }

/* operator++ */
        CR_NO_THROW inline iterator operator++(int)
        {
            iterator tmp = *this;

            ++*this;
            return tmp;
        }

/* operator== */
        CR_NO_THROW inline bool operator==(const iterator& itr) const
        {
            return(m_itr == itr.m_itr);
        }

/* operator!= */
        CR_NO_THROW inline bool operator!=(const iterator& itr) const
        {
            //printf("calling operator!= in iterator.\n");
            //printf("would return %d\n", (m_itr!=itr.m_itr));
            return(m_itr != itr.m_itr);
        }

protected:
        MapPtr_type m_mapPtr;
        btreeitr_type m_itr;
    } iterator;

    typedef const iterator const_iterator;

    typedef class value_compare : public binary_function<value_type, value_type, bool>
    {
        friend class cr_map<Key, Data, Compare>;
protected:
        Compare comp;

        value_compare(Compare c) : comp(c)
        {
        }

public:
        CR_NO_THROW bool operator()(const value_type& x, const value_type& y) const
        {
            return comp(x.first, y.first);
        }
    } value_compare;

//protected member variables.
protected:

    size_type m_num_elements;
    btree_type btree;
    key_compare m_comp;
    value_compare   m_vcomp(cr_less<Key>);
    const static int MIN_CR_MAP_RESERVE = 500;


//public methods.
public:
/* default constructor */
    CR_NO_THROW cr_map()
    {
        btree.reserve(MIN_CR_MAP_RESERVE);
        m_num_elements = 0;
    }

/* constructor */
    CR_NO_THROW cr_map(const key_compare& comp)
    {
        btree.reserve(MIN_CR_MAP_RESERVE);
        m_num_elements = 0;
        m_comp = comp;
    }

/* constructor */
    CR_NO_THROW cr_map(iterator f, iterator l)
    {
        btree.reserve(MIN_CR_MAP_RESERVE);
        m_num_elements = 0;

        for(iterator i = f; i != l; i++)
        {
            pointer tmpPtr = get_new_node(value_type((*i).first, (*i).second));
            insert_to_vector(tmpPtr);
        }
    }

/* constructor */
    CR_NO_THROW cr_map(iterator f, iterator l, const key_compare& comp)
    {
        btree.reserve(MIN_CR_MAP_RESERVE);
        m_num_elements = 0;
        m_comp = comp;

        for(iterator i = f; i != l; i++)
        {
            pointer tmpPtr = get_new_node(value_type((*i).first, (*i).second));
            insert_to_vector(tmpPtr);
        }
    }

/* copy constructor */
    CR_NO_THROW cr_map(const cr_map& mp)
    {
        btree.clear();
        btree.resize(mp.btree.size());

        for(size_type i = 0; i < mp.btree.size(); i++)
        {
            pointer tmpPtr = get_new_node(value_type((mp.btree[i])->first, (mp.btree[i])->second));

            btree[i] = tmpPtr;
        }

        m_num_elements = mp.m_num_elements;
    }

/* destructor */
    CR_NO_THROW ~cr_map()
    {
        clear();
    }

    CR_NO_THROW inline iterator begin(void)
    {
        //printf("calling iterator constructor in map::iterator begin().\n");
        return iterator(btree.begin(), this);
    }

    CR_NO_THROW inline iterator end(void)
    {
        //printf("calling iterator constructor in map::iterator end().\n");
        return iterator(btree.end(), this);
    }

#if 0
    CR_NO_THROW inline const_iterator begin(void) const
    {
        return (const_iterator)iterator(btree.begin(), this);
    }

    CR_NO_THROW inline const_iterator end(void) const
    {
        return (const_iterator)end();
    }
#endif

    CR_NO_THROW inline size_type size(void) const
    {
        return m_num_elements;
    }

    CR_NO_THROW inline size_type max_size(void) const
    {
        return size_type(-1);
    }

    CR_NO_THROW inline bool empty(void) const
    {
        return(size() == 0);
    }

    CR_NO_THROW inline key_compare key_comp(void) const
    {
        return m_comp;
    }

    CR_NO_THROW inline value_compare value_comp(void) const
    {
        return m_vcomp;
    }

/* assignment operator */
    CR_NO_THROW inline cr_map& operator=(const cr_map& mp)
    {
        /* assign only if not ourselves */
        if(&mp != this)
        {
            clear();
            m_comp = mp.m_comp;

            btree.resize(mp.btree.size());

            for(size_type i = 0; i < mp.btree.size(); i++)
            {
                pointer tmpPtr = get_new_node(value_type((mp.btree[i])->first, (mp.btree[i])->second));

                btree[i] = tmpPtr;
            }

            m_num_elements = mp.m_num_elements;
        }

        return *this;
    }

    CR_NO_THROW inline void swap(cr_map& mp)
    {
        /* First swap the compare function */
        key_compare tmpComp = m_comp;

        m_comp = mp.m_comp;
        mp.m_comp = tmpComp;

        /* Now swap the btree */
        btree.swap(mp.btree);

        /* Now swap the num of elements */
        size_type s = m_num_elements;
        m_num_elements = mp.m_num_elements;
        mp.m_num_elements = s;
    }

    CR_NO_THROW inline size_type erase(const key_type& k)
    {
        if(erase_from_vector(k))
            return (size_type)1;
        else
            return (size_type)0;
    }

    CR_NO_THROW inline size_type count(const key_type& k)
    {
        size_type i;
        size_type s = 0;

        if(find_index(k, i))
        {
            s = 1;
        }

        return s;
    }

    CR_NO_THROW inline data_type& operator[](const key_type& k)
    {
        //printf("calling map operator[]\n");
        size_type i;

        if(!find_index(k, i))
        {
            //printf("calling get_new_node.\n");
            pointer tmpPtr = get_new_node(value_type(k, data_type()));
            //printf("calling insert_to_vector.\n");
            i = insert_to_vector(tmpPtr);
        }

        return btree[i]->second;
    }

    CR_NO_THROW inline void clear(void)
    {
        //Just return if nothing to clear.
        if(m_num_elements == 0)
            return;

        /* delete the pointers */
        btreeitr_type itr;

        for(itr = btree.begin(); itr != btree.end(); itr++)
        {
            // fix KW NPD.FUNC.MIGHT issue
            if (itr == NULL)
                break;

            delete_node(*itr);
        }

        btree.clear();

        m_num_elements = 0;
    }

    CR_NO_THROW inline iterator find(const key_type& k)
    {
        size_type i;

        if(find_index(k, i))
        {
            return iterator((btree.begin() + i), this);
        }
        else
        {
            return end();
        }
    }

    CR_NO_THROW inline const_iterator find(const key_type& k) const
    {
        int i;

        if(find_index(k, i))
        {
            return (const_iterator)iterator((btree.begin() + i), this);
        }
        else
        {
            return end();
        }
    }

    CR_NO_THROW inline iterator lower_bound(const key_type& k)
    {
        size_type i;

        find_index(k, i);
        return iterator(btree.begin() + i, this);
    }

    CR_NO_THROW inline iterator upper_bound(const key_type& k)
    {
        size_type i;

        if(find_index(k, i))
            i++;

        return iterator(btree.begin() + i, this);
    }

    CR_NO_THROW inline cr_pair<iterator, bool> insert(const value_type& x)
    {
        iterator i;
        bool inserted = false;

        size_t old_size = size();

        pointer tmpPtr = get_new_node(value_type(x.first, x.second));
        size_t index = insert_to_vector(tmpPtr);

        if(size() == (old_size + 1))
        {
            inserted = true;
            i = iterator(btree.begin() + index, this);
        }
        else
        {
            delete_node(tmpPtr);
            i = end();
        }

        return cr_pair<iterator, bool>(i, inserted);
    }

/* This function is provided for completeness sake */
    CR_NO_THROW inline iterator insert(iterator pos, const value_type& x)
    {
        pos;
        cr_pair<iterator, bool> i = insert(x);

        return i.first;
    }

    CR_NO_THROW inline void insert(iterator first, iterator last)
    {
        for(iterator i = first; i != last; i++)
        {
            insert(*i);
        }
    }

    CR_NO_THROW inline void erase(iterator pos)
    {
        if(pos != end())
        {
            erase((*pos).first);
        }
    }

    CR_NO_THROW inline void erase(iterator first, iterator last)
    {
        size_type f, i;

        find_index((*first).first, f);
        iterator mItr = first;
        for(i = 0; mItr != last; i++, mItr++)
        {
            pointer ptr = btree[f + i];
            delete_node(ptr);
        }

        btreeitr_type itr = btree.begin() + f;
        for(size_type j = 0; j < i; j++)
        {
            itr = btree.erase(itr);
        }

        m_num_elements -= i;
    }

//protected methods.
protected:

    CR_NO_THROW inline pointer get_new_node(const value_type& val)
    {
        //printf("Creating new node to map\n");
        pointer nPtr = (pointer)MALLOC(sizeof(value_type));

        if(nPtr != (pointer)0)
        {
            /* Copy the contents */

            /* Now call constructors */
            //printf("calling constructor first.\n");
            new((void*)&(nPtr->first))key_type(val.first);
            //printf("calling constructor second.\n");
            new(&(nPtr->second))data_type(val.second);
            //printf("came out of constructor.\n");
        }
        else
        {
            printf("ERROR:: Unable to Malloc memory in file %s at line %d\n", __FILE__, __LINE__);
        }
        //printf("returing the pointer %x\n", (int)nPtr);
        return nPtr;
    }

    CR_NO_THROW inline void delete_node(pointer nPtr)
    {
        if(nPtr)
        {
            /* call destructors */
            const key_type* kp = &(nPtr->first);
            kp->~Key();

            data_type* dp = &(nPtr->second);
            dp->~Data();

            FREE(nPtr);

            nPtr = 0;
        }
    }

    CR_NO_THROW inline bool find_index(const key_type& key, size_t& i)
    {
        //printf("%s function called.\n", __FUNCTION__);
        size_t j, k;

        i = k = 0;
        j = btree.size();
        bool found = false;

        if(!btree.empty())
        {
            while(i < j)
            {
                k = i + ((j - i) / 2);

                if(btree[k])
                {
                    if(m_comp(btree[k]->first, key))
                    {
                        i = k + 1;
                    }
                    else
                    {
                        j = k;
                    }
                }
                else
                {
                    break;
                }
            }
        }

        if(i < btree.size() && !m_comp(key, btree[i]->first))
        {
            found = true;
        }

        //printf("%s exiting with found = %d.\n", __FUNCTION__, found);
        return found;
    }

    CR_NO_THROW inline size_type insert_to_vector(const pointer& ptr)
    {
        size_type i;
        //printf("%s function called.\n", __FUNCTION__);

        bool found = find_index(ptr->first, i);

        if(!found)
        {
            btreeitr_type itr = btree.begin();
            itr += i;
            btree.insert(itr, ptr);
            m_num_elements++;
        }
        else
        {
            btree[i]->second = ptr->second;
        }

        return i;
    }

    CR_NO_THROW inline bool erase_from_vector(const key_type& key)
    {
        size_type i;
        bool found = find_index(key, i);

        if(found)
        {
            pointer ptr = btree[i];
            delete_node(ptr);
            btree.erase(btree.begin() + i);

            m_num_elements--;
        }

        return found;
    }
};
} //end of namespace osal_stl
#endif //__CR_HASH_MAP_H_
