/******************************************************************************
 *
 *                   INTEL CORPORATION PROPRIETARY INFORMATION
 *       This software is supplied under the terms of a license agreement or
 *       nondisclosure agreement with Intel Corporation and may not be copied
 *       or disclosed except in accordance with the terms of that agreement.
 *
 *            Copyright (c) 2008-2009 Intel Corporation. All Rights Reserved.
 *
 *       All rights reserved.  No part of this program or publication may be
 *       reproduced, transmitted, transcribed, stored in a retrieval system,
 *       or translated into any language or computer language, in any form or
 *       by any means, electronic, mechanical, magnetic, optical, chemical,
 *       manual, or otherwise, without the prior written permission of Intel
 *       Corporation.
 *
 *
 *******************************************************************************/
#ifndef __CR_DEFAULT_FUNCTORS_H_
#define __CR_DEFAULT_FUNCTORS_H_

#include <os/include/cr_malloc.h>
#include <string.h>

namespace osal_stl {
/* unary functor definition */
template<typename arg, typename res>
struct unary_function
{
    typedef     arg argument_type;
    typedef     res result_type;
};

/* binary functor definition */
template<typename arg1, typename arg2, typename res>
struct binary_function
{
    typedef arg1 first_argument_type;
    typedef arg2 second_argument_type;
    typedef res result_type;
};

/* binary plus functor definition */
template<typename T>
struct cr_plus : public binary_function<T, T, T>
{
    CR_NO_THROW T
    operator()(const T& x, const T& y) const
    {
        return(x + y);
    }
};

/* binary minus functor definition */
template<typename T>
struct cr_minus : public binary_function<T, T, T>
{
    CR_NO_THROW T
    operator()(const T& x, const T& y) const
    {
        return(x - y);
    }
};

/* binary multiply functor definition */
template<typename T>
struct cr_multiplies : public binary_function<T, T, T>
{
    CR_NO_THROW T
    operator()(const T& x, const T& y) const
    {
        return(x * y);
    }
};

/* binary divides functor definition */
template<typename T>
struct cr_divides : public binary_function<T, T, T>
{
    CR_NO_THROW T
    operator()(const T& x, const T& y) const
    {
        return(x / y);
    }
};

/* binary modulus functor definition */
template<typename T>
struct cr_modulus : public binary_function<T, T, T>
{
    CR_NO_THROW T
    operator()(const T& x, const T& y) const
    {
        return(x % y);
    }
};

/* unary negate functor definition */
template<typename T>
struct cr_negate : public unary_function<T, T>
{
    CR_NO_THROW T
    operator()(const T& x) const
    {
        return -x;
    }
};

/* binary comparator == functor */
template<typename E> struct cr_equal_to : public binary_function<E, E, bool>
{
    CR_NO_THROW bool
    operator()(const E& x, const E& y) const
    {
        return(x == y);
    }
};

/* binary comparator != functor */
template<typename E> struct cr_not_equal_to : public binary_function<E, E, bool>
{
    CR_NO_THROW bool
    operator()(const E& x, const E& y) const
    {
        return(x != y);
    }
};

/* binary comparator > functor */
template<typename E> struct cr_greater : public binary_function<E, E, bool>
{
    CR_NO_THROW bool
    operator()(const E& x, const E& y) const
    {
        return(x > y);
    }
};

/* binary comparator < functor */
template<typename E> struct cr_less : public binary_function<E, E, bool>
{
    CR_NO_THROW bool
    operator()(const E& x, const E& y) const
    {
        return(x < y);
    }
};

/* binary comparator >= functor */
template<typename E> struct cr_greater_equal : public binary_function<E, E, bool>
{
    CR_NO_THROW bool
    operator()(const E& x, const E& y) const
    {
        return(x >= y);
    }
};

/* binary comparator <= functor */
template<typename E> struct cr_less_equal : public binary_function<E, E, bool>
{
    CR_NO_THROW bool
    operator()(const E& x, const E& y) const
    {
        return(x <= y);
    }
};

/* binary logical and (&&) functor */
template<typename E> struct cr_logical_and : public binary_function<E, E, bool>
{
    CR_NO_THROW bool
    operator()(const E& x, const E& y) const
    {
        return(x && y);
    }
};

/* binary logical or (|| functor */
template<typename E> struct cr_logical_or : public binary_function<E, E, bool>
{
    CR_NO_THROW bool
    operator()(const E& x, const E& y) const
    {
        return(x || y);
    }
};

/* unary logical not (!) functor */
template<typename E> struct cr_logical_not : public unary_function<E, bool>
{
    CR_NO_THROW bool
    operator()(const E& x) const
    {
        return !x;
    }
};

/* binary bit and (&) functor */
template<typename E> struct cr_bit_and : public binary_function<E, E, bool>
{
    CR_NO_THROW bool
    operator()(const E& x, const E& y) const
    {
        return(x & y);
    }
};

/* binary bit or (|) functor */
template<typename E> struct cr_bit_or : public binary_function<E, E, bool>
{
    CR_NO_THROW bool
    operator()(const E& x, const E& y) const
    {
        return(x | y);
    }
};

/* binary bit xor (|) functor */
template<typename E> struct cr_bit_xor : public binary_function<E, E, bool>
{
    CR_NO_THROW bool
    operator()(const E& x, const E& y) const
    {
        return(x ^ y);
    }
};

/* equal_to specialization for char* */
template<>
struct cr_equal_to<char*>
{
    CR_NO_THROW bool
    operator()(const char* ch1, const char* ch2) const
    {
        return(strcmp(ch1, ch2) == 0);
    }
};

/* equal_to specialization for const char* */
template<>
struct cr_equal_to<const char*>
{
    CR_NO_THROW bool
    operator()(const char* ch1, const char* ch2) const
    {
        return(strcmp(ch1, ch2) == 0);
    }
};

/* equal_to specialization for STRING */
template<>
struct cr_equal_to<STRING>
{
    CR_NO_THROW bool
    operator()(const STRING ch1, const STRING ch2) const
    {
        return(strcmp(ch1.c_str(), ch2.c_str()) == 0);
    }
};

/* equal_to specialization for const STRING */
template<>
struct cr_equal_to<const STRING>
{
    CR_NO_THROW bool
    operator()(const STRING ch1, const STRING ch2) const
    {
        return(strcmp(ch1.c_str(), ch2.c_str()) == 0);
    }
};

/* not_equal_to specialization for char* */
template<>
struct cr_not_equal_to<char*>
{
    CR_NO_THROW bool
    operator()(const char* ch1, const char* ch2) const
    {
        return(strcmp(ch1, ch2) != 0);
    }
};

/* not_equal_to specialization for const char* */
template<>
struct cr_not_equal_to<const char*>
{
    CR_NO_THROW bool
    operator()(const char* ch1, const char* ch2) const
    {
        return(strcmp(ch1, ch2) != 0);
    }
};

/* less specialization for char* */
template<>
struct cr_less<char*>
{
    CR_NO_THROW bool
    operator()(const char* ch1, const char* ch2) const
    {
        return(strcmp(ch1, ch2) < 0);
    }
};

/* less specialization for const char* */
template<>
struct cr_less<const char*>
{
    CR_NO_THROW bool
    operator()(const char* ch1, const char* ch2) const
    {
        return(strcmp(ch1, ch2) < 0);
    }
};

/* less specialization for char* */
template<>
struct cr_less<STRING>
{
    CR_NO_THROW bool
    operator()(const STRING ch1, const STRING ch2) const
    {
        return(strcmp(ch1.c_str(), ch2.c_str()) < 0);
    }
};

/* less specialization for const char* */
template<>
struct cr_less<const STRING>
{
    CR_NO_THROW bool
    operator()(const STRING ch1, const STRING ch2) const
    {
        return(strcmp(ch1.c_str(), ch2.c_str()) < 0);
    }
};

/* greater specialization for char* */
template<>
struct cr_greater<char*>
{
    CR_NO_THROW bool
    operator()(const char* ch1, const char* ch2) const
    {
        return(strcmp(ch1, ch2) > 0);
    }
};

/* greater specialization for const char* */
template<>
struct cr_greater<const char*>
{
    CR_NO_THROW bool
    operator()(const char* ch1, const char* ch2) const
    {
        return(strcmp(ch1, ch2) > 0);
    }
};

/* less_equal specialization for char* */
template<>
struct cr_less_equal<char*>
{
    CR_NO_THROW bool
    operator()(const char* ch1, const char* ch2) const
    {
        return(strcmp(ch1, ch2) <= 0);
    }
};

/* less_equal specialization for const char* */
template<>
struct cr_less_equal<const char*>
{
    CR_NO_THROW bool
    operator()(const char* ch1, const char* ch2) const
    {
        return(strcmp(ch1, ch2) <= 0);
    }
};

/* greater_equal specialization for char* */
template<>
struct cr_greater_equal<char*>
{
    CR_NO_THROW bool
    operator()(const char* ch1, const char* ch2) const
    {
        return(strcmp(ch1, ch2) >= 0);
    }
};

/* greater_equal specialization for const char* */
template<>
struct cr_greater_equal<const char*>
{
    CR_NO_THROW bool
    operator()(const char* ch1, const char* ch2) const
    {
        return(strcmp(ch1, ch2) >= 0);
    }
};

// MurmurHash2A
    #define mmix(h, k) { k *= m; k ^= k >> r; k *= m; h *= m; h ^= k; }

unsigned int inline MurmurHash2A(const void * Key, int Length, unsigned int Seed)
{
    const unsigned int m = 0x5bd1e995;
    const int r = 24;
    unsigned int len = Length;

    const unsigned char * Data = (const unsigned char*)Key;

    unsigned int h = Seed;

    while(Length >= 4)
    {
        unsigned int k = *(unsigned int*)Data;

        mmix(h, k);

        Data += 4;
        Length -= 4;
    }

    unsigned int temp = 0;

    switch(Length)
    {
    case 3: temp ^= Data[2] << 16;

    case 2: temp ^= Data[1] << 8;

    case 1: temp ^= Data[0];
    }
    ;

    mmix(h, temp);
    mmix(h, len);

    h ^= h >> 13;
    h *= m;
    h ^= h >> 15;

    return h;
}

/* hash function object definition */
template<class Key> struct cr_hash { };

CR_NO_THROW inline size_t
_cr_hash_string(const char* ch)
{
    size_t hash = 0;
    int length = strlen(ch);

    hash = MurmurHash2A(ch, length, 0);
    return hash;
}

/* cr_hash specialization for char */
template<>
struct cr_hash<char>
{
    CR_NO_THROW size_t
    operator()(const char ch) const
    {
        return size_t(ch);
    }
};

/* cr_hash specialization for const char */
template<>
struct cr_hash<const char>
{
    CR_NO_THROW size_t
    operator()(const char ch) const
    {
        return size_t(ch);
    }
};

/* cr_hash specialization for int */
template<>
struct cr_hash<int>
{
    CR_NO_THROW size_t
    operator()(const int i) const
    {
        return size_t(i);
    }
};

/* cr_hash specialization for const int */
template<>
struct cr_hash<const int>
{
    CR_NO_THROW size_t
    operator()(const int i) const
    {
        return size_t(i);
    }
};


/* cr_hash specialization for char* */
template<>
struct cr_hash<char*>
{
    CR_NO_THROW size_t
    operator()(const char* ch) const
    {
        return _cr_hash_string(ch);
    }
};

/* cr_hash specialization for const char* */
template<>
struct cr_hash<const char*>
{
    CR_NO_THROW size_t
    operator()(const char* ch) const
    {
        return _cr_hash_string(ch);
    }
};

/* cr_hash specialization for UINT8 */
template<>
struct cr_hash<UINT8>
{
    CR_NO_THROW size_t
    operator()(UINT8 i) const
    {
        return size_t(i);
    }
};

/* cr_hash specialization for UINT16 */
template<>
struct cr_hash<UINT16>
{
    CR_NO_THROW size_t
    operator()(UINT16 i) const
    {
        return size_t(i);
    }
};

/* cr_hash specialization for UINTN */
template<>
struct cr_hash<UINTN>
{
    CR_NO_THROW size_t
    operator()(UINTN i) const
    {
        return size_t(i);
    }
};
} //end of namespace osal_stl
#endif // __CR_DEFAULT_FUNCTORS_H_
