/******************************************************************************
 *
 *                   INTEL CORPORATION PROPRIETARY INFORMATION
 *       This software is supplied under the terms of a license agreement or
 *       nondisclosure agreement with Intel Corporation and may not be copied
 *       or disclosed except in accordance with the terms of that agreement.
 *
 *            Copyright (c) 2008-2009 Intel Corporation. All Rights Reserved.
 *
 *       All rights reserved.  No part of this program or publication may be
 *       reproduced, transmitted, transcribed, stored in a retrieval system,
 *       or translated into any language or computer language, in any form or
 *       by any means, electronic, mechanical, magnetic, optical, chemical,
 *       manual, or otherwise, without the prior written permission of Intel
 *       Corporation.
 *
 *
 *******************************************************************************/
#ifndef __CR_LIST_H
#define __CR_LIST_H

#include <os/include/cr_malloc.h>
#include <string.h>
#include "cr_vector.h"

namespace osal_stl {
template<class T> class cr_list {
//public typedefs and member variables.
public:
    typedef T value_type;
    typedef const value_type const_value_type;
    typedef value_type*         pointer;
    typedef value_type&         reference;
    typedef const value_type&   const_reference;
    typedef size_t size_type;
    typedef size_t difference_type;
    typedef typename            cr_vector<T>::iterator vector_iterator;
    typedef cr_equal_to<value_type>             value_compare;

    const static long long MAX_CR_LIST_SIZE = 1073741823;
    const static unsigned int NPOS_LENGTH = 163613304U;

    typedef class iterator {
public:
        typedef typename cr_list<T>::NodePtr NodePtr;
        typedef typename cr_list<T>::value_type value_type;
        typedef typename cr_list<T>::reference reference;
        typedef typename cr_list<T>::pointer pointer;

/* default constructor */
        CR_NO_THROW inline iterator()
        {
            m_nptr = (NodePtr)NULL;
            m_tail = (NodePtr)NULL;
            m_head = (NodePtr)NULL;
        }

/* copy constructor */
        CR_NO_THROW inline iterator(const iterator& aItr)
        {
            m_nptr = aItr.m_nptr;
            m_tail = aItr.m_tail;
            m_head = aItr.m_head;
        }

/* constructor */
        CR_NO_THROW inline iterator(const NodePtr nptr, const NodePtr tail, const NodePtr head)
        {
            m_nptr = nptr;
            m_tail = tail;
            m_head = head;
        }

        CR_NO_THROW inline void set_nptr(const NodePtr nptr)
        {
            m_nptr = nptr;
        }
        CR_NO_THROW inline void set_tail(const NodePtr tail)
        {
            m_tail = tail;
        }
        CR_NO_THROW inline void set_head(const NodePtr head)
        {
            m_head = head;
        }

        CR_NO_THROW inline NodePtr get_nptr(void) const
        {
            NodePtr nptr;

            if((m_nptr - MAX_CR_LIST_SIZE) == m_tail)
                nptr = m_tail;
            else
                nptr = m_nptr;

            return nptr;
        }

        CR_NO_THROW inline NodePtr get_back(void)
        {
            return m_nptr->back;
        }

        CR_NO_THROW inline NodePtr get_front(void)
        {
            NodePtr front;

            if((m_nptr + sizeof(value_type)) == m_head)
                front = m_head->front;
            else if((m_nptr - MAX_CR_LIST_SIZE) == m_tail)
                front = m_head;
            else
                front = m_nptr->front;

            return front;
        }


        CR_NO_THROW inline iterator& operator=(const iterator& aItr)
        {
            m_nptr = aItr.m_nptr;
            m_tail = aItr.m_tail;
            m_head = aItr.m_head;
            return *this;
        } //end of operator=

        CR_NO_THROW inline iterator& operator++()
        {
            if(m_nptr != m_tail)
                m_nptr = (NodePtr)m_nptr->front;
            else
                m_nptr = m_tail + MAX_CR_LIST_SIZE;

            return *this;
        }

        CR_NO_THROW inline iterator operator++(int)
        {
            iterator tmpItr = *this;

            if(m_nptr != m_tail)
                m_nptr = (NodePtr)m_nptr->front;
            else
                m_nptr = m_tail + MAX_CR_LIST_SIZE;

            return tmpItr;
        }

        CR_NO_THROW inline iterator& operator--()
        {
            if(m_nptr == (m_tail + MAX_CR_LIST_SIZE))
                m_nptr -= MAX_CR_LIST_SIZE;
            else if(m_nptr == m_head)
                m_nptr -= sizeof(value_type);
            else
                m_nptr = (NodePtr)m_nptr->back;

            return *this;
        }

        CR_NO_THROW inline iterator operator--(int)
        {
            iterator tmpItr = *this;

            if(m_nptr == (m_tail + MAX_CR_LIST_SIZE))
                m_nptr -= MAX_CR_LIST_SIZE;
            else if(m_nptr == m_head)
                m_nptr -= sizeof(value_type);
            else
                m_nptr = (NodePtr)m_nptr->back;

            return tmpItr;
        }

        CR_NO_THROW inline reference operator*() const
        {
            return m_nptr->val;
        }

        CR_NO_THROW inline pointer operator->() const
        {
            return (pointer) & m_nptr->val;
        }

        CR_NO_THROW inline bool
        operator!=(const iterator& x) const
        {
            return m_nptr != x.m_nptr;
        }

        CR_NO_THROW inline bool
        operator==(const iterator& x) const
        {
            return m_nptr == x.m_nptr;
        }

protected:
        NodePtr m_nptr;
        NodePtr m_tail;
        NodePtr m_head;
    } iterator;

    typedef const iterator const_iterator;

    typedef class reverse_iterator {
public:
        typedef typename cr_list<T>::NodePtr NodePtr;
        typedef typename cr_list<T>::value_type value_type;
        typedef typename cr_list<T>::reference reference;
        typedef typename cr_list<T>::pointer pointer;
        typedef typename cr_list<T>::iterator iterator;

/* default constructor */
        CR_NO_THROW inline reverse_iterator()
        {
        }

/* constructor */
        CR_NO_THROW inline reverse_iterator(const iterator& aItr)
        {
            itr = aItr;
        }

        CR_NO_THROW inline reverse_iterator& operator=(const reverse_iterator& aItr)
        {
            itr = aItr.itr;

            return *this;
        } //end of operator=

        CR_NO_THROW inline reverse_iterator& operator++()
        {
            --itr;

            return *this;
        }

        CR_NO_THROW inline reverse_iterator operator++(int)
        {
            reverse_iterator tmpItr = *this;

            --itr;

            return tmpItr;
        }

        CR_NO_THROW inline reverse_iterator& operator--()
        {
            ++itr;
            return *this;
        }

        CR_NO_THROW inline reverse_iterator operator--(int)
        {
            reverse_iterator tmpItr = *this;

            ++itr;

            return tmpItr;
        }

        CR_NO_THROW inline reference operator*() const
        {
            iterator tmp = itr;

            return *--tmp;
        }

        CR_NO_THROW inline pointer operator->() const
        {
            return &(operator*());
        }

        CR_NO_THROW inline bool
        operator!=(const reverse_iterator& x) const
        {
            return itr != x.itr;
        }

        CR_NO_THROW inline bool
        operator==(const reverse_iterator& x) const
        {
            return itr == x.itr;
        }

protected:
        iterator itr;
    } reverse_iterator;

    typedef const reverse_iterator const_reverse_iterator;

//public methods.
public:
/* default constructor */
    CR_NO_THROW inline cr_list()
    {
        cr_list_head = (NodePtr)NULL;
        cr_list_tail = (NodePtr)NULL;
        aloc_size = (size_type)0;
    }

/* constructor */
    CR_NO_THROW inline cr_list(size_type n)
    {
        cr_list_head = (NodePtr)NULL;
        cr_list_tail = (NodePtr)NULL;
        aloc_size = (size_type)0;
        resize(n);
    }

    CR_NO_THROW inline cr_list(size_type n, const T& t)
    {
        cr_list_head = (NodePtr)NULL;
        cr_list_tail = (NodePtr)NULL;
        aloc_size = (size_type)0;

        resize(n, t);
    }

/* The copy constructor */
    CR_NO_THROW inline cr_list(const cr_list& t)
    {
        cr_list_head = (NodePtr)NULL;
        cr_list_tail = (NodePtr)NULL;
        aloc_size = t.size();

        /* Copy all the elements of t to this */
        NodePtr tmpPtr = t.cr_list_head;
        for(size_type i = 0; i < aloc_size; i++)
        {
            NodePtr nodePtr = (NodePtr)MALLOC(sizeof(Node));
            if(nodePtr == (NodePtr)0)
            {
                printf("ERROR:: Unable to Malloc memory in file %s at line %d\n", __FILE__, __LINE__);
            }

            //memcpy((void*)nodePtr, (void*)tmpPtr, sizeof(Node));
            new((void*)(&nodePtr->val))value_type(tmpPtr->val);

            if(cr_list_tail == (NodePtr)NULL)
            {
                /* list is empty */
                cr_list_head = nodePtr;
                cr_list_tail = nodePtr;

                nodePtr->back = nodePtr;
                nodePtr->front = nodePtr;
            }
            else
            {
                /* Add to the tail */
                nodePtr->back = cr_list_tail;
                nodePtr->front = cr_list_head;

                /* update the previous tails front ptr */
                cr_list_tail->front = nodePtr;

                /* change the heads back to new node*/
                cr_list_head->back = nodePtr;

                /* Update the tail */
                cr_list_tail = nodePtr;
            } //end of else

            tmpPtr = tmpPtr->front;
        } //end of for
    } //end of cr_list copy constructor

/* list iterator constructor */
    CR_NO_THROW cr_list(iterator begin, iterator end)
    {
        cr_list_head = (NodePtr)NULL;
        cr_list_tail = (NodePtr)NULL;
        aloc_size = (size_type)0;

        for(iterator i = begin; i != end; i++)
        {
            push_back(*i);
        }
    } //end of list iterator constructor.

/* array iterator constructor */
    CR_NO_THROW cr_list(pointer begin, pointer end)
    {
        cr_list_head = (NodePtr)NULL;
        cr_list_tail = (NodePtr)NULL;
        aloc_size = (size_type)0;

        for(pointer i = begin; i != end; i++)
        {
            push_back(*i);
        }
    } //end of list iterator constructor.


/* destructor */
    CR_NO_THROW inline ~cr_list()
    {
        clear();
    }

    CR_NO_THROW inline iterator begin(void)
    {
        iterator itr(cr_list_head, cr_list_tail, cr_list_head);

        return itr;
    }

    CR_NO_THROW inline const_iterator begin(void) const
    {
        return (const_iterator)begin();
    }

    CR_NO_THROW inline iterator end(void)
    {
        if(cr_list_tail != NULL)
            return iterator(cr_list_tail + (MAX_CR_LIST_SIZE), cr_list_tail, cr_list_head);
        else
            return iterator(cr_list_tail, cr_list_tail, cr_list_head);
    }

    CR_NO_THROW inline const_iterator end(void) const
    {
        return (const_iterator)end();
    }

    CR_NO_THROW inline reverse_iterator rbegin(void)
    {
        reverse_iterator rItr(end());

        return rItr;
    }

    CR_NO_THROW inline const_reverse_iterator rbegin(void) const
    {
        reverse_iterator rItr(end());

        return (const_reverse_iterator)rItr;
    }

    CR_NO_THROW inline reverse_iterator rend(void)
    {
        reverse_iterator rItr(begin());

        return rItr;
    }

    CR_NO_THROW inline const_reverse_iterator rend(void) const
    {
        reverse_iterator rItr(begin());

        return (const_reverse_iterator)rItr;
    }

    CR_NO_THROW inline iterator erase(iterator position)
    {
        /* Remove the element the position */
        NodePtr tmpPtr = position.get_nptr();
        NodePtr tmpFrontPtr = tmpPtr->front;
        NodePtr tmpBackPtr = tmpPtr->back;

        tmpBackPtr->front = tmpFrontPtr;
        tmpFrontPtr->back = tmpBackPtr;

        /* Call destructor & free the pointer */
        pointer pN = &tmpPtr->val;
        pN->~T();
        FREE(tmpPtr);

        aloc_size--;

        if(empty())
        {
            cr_list_head = cr_list_tail = (NodePtr)NULL;
            tmpFrontPtr = (NodePtr)NULL;
        }
        else
        {
            if(tmpPtr == cr_list_tail)
            {
                tmpFrontPtr = cr_list_tail + (MAX_CR_LIST_SIZE);
                cr_list_tail = tmpBackPtr;
            }
            else if(tmpPtr == cr_list_head)
            {
                cr_list_head = tmpFrontPtr;
            }
        }

        return iterator(tmpFrontPtr, cr_list_tail, cr_list_head);
    }

    CR_NO_THROW inline iterator erase(iterator first, iterator last)
    {
        iterator i = first;

        do
        {
            i = erase(i);
        } while(i != last);

        return i;
    }

    CR_NO_THROW inline iterator insert(iterator position, const T& x)
    {
        NodePtr tmpPtr = position.get_nptr();
        NodePtr tmpBackPtr = tmpPtr->back;

        /* Add the new element */
        NodePtr nodePtr = (NodePtr)MALLOC(sizeof(Node));

        if(nodePtr == (NodePtr)0)
        {
            printf("ERROR:: Unable to Malloc memory in file %s at line %d\n", __FILE__, __LINE__);
        }

        pointer ptr = &nodePtr->val;

        nodePtr->front = tmpPtr;
        nodePtr->back = tmpBackPtr;

        tmpBackPtr->front = nodePtr;
        tmpPtr->back = nodePtr;

        /* Now copy the contents of value_type into node */
        //memcpy((void*)ptr, (void *)&x, sizeof(value_type));
        new((void*)(&ptr))value_type(x);

        if(cr_list_head == tmpPtr)
        {
            cr_list_head = nodePtr;
        }

        if(cr_list_tail == tmpPtr)
        {
            cr_list_tail = nodePtr;
        }

        aloc_size++;

        return iterator(nodePtr, cr_list_tail, cr_list_head);
    }

    CR_NO_THROW inline void insert(iterator position, size_type n, const T& x)
    {
        for(size_type i = 0; i < n; i++)
        {
            insert(position, x);
        }
    }

/* list iterator insert */
    CR_NO_THROW inline void insert(iterator position, const iterator first, const iterator last)
    {
        for(vector_iterator i = first; i != last; i++)
            insert(position, *i);
    }

/* vector iterator insert */
    CR_NO_THROW inline void insert(iterator position, vector_iterator first, vector_iterator last)
    {
        for(vector_iterator i = first; i != last; i++)
            insert(position, *i);
    }

    CR_NO_THROW inline size_type size(void) const
    {
        return aloc_size;
    }

    CR_NO_THROW inline size_type max_size(void) const
    {
        return (size_type)cr_list::MAX_CR_LIST_SIZE;
    }

    CR_NO_THROW inline bool empty(void) const
    {
        return(aloc_size == (size_type)0);
    }

    CR_NO_THROW inline void resize(size_type n, T t = T())
    {
        size_type len = size();

        if((n <= cr_list::MAX_CR_LIST_SIZE) && (len != n))
        {
            if(len < n)
            {
                /* The current list is less than resize list size */
                /* This means we have to add (n-len) new elements */
                for(size_type i = 0; i < (n - len); i++)
                {
                    NodePtr nodePtr = (NodePtr)MALLOC(sizeof(Node));
                    if(nodePtr == (NodePtr)0)
                    {
                        printf("ERROR:: Unable to Malloc memory in file %s at line %d\n", __FILE__, __LINE__);
                    }

                    pointer ptr = &nodePtr->val;

                    /* Now copy the contents of value_type into node */
                    //memcpy((void*)ptr, (void *)&t, sizeof(value_type));
                    new((void*)(ptr))value_type(t);

                    /* Add this node to the tail */
                    /* Here dont use size or empty functions
                     * to check for emptiness, since the aloc_size
                     * is not updated, it could lead to mem leak.*/
                    if(cr_list_tail == (NodePtr)NULL)
                    {
                        /* list is empty */
                        cr_list_head = nodePtr;
                        cr_list_tail = nodePtr;

                        nodePtr->back = nodePtr;
                        nodePtr->front = nodePtr;
                    }
                    else
                    {
                        /* Add to the tail */
                        nodePtr->back = cr_list_tail;
                        nodePtr->front = cr_list_head;

                        /* update the previous tails front ptr */
                        cr_list_tail->front = nodePtr;

                        /* change the previous back to new node*/
                        cr_list_head->back = nodePtr;

                        /* Update the tail */
                        cr_list_tail = nodePtr;
                    }
                } //end of for

                aloc_size += (n - len);
            }
            else
            {
                /* The current list is greater than resize list size */
                /* This means we have to remove elements from tail */
                for(size_type i = 0; i < (len - n); i++)
                {
                    /* Remove elements */
                    NodePtr tmpPtr = cr_list_tail;

                    NodePtr tailsBack = cr_list_tail->back;

                    tailsBack->front = cr_list_head;

                    cr_list_tail = tailsBack;

                    cr_list_head->back = cr_list_tail;

                    /* call destrutor and free pointer */
                    pointer pN = &tmpPtr->val;
                    pN->~T();
                    FREE(tmpPtr);
                }

                aloc_size -= (len - n);

                if(empty())
                {
                    cr_list_head = cr_list_tail = (NodePtr)NULL;
                } //end of if
            } //end of else
        } //end of if
    } //end of resize()

    CR_NO_THROW inline cr_list& operator=(const cr_list& aList)
    {
        //assign only if not self to avoid redundant operation
        if(this != &aList)
        {
            size_type s = aList.size();
            resize(s);

            /* copy all the contents of aList to this */
            NodePtr tmpPtr_dst = cr_list_head;
            NodePtr tmpPtr_src = aList.cr_list_head;
            for(size_type i = 0; i < s; i++)
            {
                pointer tmpDst_valPtr = &tmpPtr_dst->val;
                pointer tmpSrc_valPtr = &tmpPtr_src->val;
                //memcpy((void*)tmpDst_valPtr, (void*)tmpSrc_valPtr, sizeof(value_type));
                new((void*)(tmpDst_valPtr))value_type(tmpPtr_src->val);

                tmpPtr_dst = tmpPtr_dst->front;
                tmpPtr_src = tmpPtr_src->front;
            }
        }
        return *this;
    }

    CR_NO_THROW inline void assign(size_type n, const T& u)
    {
        /* Initially clear all elements in list */
        clear();

        /* Now add n, u elements in the list*/
        resize(n, u);
    }

    CR_NO_THROW inline void assign(const iterator begin, const iterator end)
    {
        /* Initially clear all elements in list */
        clear();

        for(iterator i = begin; i != end; i++)
        {
            push_back(*i);
        }
    }

    CR_NO_THROW inline void assign(const vector_iterator begin, const vector_iterator end)
    {
        /* Initially clear all elements in list */
        clear();

        for(vector_iterator i = begin; i != end; i++)
        {
            push_back(*i);
        }
    }

    CR_NO_THROW inline reference front(void)
    {
        nextPtr = cr_list_head;
        return (reference)cr_list_head->val;
    }

    CR_NO_THROW inline const_reference front(void) const
    {
        nextPtr = cr_list_head;
        return (const_reference)cr_list_head->val;
    }

    CR_NO_THROW inline reference back(void)
    {
        return (reference)cr_list_tail->val;
    }

    CR_NO_THROW inline const_reference back(void) const
    {
        return (const_reference)cr_list_tail->val;
    }

    CR_NO_THROW inline void push_front(const T& t)
    {
        NodePtr nodePtr = (NodePtr)MALLOC(sizeof(Node));

        if(nodePtr == (NodePtr)0)
        {
            printf("ERROR:: Unable to Malloc memory in file %s at line %d\n", __FILE__, __LINE__);
        }

        pointer ptr_dst = &nodePtr->val;

        /* Now copy the contents of value_type into node */
        //memcpy((void*)ptr_dst, (void *)&t, sizeof(value_type));
        new((void*)(ptr_dst))value_type(t);

        /* Add this node to the head */
        /* Here dont use size or empty functions
         * to check for emptiness, since the aloc_size
         * is not updated, it could lead to mem leak.*/
        if(cr_list_head == (NodePtr)NULL)
        {
            /* list is empty */
            cr_list_head = nodePtr;
            cr_list_tail = nodePtr;

            nodePtr->back = nodePtr;
            nodePtr->front = nodePtr;
        }
        else
        {
            /* Add to the head */
            nodePtr->front = cr_list_head;
            nodePtr->back = cr_list_tail;

            /* update the previous tails front ptr */
            cr_list_head->back = nodePtr;

            /* update the tails front */
            cr_list_tail->front = nodePtr;

            /* Update the head */
            cr_list_head = nodePtr;
        }

        aloc_size++;
        nextPtr = cr_list_head;
    }

    CR_NO_THROW inline void push_back(const T& t)
    {
        NodePtr nodePtr = (NodePtr)MALLOC(sizeof(Node));

        if(nodePtr == (NodePtr)0)
        {
            printf("ERROR:: Unable to Malloc memory in file %s at line %d\n", __FILE__, __LINE__);
            //KW issue fix: NPD.CHECK.MUST
            exit(0xFF);
        }

        pointer ptr = &nodePtr->val;

        /* Now copy the contents of value_type into node */
        //memcpy((void*)ptr, (void *)&t, sizeof(value_type));
        new((void*)(ptr))value_type(t);

        if(cr_list_tail == (NodePtr)NULL)
        {
            /* list is empty */
            cr_list_head = cr_list_tail = nodePtr->back = nodePtr->front = nodePtr;
        }
        else
        {
            /* Add to the tail */
            nodePtr->back = cr_list_tail;
            nodePtr->front = cr_list_head;

            /* update the previous tails front ptr */
            cr_list_tail->front = nodePtr;

            /* change the previous back to new node */
            cr_list_head->back = nodePtr;

            /* Update the tail */
            cr_list_tail = nodePtr;
        }

        aloc_size++;
    }

    CR_NO_THROW inline void pop_front(void)
    {
        /* Remove one element from the front */
        NodePtr tmpPtr = cr_list_head;

        NodePtr headsFront = cr_list_head->front;

        headsFront->back = cr_list_tail;

        cr_list_head = headsFront;

        aloc_size--;

        pointer pN = &tmpPtr->val;
        pN->~T();
        FREE(tmpPtr);

        nextPtr = cr_list_head;
    }

    CR_NO_THROW inline void pop_back(void)
    {
        size_type s = size();

        s--;
        resize(s);
    }

    CR_NO_THROW inline void clear(void)
    {
        size_type len = size();

        /* Remove elements from tail */
        for(size_type i = 0; i < len; i++)
        {
            /* Remove elements */
            NodePtr tmpPtr = cr_list_tail;

            NodePtr tailsBack = cr_list_tail->back;

            tailsBack->front = cr_list_head;

            cr_list_tail = tailsBack;

            cr_list_head->back = cr_list_tail;

            pointer pN = &tmpPtr->val;
            pN->~T();
            FREE(tmpPtr);
        }

        aloc_size = 0;

        cr_list_head = cr_list_tail = (NodePtr)NULL;
    } //end of clear()

    CR_NO_THROW inline void swap(cr_list& aList)
    {
        NodePtr tmp_head = aList.cr_list_head;
        NodePtr tmp_tail = aList.cr_list_tail;
        size_type tmp_size = aList.aloc_size;

        aList.cr_list_head = cr_list_head;
        aList.cr_list_tail = cr_list_tail;
        aList.aloc_size = aloc_size;

        cr_list_head = tmp_head;
        cr_list_tail = tmp_tail;
        aloc_size = tmp_size;
    }

    CR_NO_THROW inline void remove(const T& val)
    {
        /* Remove all the elements of this that match val*/
        NodePtr tmpPtr = cr_list_head;

        for(size_type i = 0; i < size(); i++)
        {
            NodePtr tmpPtrBack = tmpPtr->back;
            NodePtr tmpPtrFront = tmpPtr->front;

            value_type* tValPtr = &(tmpPtr->val);

            if(m_comp(tmpPtr->val, val))
            {
                /* Remove This element */
                aloc_size--;
                i--;

                if(aloc_size > 0)
                {
                    tmpPtrBack->front = tmpPtrFront;
                    tmpPtrFront->back = tmpPtrBack;

                    if(cr_list_tail == tmpPtr)
                    {
                        cr_list_tail = tmpPtrBack;
                    }

                    if(cr_list_head == tmpPtr)
                    {
                        cr_list_head = tmpPtrFront;
                    }
                }
                else
                {
                    cr_list_head = cr_list_tail = NULL;
                } //end of if

                pointer pN = &tmpPtr->val;
                pN->~T();
                FREE(tmpPtr);
            } //end of if

            tmpPtr = tmpPtrFront;
        } //end of for
    } //end of remove

    CR_NO_THROW inline void reverse(void)
    {
        size_type s = size();
        NodePtr tmpHead;

        tmpHead = cr_list_tail;

        cr_list_tail = cr_list_head;
        cr_list_head = tmpHead;

        for(size_type i = 0; i < s; i++)
        {
            NodePtr tmp = tmpHead->front;
            tmpHead->front = tmpHead->back;
            tmpHead->back = tmp;

            tmpHead = tmpHead->front;
        }
    } //end of reverse

    CR_NO_THROW inline const_reference next(void)
    {
        const_reference t = (const_reference)nextPtr->val;

        nextPtr = nextPtr->front;

        return t;
    }

    template<class Predicate>
    CR_NO_THROW void remove_if(Predicate p)
    {
        iterator i = begin();

        do
        {
            if(p(*i))
            {
                i = erase(i);
            }
            else
                ++i;
        } while(i != end());
    }

protected:

    typedef class Node {
public:
        Node*       back;
        value_type val;
        Node*       front;
    } Node;

    typedef Node*   NodePtr;

    NodePtr cr_list_head;         //Head
    NodePtr cr_list_tail;         //tail
    size_type aloc_size;
    NodePtr nextPtr;
    value_compare m_comp;
};

template<class T>
CR_NO_THROW inline bool operator==(cr_list<T>& l1, cr_list<T>& l2)
{
    bool result = true;
    value_compare m_comp;
    size_t s1 = l1.size();
    size_t s2 = l2.size();

    if(s1 == s2)
    {
        T tp1 = l1.front();
        T tp2 = l2.front();
        for(size_t s = 0; s < s1; s++)
        {
            if(m_comp(tp1, tp2))
            {
                result = false;
                break;
            }

            tp1 = l1.next();
            tp2 = l2.next();
        }
    }
    else
    {
        result = false;
    }

    return result;
} //end of operator==
} //end of namespace osal_stl
#endif // __CR_LIST_H
