/******************************************************************************
 *
 *                   INTEL CORPORATION PROPRIETARY INFORMATION
 *       This software is supplied under the terms of a license agreement or
 *       nondisclosure agreement with Intel Corporation and may not be copied
 *       or disclosed except in accordance with the terms of that agreement.
 *
 *            Copyright (c) 2008-2009 Intel Corporation. All Rights Reserved.
 *
 *       All rights reserved.  No part of this program or publication may be
 *       reproduced, transmitted, transcribed, stored in a retrieval system,
 *       or translated into any language or computer language, in any form or
 *       by any means, electronic, mechanical, magnetic, optical, chemical,
 *       manual, or otherwise, without the prior written permission of Intel
 *       Corporation.
 *
 *
 *******************************************************************************/
#ifndef __CR_VECTOR_H_
#define __CR_VECTOR_H_

#include <os/include/cr_malloc.h>
#include <string.h>
#include <new>
#include "cr_allocator.h"

namespace osal_stl {
template<class T> class cr_vector {
//public typedefs and member variables.
public:
    typedef T value_type;
    typedef const value_type const_value_type;
    typedef value_type*         pointer;
    typedef value_type&         reference;
    typedef const value_type&   const_reference;
    typedef size_t size_type;
    typedef size_t difference_type;
    typedef value_type*         iterator;
    typedef const value_type*   const_iterator;

    const static int MAX_CR_VECTOR_SIZE = 1073741823;
    const static unsigned int NPOS_LENGTH = 4294967295U;


//public methods.
public:
/* default constructor */
    cr_vector()
    {
        cr_vector_ptr = (pointer)NULL;
        aloc_capacity = (size_type)0;
        aloc_size = (size_type)0;

#ifdef CR_USE_ALLOCATOR
        cr_alloc_ptr = (cr_allocator<T>*) 0;
#endif
    }

/* constructor */
    cr_vector(size_type n)
    {
        cr_vector_ptr = (pointer)NULL;
        aloc_capacity = (size_type)0;
        aloc_size = (size_type)0;

#ifdef CR_USE_ALLOCATOR
        cr_alloc_ptr = (cr_allocator<T>*) 0;
#endif
        resize(n);
    }

/* constructor */
    cr_vector(size_type n, const T& t)
    {
        cr_vector_ptr = (pointer)NULL;
        aloc_capacity = (size_type)0;
        aloc_size = (size_type)0;

#ifdef CR_USE_ALLOCATOR
        cr_alloc_ptr = (cr_allocator<T>*) 0;
#endif
        resize(n, t);
    }

/* copy constructor */
    cr_vector(const cr_vector& aVec)
    {
        //printf("%s: function called.\n", __FUNCTION__);
        cr_vector_ptr = (pointer)NULL;

#ifdef CR_USE_ALLOCATOR
        cr_alloc_ptr = aVec.cr_alloc_ptr;

        if(cr_alloc_ptr)
        {
            cr_alloc_ptr->incr_refCount();
            cr_vector_ptr = cr_alloc_ptr->get_pointer();
        }

        aloc_capacity = aVec.capacity();
#else
        aloc_capacity = (size_type)0;
        aloc_size = (size_type)0;

        size_type s = aVec.size();

        reserve(s);

        for(size_type j = 0; j < aVec.size(); j++)
        {
            new((void*)(cr_vector_ptr + j))value_type(aVec[j]);
        }
#endif

        aloc_size = aVec.size();
    }

    cr_vector(iterator f, iterator l)
    {
        cr_vector_ptr = (pointer)NULL;

#ifdef CR_USE_ALLOCATOR
        cr_alloc_ptr = (cr_allocator<T>*) 0;
#endif

        aloc_capacity = (size_type)0;
        aloc_size = (size_type)0;

        size_type s = (l - f);
        reserve(s);

        for(iterator i = f; i != l; i++)
        {
            push_back(*i);
        }
    }

    inline size_type size(void) const
    {
        return aloc_size;
    }

    inline size_type max_size(void) const
    {
        return (size_type)cr_vector::MAX_CR_VECTOR_SIZE;
    }

    inline size_type capacity(void) const
    {
        return aloc_capacity;
    }

    inline bool empty(void) const
    {
        return((const size_type)size() == 0);
    }

    inline reference operator[](size_type n)
    {
#ifdef CR_USE_ALLOCATOR
        check_modify_allocator();
#endif

        if(n >= size())
        {
            resize(n + 1);
        }

        iterator cv = begin();

        return (reference) * (cv + n);
    }

    inline const_reference operator[](size_type n) const
    {
        const_iterator cv = begin();

        if(aloc_capacity > 0)
        {
            if(n < size())
            {
                return (const_reference) * (cv + n);
            }
        }

        return (const_reference) * cv;
    }

    inline iterator begin(void)
    {
        if(cr_vector_ptr)
            return &cr_vector_ptr[0];
        else
            return (iterator)0;
    }
    inline iterator end(void)
    {
        if(cr_vector_ptr)
            return &cr_vector_ptr[size()];
        else
            return (iterator)0;
    }
//protected:
    inline void set_capacity(size_type c)
    {
        aloc_capacity = c;
    }

    inline void set_size(size_type s)
    {
        aloc_size = s;
    }

    inline void set_pointer(pointer ptr)
    {
        cr_vector_ptr = ptr;
    }

    inline pointer get_pointer(void)
    {
        return cr_vector_ptr;
    }
/* destructor*/
    ~cr_vector()
    {
        clear();
    }


    inline void resize(size_type n, T t = T())
    {
#ifdef CR_USE_ALLOCATOR
        check_modify_allocator();
#endif

        size_type len = size();

        if((n <= cr_vector::MAX_CR_VECTOR_SIZE) && (len != n))
        {
            if(len < n)
            {
                size_type aloc = n * sizeof(T);

                //Increasing the memory, only if current capacity is less.
                if(aloc_capacity < n)
                {
                    cr_vector_ptr = (pointer)REALLOC(cr_vector_ptr, aloc);

                    if(cr_vector_ptr == (pointer)0)
                    {
                        printf("ERROR:: Unable to Malloc memory in file %s at line %d\n", __FILE__, __LINE__);
                    }

                    aloc_capacity = n;
                }

                for(size_type i = len; i < n; i++)
                {
                    new((void*)(cr_vector_ptr + i))value_type(t);
                }

                aloc_size = n;
            }
            else
            {
                //Dropping extra elements.
                for(size_type i = n; i < len; i++)
                    (cr_vector_ptr + i)->~T();

                aloc_size = n;
            }

#ifdef CR_USE_ALLOCATOR
            if(cr_alloc_ptr->get_refCount() == 1)
            {
                /* set the vector pointer */
                cr_alloc_ptr->set_pointer(cr_vector_ptr);
            }
#endif
        }
    }
    void reserve(size_type n)
    {
#ifdef CR_USE_ALLOCATOR
        check_modify_allocator();
#endif
        if(n > capacity())
        {
            cr_vector_ptr = (pointer)REALLOC(cr_vector_ptr, n * sizeof(T));
            if(cr_vector_ptr == (pointer)0)
            {
                printf("ERROR:: Unable to Malloc memory in file %s at line %d\n", __FILE__, __LINE__);
            }

            aloc_capacity = n;
        }
#ifdef CR_USE_ALLOCATOR
        if(cr_alloc_ptr->get_refCount() == 1)
        {
            /* set the vector pointer */
            cr_alloc_ptr->set_pointer(cr_vector_ptr);
        }
#endif
    }

    inline void clear(void)
    {
#ifdef CR_USE_ALLOCATOR
        if(cr_alloc_ptr && cr_alloc_ptr->get_refCount() >= 1)
        {
            cr_alloc_ptr->dec_refCount();
        }
#endif

        if(cr_vector_ptr != (pointer)0)
        {
#ifdef CR_USE_ALLOCATOR
            if(cr_alloc_ptr && cr_alloc_ptr->get_refCount() == 0)
            {
#endif
            /* Call destructor & free the pointer */
            for(size_t i = 0; i < aloc_size; i++)
                (cr_vector_ptr + i)->~T();

            FREE(cr_vector_ptr);


#ifdef CR_USE_ALLOCATOR
        }    //end of if
#endif
        }

#ifdef CR_USE_ALLOCATOR
        if(cr_alloc_ptr && cr_alloc_ptr->get_refCount() == 0)
        {
            cr_alloc_ptr->set_pointer(NULL);
            cr_allocator<T>::delete_node(cr_alloc_ptr);
            cr_alloc_ptr = NULL;
        }
#endif

        cr_vector_ptr = (pointer)NULL;
        aloc_capacity = (size_type)0;
        aloc_size = (size_type)0;
    }

    void push_back(const T& t)
    {
#ifdef CR_USE_ALLOCATOR
        check_modify_allocator();
#endif
        size_type s = this->size();
        size_type capcty = this->capacity();
        size_type aloc = capcty;

        while(aloc < (s + 1))
        {
            //printf("aloc = %d s = %d\n", aloc, s);
            aloc = (aloc != 0) ? aloc * 2 : 1;
            //printf("new aloc = %d\n", aloc);
        }


        if(aloc != capcty)
        {
            cr_vector_ptr = (pointer)REALLOC(cr_vector_ptr, (aloc) * sizeof(T));
            if(cr_vector_ptr == (pointer)0)
            {
                printf("ERROR:: Unable to Malloc memory in file %s at line %d\n", __FILE__, __LINE__);
            }

            aloc_capacity = aloc;
        }

        new((void*)(cr_vector_ptr + s))value_type(t);

        aloc_size++;

#ifdef CR_USE_ALLOCATOR
        if(cr_alloc_ptr->get_refCount() == 1)
        {
            /* set the vector pointer */
            cr_alloc_ptr->set_pointer(cr_vector_ptr);
        }
#endif
    }

    void pop_back(void)
    {
#ifdef CR_USE_ALLOCATOR
        check_modify_allocator();
#endif

        size_type s = size();
        if(s > 0)
        {
            pointer pElem = (cr_vector_ptr + s - 1);
            pElem->~T();
            aloc_size--;
        }
    }

    void swap(cr_vector& aVec)
    {
#ifdef CR_USE_ALLOCATOR
        check_modify_allocator();
#endif

        size_type s1 = size();
        size_type s2 = aVec.size();
        size_type c1 = capacity();
        size_type c2 = aVec.capacity();

        pointer ptr = cr_vector_ptr;
        cr_vector_ptr = aVec.get_pointer();
        aloc_size = s2;
        aloc_capacity = c2;

        aVec.set_pointer(ptr);
        aVec.set_size(s1);
        aVec.set_capacity(c1);

#ifdef CR_USE_ALLOCATOR
        cr_allocator<T>* aloc_ptr = cr_alloc_ptr;
        cr_alloc_ptr = aVec.cr_alloc_ptr;
        aVec.cr_alloc_ptr = aloc_ptr;
#endif
    }

    inline iterator insert(iterator pos, const T& x)
    {
        iterator r = (iterator)npos;

        size_t diff = pos - begin();

        if((pos >= begin()) && (pos <= end()))
        {
            insert(pos, 1, x);

            r = begin() + diff;
        }

        return r;
    }

    inline void insert(iterator pos, size_type n, const T& x)
    {
#ifdef CR_USE_ALLOCATOR
        size_t diff = pos - begin();
        check_modify_allocator();

        //If check modify reallocated new memory
        //Then iterators get invalidated, so set
        //new iterators.
        pos = begin() + diff;
#endif

        iterator b = begin();
        iterator e = end();

        if(pos >= b && pos <= e)
        {
            size_type c = capacity();
            size_type s = size();
            size_type elSize = 1; //sizeof(T);
            size_type diff = (size_type)(pos - cr_vector_ptr);
            size_type s_pos = diff / elSize;

            if(c < (s + n))
            {
                size_type aloc = c;

                do
                {
                    aloc = (aloc != 0) ? aloc * 2 : 1;
                } while(aloc < (s + n));

                cr_vector_ptr = (pointer)REALLOC(cr_vector_ptr, (aloc) * sizeof(T));
                if(cr_vector_ptr == (pointer)0)
                {
                    printf("ERROR:: Unable to Malloc memory in file %s at line %d\n", __FILE__, __LINE__);
                }

                aloc_capacity = aloc;
            }

            size_type rem = (s - s_pos);
            if(rem != 0)
            {
                char* src = (char*)(cr_vector_ptr + s_pos);
                char* dst = (char*)src + n * sizeof(T);
                size_type bytes = rem * sizeof(T);
                memmove(dst, src, bytes);
            }

            pointer ptr = cr_vector_ptr + s_pos;
            for(size_type i = 0; i < n; i++)
            {
                new((void*)(ptr))value_type(x);
                ptr++;
            }

            aloc_size = s + n;
        }

#ifdef CR_USE_ALLOCATOR
        if(cr_alloc_ptr->get_refCount() == 1)
        {
            /* set the vector pointer */
            cr_alloc_ptr->set_pointer(cr_vector_ptr);
        }
#endif
    }

    inline void insert(iterator pos, iterator f, iterator l)
    {
#ifdef CR_USE_ALLOCATOR
        size_t diff = pos - begin();
        check_modify_allocator();

        //If check modify reallocated new memory
        //Then iterators get invalidated, so set
        //new iterators.
        pos = begin() + diff;
#endif

        iterator b = begin();
        iterator e = end();

        if(pos >= b && pos <= e)
        {
            size_type c = capacity();
            size_type s = size();
            size_type n = (l - f);
            size_type s_pos = (size_type)(pos - cr_vector_ptr) / sizeof(T);

            if(c < (s + n))
            {
                size_type aloc = c;

                do
                {
                    aloc = (aloc != 0) ? aloc * 2 : 1;
                } while(aloc < (s + n));

                cr_vector_ptr = (pointer)REALLOC(cr_vector_ptr, (aloc) * sizeof(T));
                if(cr_vector_ptr == (pointer)0)
                {
                    printf("ERROR:: Unable to Malloc memory in file %s at line %d\n", __FILE__, __LINE__);
                }

                aloc_capacity = aloc;
            }

            pos = begin() + diff;

            for(iterator i = f; i != l; i++)
            {
                insert(pos++, 1, *i);
            }
        }

#ifdef CR_USE_ALLOCATOR
        if(cr_alloc_ptr->get_refCount() == 1)
        {
            /* set the vector pointer */
            cr_alloc_ptr->set_pointer(cr_vector_ptr);
        }
#endif
    }

    inline iterator erase(iterator pos)
    {
        return erase(pos, pos);
    }

    inline iterator erase(iterator first, iterator last)
    {
#ifdef CR_USE_ALLOCATOR
        size_t diff1 = first - begin();
        size_t diff2 = last - begin();

        check_modify_allocator();

        //If check modify reallocated new memory
        //Then iterators get invalidated, so set
        //new iterators.
        first = begin() + diff1;
        last = begin() + diff2;
#endif
        iterator b = begin();
        iterator e = end();

        if((first <= last) && (first >= b) && (first <= e) && (last >= b) && (last <= e))
        {
            size_type s = size();
            size_type elSize = 1; //sizeof(T)
            size_type s_pos = (size_type)(first - cr_vector_ptr) / elSize;
            size_type e_pos = (size_type)(last - cr_vector_ptr) / elSize;
            size_type n = (s_pos == e_pos) ? 1 : (e_pos - s_pos);
            size_type rem = s - e_pos;

            if(rem)
            {
                char * dst = (char*)(cr_vector_ptr + s_pos);

                /* Call destructor & free the pointer */
                for(size_t i = 0; i < n; i++)
                    (cr_vector_ptr + s_pos + i)->~T();

                size_type bytes = (rem - 1) * sizeof(T);
                char * src = dst + n * sizeof(T);
                memmove(dst, src, bytes);
            }

            aloc_size -= n;
        }

        return first;
    }
    inline cr_vector& operator=(const cr_vector& aVec)
    {
        //printf("%s: function called.\n", __FUNCTION__);

        //First clear
        clear();
        size_type s = aVec.size();

#ifdef CR_USE_ALLOCATOR
        cr_alloc_ptr = aVec.cr_alloc_ptr;

        if(cr_alloc_ptr)
        {
            cr_alloc_ptr->incr_refCount();
            cr_vector_ptr = cr_alloc_ptr->get_pointer();
        }

        aloc_capacity = aVec.capacity();
#else
        reserve(s);

        for(size_type j = 0; j < s; j++)
        {
            new((void*)(cr_vector_ptr + j))value_type(aVec[j]);
        }
#endif
        aloc_size = s;

        return *this;
    }

protected:
    static const size_type npos = (size_type)NPOS_LENGTH;

//private methods.
private:
#ifdef CR_USE_ALLOCATOR
    void check_modify_allocator(void)
    {
        /* check if allocator is already malloc'd */
        if(cr_alloc_ptr != NULL)
        {
            /* allocator already malloc'd */
            if(cr_alloc_ptr->get_refCount() < 0)
            {
                printf("Error: refCount is < 0\n");
                exit(0);
            }

            /* check if more than one reference is present */
            if(cr_alloc_ptr->get_refCount() > 1)
            {
                /* create a new alloc_ptr, copy the contents and
                 * decrement the old reference and increment the new
                 * reference count */
                cr_allocator<T>* tmp = cr_allocator<T>::create_node();

                /* malloc the size of vector and copy */
                pointer tmpPtr = (pointer)MALLOC(aloc_capacity * sizeof(T));
                if(tmpPtr == NULL)
                {
                    printf("Error: unable to malloc.\n");
                    exit(0);
                }

                pointer oldPtr = cr_alloc_ptr->get_pointer();

                /* copy old memory to new */
                memcpy((void*)tmpPtr, (void*)oldPtr, (aloc_size * sizeof(T)));

                cr_alloc_ptr->dec_refCount();
                tmp->incr_refCount();

                /* set the copied memory as the new pointer */
                tmp->set_pointer(tmpPtr);

                /* use the tmp as the new aloc_ptr */
                cr_alloc_ptr = tmp;

                /* set the vector pointer to the new ptr */
                cr_vector_ptr = cr_alloc_ptr->get_pointer();
            }
        }
        else
        {
            /* allocator is not malloc'd */
            cr_alloc_ptr = cr_allocator<T>::create_node();

            /* This will set refCount to 1 */
            cr_alloc_ptr->incr_refCount();
        }
    } //end of check_modify_allocator()
#endif

//private member variables.
private:
    pointer cr_vector_ptr;
    size_type aloc_capacity;
    size_type aloc_size;

#ifdef CR_USE_ALLOCATOR
protected:
    cr_allocator<T>* cr_alloc_ptr;
#endif
};

template <class T>
bool operator==(const cr_vector<T>& a, const cr_vector<T>& b)
{
    typedef cr_vector<T> _Self;
    typedef typename _Self::size_type size_type;
    typedef typename _Self::const_iterator const_iterator;

    size_type s1 = a.size();
    size_type s2 = b.size();

    bool result = false;
    int r = 1;

    if(s1 == s2)
    {
        const_iterator j = b.begin();
        result = true;
        for(const_iterator i = a.begin(); i != a.end(); i++, j++)
        {
            r = memcmp(i, j, sizeof(T));

            if(r != 0)
            {
                result = false;
                break;
            }
        }
    }

    return result;
}

template <class T>
bool operator!=(const cr_vector<T>& a, const cr_vector<T>& b)
{
    return !(a == b);
}

template <class T>
bool operator<(const cr_vector<T>& a, const cr_vector<T>& b)
{
    typedef cr_vector<T> _Self;
    typedef typename _Self::size_type size_type;
    typedef typename _Self::const_iterator const_iterator;

    size_type s1 = a.size();
    size_type s2 = b.size();

    bool result = false;
    int r = -1;

    const_iterator i;
    if(s1 < s2)
    {
        const_iterator j = b.begin();
        for(i = a.begin(); i != a.end(); i++, j++)
        {
            if(*i >= *j)
            {
                break;
            }
        }

        if(i == a.end())
        {
            result = true;
        }
    }
    else
    {
        const_iterator j = a.begin();
        for(i = b.begin(); i != b.end(); i++, j++)
        {
            size_type s = sizeof(T);

            if(*j >= *i)
            {
                result = false;
                break;
            }
        }

        if(s1 != 0)
        {
            if(i == b.end())
            {
                result = true;
            }
        }
    }

    return result;
}

template <class T>
bool operator>(const cr_vector<T>& a, const cr_vector<T>& b)
{
    typedef cr_vector<T> _Self;
    typedef typename _Self::size_type size_type;
    typedef typename _Self::const_iterator const_iterator;

    size_type s1 = a.size();
    size_type s2 = b.size();

    bool result = false;
    int r = 1;

    const_iterator i;
    if(s1 < s2)
    {
        const_iterator j = b.begin();
        for(i = a.begin(); i != a.end(); i++, j++)
        {
            if(*i <= *j)
            {
                break;
            }
        }

        if(i == a.end())
        {
            result = true;
        }
    }
    else
    {
        const_iterator j = a.begin();
        for(i = b.begin(); i != b.end(); i++, j++)
        {
            if(*j <= *i)
            {
                result = false;
                break;
            }
        }

        if(s1 != 0)
        {
            if(i == b.end())
            {
                result = true;
            }
        }
    }

    return result;
}
} //end of namespace osal_stl
#endif //__CR_VECTOR_H_
