#!/bin/bash

###############################################################################
#
# Copyright (c) 2016-2017, Intel Corporation
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#  1. Redistributions of source code must retain the above copyright notice,
#     this list of conditions and the following disclaimer.
#
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in the
#     documentation and/or other materials provided with the distribution.
#
#  3. Neither the name of the copyright holder nor the names of its
#     contributors may be used to endorse or promote products derived from
#     this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#
###############################################################################

#
# Description:
# Integrate 6LoWPAN into APL-I Yocto BSP build.
#

# Terminal color definitions
color_text_normal="\e[0m"
color_text_red="\e[31m"
color_text_green="\e[32m"
color_text_yellow="\e[33m"

color_text_pass=$color_text_green
color_text_warn=$color_text_yellow
color_text_error=$color_text_red

# Arguments:
# $1: Pass message
show_pass () {
	echo -e $color_text_pass$1"."$color_text_normal
}

# Arguments:
# $1: Warning message
warn_and_exit () {
	echo -e $color_text_warn$1"!"$color_text_normal
	exit 1
}

# Arguments:
# $1: Command (including arguments)
# $2: Message before running command
# $3: Warning message if command failed
run_and_check () {
	echo $2
	$1
	if [ $? -ne 0 ]
	then
		warn_and_exit "$3"
	fi
}

# Arguments:
# $1: File name to insert text
# $2: Text to add
# $3: Path to text to add
insert_bblayer () {
	grep $2 $1 1>/dev/null
	if [ $? -ne 0 ]
	then
		sed -i "/iot-middleware/a\  $3\/$2 \\\\" $1
	fi
}

# Arguments:
# $1: Yocto BSP repo directory
# $2: Meta-netcontiki directory
# $3: List of files to copy
prepare_meta_netcontiki () {
	run_and_check "mkdir -p $1/$2" "Creating meta-netcontiki directory '$1/$2'..." "Failed to create '$1/$2'"
	run_and_check "cp -rf $3 $1/$2" "Copying '$3' to '$1/$2'..." "Failed to copy '$3' to '$1/$2'"

	show_pass "Meta-netcontiki is ready"
	echo
}

# Arguments:
# $1: Yocto BSP repo directory
# $2: Meta-wakaama directory
# $3: List of files to copy
prepare_meta_wakaama () {
	run_and_check "mkdir -p $1/$2" "Creating meta-wakaama directory '$1/$2'..." "Failed to create '$1/$2'"
	run_and_check "cp -rf $3 $1/$2" "Copying '$3' to '$1/$2'..." "Failed to copy '$3' to '$1/$2'"

	show_pass "Meta-wakaama is ready"
	echo
}

# Arguments:
# $1: Yocto BSP repo directory
# $2: Meta-Netcontiki and Meta-Wakaama directory
# $3: bblayers config file
update_bblayers_conf () {
	cur_dir=$(pwd)
	cd $1

	dir_repo=$(pwd)
	run_and_check "insert_bblayer $3 $2 $dir_repo" "Adding '$2' to '$3'..." "Failed to add '$2' to '$3'"

	show_pass "bblayers config done"
	echo
	cd $cur_dir
}

#Arguments:
#Adding AT_WORKSPACE for bblayers
# $1: Yocto BSP repo directory
update_bblayers_conf_workspace () {
	cur_dir=$(pwd)
	cd $1

	dir_repo=$(pwd)
#	echo "LAYERDEPENDS_networking-layer_remove = "openembedded-layer"" >>$file_bblayers_conf
#	echo "LAYERDEPENDS_networking-layer_remove = "meta-python"" >>$file_bblayers_conf

	echo "AT_WORKSPACE='$dir_repo'" >>$file_bblayers_conf
	cd $cur_dir
}

# Sparrow
sparrow_branch="master"
sparrow_srcrev="fb900f27eb27f5df4f599f4c23a25d9a2c2d61e9"
# Sparrow-Wakaama
sparrow_wakaama_branch="sparrow-lwm2m"
sparrow_wakaama_srcrev="c838cdab252c1928905dd93eb528630e6ff1597d"

clone_sparrow () {
        cur_dir=$(pwd)
	cd ../$dir_repo_yocto_bsp
        SPARROW_GIT_REPO="https://github.com/sics-iot/sparrow.git"
        if [ ! -d "$dir_repo_yocto_bsp/sparrow" ]
        then
                echo "Creating new sparrow repository"
                git clone $SPARROW_GIT_REPO sparrow
                cd sparrow
                git checkout -b $sparrow_branch --track remotes/origin/$sparrow_branch
                git reset --hard $sparrow_srcrev
                git submodule init && git submodule update
#		cd apps/tinydtls
#		git submodule init && git submodule update
		cd contiki
                git submodule init && git submodule update
                cd ..
                git tag -a 6LoWPAN_3.0-baseline -m "6LoWPAN 3.0 Baseline"
        else
                cd $dir_repo_yocto_bsp/sparrow
                SPARROW_COMMIT=`git log --oneline | grep "Merge pull request #29 from nfi/modules" | cut -d " " -f2-300`
                if [[ "$SPARROW_COMMIT" == "Merge pull request #29 from nfi/modules" ]]
                then
                        echo "Commit message matched. HEAD is at $SPARROW_COMMIT"
                else
                        git checkout -b $sparrow_branch --track remotes/origin/$sparrow_branch
                        git reset --hard $sparrow_srcrev
			git submodule init && git submodule update
#			cd apps/tinydtls
#			git submodule init && git submodule update
			cd contiki
                        git submodule init && git submodule update
                        cd ..
			git tag -a 6LoWPAN_3.0-baseline -m "6LoWPAN 3.0 Baseline"
		fi
        fi
        echo "Fetching latest changes for sparrow"
        git fetch origin
        cd ${cur_dir}
}

clone_sparrow_wakaama () {
        cur_dir=$(pwd)
	cd ../$dir_repo_yocto_bsp
        SPARROW_WAKAAMA_GIT_REPO="https://github.com/sics-iot/sparrow-wakaama.git"
        if [ ! -d "$dir_repo_yocto_bsp/sparrow-wakaama" ]
        then
                echo "Creating new sparrow-wakaama repository"
                git clone $SPARROW_WAKAAMA_GIT_REPO sparrow-wakaama
                cd sparrow-wakaama
                git checkout -b $sparrow_wakaama_branch --track remotes/origin/$sparrow_wakaama_branch
                git reset --hard $sparrow_wakaama_srcrev
                cd examples/shared/tinydtls
                git submodule init
                git submodule update
		cd ../../../
		git tag -a Sparrow-Wakaama -m "Sparrow-Wakaama LWM2M servers"
        else
                cd sparrow-wakaama
		SPARROW_WAKAAMA_COMMIT=`git log --oneline | grep "Disable inactivity rehandshake for LWM2M server" | cut -d " " -f2-300`
                if [[ "$SPARROW_WAKAAMA_COMMIT" == "Disable inactivity rehandshake for LWM2M server" ]]
                then
                        echo "Commit message matched. HEAD is at $SPARROW_WAKAAMA_COMMIT"
                else
                        git checkout -b $sparrow_wakaama_branch --track remotes/origin/$sparrow_wakaama_branch
                        git reset --hard $sparrow_wakaama_srcrev
                        cd examples/shared/tinydtls
                	git submodule init && git submodule update
			cd ../../../
			git tag -a Sparrow-Wakaama -m "Sparrow-Wakaama LWM2M servers"
                fi
        fi
        echo "Fetching latest changes for Sparrow-wakaama"
        git fetch origin
        cd ${cur_dir}
}

# List of global variables
#
dir_base=..
dir_meta_netcontiki=meta-netcontiki
dir_meta_wakaama=meta-wakaama
dir_repo_yocto_bsp=yocto_build
dir_sparrow=sparrow
dir_sparrow_wakaama=sparrow-wakaama
file_bblayers_conf=build/conf/bblayers.conf
files_to_copy="$dir_meta_netcontiki/*"
files_to_copy1="$dir_meta_wakaama/*"

# Start executing from here
prepare_meta_netcontiki $dir_base/$dir_repo_yocto_bsp $dir_meta_netcontiki "$files_to_copy"
prepare_meta_wakaama $dir_base/$dir_repo_yocto_bsp $dir_meta_wakaama "$files_to_copy1"
update_bblayers_conf $dir_base/$dir_repo_yocto_bsp $dir_meta_netcontiki $file_bblayers_conf
update_bblayers_conf $dir_base/$dir_repo_yocto_bsp $dir_meta_wakaama $file_bblayers_conf
update_bblayers_conf_workspace $dir_base/$dir_repo_yocto_bsp
clone_sparrow
clone_sparrow_wakaama

exit 0
