/*
 * Copyright (C) 2012 Intel Corporation. All rights reserved.
 *
 * This software is supplied under the terms of a license
 * agreement or nondisclosure agreement with Intel Corp.
 * and may not be copied or disclosed except in accordance
 * with the terms of that agreement.
 */

/******************************************************************************
 *  $Header:  $
 *  $DateTime:  $
 *  $Author:  rcshipmx$
 *****************************************************************************/

#include "getopt_win.h"

// IMPLEMENT_GETOPT is defined inside getopt_win.h, so this must follow its inclusion
#if defined(IMPLEMENT_GETOPT)
#include <assert.h>
#endif

// For our testing purposes, we want the original as well as our versions
#if defined(TESTING)
#include <ctype.h>
#endif


/// \file
/// Implement getopt() and getopt_long() to match the GNU version (which should match the POSIX.11 standard).
/// Where the library implements the functions, no code is generated unless for test purposes.
/// The test code should define IMPLEMENT_GETOPT before compiling this file.

#ifdef __cplusplus
extern "C"
{
#endif

#if defined(WINDOWS)

//////////////////////////////////////////////////////////////////////////////
// Global values used by getopt() and getopt_long()
//////////////////////////////////////////////////////////////////////////////

/// Pointer to the argument processed
char *optarg = NULL;

/// The index of the next argument to process.  0 to rescan \p optstring, 1 to restart
int optind = 0;

/// Non-zero specifies to print an error; 0 to suppress error output
int opterr = 1;

/// The option char indicating the option or error found
int optopt = NO_MORE_ARGUMENTS;

#endif

#if defined(IMPLEMENT_GETOPT)

// These must be global to match the expected behavior, but only to this module

// POSIX used
static bool getopt_posix = false;

// No permuting of elements, just return option char \1 for non-options
static bool getopt_strict_ordering = false;

// Used to permute the arguments, when not one of getopt_posix or getopt_strict_ordering
static int getopt_argbegin = 0;
static int getopt_argend = 0;

// True when a missing argument should be indicated by ':' instead of '?'
static bool missingarg_return = false;

// The array index into an argument of the next character to process
int getopt_nextchar = 0;

// Applies to int types only
#define swap(a, b)    { int t = a; a = b; b = t; }

// Move processed option elements ahead of unprocessed non-option elements
// If no elements were skipped, this code will have no effect
static void _getopt_permute(char **argv)
{
    int arg = 0;
    char *element = NULL;

    // This can only be done after completely processing an option and its arguments
    if (getopt_nextchar == 0 && getopt_argbegin != getopt_argend) {
        do {
            element = argv[getopt_argend];
            for (arg = getopt_argend; arg > getopt_argbegin; --arg)
                argv[arg] = argv[arg - 1];
            argv[getopt_argbegin] = element;

            ++getopt_argbegin;
            ++getopt_argend;
        } while (getopt_argend < optind);
        optind -= getopt_argend - getopt_argbegin;
        getopt_argbegin = getopt_argend;
    }
}

// Point to the next option element, skipping non-options; return 0 to continue processing
static int _getopt_nextoption(int argc, char *const argv[],
                              const char *optstring)
{
    const char *arg = NULL;

    optarg = NULL;

    if (optind == 0) {
// RCSFIX: This should actually get the environment variable...
#ifdef POSIXLY_CORRECT
        getopt_posix = true;
#else
        // The first character is '+'
        getopt_posix = *optstring == '+';
#endif

        getopt_strict_ordering = *optstring == '-';

        getopt_nextchar = 0;

        // Ignore lead-in characters
        if (*optstring == '+' || *optstring == '-')
            ++optstring;

        missingarg_return = *optstring == ':';

        // Start with argument 1
        ++optind;
        getopt_argbegin = optind;
        getopt_argend = optind;
    }

    // If, the last time we were here, we needed to move arguments, do so now
    _getopt_permute((char **)argv);

    // Are we done?
    if (optind >= argc)
        return NO_MORE_ARGUMENTS;

    // We assume that everything is in order, and that the argument is non-NULL
    arg = argv[optind];
    assert(NULL != arg && '\0' != *arg);

    if (getopt_nextchar == 0 && (arg[0] != '-' || '\0' == arg[1])) {
        // Not an option
        if (getopt_posix)
            return NO_MORE_ARGUMENTS;

        // Never change the order of arguments
        if (getopt_strict_ordering) {
            optarg = &argv[optind][0];
            ++optind;       // Point to the next argument for the next iteration
            return 1;       // Per the GNU getopt()/getopt_long() man page
        }

        // At this point, we have an argument that is not an option; find the end of the sequence of non-options
        getopt_argbegin = optind;
        for (getopt_argend = optind; getopt_argend < argc; ++getopt_argend) {
            arg = argv[getopt_argend];

            // A lone dash is not an option, but two lone dashes is
            if (arg[0] == '-' && arg[1] != '\0') {
                optind = getopt_argend;
                return !NO_MORE_ARGUMENTS;
            }
        }

        // Nothing to move
        getopt_argend = optind;
        return NO_MORE_ARGUMENTS;
    }

    // This is an option element
    return !NO_MORE_ARGUMENTS;
}


//////////////////////////////////////////////////////////////////////////////
// getopt_internal()
//////////////////////////////////////////////////////////////////////////////

// We are here because we know that we have a short option
static int _getopt_internal(int argc, char *const argv[],
                            const char *optstring) __THROW
{
    // Skip the dash
    char *arg = argv[optind] + 1;
    char *option = NULL;
    char opt = arg[getopt_nextchar++];

    // To prevent a bug, we must ignore lead-in characters, which have already been handled
    if (*optstring == '+' || *optstring == '-')
        ++optstring;
    if (*optstring == ':')
        ++optstring;

    // Short option found: scan optstring (argument is not "-"; that was eliminated above)
    option = strchr((char *)optstring, opt);

    if (opt == ':' || NULL == option) {
        if (opterr)
            fprintf(stderr, "getopt: invalid option '-%c'\n", opt);

        // Set up for next time
        if ('\0' == arg[getopt_nextchar]) {
            ++optind;
            getopt_nextchar = 0;
        }
        optopt = opt;
        return '?';
    }

    // Is an option argument wanted?
    if (*++option == ':') {
        // NULL if no option included (it may be optional as checked right after this)
        optarg = ('\0' == arg[getopt_nextchar]) ? NULL : &arg[getopt_nextchar];

        // Since an argument was wanted, we continue with the next argv-element afterward
        ++optind;
        getopt_nextchar = 0;

        // If an option argument is required but missing, check the next argv-element
        if (*++option != ':' && NULL == optarg) {
            // It is an error if there are no more argv-elements
            if (optind >= argc) {
                optopt = opt;
                if (missingarg_return)
                    return ':';
                if (opterr)
                    fprintf(stderr,
                            "getopt: Missing command-line argument: '-%c'\n",
                            opt);
                return '?';
            }

            // Note that this argument can be anything, including what looks like an option
            optarg = argv[optind];
            ++optind;
        }

        return opt;
    }

    // Set up for next time
    if ('\0' == arg[getopt_nextchar]) {
        ++optind;
        getopt_nextchar = 0;
    }
    return opt;
}


//////////////////////////////////////////////////////////////////////////////
// getopt_long_internal()
//////////////////////////////////////////////////////////////////////////////

// We are here because we know that we have an option, long or short
// Note that if longopts is NULL, _getopt_internal will be called instead to process the short options.
static int _getopt_long_internal(int argc, char *const argv[],
                                 const char *optstring,
                                 const struct option *longopts,
                                 int *longindex) __THROW
{
    char *arg = argv[optind];
    int optlen = 0;
    const struct option *found = NULL;
    const struct option *lop = NULL;

    // Short option?
    if (NULL == longopts || '-' != arg[1])
        return _getopt_internal(argc, argv, optstring);

    // To prevent a bug, we must ignore lead-in characters, which have already been handled
    if (*optstring == '+' || *optstring == '-')
        ++optstring;
    if (*optstring == ':')
        ++optstring;

    // Since this is a long option, we will use the whole argument, so
    // point to the next argument for the next iteration, if any
    ++optind;

    // Two lone dashes always terminate processing
    if ('\0' == arg[2])
        return NO_MORE_ARGUMENTS;

    // Skip the dashes
    arg += 2;

    // Long option found: scan longopts looking for it
    optlen = (int)(strchrnul(arg, '=') - arg);


    for (lop = longopts; NULL != lop->name; ++lop) {
        // If not a match, continue
        if (0 != strncmp(lop->name, arg, optlen))
            continue;

        // If an exact match, we found it
        if ('\0' == lop->name[optlen]) {
            found = lop;
            break;
        }

        if (NULL == found) {
            found = lop;
        } else {
            if (opterr)
                fprintf(stderr, "getopt_long: ambiguous option: '--%*s'\n",
                        optlen, arg);

            // Ambiguous match
            optopt = 0;
            return '?';
        }
    }

    // No match?
    if (NULL == found) {
        optopt = 0;
        return '?';
    }

    switch (found->has_arg) {
    case no_argument:
        if (arg[optlen] == '=') {
            if (opterr)
                fprintf(stderr,
                        "getopt_long: option `--%s' takes no argument\n",
                        found->name);
            optopt = found->val;
            return '?';
        }
        break;

    case optional_argument:
        // NULL if no option included
        optarg = ('\0' == arg[optlen]) ? NULL : &arg[optlen + 1];       // Skip the '='
        break;

    case required_argument:
        if ('=' == arg[optlen]) {
            optarg = &arg[optlen + 1];       // Skip the '='
        } else {
            // If a required option argument is missing, check the next argv-element
            // Error if no more arguments
            if (optind >= argc) {
                optarg = NULL;
                optopt = found->val;
                if (missingarg_return)
                    return ':';
                if (opterr)
                    fprintf(stderr,
                            "getopt_long: missing command-line argument: '--%s'\n",
                            found->name);
                return '?';
            }

            // Use the next argv-element
            optarg = argv[optind];
            ++optind;
        }
        break;

    default:
        if (opterr)
            fprintf(stderr, "getopt_long: invalid argument specification\n",
                    found->name);
        break;
    }

    // A long option was found: set the index value
    if (NULL != longindex)
        *longindex = (int)(found - longopts);

    if (NULL == found->flag)
        return found->val;

    *found->flag = found->val;
    return 0;
}


//////////////////////////////////////////////////////////////////////////////
// getopt()
//////////////////////////////////////////////////////////////////////////////

/// Create our own function, for Windows.
/// \param argc The number of arguments.
/// \param argv The argument string.
/// \param optstring The short options, plus possible mode and argument flags.
/// \return -1 when done, the option character, ':' or '?' for a missing argument.
int _getopt(int argc, char *const argv[], const char *optstring) __THROW
{
    int retVal = _getopt_nextoption(argc, argv, optstring);

    if (!retVal)
        retVal = _getopt_internal(argc, argv, optstring);
    return retVal;
}


//////////////////////////////////////////////////////////////////////////////
// getopt_long()
//////////////////////////////////////////////////////////////////////////////

/// Create our own function for Windows (and to compare to the GNU function, when on Linux).
/// \param argc The number of arguments.
/// \param argv The argument string.
/// \param optstring The short options, plus possible mode and argument flags.
/// \param longopts An array of long options.
/// \param longindex If non-NULL, a pointer to an integer receiving the return value, which will be 0 if valid.
/// \return -1 when done, the option character, ':' or '?' for a missing argument, ambiguous match or extraneous parameter.
int _getopt_long(int argc, char *const argv[], const char *optstring,
                 const struct option *longopts, int *longindex) __THROW
{
    // Find the next option, if any
    int retVal = _getopt_nextoption(argc, argv, optstring);

    if (!retVal) {
        // We have an option
        retVal = _getopt_long_internal(argc, argv, optstring, longopts,
                                       longindex);
    }
    return retVal;
}

// End of Windows-or-test implementation
#endif


#ifdef __cplusplus
}
#endif


/*
 * Copyright (C) 2012 Intel Corporation. All rights reserved.
 *
 * This software is supplied under the terms of a license
 * agreement or nondisclosure agreement with Intel Corp.
 * and may not be copied or disclosed except in accordance
 * with the terms of that agreement.
 */

/******************************************************************************
 *  $Header:  $
 *  $DateTime:  $
 *  $Author:  rcshipmx$
 *****************************************************************************/

/// \file
/// This file contains the source for OS-agnostic code for Windows and Linux.

/*
 * Copyright (C) 2012 Intel Corporation. All rights reserved.
 *
 * This software is supplied under the terms of a license
 * agreement or nondisclosure agreement with Intel Corp.
 * and may not be copied or disclosed except in accordance
 * with the terms of that agreement.
 */

/******************************************************************************
 *  $Header:  $
 *  $DateTime:  $
 *  $Author:  rcshipmx$
 *****************************************************************************/

#ifndef OS_AGNOSTIC_H
#define OS_AGNOSTIC_H

/*
 * Copyright (C) 2010-2012 Intel Corporation. All rights reserved.
 *
 * This software is supplied under the terms of a license
 * agreement or nondisclosure agreement with Intel Corp.
 * and may not be copied or disclosed except in accordance
 * with the terms of that agreement.
 */

#ifndef _OSAL_H_
#define _OSAL_H_

//////////////////////////////////////////////////////////////////////////////
//  OSAL.h
//      Definitions to make code common to all operating systems.
//
//  Usage:
//      Use defines WINDOWS and LINUX to select an operating system.
//      Avoid local definitions of OS-specific code whenever possible.
//      Choose functions that begin with os_ to ensure OS-agnostic code;
//      many of these functions have a slightly redefined interface.
//
//  Function index:
//      void Sleep(ms)
//              Sleep the given number of milliseconds
//
//      bool os_fopen(FILE descriptor, name, mode)
//              Open a file descriptor for the given name and mode.
//              Return true if open succeeded.
//
//  C-string functions:
//      int os_vcprintf
//              Get the buffer size needed to write a printf-style string.
//
//      int os_vsnprintf(s, size, count, format, args)
//              Safe version of vsnprintf.
//
//      int os_strncpy(destination, size, source, num)
//      int os_sprintf(str, size, format, ...)
//      int os_vsprintf(str, size, args)
//      void *os_memcpy(dst, dstSize, src, maxCount)
//      void *os_memmove(dst, dstSize, src, maxCount)
//
//      void *VIRTUAL_ALLOC(size)
//      void VIRTUAL_FREE(address)
//
//  Where equivalent functions exist, prefer the Linux names.
//    Name to use         Windows equivalent
//      strncasecmp         strnicmp
//      popen               _popen
//      pclose              fclose
//
// Changed names; unchanged functionality
//////////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////////
// The preferred definitions to use if needed: LINUX or WINDOWS
//////////////////////////////////////////////////////////////////////////////

#if !defined(WINDOWS)
#if defined(_WIN32) || defined(_WIN64)
#define WINDOWS
#endif
#endif

#if !defined(LINUX) && defined(__linux__)
#define LINUX
#endif


//////////////////////////////////////////////////////////////////////////////
// Includes
//////////////////////////////////////////////////////////////////////////////

#include <string.h>
#include <stdio.h>
/*
 * Copyright (C) 2001-2012 Intel Corporation. All rights reserved.
 *
 * This software is supplied under the terms of a license
 * agreement or nondisclosure agreement with Intel Corp.
 * and may not be copied or disclosed except in accordance
 * with the terms of that agreement.
 */

#ifndef _TYPES_H
#define _TYPES_H

typedef unsigned short USHORT;
typedef unsigned char UCHAR;
#ifdef __linux__

#include <stdint.h>

typedef unsigned int DWORD;
#define stricmp      strcasecmp
#define strnicmp     strncasecmp
#define _stricmp     strcasecmp
#define _strnicmp    strncasecmp
#define getch        getchar

// Windows defines these, so only needed for linux
typedef uint8_t BYTE;
typedef uint16_t WORD;
typedef uint32_t ULONG;

#else // _WIN32
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#if _MSC_VER >= 1600
#include <stdint.h>
#else
// J.Potts: #error directive will not compile
//#error Unsupported version of Visual Studio! Must use VS2010 or greater.
#endif

typedef unsigned long DWORD;
#endif

typedef void *HANDLE;
typedef void *PVOID;
typedef uint64_t QWORD;
typedef uint64_t ULONG64;

#define ARG_USED(x)    while ((void)(x), 0)

#if !defined(__cplusplus)
typedef uint8_t bool;
#endif

typedef          int8_t I8;      //  8-bit signed integer
typedef          int64_t I64;    // 64-bit signed integer
typedef          uint64_t U64;   // 64-bit unsigned integer
typedef          int16_t I16;    // 16-bit signed integer
typedef          int32_t I32;    // 32-bit signed integer

typedef          uint8_t U8;     //  8-bit unsigned integer
typedef          uint16_t U16;   // 16-bit unsigned integer
typedef          uint32_t U32;   // 32-bit unsigned integer

typedef            float F32;    // 32-bit single precision floating point value
typedef            double F64;   // 64-bit double precision floating point value

#endif // _TYPES_H


#if defined(WINDOWS)
#include <errno.h>
#elif defined(LINUX)
#include <unistd.h>     // For getopt_long(), fsync(), etc.
#endif //__linux__


//////////////////////////////////////////////////////////////////////////////
// Not yet implemented
//////////////////////////////////////////////////////////////////////////////

#define os_sscanf1    sscanf
#define os_sscanf2    sscanf
#define os_strcat     strcat


//////////////////////////////////////////////////////////////////////////////
// Windows specifics
//////////////////////////////////////////////////////////////////////////////

#if defined(WINDOWS)

// Use Windows equivalent
#define popen          _popen
#define pclose         fclose
#define strncasecmp    _strnicmp
#define strtok_r       strtok_s

// Do not use these Windows values
#define __THROW        // Ignore this common modifier
#define ssize_t        SSIZE_T

// Redefined functions
#define os_fopen(fd, name,                                                 \
                 mode)                                (!fopen_s(&fd, name, \
                                                                mode))

// Windows needs this, but Linux does not
#define va_copy(ap, args)                             (ap) = (args)

// Use safe version (typically defined on Windows already)
#define os_vcprintf(fmt, ap)                          _vscprintf(fmt, ap)
#define os_vsnprintf(s, size, count, format, args)    _vsnprintf_s(s, size, \
                                                                   count,   \
                                                                   format, args)
#define os_strncpy(destination, size, source, num)    strncpy_s(destination,  \
                                                                size, source, \
                                                                num)
#define os_strncat(destination, size, source, num)    strncat_s(destination,  \
                                                                size, source, \
                                                                num)
#define os_sprintf(str, size, format, ...)            sprintf_s(str, size, \
                                                                format,    \
                                                                __VA_ARGS__)
#define os_vsprintf(str, size, args)                  vsprintf_s(str, size, \
                                                                 args)
#define os_memcpy(dst, dstSize, src, maxCount)        ::memcpy_s(dst, dstSize, \
                                                                 src, maxCount)
#define os_memmove(dst, dstSize, src, maxCount)       ::memmove_s(dst, dstSize, \
                                                                  src, maxCount)


#define VIRTUAL_ALLOC(size)                           VirtualAlloc(NULL, size,    \
                                                                   MEM_COMMIT,    \
                                                                   PAGE_READWRITE \
                                                                   |              \
                                                                   PAGE_NOCACHE)
#define VIRTUAL_FREE(address)                         VirtualFree(address, 0, \
                                                                  MEM_RELEASE)


//////////////////////////////////////////////////////////////////////////////
// Linux specifics
//////////////////////////////////////////////////////////////////////////////

#elif defined(LINUX)

// Redefined functions
#define os_fopen(fd, name, mode)                      (NULL != \
                                                       (fd = fopen(name, mode)))

#define os_vcprintf(fmt, ap)                          vsnprintf(NULL, 0, fmt, \
                                                                ap)
#define os_vsnprintf(s, size, count, format, args)    vsnprintf(s, count, \
                                                                format, args)
#define os_strncpy(destination, size, source, num)    strncpy(destination, \
                                                              source, num);
#define os_strncat(destination, size, source, num)    strncat(destination, \
                                                              source, num)
#define os_sprintf(str, size, format, ...)            sprintf(str, format, \
                                                              ## __VA_ARGS__)
#define os_vsprintf(str, size, args)                  vsprintf(str, args)
#define os_memcpy(dst, dstSize, src,                                    \
                  maxCount)                           (memcpy(dst, src, \
                                                              maxCount), 0)
#define os_memmove(dst, dstSize, src,                                    \
                   maxCount)                          (memmove(dst, src, \
                                                               maxCount), 0)

#define VIRTUAL_ALLOC(size)                           memalign(sysconf(           \
                                                                   _SC_PAGESIZE), \
                                                               size)
#define VIRTUAL_FREE(address)                         free(address)

#define Sleep(ms)                                     usleep((ms) * 1000)


//////////////////////////////////////////////////////////////////////////////
// Unsupported
//////////////////////////////////////////////////////////////////////////////

#else

#error Unsupported O/S! Not Linux or Windows


// Legacy code

// This is not safe: to make safe replace all uses of os_sprintf, os_strncpy and os_vsprintf
const size_t MAX_STRING_LEN = 1024 * 1024;

// [todo] Write a general replacement vcscanf with variable nmber of args
// [todo] This is not available on all systems

inline int os_sscanf1(const char *str, const char *format, void *p)
{
    return sscanf_s(str, format, p);
}

inline int os_sscanf2(const char *str, const char *format, void *p, void *q)
{
    return sscanf_s(str, format, p, q);
}

#endif


// Ensure needed value is defined to a reasonable value
#if !defined(PATH_MAX)
#define PATH_MAX    4096
#elif PATH_MAX > 4096
#undef PATH_MAX
#define PATH_MAX    4096
#endif


#endif // _OSAL_H_

#ifdef __cplusplus
extern "C"
{
#endif
//////////////////////////////////////////////////////////////////////////////
// List of common values:
//
// strchrnul()
// fileno()
//
//////////////////////////////////////////////////////////////////////////////

#ifdef UNICODE
#undef UNICODE
#endif

#ifdef _UNICODE
#undef _UNICODE
#endif


//////////////////////////////////////////////////////////////////////////////
// Windows definitions
//////////////////////////////////////////////////////////////////////////////


#if defined(WINDOWS)

// Windows-specific
#define _TCHAR    char

// Types

// String functions

#define fileno(f)    _fileno(f)


/// Return a pointer to the first occurrence of the character in the string, or the end of the string.
/// \attention Null is returned if null was passed for the string.
char *strchrnul(const char *s, int c);

/// Return the filename of a path, just like on Linux.
const char *basename(const char *name);

// OS-specific functions

// Flush data to disk
int fsync(int fd);

#endif  // WINDOWS
#ifdef __cplusplus
}
#endif
#endif  // OS_AGNOSTIC_H


#ifdef __cplusplus
extern "C"
{
#endif

#if defined(WINDOWS)

char *strchrnul(const char *s, int c)
{
    if (NULL != s) {
        while (*s && *s != c)
            ++s;
    }
    return (char *)s;
}

// Return the last component of a pathname.
// Based on code that is in the public domain.
//
// Given a pointer to a string containing a typical pathname (C:\Users\mic\src\codefile.cpp for example),
// return a pointer to the last component of the pathname ("codefile.cpp" in this case).
//
// The code presumes a DOS/Windows style path with DOS/Windows style separators.

#ifndef DIR_SEPARATOR
#define DIR_SEPARATOR    '/'
#endif

#if (defined (_WIN32) || defined (__MSDOS__) || defined (__DJGPP__))
#define HAVE_DOS_BASED_FILE_SYSTEM
#ifndef DIR_SEPARATOR_2
#define DIR_SEPARATOR_2    '\\'
#endif
#endif

// Define IS_DIR_SEPARATOR.
#ifndef DIR_SEPARATOR_2
#define IS_DIR_SEPARATOR(ch)    ((ch) == DIR_SEPARATOR)
#else
#define IS_DIR_SEPARATOR(ch)    ((ch) == DIR_SEPARATOR || (ch) == \
                                 DIR_SEPARATOR_2)
#endif

const char *basename(const char *name)
{
    const char *base;

    if (name == NULL)
        return NULL;

#if defined (HAVE_DOS_BASED_FILE_SYSTEM)
    // Skip over the disk name in MSDOS pathnames.
    if (isalpha(name[0]) && name[1] == ':')
        name += 2;
#endif

    for (base = name; *name; name++) {
        if (IS_DIR_SEPARATOR(*name))
            base = name + 1;
    }
    return base;
}

int fsync(int fd)
{
    fd;

    return 0;
}


#endif
#ifdef __cplusplus
}
#endif
