"""
    preprocess_micp.py - pre-process placeholders in the micp/* scripts
    and documentation depending on the build target. This script should
    be executed in the directory where micp/ is located.

    This script also creates the micp/version.py script for knc linux,
    knllb windows, knllb linux and knlsb linux.

Usage:

    preprocess_micp.py <mipcdir> <target> <mpsx_version> <datadir> <libexecdir>

Where <target> can be:
    - knllb
    - knlsb
    - knc

and <micpdir> is the directory where micperf is being compiled within the build
system
"""

import fileinput
import os
import sys
import platform
import ftplib

################################################################################
# placeholders in the micp/* scripts and documentation, names of the variables
# match the name of the placeholders in the files (INSTALL.txt, README.txt, etc)
################################################################################

KNX_MICPERF_HELP = 'KNX_MICPERF_HELP'
MPSSVERSION = 'MPSSVERSION'
README_KNX_PRODUCT = 'README_KNX_PRODUCT'
WIN_LIBEXEC_KEY = 'README_WINDOWS_LIBEXEC_PATH'
README_WINDOWS_TUTORIALS_PATH = 'README_WINDOWS_TUTORIALS_PATH'
README_WINDOWS_VISUAL_STUDIO = 'README_WINDOWS_VISUAL_STUDIO'
README_LINUX_EULA = 'README_LINUX_EULA'
README_WINDOWS_EULA = 'README_WINDOWS_EULA'


################################################################################
#                     potential values for the placeholders
################################################################################
KNL_HELP = """NAME
    micprun - Run the micperf workloads.

SYNOPSIS
    micprun -h | --help
      Print this help message.

    micprun --version
      Print the version.

    micprun [-v level] [-o outdir] [-t outtag] [-x offload]* [-d device] [-e plugin] -k kernel -p params
      Run a single kernel with command line parameters.

    micprun [-v level] [-o outdir] [-t outtag] [-x offload]* [-d device] [-e plugin] [-k kernels] [-c category]
      Run on all or a subset of the kernels with parameter category.

    micprun [-v level] [-o outdir] [-t outtag] [-x offload]* [-d device] [-e plugin] [-k kernels] [-c category] [-m margin] -r pickle
      Repeat a previously executed run and compare results.

    * Command line option only available for Intel(R) Xeon Phi(TM) X100/X200 Coprocessors.

DESCRIPTION
    Runs the micperf workloads, prints rolled up statistics, plots
    results and can compare results to previous runs.  When run with
    -o the resulting .pkl file can be used with the other micp
    applications (micpinfo, micpprint, micpcsv, micpplot).

OPTIONS
    -k kernel
       Specifies the name of the kernel to be run. Multiple kernels
       can be specified separated by ":" e.g "-k dgemm:sgemm". On
       knl/knc coprocessors it defaults to all kernels available for
       the offload method (also done when "-k all" is specified).
       On knl processors offload methods are not supported kernels
       are directly executed in the processor.
       If "-k help" is specified then all available kernels are listed.
    -D (Only for Intel(R) Xeon Phi(TM) Processors X200)
       Memory selection, by default micprun will attempt to allocate
       MCDRAM memory to execute the workloads, on failure (e.g. memory
       mode set to Cache) micprun will continue with the execution but
       using only DDR memory. -D forces micprun to use directly DDR memory
       without performing any MCDRAM memory availability check. This option
       only works in Flat mode as in cache mode MCDRAM is used transparently.
    -p params
       Command line arguments to be passed to the kernel.  These can
       be positional or --parameter value.  To find the argument
       names, positional ordering and default values for a kernel
       specify "-p --help".
    -c category
       Category of command line arguments to be used.  Defaults to
       "optimal".  All kernels have an "optimal", "scaling" and "test"
       parameter category.  The distributed kernels have three
       additional categories: "scaling_quick", "optimal_quick",
       "scaling_core".  The quick variants complete in less time but
       may not be as high performance.  The "scaling_core" category
       does a strong core scaling test where possible (rather than a
       data scaling test).
    -x method (Only for Intel(R) Xeon Phi(TM) X100/X200 Coprocessors)
       Specifies the offload technique default is "native:scif",
       options for the method are "native", "scif", "pragma", "coi",
       or "myo".  Multiple offload methods can be specified separated
       by ":" e.g. "-x native:scif:pragma"
       On a Intel(R) Xeon Phi(TM) Processors X200 the kernels can NOT be
       offloaded since there are no coprocessor(s) installed in the system,
       in this case an error will be returned.
    -d device (Only for Intel(R) Xeon Phi(TM) X100/X200 Coprocessors)
       Enables the selection of the device used to run the workloads
       in a multi-card system.  For a single card system, or to run on
       the zero indexed device, -d is not required.  The parameter
       given is the zero based card index, resolvable host name or an
       IP address.
    -v level
       Level of detail given in the report: 0 raw output to standard
       out, 1 reprints rolled up data to standard out, 2 creates
       plots, 3 plots all of the kernels onto a single graph if
       possible.  Defaults to 0 unless -r is given, then defaults to
       1.
    -o outdir
       Specify a directory in which to create output files.  If not
       specified then no files are created and plots are displayed on
       the screen.  If specified, a pickle file for comparison is
       created, plots are written to png files and not displayed on
       the screen.  If verbosity is set to 1 or higher then a comma
       separated value (csv) file of rolled up statistics is created,
       and -v 2 creates a csv file with all statistics.
       When used on Intel(R) Xeon Phi(TM) Processors X200, the pickle
       file name includes the word 'mcdram' or 'ddr' to indicate the
       type of memory used to execute the workloads.
    -t tag
       Name used to identify the run in pickle file.  The tag is
       included in the name of all output files.  Tags may contain
       only alphanumeric characters underscores and dashes.  All
       other characters are replaced with the dash character.  If -t
       is not specified then a default tag is created.  This default
       tag is an underscore separated list of the coprocessor SKU, the
       mpss version, the offload method, the parameter category and
       the device name.  If more than one offload method or parameter
       category are given these are separated by a dashes.
    -r pickle
       Compare against a pickle file produced by running micprun
       previously with -o specified.  If the -k, -p, -c or -x options
       are not specified on the command line, then options specified
       for these in the previous run are used.
       Intel(R) Xeon Phi(TM) Processor X200 pickle files are not compatible
       with Intel(R) Xeon Phi(TM) coprocessor X200 or Intel(R) Xeon Phi(TM)
       coprocessor X100 pickle files. As offload methods differ among these
       platforms micperf behavior is undefined.
    -R tag
       Compare against a tagged performance measurement included in the
       distribution.  Running with "-R help" will print a list of
       available tags.  If the -k, -p, -c or -x options are not
       specified on the command line, then options specified for these
       in the tagged run are used.
    -m margin
       When run with -r or -R, the -m option compares performance of
       the run against the values stored in the pickle.  If the
       performance of rolled statistics drops by a relative factor
       larger than margin, then a failure message in gtest format is
       printed and a micprun gives a non-zero return code.  If the
       acceptable margin of error is 4% than margin should be
       specified as 0.04.
    -e plugin
       Extend the available kernels with the plug-in package given.
       Note that the plug-in package must be in a directory included
       in the PYTHONPATH, and must contain an __init__.py file which
       declares an __all__ attribute which lists all of the kernel
       modules in the plug-in package.  Each kernel module must have a
       class that inherits from micp.Kernel that has the same name as
       the module that contains it.
    --sudo
       Allow to run benchmarks in privileged mode if they require it.
       The executing user has to be added to the sudoers list.
       Note that depending on system settings user may be prompted for
       root password. In such case the micprun execution will halt until
       the password is provided.

EXIT STATUS
    If a call to a kernel executable gives a non-zero exit code, this
    exit code will be used as the exit code for micprun as well and a
    message is written to standard error that reads:

    ERROR:  Command 'COMMAND' returned non-zero exit status VALUE

    Otherwise, the following table defines the return codes:

    0    No error
    1    Unhandled python exception
    2    Command line parse error
    3    File I/O error
    88   Performance regression error
    89   MPSS service not available error
    90   Kernel or offload lookup error
    91   Linpack kernel could not be executed (missing dependencies).
    127  Missing shared object libraries error

ENVIRONMENT
    Intel(R) Xeon Phi(TM) X100/X200 Coprocessors
        INTEL_MPSS_USER (default host user ID)
            If set this determines the user name for the login to the MIC
            device.
        INTEL_MPSS_SSH_KEY (default is OS dependent)
            If set this determines the path to the private SSH key used to
            log into the MIC device.
        MIC_PERF_EXEC (default defined in micp.version)
            If set the binary executables will be located assuming this is
            path that contains the mic perf bin directory.  Host side
            binaries are located in "$MIC_PERF_EXEC/x86_64" and device
            binaries are located in "$MIC_PERF_EXEC/k1om".

    Intel(R) Xeon Phi(TM) Processors X200 and Intel(R) Xeon Phi(TM) X100/X200 Coprocessors
        MIC_PERF_DATA (default defined in micp.version)
            If set the reference data located in this directory will be
            used with the -R flag.

EXAMPLES
    Intel(R) Xeon Phi(TM) X100/X200 Coprocessors
        micprun -k all -c optimal -x native -v 0
            Equivalent to running without command line parameters (defaults).
        micprun -k shoc_download:shoc_readback -x pragma:scif -v 3
            Run the scaling test for shoc download and readback comparing pragma
            against scif offload, reprint rolled up statistics, and plot
            to screen.

    Intel(R) Xeon Phi(TM) Processors X200
        micprun -k all -c optimal -v 0
            Equivalent to running without command line parameters (defaults).
        micprun -k sgemm:dgemm -v 3 -o .
            Run the scaling test for sgemm and dgemm comparing results,
            reprint rolled up statistics, and create graph in png format
            in the current working directory.
        micprun -D -k sgemm:dgemm:stream -o .
            Run the sgemm, dgemm and stream kernels with DDR memory only
            and creates a pickle file in the current working directory.
        micprun -k hplinpack -c scaling -o .
            Run the hplinpack kernel with MCDRAM memory (if available, DDR
            memory otherwise) using the scaling parameter category, and creates
            a pickle file in the current working directory.

    Intel(R) Xeon Phi(TM) Processors X200 and Intel(R) Xeon Phi(TM) X100/X200 Coprocessors
        micprun
            Run all of the Linux native kernels with optimal parameters.
        micprun -k sgemm -c scaling -v 1
            Run the scaling test for Linux native sgemm and reprint rolled
            up statistics.
        micprun -k sgemm -c scaling -v 2 -o . -t example
            Run the scaling test for sgemm, reprint rolled up statistics
            and create graph in png format in the current working
            directory.  This also produces micp_run_stats_example.pkl in
            the current working directory.
        micprun -r ./micp_run_stats_example.pkl -v 2
            Rerun using parameters from run which produced the pickle
            file, compare the rolled up statistics and create comparative
            graphs on the screen.
        micprun -k sgemm -p --help
            Display the parameters and default values for the sgemm
            kernel.
        micprun -k sgemm -p '--n_num_thread 16'
            Run the sgemm kernel with the default kernel parameters, except
            for n_num_thread which is set to 16 instead of the optimal value.
        micprun -k sgemm -p '4096 10 0 128 NN 4096 4096'
            Run the sgemm kernel with positional parameters.
        micprun  -k fio --sudo
            Run the fio benchmark.

COPYRIGHT
    Copyright 2012-2017, Intel Corporation, All Rights Reserved.

SEE ALSO
    micpinfo, micpprint, micpplot, micpcsv
"""


KNC_HELP = """NAME
    micprun - Run the micperf workloads.

SYNOPSIS
    micprun -h | --help
      Print this help message.

    micprun --version
      Print the version.

    micprun [-v level] [-o outdir] [-t outtag] [-x offload] [-d device] [-e plugin] -k kernel -p params
      Run a single kernel with command line parameters.

    micprun [-v level] [-o outdir] [-t outtag] [-x offload] [-d device] [-e plugin] [-k kernels] [-c category]
      Run on all or a subset of the kernels with parameter category.

    micprun [-v level] [-o outdir] [-t outtag] [-x offload] [-d device] [-e plugin] [-k kernels] [-c category] [-m margin] -r pickle
      Repeat a previously executed run and compare results.

DESCRIPTION
    Runs the micperf workloads, prints rolled up statistics, plots
    results and can compare results to previous runs.  When run with
    -o the resulting .pkl file can be used with the other micp
    applications (micpinfo, micpprint, micpcsv, micpplot).

OPTIONS
    -k kernel
       Specifies the name of the kernel to be run.  Multiple kernels
       can be specified separated by ":" e.g "-k dgemm:sgemm" and it
       defaults to all kernels available for the offload method (also
       done when "-k all" is specified).  If "-k help" is specified
       then all available kernels are listed.
    -p params
       Command line arguments to be passed to the kernel.  These can
       be positional or --parameter value.  To find the argument
       names, positional ordering and default values for a kernel
       specify "-p --help".
    -c category
       Category of command line arguments to be used.  Defaults to
       "optimal".  All kernels have an "optimal", "scaling" and "test"
       parameter category.  The distributed kernels have three
       additional categories: "scaling_quick", "optimal_quick",
       "scaling_core".  The quick variants complete in less time but
       may not be as high performance.  The "scaling_core" category
       does a strong core scaling test where possible (rather than a
       data scaling test).
    -x method
       Specifies the offload technique default is "native:scif",
       options for the method are "native", "scif", "pragma", "coi",
       or "myo".  Multiple offload methods can be specified separated
       by ":" e.g. "-x native:scif:pragma"
    -d device
       Enables the selection of the device used to run the workloads
       in a multi-card system.  For a single card system, or to run on
       the zero indexed device, -d is not required.  The parameter
       given is the zero based card index, resolvable host name or an
       IP address.
    -v level
       Level of detail given in the report: 0 raw output to standard
       out, 1 reprints rolled up data to standard out, 2 creates
       plots, 3 plots all of the kernels onto a single graph if
       possible.  Defaults to 0 unless -r is given, then defaults to
       1.
    -o outdir
       Specify a directory in which to create output files.  If not
       specified then no files are created and plots are displayed on
       the screen.  If specified, a pickle file for comparison is
       created, plots are written to png files and not displayed on
       the screen.  If verbosity is set to 1 or higher then a comma
       separated value (csv) file of rolled up statistics is created,
       and -v 2 creates a csv file with all statistics.
    -t tag
       Name used to identify the run in pickle file.  The tag is
       included in the name of all output files.  Tags may contain
       only alphanumeric characters underscores and dashes.  All
       other characters are replaced with the dash character.  If -t
       is not specified then a default tag is created.  This default
       tag is an underscore separated list of the coprocessor SKU, the
       mpss version, the offload method, the parameter category and
       the device name.  If more than one offload method or parameter
       category are given these are separated by a dashes.
    -r pickle
       Compare against a pickle file produced by running micprun
       previously with -o specified.  If the -k, -p, -c or -x options
       are not specified on the command line, then options specified
       for these in the previous run are used.
    -R tag
       Compare against a tagged performance measurement included in the
       distribution.  Running with "-R help" will print a list of
       available tags.  If the -k, -p, -c or -x options are not
       specified on the command line, then options specified for these
       in the tagged run are used.
    -m margin
       When run with -r or -R, the -m option compares performance of
       the run against the values stored in the pickle.  If the
       performance of rolled statistics drops by a relative factor
       larger than margin, then a failure message in gtest format is
       printed and a micprun gives a non-zero return code.  If the
       acceptable margin of error is 4% than margin should be
       specified as 0.04.
    -e plugin
       Extend the available kernels with the plug-in package given.
       Note that the plug-in package must be in a directory included
       in the PYTHONPATH, and must contain an __init__.py file which
       declares an __all__ attribute which lists all of the kernel
       modules in the plug-in package.  Each kernel module must have a
       class that inherits from micp.Kernel that has the same name as
       the module that contains it.

EXIT STATUS
    If a call to a kernel executable gives a non-zero exit code, this
    exit code will be used as the exit code for micprun as well and a
    message is written to standard error that reads:

    ERROR:  Command 'COMMAND' returned non-zero exit status VALUE

    Otherwise, the following table defines the return codes:

    0    No error
    1    Unhandled python exception
    2    Command line parse error
    3    File I/O error
    88   Performance regression error
    89   MPSS service not available error
    90   Kernel or offload lookup error
    127  Missing shared object libraries error

ENVIRONMENT
    INTEL_MPSS_USER (default host user ID)
        If set this determines the user name for the login to the MIC
        device.
    INTEL_MPSS_SSH_KEY (default is OS dependent)
        If set this determines the path to the private SSH key used to
        log into the MIC device.
    MIC_PERF_EXEC (default defined in micp.version)
        If set the binary executables will be located assuming this is
        path that contains the mic perf bin directory.  Host side
        binaries are located in "$MIC_PERF_EXEC/x86_64" and device
        binaries are located in "$MIC_PERF_EXEC/k1om".
    MIC_PERF_DATA (default defined in micp.version)
        If set the reference data located in this directory will be
        used with the -R flag.

EXAMPLES
    micprun
        Run all of the Linux native kernels with optimal parameters.
    micprun -k all -c optimal -x native -v 0
        Equivalent to running without command line parameters
        (defaults).
    micprun -k sgemm -c scaling -v 1
        Run the scaling test for Linux native sgemm and reprint rolled
        up statistics.
    micprun -k sgemm -c scaling -v 2 -o . -t example
        Run the scaling test for sgemm, reprint rolled up statistics
        and create graph in png format in the current working
        directory.  This also produces micp_run_stats_example.pkl in
        the current working directory.
    micprun -k shoc_download:shoc_readback -x pragma:scif -v 3
        Run the scaling test for shoc download and readback comparing pragma
        against scif offload, reprint rolled up statistics, and plot
        to screen.
    micprun -r ./micp_run_stats_example.pkl -v 2
        Rerun using parameters from run which produced the pickle
        file, compare the rolled up statistics and create comparative
        graphs on the screen.
    micprun -k sgemm -p --help
        Display the parameters and default values for the sgemm
        kernel.
    micprun -k sgemm -p '--n_num_thread 16'
        Run the sgemm kernel with the default kernel parameters, except
        for n_num_thread which is set to 16 instead of the optimal value.
    micprun -k sgemm -p '4096 10 0 128 NN 4096 4096'
        Run the sgemm kernel with positional parameters.

COPYRIGHT
    Copyright 2012-2017, Intel Corporation, All Rights Reserved.

SEE ALSO
    micpinfo, micpprint, micpplot, micpcsv
"""

# MPSSVERSION is defined by the user

# README_KNX_PRODUCT
KNC_PRODUCT_NAME = "Coprocessor X100 Product Family"
KNLLB_PRODUCT_NAME = "Coprocessor X200 Product Family"
KNLSB_PRODUCT_NAME = "Processor X200 Product Family"

# WIN_LIBEXEC_KEY and README_WINDOWS_TUTORIALS_PATH
_WIN_WORKLOAD_LOCATION_MSG = "\nand for the Windows distribution these are installed to:\n\n{0}"
WIN_LIBEXEC_VALUE = _WIN_WORKLOAD_LOCATION_MSG.format("C:\\Program Files\\Intel\\MPSS\\sdk\\micperf\\libexec\n")
WIN_TUTORIALS_VALUE = _WIN_WORKLOAD_LOCATION_MSG.format("C:\\Program Files\\Intel\\MPSS\\sdk\\tutorials\\micperf\n")

# README_WINDOWS_VISUAL_STUDIO
WIN_VISUAL_STUDIO_VALUE = ("In Windows each sub-directory contains Microsoft Visual Studio project\n"
                           "files that will compile the workload with the Intel(R) Composer XE 14.0\n"
                           "compiler.")

# README_LINUX_EULA and README_WINDOWS_EULA
KNC_LINUX_EULA_VALUE = "{docdir}/mpss-license-{version}/license.txt"
KNLSB_LINUX_EULA_VALUE = "{docdir}/micperf/EULA"
KNLLB_LINUX_EULA_VALUE = "{docdir}/packages/mpss-{version}/EULA"
WINDOWS_EULA_VALUE = "and on windows at:\n\nC:\\Program Files\\Intel\\MPSS\\EULA.rtf"


################################################################################
#                            Preprocessor Classes
################################################################################

class Preprocessor(object):
    """
    Abstract class, this class plays a role similar to the C/C++ preprocessor
    it replaces a set of placeholders (in the docuemnation and scripts) with
    its correct value depending on the version of micperf that is being
    compiled: knc, knlsb or knllb.

    The most important private members are:
      _patterns_to_replace, dictionary that maps the placeholder to its value
      _patterns_to_delete, tuple of placeholders that should be removed
      _files_to_update, tuple of files to be processed
      _micp_path, path to the micperf micp/ directory
    """

    def __init__(self):
        """defining private members to prevent complaints from pylint,
        but this is an abstract class"""
        self._patterns_to_replace = None
        self._patterns_to_delete = None
        self._files_to_preprocess = None
        self._micp_path = None
        raise NotImplementedError('ERROR: Preprocessor is an abstract class')

    def do_preprocesing(self, target):
        """process the templates within the micp/ directory, it is expected
        that this directory in the same level as this script"""

        self._files_to_preprocess = ['README.txt', 'INSTALL.txt', 'INSTALL_WIN.txt',
                                     'micperf_faqs.txt']

        is_knllb_windows = platform.system() == 'Windows' and target == 'knllb'
        is_knc_windows = platform.system() == 'Windows' and target == 'knc'
        if is_knllb_windows or is_knc_windows:
            self._files_to_preprocess.append('micprun.py')
        else:
            self._files_to_preprocess.append('micprun')

        all_files = [os.path.join(self._micp_path, file_name)
                     for file_name in self._files_to_preprocess]

        for file_name in all_files:
            print "updating file", file_name
            self._update_file(file_name)

    def _update_file(self, file_name):
        """given an input file "file_name" replace all the placeholders
        with its correct value or delete the lines containing them"""
        for pattern, value in self._patterns_to_replace.items():
            print "file", file_name, "replacing", pattern, "with", value
            self.search_and_replace(file_name, pattern, value)

        for pattern in self._patterns_to_delete:
            print "file removing", pattern
            self.search_and_delete(file_name, pattern)

    @staticmethod
    def search_and_replace(input_file, pattern, value):
        """search instances of 'pattern' in every line of the given
        'input_file' and replace them by 'value'"""
        for line in fileinput.input(input_file, inplace=True):
            if pattern in line:
                line = line.replace(pattern, value)
            sys.stdout.write(line)

    @staticmethod
    def search_and_delete(input_file, pattern):
        """search and remove lines that contain 'pattern' in
        the given 'input_file'"""
        for line in fileinput.input(input_file, inplace=True):
            if pattern not in line:
                sys.stdout.write(line)


class KNCPreprocessor(Preprocessor):
    """Concrete implementation of Preprocessor for KNC (Linux & Windows)"""
    def __init__(self, version, docdir, micp_path):
        print "entering KNC preprocessor_factory"
        self._micp_path = micp_path
        self._patterns_to_delete = ()
        self._patterns_to_replace = {}

        self._patterns_to_replace[MPSSVERSION] = version
        self._patterns_to_replace[KNX_MICPERF_HELP] = KNC_HELP
        self._patterns_to_replace[README_KNX_PRODUCT] = KNC_PRODUCT_NAME
        self._patterns_to_replace[WIN_LIBEXEC_KEY] = WIN_LIBEXEC_VALUE
        self._patterns_to_replace[README_WINDOWS_TUTORIALS_PATH] = WIN_TUTORIALS_VALUE
        self._patterns_to_replace[README_WINDOWS_VISUAL_STUDIO] = WIN_VISUAL_STUDIO_VALUE
        self._patterns_to_replace[README_WINDOWS_EULA] = WINDOWS_EULA_VALUE
        self._patterns_to_replace[README_LINUX_EULA] = \
                KNC_LINUX_EULA_VALUE.format(docdir=docdir, version=version)

        print "to replace"
        print self._patterns_to_replace

class KNLSBPreprocessor(Preprocessor):
    """Concrete implementation of Preprocessor for KNL SB (Linux)"""
    def __init__(self, version, docdir, micp_path):
        self._micp_path = micp_path
        self._patterns_to_replace = {}
        self._patterns_to_delete = (WIN_LIBEXEC_KEY, README_WINDOWS_TUTORIALS_PATH,
                                    README_WINDOWS_VISUAL_STUDIO, README_WINDOWS_EULA)

        self._patterns_to_replace[MPSSVERSION] = version
        self._patterns_to_replace[KNX_MICPERF_HELP] = KNL_HELP
        self._patterns_to_replace[README_KNX_PRODUCT] = KNLSB_PRODUCT_NAME
        self._patterns_to_replace[README_LINUX_EULA] = \
                KNLSB_LINUX_EULA_VALUE.format(docdir=docdir, version=version)


class KNLLBPreprocessor(Preprocessor):
    """Concrete implementation of Preprocessor for KNL LB (Linux & Windows)"""
    def __init__(self, version, docdir, micp_path):
        self._micp_path = micp_path
        self._patterns_to_delete = ()
        self._patterns_to_replace = {}

        self._patterns_to_replace[MPSSVERSION] = version
        self._patterns_to_replace[KNX_MICPERF_HELP] = KNL_HELP
        self._patterns_to_replace[README_KNX_PRODUCT] = KNLLB_PRODUCT_NAME
        self._patterns_to_replace[WIN_LIBEXEC_KEY] = WIN_LIBEXEC_VALUE
        self._patterns_to_replace[README_WINDOWS_TUTORIALS_PATH] = WIN_TUTORIALS_VALUE
        self._patterns_to_replace[README_WINDOWS_VISUAL_STUDIO] = WIN_VISUAL_STUDIO_VALUE
        self._patterns_to_replace[README_WINDOWS_EULA] = WINDOWS_EULA_VALUE
        self._patterns_to_replace[README_LINUX_EULA] = \
                KNLLB_LINUX_EULA_VALUE.format(docdir=docdir, version=version)


def preprocessor_factory(name, version, docdir, micp_path):
    """Factory to create KNC/KNLB/KNLLB Preprocessor objects.
    Receives the name of kind of object to build, the MPSS
    version and the docdir path"""
    available_classes = {}
    available_classes['knc'] = KNCPreprocessor
    available_classes['knllb'] = KNLLBPreprocessor
    available_classes['knlsb'] = KNLSBPreprocessor

    if name not in available_classes:
        error_message = "ERROR: '{0}' is not a valid Preprocessor class."
        raise ValueError(error_message.format(name))

    return available_classes[name](version, docdir, micp_path)


################################################################################
#                                    main()
################################################################################


def linux_micp_version(micp_path, version, micperf_exec, micperf_data, card_arch, host_arch):
    """creates the configuration file version.py inside the directory
    micp_path/micp/. This method provides a generic template and requires:
        - version, MPSS version
        - micperf_exec, path where micperf workloads are installed
        - micperf_data, path where micperf golden pickle files are installed
        - card_arch, k1om or x86_64_AVX512
        - host_arch, x86_64 or x86_64_AVX512
    """
    micp_version_path = os.path.join(micp_path, 'micp', 'version.py')
    with open(micp_version_path, 'w') as fid:
        fid.write("__version__ = '{0}'\n".format(version))
        fid.write("MIC_PERF_EXEC = '{0}'\n".format(micperf_exec))
        fid.write("MIC_PERF_DATA = '{0}'\n".format(micperf_data))
        fid.write("MIC_PERF_CARD_ARCH = '{0}'\n".format(card_arch))
        fid.write("MIC_PERF_HOST_ARCH = '{0}'\n".format(host_arch))


def windows_micp_version(micp_path, version):
    """creates the configuration file version.py inside the directory
    micp_path\\micp\\. For knllb windows we only need the MPSS version.
    YOCTO creates the corresponding file for KNC."""
    micp_version_path = os.path.join(micp_path, 'micp', 'version.py')
    with open(micp_version_path, 'w') as fid:
        fid.write('import os\n')
        fid.write("__version__ = '{0}'\n".format(version))
        fid.write("MIC_PERF_EXEC = '{0}\\sdk\\micperf\\libexec'.format(os.environ.get('INTEL_MPSS_HOME','C:\\Program Files\\intel\\MPSS'))\n")
        fid.write("MIC_PERF_DATA = '{0}\\sdk\\micperf\\data'.format(os.environ.get('INTEL_MPSS_HOME', 'C:\\Program Files\\intel\\MPSS'))\n")
        fid.write("MIC_PERF_CARD_ARCH = 'x86_64_AVX512'\n")
        fid.write("MIC_PERF_HOST_ARCH = 'x86_64'\n")


def main():
    """main function, verifies user input arguments, pre-processes the
    micperf documentation and creates the micp/micp/version.py script"""
    if len(sys.argv) != 6:
        sys.stderr.write("ERROR: Invalid number of arguments\n")
        sys.stderr.write(__doc__)
        sys.exit(-1)

    micpdir = sys.argv[1]
    target = sys.argv[2]
    version = sys.argv[3]
    datadir = sys.argv[4]
    libexecdir = sys.argv[5]

    # validate we received a valid target
    is_valid_target = [target == good_target
                       for good_target in
                       ('knc', 'knllb', 'knlsb')]

    if not any(is_valid_target):
        sys.stderr.write("ERROR: Invalid target '{0}'\n".format(target))
        sys.stderr.write(__doc__)
        sys.exit(-1)

    # this script should be executed in the same directory where micp/ is located
    micp_directory = os.path.dirname(os.path.abspath(__file__))
    micp_directory = os.path.join(micp_directory, micpdir)

    if not os.path.exists(micp_directory):
        message = "ERROR: couldn't find the micp/ directory in the path '{0}'"
        raise RuntimeError(message.format(micp_directory))

    # create micp/micp/version.py
    # *** future improvement move this code to a class or a method ***
    if platform.system() == 'Linux':
        micperf_exec = os.path.join(libexecdir, 'micperf')
        micperf_data = os.path.join(datadir, 'micperf', 'data')

        if target == 'knlsb':
            linux_micp_version(
                micp_directory,
                version,
                micperf_exec,
                micperf_data,
                'NONE',
                'x86_64_AVX512')

        elif target == 'knllb':
            linux_micp_version(
                micp_directory,
                version,
                micperf_exec,
                micperf_data,
                'x86_64_AVX512',
                'x86_64')

        elif target == 'knc':
            linux_micp_version(
                micp_directory,
                version,
                micperf_exec,
                micperf_data,
                'k1om',
                'x86_64')
    else:
        # Yocto creates the Windows KNC micp/version.py script and adds the
        # '.py' extension to the micp/ scripts
        if target != 'knc':
            windows_micp_version(micp_directory, version)

            micp_scripts = ('micpcsv', 'micpinfo', 'micpplot', 'micpprint', 'micprun')
            micp_scripts = [os.path.join(micp_directory, script) for script in micp_scripts]
            for script in micp_scripts:
                try:
                    os.rename(script, '{0}.py'.format(script))
                except OSError:
                    pass

        else:
            print "micp/version.py won't be created nor micp/ scripts renamed"


    # update documentation README.txt, INSTALL.txt, etc.
    docdir = '{0}/doc'.format(datadir)
    preprocessor = preprocessor_factory(target, version, docdir, micp_directory)
    preprocessor.do_preprocesing(target)


if __name__ == "__main__":
    main()
