//===========================================================================
//
// This example from a prerelease of the Scalable HeterOgeneous Computing
// (SHOC) Benchmark Suite Alpha v1.1.1i for Intel MIC architecture
// Contact: Kyle Spafford <kys@ornl.gov>
//         Rezaur Rahman <rezaur.rahman@intel.com>
//
// Copyright (c) 2011, UT-Battelle, LLC
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//   
//  * Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
//  * Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//  * Neither the name of Oak Ridge National Laboratory, nor UT-Battelle, LLC, nor
//    the names of its contributors may be used to endorse or promote products
//    derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// ==============================================================================

#include <iostream>
#include <cstdio>
#include <cstdlib>
#include <cassert>
#include <cmath>
#include <vector>
#include <string>
#include "omp.h"

#ifndef _WIN32

#include "offload.h"
#define T_SLEEP(X)	sleep(X)
#else

#include <Windows.h>
#define T_SLEEP(X)	Sleep(1000*X)

#endif

#include "OptionParser.h"
#include "ResultDatabase.h"

#if defined  __MIC__
#include <pthread.h>
#endif


// Forward Declarations
#ifndef SCIF_SHOC
void RunBenchmark(OptionParser &op, ResultDatabase& resultDB);
#else
#include "scif.h"
void RunBenchmark(OptionParser &op, ResultDatabase& resultDB, scif_epd_t epd);
void addBenchmarkSpecOptions(OptionParser &op);
#endif

#ifndef SCIF_SHOC
int main(int argc, char **argv)
{

    OptionParser op;
    op.addOption("verbose", OPT_BOOL, "", "enable verbose output", 'v');
    op.addOption("passes", OPT_INT, "10", "specify number of passes", 'n');
    op.addOption("size", OPT_INT, "1", "specify problem size", 's');
    op.addOption("target", OPT_INT, "0", "specify MIC target device number", 't');
  
    if (!op.parse(argc, argv))
    {
        op.usage();
        return -1;
    }

    ResultDatabase resultDB;
    // Run the test
    RunBenchmark(op, resultDB);

    // Print out results to stdout
    resultDB.DumpDetailed(cout);

    return 0;
}
#else
int main(int argc, char *argv[])
{
    // Get args
    OptionParser op;

    //Add shared options to the parser
    op.addOption("verbose", OPT_BOOL, "", "enable verbose output", 'v');
    op.addOption("passes", OPT_INT, "3", "specify number of passes", 'n');
    op.addOption("target", OPT_INT, "0", "specify MIC target device number (0 based)", 't');
    op.addOption("size", OPT_INT, "1", "specify size of matrix for SGEMM", 's');


    addBenchmarkSpecOptions(op);

    if (!op.parse(argc, argv))
        return 0;

    ResultDatabase resultDB;

    int err, i;
    portID_t portID;
    scif_epd_t epd;
    int req_pn = 3011;
    int con_pn = 3010;

    //* Initialize the Device *//
    portID.node = (uint16_t)op.getOptionInt("target") + 1;
    portID.port = con_pn;

    //* Initialize the Device *//
    epd = scif_open();
    con_pn = scif_bind(epd, req_pn);
    /* We try to connect to the server and if the connection is refused we keep trying */
    /* in one second intervals, in a 30 seconds timeframe, for a successful connection */
    /* before failing the test. */
    for(i=0,err=-1; err<0 && i<15; i++) {
        err = scif_connect(epd, &portID);

        /* Check for a connection refused error*/
        if(err<0 && errno!=ECONNREFUSED) {
            printf("(Barrier) Error in scif_connect\n");
            return errno;
        } else if(errno==ECONNREFUSED) {
            printf("Connection unsuccessful. Attempt %i\n", i);
            /* Sleep for one second before attempting another connection */
            T_SLEEP(1);
        }
    }

    /* Check if a connection was successfully established */
    if(err<0) {
        printf("Client: (Barrier) Error in scif_connect: %i\n", errno);
        return errno;
    }

    // Run the benchmark
    RunBenchmark(op, resultDB, epd);
    resultDB.DumpDetailed(cout);

    scif_close(epd);
 
    return 0;
}
#endif //SCIF_SHOC
