# Copyright 2010-2017 Intel Corporation.
# 
# This library is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, version 2.1.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
# 
# Disclaimer: The codes contained in these modules may be specific
# to the Intel Software Development Platform codenamed Knights Ferry,
# and the Intel product codenamed Knights Corner, and are not backward
# compatible with other Intel products. Additionally, Intel will NOT
# support the codes or instruction set in future products.
# 
# Intel offers no warranty of any kind regarding the code. This code is
# licensed on an "AS IS" basis and Intel is not obligated to provide
# any support, assistance, installation, training, or other services
# of any kind. Intel is also not obligated to provide any updates,
# enhancements or extensions. Intel specifically disclaims any warranty
# of merchantability, non-infringement, fitness for any particular
# purpose, and any other warranty.
# 
# Further, Intel disclaims all liability of any kind, including but
# not limited to liability for infringement of any proprietary rights,
# relating to the use of the code, even if Intel is notified of the
# possibility of such liability. Except as expressly stated in an Intel
# license agreement provided with this code and agreed upon with Intel,
# no license, express or implied, by estoppel or otherwise, to any
# intellectual property rights is granted herein.

"""UT's for micp/kernel.py modules, since kernel.py provides
an abstract class Kernel this module implements DummyKernel"""

import pytest

import micp.kernel as micp_kernel
import micp.common as micp_common
import micp.version as micp_version


class DummyKernel(micp_kernel.Kernel):
    """Minimal implementation of abstract class Kernel
    to unit test methods provided by this class"""
    def __init__(self):
        self.name = "DummyKernel"
        self._paramNames = ['test_arg0', 'test_arg1']
        self._paramDefaults = {'test_arg0':'arg0', 'test_arg1':'arg1'}
        self._categoryParams = {'scaling': range(10)}


def test_help_default_message():
    """validate default help message return by kernel"""
    expected = '''Parameter help for kernel DummyKernel <param:default>:
<test_arg0:arg0> <test_arg1:arg1>
'''
    dummy_kernel = DummyKernel()
    actual = dummy_kernel.help()
    assert expected == actual


def test_help_custom_message():
    """validate custom help message return by kernel"""
    expected = '''Parameter help for kernel DummyKernel:
this is the help
'''
    dummy_kernel = DummyKernel()
    actual = dummy_kernel.help("this is the help")
    assert expected == actual


def test_path_exec(init_basic_knxlb_object):
    """validate kernel._path_exec() is able to determine if a kernel binary
    is in the system's PATH or MIC_PERF_EXEC and returns its location"""

    if micp_common.is_platform_windows():
        binary_name = 'sgemm_cpu.exe'
        expected = 'C:\\MICPERF\\libexec\\{0}\\sgemm_cpu.exe'.format(micp_version.MIC_PERF_HOST_ARCH)
    else:
        binary_name = 'sgemm_cpu.x'
        expected = '/usr/libexec/micperf/{0}/sgemm_cpu.x'.format(micp_version.MIC_PERF_HOST_ARCH)

    dummy_kernel = DummyKernel()
    kernel_binary_path = dummy_kernel._path_exec(micp_version.MIC_PERF_HOST_ARCH, binary_name)
    assert kernel_binary_path == expected


def test_path_exec_exception(init_basic_knxlb_object):
    """validate kernel._path_exec() throws an exception NoExecutableError
    when the kernel binary is not in the system's PATH nor MIC_PERF_EXEC"""
    dummy_kernel = DummyKernel()
    with pytest.raises(micp_kernel.NoExecutableError):
        dummy_kernel._path_exec('x86_64', 'invalid_binary')


def test_dummy_kernel_host_exec_path(init_basic_knxlb_object):
    """validate _host_exec_path() finds the kernel
    host side binary in the system"""
    if micp_common.is_platform_windows():
        expected = 'C:\\MICPERF\\libexec\\x86_64\\DummyKernel'
    else:
        expected = '/usr/libexec/micperf/x86_64/DummyKernel'

    dummy_kernel = DummyKernel()
    assert dummy_kernel.path_host_exec(None) == expected


def test_dummy_kernel_dev_exec_path(init_basic_knxlb_object):
    """validate _host_exec_path() finds the kernel
    card side binary in the system"""
    if micp_common.is_platform_windows():
        expected = 'C:\\MICPERF\\libexec\\{0}\\DummyKernel'
    else:
        expected = '/usr/libexec/micperf/{0}/DummyKernel'

    expected = expected.format(micp_version.MIC_PERF_CARD_ARCH)
    dummy_kernel = DummyKernel()
    assert dummy_kernel.path_dev_exec(None) == expected


def test_dummy_kernel_aux_path(init_basic_knxlb_object):
    """validates that for dummy kernel, aux_paths is an empty list"""
    dummy_kernel = DummyKernel()
    aux_paths = dummy_kernel.path_aux_data(None)
    assert isinstance(aux_paths, list)
    assert not aux_paths


@pytest.mark.parametrize("output, expected",
    [(" ...[ DESCRIPTION ] DummyKernel size AxB\n...", 'DummyKernel size AxB'),
     ('No Description avaliable','')])
def test_parse_desc(output, expected, init_basic_knxlb_object):
    """validate parse_desc() is able to parse the
    kernel description from the kernel output"""
    dummy_kernel = DummyKernel()
    desc = dummy_kernel.parse_desc(output)
    assert desc == expected


def test_parse_perf(init_basic_knxlb_object):
    """validate parse_perf() is able to parse the kernel
    perfomance results from the kernel output"""
    output = " ... [ PERFORMANCE ] tag value units R ... "
    dummy_kernel = DummyKernel()
    perf = dummy_kernel.parse_perf(output)
    assert 'tag' in perf

    data = perf['tag']
    assert data['value'] == 'value'
    assert data['units'] == 'units'
    assert data['rollup']


def test_parse_perf_multiple(init_basic_knxlb_object):
    """validate parse_perf() is able to parse the kernel perfomance
    results from the kernel output for multiple runs"""
    output = (" ... [ PERFORMANCE ] tag0 value0 units0 R ...\n"
              " ... [ PERFORMANCE ] tag1 value1 units1 ... ")
    dummy_kernel = DummyKernel()
    perf = dummy_kernel.parse_perf(output)
    assert 'tag0' in perf and 'tag1' in perf

    data = perf['tag0']
    assert data['value'] == 'value0'
    assert data['units'] == 'units0'
    assert data['rollup']

    data = perf['tag1']
    assert data['value'] == 'value1'
    assert data['units'] == 'units1'
    assert not data['rollup']


def test_parse_perf_exception(init_basic_knxlb_object, monkeypatch):
    """validate parse_perf() raises and exception for kernels that support 
    'internal_scaling' by default this setting is off, derived classes should
    implement it if required."""
    monkeypatch.setattr(
        micp_kernel.Kernel, 'internal_scaling',
        lambda self:
            True
    )
    dummy_kernel = DummyKernel()

    with pytest.raises(NotImplementedError):
        dummy_kernel.parse_perf("")


@pytest.mark.parametrize("is_full", [True, False])
def test_param_names(is_full, init_basic_knxlb_object):
    """validate param_names() returns a list with the
    names of the parameters used by the kernel"""
    dummy_kernel = DummyKernel()
    param_names = dummy_kernel.param_names(full=is_full)

    assert param_names == ['test_arg0', 'test_arg1']


@pytest.mark.parametrize("library_name, win_expected, linux_expected", [
    ('libiomp5.so', 'C:\\lib\\mic\\libiomp5.so', '/lib/mic/libiomp5.so'),
    ('libabc.so', 'C:\\lib\\mic\\libabc.so', '/lib/mic/libabc.so')
])
def test_mic_library_find(library_name, win_expected,
                          linux_expected, init_basic_knxlb_object):
    """validates mic_library_find() is able to find a given shared library
    in any of the paths included in MIC_LD_LIBRARY_PATH"""
    if micp_common.is_platform_windows():
        expected = win_expected
    else:
        expected = linux_expected

    dummy_kernel = DummyKernel()
    path_to_library = dummy_kernel.mic_library_find(library_name)
    assert path_to_library == expected


def test_mic_library_find_nonexistent(init_basic_knxlb_object):
    """validates mic_library_find() throws an exception when it is
    unable to find a given shared library in any of the paths included
    in MIC_LD_LIBRARY_PATH"""
    dummy_kernel = DummyKernel()
    with pytest.raises(micp_common.MissingDependenciesError):
        dummy_kernel.mic_library_find('libnonexistent.so')


def test_mic_ld_library_path(init_basic_knxlb_object):
    """validate mic_ld_library_path() gets the card's
    enviroment variable MIC_LD_LIBRARY_PATH and appends
    the path where COI and MYO shared libraries are kept"""
    if micp_common.is_platform_windows():
        expected = ('C:\\lib\\mic;C:\\Users\\lib\\mic;'
                    'C:\\Program Files\\Intel\\MPSS\\sdk\\lib')
    else:
        expected = '/lib/mic:/usr/lib/mic'

    dummy_kernel = DummyKernel()
    mic_shared_libraries = dummy_kernel.mic_ld_library_path()
    assert mic_shared_libraries == expected


def test_ld_library_path(init_basic_knxlb_object):
    """validate ld_library_path() gets the host's
    enviroment variable LD_LIBRARY_PATH"""
    if micp_common.is_platform_windows():
        expected = 'C:\\lib\\intel64;C:\\Users\\lib\\intel64'
    else:
        expected = '/lib/intel64:/usr/lib/intel64'

    dummy_kernel = DummyKernel()
    shared_libraries = dummy_kernel.ld_library_path()
    assert shared_libraries == expected
