# Copyright 2010-2017 Intel Corporation.
# 
# This library is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, version 2.1.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
# 
# Disclaimer: The codes contained in these modules may be specific
# to the Intel Software Development Platform codenamed Knights Ferry,
# and the Intel product codenamed Knights Corner, and are not backward
# compatible with other Intel products. Additionally, Intel will NOT
# support the codes or instruction set in future products.
# 
# Intel offers no warranty of any kind regarding the code. This code is
# licensed on an "AS IS" basis and Intel is not obligated to provide
# any support, assistance, installation, training, or other services
# of any kind. Intel is also not obligated to provide any updates,
# enhancements or extensions. Intel specifically disclaims any warranty
# of merchantability, non-infringement, fitness for any particular
# purpose, and any other warranty.
# 
# Further, Intel disclaims all liability of any kind, including but
# not limited to liability for infringement of any proprietary rights,
# relating to the use of the code, even if Intel is notified of the
# possibility of such liability. Except as expressly stated in an Intel
# license agreement provided with this code and agreed upon with Intel,
# no license, express or implied, by estoppel or otherwise, to any
# intellectual property rights is granted herein.

"""Generic class for testing kernels"""

import pytest
import re
import os

import common_utils

import micp.info as micp_info
import micp.common as micp_common
import micp.kernel as micp_kernel
import micp.version as micp_version
import micp.kernels.libxsmm_conv as micp_kernels_libxsmm_conv
import micp.kernels.mkl_conv as micp_kernels_mkl_conv
import micp.kernels.fio as micp_kernels_fio
from micp.kernel import NoExecutableError
from micp_kernels_test_data import *


## Those generic tests call basic methods of kernel.py class API,
## the output is then compared to fixed expected result stored in
## container kernels_test_data and file micp_kernels_test_data.py
## This class test only subset of kernels mentioned in table kernels_test_data
## Note that meaning of kernel.py API is explained in kernel.py

## below indexes define meaning of data in kernels_test_data container
## for more details refer to kernel.py
# supported offloads
D_KERNEL_OFFLOADS = 0
# kernel command line parameter type
D_PARAM_TYPE = 1
# independent var for scaling plot
D_INDEPENDENT_VAR = 2
# shell OS variables pass to kernel
D_PARAM_FOR_ENV = 3
# environmental vars for kernel
D_ENVIRONMENT_HOST = 4
# kernel output when no error occurred
D_OUT_GOOD = 5
# expected result of parsing D_OUT_GOOD (description)
D_OUT_GOOD_DESC = 6
# expected result of parsing D_OUT_GOOD (performance)
D_OUT_GOOD_PERF = 7

## defines expected results of kernel.py API calls
kernels_test_data = {

    'libxsmm_conv': [
        ['local'],
        'pos',
        {None:'omp_num_threads'},
        ['omp_num_threads'],
        {'LD_LIBRARY_PATH':'stub', 'KMP_PLACE_THREADS':'1T',
            'KMP_AFFINITY':'compact,granularity=fine'},
        [libxsmm_layer_out_good],
        [libxsmm_layer_out_good_desc],
        [libxsmm_layer_out_good_perf]],

    'mkl_conv': [
        ['local'],
        'pos',
        {None:'omp_num_threads'},
        ['omp_num_threads'],
        {'LD_LIBRARY_PATH':'stub', 'KMP_PLACE_THREADS':'1T',
            'KMP_AFFINITY':'compact,granularity=fine'},
        [std_conv_bench_out_good],
        [std_conv_bench_out_good_desc],
        [std_conv_bench_out_good_perf]],

    'fio': [
        ['local'],
        'file',
        {'scaling_core':'numjobs','scaling':'size'},
        [],
        {},
        [fio_out_good_1, fio_out_good_2],
        [fio_out_good_desc_1, fio_out_good_desc_2],
        [fio_out_good_perf_1, fio_out_good_perf_2]]
    }

def pytest_generate_tests(metafunc):
    """ test generator generates tests for all kernels from
    kernels_test_data container and for all offloads"""
    if 'kernel_name' in metafunc.fixturenames:
        metafunc.parametrize("kernel_name",
                [kname for kname in kernels_test_data])
    if 'offload_method' in metafunc.fixturenames:
        metafunc.parametrize('offload_method', ['local', 'scif',
                'pragma', 'coi', 'myo', 'native'])

## mocks ##
###########

@pytest.fixture
def info_mock(monkeypatch, request, init_basic_knxlb_object):
    """monkypatches to mock config if needed"""
    return None

class ParamsMock(object):
    """class to simluate Params class, for details refer to params.py"""
    def get_named(self, *args, **kwargs):
        """ for the purpose of unit testing just mock all parameters with
        dummy value"""
        return 'DUMMY_VALUE'

## tests ##
###########

def test_kernel_string_name(info_mock, kernel_name):
    """validate kernel object is created successfully
    and its name is as expected"""
    kernel_class = eval("micp_kernels_{0}.{0}()".format(kernel_name))
    assert kernel_class.name == kernel_name

def test_kernel_path_dev_exec(offload_method, info_mock, kernel_name):
    """checks if path_dev_exec() API returns None for non-supported
    offload method"""
    kernel_class = eval("micp_kernels_{0}.{0}()".format(kernel_name))
    path_to_binary = kernel_class.path_dev_exec(offload_method)
    assert path_to_binary is None

def test_kernel_path_host_exec(offload_method, info_mock, kernel_name):
    """checks if path_host_exec() API returns None for non-supported
    offload method """
    kernel_class = eval("micp_kernels_{0}.{0}()".format(kernel_name))
    if offload_method in kernels_test_data[kernel_name][D_KERNEL_OFFLOADS]:
        try:
            path_to_binary = kernel_class.path_host_exec(offload_method)
        except NoExecutableError:
            # expected, this test does not necessarily run on machine
            # with micperf installed
            assert True
    else:
        path_to_binary = kernel_class.path_host_exec(offload_method)
        assert path_to_binary is None


def test_offload_methods(info_mock, kernel_name):
    """checks if offload_methods() API returns the list containing
    supported offload methods """
    kernel_class = eval("micp_kernels_{0}.{0}()".format(kernel_name))
    assert set(kernel_class.offload_methods()) == \
        set(kernels_test_data[kernel_name][D_KERNEL_OFFLOADS])


def test_param_type(info_mock, kernel_name):
    """checks if param_type() API returns expected type"""
    kernel_class = eval("micp_kernels_{0}.{0}()".format(kernel_name))
    assert kernel_class.param_type() == \
        kernels_test_data[kernel_name][D_PARAM_TYPE]

def test_independent_var(info_mock, kernel_name):
    """checks if independent_var() returns expected variable"""
    kernel_class = eval("micp_kernels_{0}.{0}()".format(kernel_name))
    for category in kernels_test_data[kernel_name][D_INDEPENDENT_VAR]:
        assert kernel_class.independent_var(category) == \
            kernels_test_data[kernel_name][D_INDEPENDENT_VAR][category], \
            "For category: {0}".format(category)

def test_param_for_env(info_mock, kernel_name):
    """checks if param_for_env() API returns expected variables"""
    kernel_class = eval("micp_kernels_{0}.{0}()".format(kernel_name))
    assert kernel_class.param_for_env() == \
        kernels_test_data[kernel_name][D_PARAM_FOR_ENV]

def test_environment_host(info_mock, kernel_name):
    """checks if environment_host() API returns expected variables"""
    library_path = common_utils.get_host_library_path()
    kernel_class = eval("micp_kernels_{0}.{0}()".format(kernel_name))
    expected_variables = kernels_test_data[kernel_name][D_ENVIRONMENT_HOST]
    # LD_LIBRARY_PATH is a special case it's value is not fixed and
    # kernel specific, it has to be obtained from common_utils
    if 'LD_LIBRARY_PATH' in expected_variables:
        expected_variables['LD_LIBRARY_PATH'] = library_path
    assert kernel_class.environment_host() == expected_variables

def test_parse_dec(info_mock, kernel_name):
    """checks if result of parsing fixed input by parse_desc() API matches
    expected result"""
    kernel_class = eval("micp_kernels_{0}.{0}()".format(kernel_name))
    for idx, sample_out in enumerate(kernels_test_data[kernel_name][D_OUT_GOOD]):
        description = kernel_class.parse_desc(sample_out)
        assert description == kernels_test_data[kernel_name][D_OUT_GOOD_DESC][idx]

def test_parse_dec_raises_exception(info_mock, kernel_name):
    """checks if parse_desc() API raises exception in the event of malformed
    input"""
    kernel_class = eval("micp_kernels_{0}.{0}()".format(kernel_name))
    with pytest.raises(micp_kernel.SelfCheckError):
        kernel_class.parse_desc('bad_input')

def test_parse_perf(info_mock, kernel_name):
    """checks if result of parsing fixed input by parse_perf() API matches
    expected result"""
    kernel_class = eval("micp_kernels_{0}.{0}()".format(kernel_name))
    for idx, sample_out in enumerate(kernels_test_data[kernel_name][D_OUT_GOOD]):
        # first parse_desc since some info may be already parsed there
        kernel_class.parse_desc(sample_out)
        kernel_result = kernel_class.parse_perf(sample_out)
        assert kernel_result == kernels_test_data[kernel_name][D_OUT_GOOD_PERF][idx]

def test_parse_perf_raises_exception(info_mock, kernel_name):
    """checks if parse_perf() API raises exception in the event of malformed
    input"""
    kernel_class = eval("micp_kernels_{0}.{0}()".format(kernel_name))
    with pytest.raises(micp_kernel.SelfCheckError):
        kernel_class.parse_perf('bad_input')

def test_config_file(kernel_name):
    """validates if param_file() creates config file and if it is
    properly removed by clean up routine"""
    kernel_class = eval("micp_kernels_{0}.{0}()".format(kernel_name))
    config_file_path = kernel_class.param_file(ParamsMock())
    param_type = kernel_class.param_type()

    # check if kernel creates config file for kernels
    # that declare file config support
    if param_type is 'file':
        assert os.path.exists(config_file_path)
        kernel_class.clean_up(None, None)
        assert not os.path.exists(config_file_path)
    else:
        assert not config_file_path
