# Copyright 2010-2017 Intel Corporation.
# 
# This library is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, version 2.1.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
# 
# Disclaimer: The codes contained in these modules may be specific
# to the Intel Software Development Platform codenamed Knights Ferry,
# and the Intel product codenamed Knights Corner, and are not backward
# compatible with other Intel products. Additionally, Intel will NOT
# support the codes or instruction set in future products.
# 
# Intel offers no warranty of any kind regarding the code. This code is
# licensed on an "AS IS" basis and Intel is not obligated to provide
# any support, assistance, installation, training, or other services
# of any kind. Intel is also not obligated to provide any updates,
# enhancements or extensions. Intel specifically disclaims any warranty
# of merchantability, non-infringement, fitness for any particular
# purpose, and any other warranty.
# 
# Further, Intel disclaims all liability of any kind, including but
# not limited to liability for infringement of any proprietary rights,
# relating to the use of the code, even if Intel is notified of the
# possibility of such liability. Except as expressly stated in an Intel
# license agreement provided with this code and agreed upon with Intel,
# no license, express or implied, by estoppel or otherwise, to any
# intellectual property rights is granted herein.

"""UT's for micp/kernels/hpcg.py"""

import pytest
import os
import shutil
import copy

import common_utils

import micp.info as micp_info
import micp.common as micp_common
import micp.version as micp_version
import micp.kernels.hpcg as micp_kernels_hpcg
import micp.kernel as micp_kernel
import micp.params as micp_params


#******************************************************************************
#                     mocks and additional infrastructure
#******************************************************************************


# memories sizes selected to cover different branches in micp/kernels/hpcg.py
MEM_8GB = 8*1024**3
MEM_16GB = 16*1024**3

# Available "CPUs" for testing
CPU72_CORES_16GB = (MEM_16GB, 72)
CPU68_CORES_16GB = (MEM_16GB, 68)
CPU68_CORES_8GB = (MEM_8GB, 68)

HPCG_OUTPUT_KEYS = ['nx', 'ny', 'nz', 'Distributed Processes', 'Threads per processes']


@pytest.fixture(params=[CPU72_CORES_16GB, CPU68_CORES_16GB, CPU68_CORES_8GB])
def info_mock(monkeypatch, request, init_basic_knxlb_object):
    """creates InfoKNXLB objects for windows and for linux, then
    monkypatches get_device_index(), num_cores() and mic_memory_size
    using the parameter provided to this fixture to test this kernel
    with different 'devices' (number of cores and memory size).
    Returns the tuple (mem_size, cores) so tests can know which
    configuration is being used"""
    mem_size, cores = request.param
    monkeypatch.setattr(micp_info.Info, 'get_device_index', lambda self: -1)
    monkeypatch.setattr(micp_info.Info, 'num_cores', lambda self: cores)
    monkeypatch.setattr(micp_info.Info, 'mic_memory_size', lambda self: mem_size)

    return mem_size, cores


@pytest.fixture
def generic_cpu_mock(monkeypatch):
    """creates a generic 'CPU' for testing purposes, the difference between this
    mock and info_mock is that generic_system_mock only fakes the number of cores
    and the memory size"""
    monkeypatch.setattr(micp_info.Info, 'num_cores', lambda self: 68)
    monkeypatch.setattr(micp_info.Info, 'mic_memory_size', lambda self: 16*1024**3)


class ParamsMock(object):
    """class to simluate Params"""
    def get_named(self, *args, **kwargs):
       return 'DUMMY_VALUE'


#******************************************************************************
#                                    tests
#******************************************************************************
def test_hpcg_init(info_mock):
    """validate hpcg kernel object is created successfully
    and its name is set correctly"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()
    assert hpcg_kernel.name == 'hpcg'


def test_scaling_parameters(info_mock):
    """validate how scaling parameters (scaling, scaling_quick, optimal_quick,
    etc.) are initialized depending on the of memory available"""

    hpcg_kernel = micp_kernels_hpcg.hpcg()

    # expected parameters depend on the memory available
    mem_size = micp_info.Info().mic_memory_size()
    if mem_size > MEM_8GB:
        max_problem = 160
    else:
        max_problem = 128

    # For HPCG optimal performance --omp_num_threads 32
    args = '--problem_size {0} --time 60 --omp_num_threads {1}'

    # problem sizes should be multiples of 32
    valid_problems = range(32, max_problem+32, 32)

    expected_scaling = [args.format(problem, 32) for problem in valid_problems]
    expected_scaling_quick = expected_scaling[:]
    expected_optimal_quick = [expected_scaling[-1]]

    # once again 32 threads yields the best HPCG performance that's the max
    expected_scaling_core = [args.format(max_problem, _threads)
                             for _threads in range(2, 32+2, 2)]

    # tests
    assert hpcg_kernel._categoryParams['scaling'] == expected_scaling
    assert hpcg_kernel._categoryParams['scaling_quick'] ==  expected_scaling_quick
    assert hpcg_kernel._categoryParams['optimal_quick'] ==  expected_optimal_quick
    assert hpcg_kernel._categoryParams['scaling_core'] == expected_scaling_core


def test_default_params(info_mock):
    """validate hpcg default parameters, they depend on the memory available"""

    hpcg_kernel = micp_kernels_hpcg.hpcg()

    # expected parameters depend on the memory available
    mem_size = micp_info.Info().mic_memory_size()
    if mem_size > MEM_8GB:
        max_problem = '160'
    else:
        max_problem = '128'

    expected = {'time':'60', 'problem_size':max_problem, 'omp_num_threads':'32'}
    assert hpcg_kernel._paramDefaults == expected


def test_unit_test(info_mock):
    """HPCG reports self-check its results, validate _do_unit_test() returns
    always True indicating the existence of such self-check"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()
    assert hpcg_kernel._do_unit_test()


def test_offload_methods(info_mock):
    """validate offload_methods() returns 'local' as the only offload method
    support"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()
    assert hpcg_kernel.offload_methods() == ['local']


def test_path_dev_exec(info_mock):
    """no binaries for the coprocessor path_dev_exec() should return None"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()
    assert hpcg_kernel.path_dev_exec('local') is None


def test_path_aux_data(info_mock):
    """No additional dependencies (e.g. libiomp5.so) so test_path_aux_data()
    should return an empty list"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()
    assert hpcg_kernel.path_aux_data('local') == []


def test_param_type(info_mock):
    """validate param_type() returns 'file'"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()
    assert hpcg_kernel.param_type() == 'file'


@pytest.mark.parametrize("offload_method", ['myo', 'coi', 'scif', 'pragma', 'auto'])
def test_path_host_exec_invalid_offload(offload_method, info_mock):
    """validate path_host_exec() returns None when the offload method is not 'local'"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()
    assert hpcg_kernel.path_host_exec(offload_method) is None


def test_path_host_exec_on_windows(info_mock, monkeypatch):
    """validate path_host_exec() raises an OSError exception when the OS is
    windows, as this OS is not supporte yet"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()
    monkeypatch.setattr(micp_common, 'is_platform_windows', lambda: True)

    with pytest.raises(OSError):
        hpcg_kernel.path_host_exec('local')


def test_path_host_exec_no_mklroot(info_mock, monkeypatch):
    """validate path_host_exec() raises an micp_kernel.NoExecutableError when
    the enviroment variable MKLROOT is not set"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()
    monkeypatch.setattr(micp_common, 'is_platform_windows', lambda: False)
    monkeypatch.setattr(os, 'environ', {})

    with pytest.raises(micp_kernel.NoExecutableError):
        hpcg_kernel.path_host_exec('local')


def test_path_host_exec_happy_path(info_mock, monkeypatch):
    """validate path_host_exec() returns the path the hpcg binary when MKLROOT
    is set"""

    expected_path = '/path/to/mkl/benchmarks/hpcg'
    hpcg_kernel = micp_kernels_hpcg.hpcg()
    monkeypatch.setattr(micp_common, 'is_platform_windows', lambda: False)
    monkeypatch.setattr(os, 'environ', {'MKLROOT':'/path/to/mkl/benchmarks'})
    monkeypatch.setattr(micp_kernels_hpcg.hpcg, '_search_path_to_file',
                        lambda arg0, arg1, arg2: expected_path)

    actual_path = hpcg_kernel.path_host_exec('local')
    assert actual_path == expected_path


def test_config_file(generic_cpu_mock):
    """Validates the configuration file 'life cycle'"""

    expected = """HPCG benchmark input file
Sandia National Laboratories; University of Tennessee, Knoxville
DUMMY_VALUE DUMMY_VALUE DUMMY_VALUE
DUMMY_VALUE"""


    hpcg_kernel = micp_kernels_hpcg.hpcg()
    config_file_path = hpcg_kernel.param_file(ParamsMock())
    config_file_dir = os.path.dirname(config_file_path)

    # read configuration file
    with open(config_file_path) as fh:
        config_file_content = fh.read()

    # tests
    assert os.path.exists(hpcg_kernel.get_working_directory())
    assert config_file_dir == hpcg_kernel.get_working_directory()
    assert config_file_content == expected

    # test clean up
    shutil.rmtree(config_file_dir)


def test_process_modifiers_mcdram_available(info_mock, monkeypatch):
    """validate the behavior of get_process_modifiers() when MCDRAM memory is
    available"""
    monkeypatch.setattr(micp_info.Info, 'is_processor_mcdram_available', lambda __: True)
    hpcg_kernel = micp_kernels_hpcg.hpcg()

    expected = ['numactl', '--membind=1', 'mpirun', '-n', '4']
    assert expected == hpcg_kernel.get_process_modifiers()


def test_process_modifiers_no_available(info_mock, monkeypatch):
    """validate the behavior of get_process_modifiers() when MCDRAM memory is NOT
    available"""
    monkeypatch.setattr(micp_info.Info, 'is_processor_mcdram_available', lambda __: False)
    hpcg_kernel = micp_kernels_hpcg.hpcg()

    expected = ['mpirun', '-n', '4']
    assert expected == hpcg_kernel.get_process_modifiers()


@pytest.mark.parametrize("category, expected", [('scaling_core', 'omp_num_threads'),
                                                ('scaling', 'problem_size'),
                                                ('test', 'problem_size'),
                                                ('scaling_quick', 'problem_size'),
                                                ('optimal', 'problem_size')])
def test_independent_var(info_mock, category, expected):
    """independent_var() should only return 'omp_num_threads' when the parameters
    category is scaling_core"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()
    assert hpcg_kernel.independent_var(category) == expected


def test_enviroment_host_knlsb(info_mock, monkeypatch):
    """validate environment variables to be set on the KNL SB host"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()
    monkeypatch.setattr(micp_common, 'is_selfboot_platform', lambda: True)

    expected = {'OMP_NUM_THREADS':'32',
                'KMP_AFFINITY':'granularity=fine,balanced',
                'KMP_HW_SUBSET':'16c,2t'}

    assert expected == hpcg_kernel.environment_host()


def test_enviroment_host_knllb(info_mock, monkeypatch):
    """validate environment variables to be set on the KNL card"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()
    monkeypatch.setattr(micp_common, 'is_selfboot_platform', lambda: False)

    expected = {}
    assert expected == hpcg_kernel.environment_host()


def test_environment_dev(info_mock):
    """validate no additional environment variables are set"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()
    assert hpcg_kernel.environment_dev() == {}


def test_parse_output_no_results_file(generic_cpu_mock):
    """validate _parse_hpcg_output() raises an micp_kernel.SelfCheckError
    exception when the configuration file is missing"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()
    hpcg_kernel.param_file(ParamsMock())

    with pytest.raises(micp_kernel.SelfCheckError):
        hpcg_kernel._parse_hpcg_output()

    shutil.rmtree(hpcg_kernel.get_working_directory())


def test_parse_output_happy_path(generic_cpu_mock, capsys):
    """validate _parse_hpcg_output() happy path"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()
    hpcg_kernel.param_file(ParamsMock())

    # create a dummy results file, file name resembles the sintax used by hpcg
    results_file_name = 'n160-4p-32t-color-2-4-2.4_2016.04.28.20.36.20.yaml'
    results_file_path = hpcg_kernel.get_working_directory()
    results_file_path = os.path.join(results_file_path, results_file_name)

    with open(results_file_path, 'w') as fh:
        fh.write("nx : 160\n")
        fh.write("ny : 160\n")
        fh.write("nz : 160\n")
        fh.write("Distributed Processes : 4\n")
        fh.write("Threads per processes : 32\n")
        fh.write('HPCG result is VALID with a GFLOP/s rating of : 50 GFlops\n')

    expected_output = """For HPCG execution details please refer to the logs:

    Log Directory: {0}
    HPCG current test log: {1}


""".format(hpcg_kernel.get_working_directory(), results_file_path)

    # test 1: first time _parse_hpcg_output() is called is should parse the file
    # and print a message to stdout, also internal members should be initialized.
    assert hpcg_kernel._hpcg_output_info == {}
    assert hpcg_kernel._lastest_hpcg_log is None

    hpcg_kernel._parse_hpcg_output()
    stdout, stderr = capsys.readouterr()

    assert not stderr
    assert stdout == expected_output
    assert hpcg_kernel._hpcg_output_info
    assert hpcg_kernel._lastest_hpcg_log

    # test 2: call _parse_hpcg_output() a second time, not output is expected
    # this time as the file should only be parsed once, the object's internal
    # state remains unchanged
    initial_hpcg_output_info = copy.deepcopy(hpcg_kernel._hpcg_output_info)
    initial_lastest_hpcg_log = copy.deepcopy(hpcg_kernel._lastest_hpcg_log)

    hpcg_kernel._parse_hpcg_output()
    stdout, stderr = capsys.readouterr()
    assert not stderr
    assert not stdout
    assert initial_hpcg_output_info == hpcg_kernel._hpcg_output_info
    assert initial_lastest_hpcg_log == hpcg_kernel._lastest_hpcg_log

    # clean up
    shutil.rmtree(hpcg_kernel.get_working_directory())



@pytest.mark.parametrize("missing", HPCG_OUTPUT_KEYS)
def test_parse_desc_negative_missing_info(generic_cpu_mock, monkeypatch, missing):
    """validate parse_desc() raises a micp_kernel.SelfCheckError exception when
    it is unable to find nx, ny, nz, 'Distributed Processes' or 'Threads per
    processes' in HPCG's output"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()

    # mocks
    monkeypatch.setattr(micp_kernels_hpcg.hpcg, '_parse_hpcg_output', lambda __: None)

    mocked_hpcg_output_info = {}
    for k in HPCG_OUTPUT_KEYS:
        if k != missing:
            mocked_hpcg_output_info[k] = 1
    hpcg_kernel._hpcg_output_info = mocked_hpcg_output_info

    with pytest.raises(micp_kernel.SelfCheckError):
        hpcg_kernel.parse_desc(None)



def test_parse_desc_happy_path(generic_cpu_mock, monkeypatch):
    """validate parse_desc() happy path"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()

    # mocks
    monkeypatch.setattr(micp_kernels_hpcg.hpcg, '_parse_hpcg_output', lambda __: None)

    mocked_hpcg_output_info = {}
    for k in HPCG_OUTPUT_KEYS:
        mocked_hpcg_output_info[k] = 1
    hpcg_kernel._hpcg_output_info = mocked_hpcg_output_info

    expected = 'hpcg Local Dimensions nx=1, ny=1, nz=1, MPI ranks 1, threads per rank 1'
    assert expected == hpcg_kernel.parse_desc(None)


def test_parse_perf_negative(generic_cpu_mock, monkeypatch):
    """validate parse_perf() raises a micp_kernel.SelfCheckError exception when
    unable to parse performance"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()

    # mocks
    monkeypatch.setattr(micp_kernels_hpcg.hpcg, '_parse_hpcg_output', lambda __: None)
    hpcg_kernel._hpcg_output_info = {}

    # run test
    with pytest.raises(micp_kernel.SelfCheckError):
       hpcg_kernel.parse_perf(None)


def test_parse_perf_happy_path(generic_cpu_mock, monkeypatch):
    """validate parse_perf() happy path"""
    hpcg_kernel = micp_kernels_hpcg.hpcg()

    # mocks
    monkeypatch.setattr(micp_kernels_hpcg.hpcg, '_parse_hpcg_output', lambda __: None)
    hpcg_kernel._hpcg_output_info = {'HPCG result is VALID with a GFLOP/s rating of' : '50'}

    expected = {}
    expected['Computation.Avg'] = {'value':'50', 'units':'GFlops', 'rollup':True}

    assert expected == hpcg_kernel.parse_perf(None)
