# Copyright 2010-2017 Intel Corporation.
# 
# This library is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, version 2.1.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
# 
# Disclaimer: The codes contained in these modules may be specific
# to the Intel Software Development Platform codenamed Knights Ferry,
# and the Intel product codenamed Knights Corner, and are not backward
# compatible with other Intel products. Additionally, Intel will NOT
# support the codes or instruction set in future products.
# 
# Intel offers no warranty of any kind regarding the code. This code is
# licensed on an "AS IS" basis and Intel is not obligated to provide
# any support, assistance, installation, training, or other services
# of any kind. Intel is also not obligated to provide any updates,
# enhancements or extensions. Intel specifically disclaims any warranty
# of merchantability, non-infringement, fitness for any particular
# purpose, and any other warranty.
# 
# Further, Intel disclaims all liability of any kind, including but
# not limited to liability for infringement of any proprietary rights,
# relating to the use of the code, even if Intel is notified of the
# possibility of such liability. Except as expressly stated in an Intel
# license agreement provided with this code and agreed upon with Intel,
# no license, express or implied, by estoppel or otherwise, to any
# intellectual property rights is granted herein.

"""UT's for micp/offload.py"""

import pytest
import os

import micp.offload as micp_offload
from micp.kernel import NoExecutableError
import micp.kernels.sgemm as micp_kernels_sgemm
import micp.kernels.dgemm as micp_kernels_dgemm
import micp.kernels.linpack as micp_linpack
import micp.params as micp_params
import micp.connect as micp_connect
import micp.common as micp_common

# micp/micp/kernels/shoc_readback.py won't be available for KNLSB
# shoc_readback tests can be skipped
try:
    import micp.kernels.shoc_readback as micp_shoc_readback
    SKIP_SHOC_TESTS = False
except ImportError:
    SKIP_SHOC_TESTS = True

import mocks.data as mocks_data


@pytest.mark.parametrize('offload_name, offload_class, on_host, on_dev', [
    ('myo', micp_offload.MYOOffload, True, True),
    ('native', micp_offload.NativeOffload, False, True),
    ('scif', micp_offload.SCIFOffload, True, True),
    ('pragma', micp_offload.PragmaOffload, True, False),
    ('auto', micp_offload.AutoOffload, True, False),
    ('local', micp_offload.LocalOffload, True, False),
    ('linux_native', micp_offload.NativeOffload, False, True)
])
def test_factory_instances(offload_name, offload_class, on_host, on_dev):
    """validate factory creates objects of the right type"""
    offload_factory = micp_offload.OffloadFactory()
    my_offload = offload_factory.create(offload_name)

    # special case
    if offload_name == 'linux_native':
        offload_name = 'native'

    assert isinstance(my_offload, offload_class)
    assert my_offload.name == offload_name
    assert my_offload._runDev == on_dev
    assert my_offload._runHost == on_host


def _init_kernel_params(kernel):
    """given a kernel, creates a list of micp_params.Params objects,
    every object in the list represents an instance of the kernel to
    be executed. The category of parameters is 'test'"""
    kernel_args = kernel.category_params('test')
    param_names = kernel.param_names(full=True)
    param_defaults = kernel.param_defaults()
    params_for_env = kernel.param_for_env()

    kernel_params = [
        micp_params.Params(args, param_names, param_defaults, params_for_env)
        for args in kernel_args]

    return kernel_params


@pytest.fixture()
def kernel_run_parameters(init_basic_knxlb_object):
    """fixture to created a sgemm kernel and parameters required to
    run the kernel in native mode, the kernel is not importat many
    times the object's behavior is monkeypatched to ease the UT
    development"""
    sgemm = micp_kernels_sgemm.sgemm()
    params = _init_kernel_params(sgemm)
    return sgemm, params


@pytest.fixture()
def local_offload_fixture(kernel_run_parameters):
    """fixture of offload.LocalOffload() to validate the run() method
    in the base class, class attributes will be monkeypatched as needed
    cover all the cases"""
    return micp_offload.OffloadFactory().create('local')


def test_offload_abstract_class():
    """validate micp.offload.Offload() is an abstract class"""
    with pytest.raises(NotImplementedError):
        micp_offload.Offload()


def test_native_offload_custom_run_exception(monkeypatch, kernel_run_parameters):
    """validate custom run() method can handle NotAttributeExceptions"""
    device_index = 1
    sgemm, kernel_params = kernel_run_parameters

    # validation function
    def validate_run(__, kernel, device, params, kernelStdOut=None):
        assert kernel == sgemm
        assert device == device_index
        assert params == kernel_params

    monkeypatch.setattr(micp_offload.Offload, 'run', validate_run)

    # do test
    native_offload = micp_offload.OffloadFactory().create('native')
    native_offload.run(sgemm, device_index, kernel_params)


def test_native_offload_custom_run(monkeypatch, kernel_run_parameters):
    """validate NativeOffload() custom run() method which should remove
    unnecessary parameters and then invoke run() in the super class"""

    device_index = 1
    sgemm, kernel_params = kernel_run_parameters

    # validation function
    def validate_run(__, kernel, device, params, kernelStdOut=None):
        assert kernel == sgemm
        assert device == device_index
        assert isinstance(params, list)
        assert len(params) == len(kernel_params)

        for param in params:
            assert isinstance(param, micp_params.ParamsDrop)

    # paramMax and paramDrop parameters are set to 1 to force
    # UT's to go through a particular branch
    setattr(sgemm, 'paramMax', {'native':1})
    setattr(sgemm, 'paramDrop', {'native':1})
    monkeypatch.setattr(micp_offload.Offload, 'run', validate_run)

    # do test
    native_offload = micp_offload.OffloadFactory().create('native')
    native_offload.run(sgemm, device_index, kernel_params)


NO_KERNEL_LINPACK_ERROR = """WARNING:  Executable for kernel {0} and offload local does not exist, skipping
          MKL Linpack is available for download here:
          http://software.intel.com/en-us/articles/intel-math-kernel-library-linpack-download
          The environment variable MKLROOT must point to the location of the
          directory extracted from the .tgz file.

"""

func_path_x_exec = lambda arg0, arg1: None
def func_path_x_exec_linp(arg0, arg1):
    raise NoExecutableError()


@pytest.mark.parametrize('on_host, on_dev, kernel_name, expected_error, path_x_exec', [
    (True, False, 'sgemm', micp_offload.CONST_NOT_SUPPORTED_KERNEL, func_path_x_exec),
    (True, False, 'linpack', NO_KERNEL_LINPACK_ERROR, func_path_x_exec_linp),
    (False, True, 'sgemm', micp_offload.CONST_NOT_SUPPORTED_KERNEL, func_path_x_exec),
    (False, True, 'linpack', NO_KERNEL_LINPACK_ERROR, func_path_x_exec_linp)
])
def test_negative_no_kernel_binary(
        local_offload_fixture,
        kernel_run_parameters,
        monkeypatch,
        capsys,
        on_dev,
        on_host,
        kernel_name,
        expected_error,
        path_x_exec):
    """validate run() returns and empty list and prints the corresponding
    error message to stderr when the tool is unable to find the binary file
    to execute the benchmark. In the case of linpack (HPCG, SMP Linpack and
    HPL) run() should raise an exception MissingDependenciesError.
    """

    kernel, params = kernel_run_parameters

    # adjust object's attributes to force run() to go through a particular branch
    local_offload_fixture._runHost = on_host
    local_offload_fixture._runDev = on_dev
    kernel.name = kernel_name

    monkeypatch.setattr(os.path, 'exists', lambda __: False)

    monkeypatch.setattr(
        micp_kernels_sgemm.sgemm,
        'path_host_exec',
        path_x_exec)

    monkeypatch.setattr(
        micp_kernels_sgemm.sgemm,
        'path_dev_exec',
        path_x_exec)

    # do test
    if kernel.name == 'sgemm':
        result = local_offload_fixture.run(kernel, -1, params)
        __, err = capsys.readouterr()

        assert not result
        assert err == \
            micp_common.mp_print(expected_error.format(kernel_name, "local"),
                micp_common.CAT_WARN)

    elif kernel.name == 'linpack':
        with pytest.raises(micp_common.MissingDependenciesError) as err:
            local_offload_fixture.run(kernel, -1, params)
            assert str(err) == expected_error.format(kernel_name)
    else:
        ValueError("Test Suite Internal Error: Invalid kernel {0}".format(kernel.name))


##################################################################
#                       offload.run() UT's
##################################################################

COPYING_FILES = "copying kernels to card..."
REMOVING_FILES = "removing kernels from cards"
KILLING_PROCESS = "killing process"

class Pid(object):
    """ class to mock objects returned by python's subprocess.Popen()"""
    returncode = 0
    def __init__(self, output):
        self._output = output

    def communicate(self):
        return (self._output, '')

    def kill(self):
        print KILLING_PROCESS

@pytest.fixture()
def mpss_connect_mock(monkeypatch):
    """fixture to mock micp.connect.MPSS()"""
    def copyto(*args):
        """copyto() mock, ignore all input arguments
        just pretend files were copied"""
        print COPYING_FILES

    def popen(self, input_args, env=None, cwd=None):
        """popen() mock, notice env is not used by the
        mock but clients may pass this argument as they
        don't expect to be 'talking' to mock"""
        args = input_args
        if isinstance(args, list):
            args = ' '.join(args)

        if 'rm -f' in args:
            print REMOVING_FILES
            return Pid('')

        if 'xlinpack_mic' in args:
            return Pid(mocks_data.LINPACK_OUTPUT)

        if 'gemm_' in args:
            return Pid(mocks_data.XGEMM_OUTPUT)

        if 'BusSpeedReadback' in args:
            return Pid(mocks_data.SHOC_OUTPUT)

        if 'BusSpeed_mic' in args:
            return Pid('')

        raise NameError('Internal test suite error, invalid key "{0}"'.format(args))

    monkeypatch.setattr(micp_connect.MPSSConnect, 'copyto', copyto)
    monkeypatch.setattr(micp_connect.MPSSConnect, 'Popen', popen)


def test_run_dgemm_local_happy_path(
        local_offload_fixture,
        kernel_run_parameters,
        mpss_connect_mock,
        capsys):
    """validate from an offload.run() point of view
    the happy path for the dgemm local execution"""
    kernel, params = kernel_run_parameters
    local_offload_fixture.run(kernel, -1, params)

    out, err = capsys.readouterr()
    assert mocks_data.XGEMM_OUTPUT in out
    assert not err


def test_run_dgemm_native_happy_path(
        kernel_run_parameters,
        mpss_connect_mock,
        monkeypatch,
        capsys):
    """validate from an offload.run() point of view
    the happy path for the dgemm native execution.

    test receives:
      - kernel_run_parameters a fixture that returns the kernel
        and the parameters objects
      - mpss_connect_mock, fixture to mock micp_connect.MPSSConnect()
        object and prevent the unit tests from communicating with the
        hardware
      - pytest built-in fixtures: monkeypatch and capsys
    """
    kernel, params = kernel_run_parameters
    native = micp_offload.OffloadFactory().create('native')
    native.run(kernel, -1, params)

    out, err = capsys.readouterr()
    assert mocks_data.XGEMM_OUTPUT in out
    assert COPYING_FILES in out
    assert not err


def test_run_shoc_readback_scif_happy_path(
        mpss_connect_mock,
        monkeypatch,
        capsys):
    """validate from an offload.run() point of viewg
    the happy path for the shoc_* execution"""

    if SKIP_SHOC_TESTS:
        pytest.skip()

    warning = ('WARNING:  kernel parameter "target" set to 0 overriding with value -1\n'
               'WARNING:  kernel parameter "target" set to 0 overriding with value -1\n')
    monkeypatch.setattr(
        micp_shoc_readback.shoc_readback, 'path_aux_data',
        lambda self, __: 'libiomp5.so')
    monkeypatch.setattr(
        micp_shoc_readback.shoc_readback, 'path_dev_exec',
        lambda self, __: '/path/to/BusSpeedReadback')

    monkeypatch.setattr(os.path, 'exists', lambda __: True)

    kernel = micp_shoc_readback.shoc_readback()
    params = _init_kernel_params(kernel)
    offload = micp_offload.OffloadFactory().create('scif')

    offload.run(kernel, -1, params)

    stdout, stderr = capsys.readouterr()
    assert COPYING_FILES in stdout
    assert mocks_data.SHOC_OUTPUT in stdout
    assert REMOVING_FILES in stdout
    assert stderr == warning


@pytest.mark.parametrize('offload', ['native', 'local'])
def test_run_linpack_native_happy_path(
        mpss_connect_mock,
        monkeypatch,
        capsys,
        offload):
    """validate from an offload.run() point of viewg
    the happy path for the linpack execution"""
    linpack_dummy_path = 'MKLROOT/benchmarks/linpack/xlinpack_mic'
    monkeypatch.setattr(
        micp_linpack.linpack, 'path_dev_exec',
        lambda self, __: linpack_dummy_path)

    monkeypatch.setattr(
        micp_linpack.linpack, 'path_host_exec',
        lambda self, __: linpack_dummy_path)

    monkeypatch.setattr(os.path, 'exists', lambda __: True)

    kernel =  micp_linpack.linpack()
    params = _init_kernel_params(kernel)
    offload = micp_offload.OffloadFactory().create(offload)

    offload.run(kernel, -1, params)

    stdout, stderr = capsys.readouterr()
    if offload == 'native':
        assert COPYING_FILES in stdout
        assert REMOVING_FILES in stdout

    assert mocks_data.LINPACK_OUTPUT in stdout
    assert not stderr


@pytest.mark.parametrize('offload_method', ['native', 'local'])
def test_run_invalid_param_type(
        local_offload_fixture,
        mpss_connect_mock,
        offload_method,
        monkeypatch):
    """validate offload.run() raises an exception if an invalid
    parameter is passed to a kernel"""
    monkeypatch.setattr(
        micp_kernels_dgemm.dgemm, 'param_type', lambda __: 'fake_param')

    kernel =  micp_kernels_dgemm.dgemm()
    params = _init_kernel_params(kernel)
    offload = micp_offload.OffloadFactory().create(offload_method)

    with pytest.raises(NameError) as err:
        offload.run(kernel, -1, params)
        assert err == 'Unknown parameter type fake_param'


@pytest.mark.parametrize('offload_method, binary', [
    ('native', 'dgemm_mic.x'),
    ('local', 'dgemm_cpu.x')
])
def test_run_workload_execution_error_clean_up(
        mpss_connect_mock,
        offload_method,
        binary,
        monkeypatch,
        capsys):
    """validate offload.run() cleans up (removes binaries, configuration files)
    the host/card if a kernel fails for any reason"""
    monkeypatch.setattr(
        micp_kernels_dgemm.dgemm, 'path_aux_data',
        lambda self, __: 'libiomp5.so')
    monkeypatch.setattr(
        micp_kernels_dgemm.dgemm, 'path_dev_exec',
        lambda self, __: '/path/to/{0}'.format(binary))

    monkeypatch.setattr(os.path, 'exists', lambda __: True)

    monkeypatch.setattr(Pid, 'returncode', -1)

    kernel =  micp_kernels_dgemm.dgemm()
    params = _init_kernel_params(kernel)
    offload = micp_offload.OffloadFactory().create(offload_method)

    with pytest.raises(micp_connect.CalledProcessError) as err:
        offload.run(kernel, -1, params)
        assert binary in str(err)

    if offload_method == 'native':
        stdout, stderr = capsys.readouterr()
        assert REMOVING_FILES in stdout
        assert COPYING_FILES in stdout


def test_run_workload_execution_error_no_shared_library(
        local_offload_fixture,
        mpss_connect_mock,
        monkeypatch,
        ):
    """validate offload.run() identifies when a kernel fails because it was
    unable to find a shared library and the error is reported accordingly"""
    monkeypatch.setattr(Pid, 'returncode', 127)

    kernel =  micp_kernels_dgemm.dgemm()
    params = _init_kernel_params(kernel)
    offload = micp_offload.OffloadFactory().create('local')

    with pytest.raises(micp_common.MissingDependenciesError):
        offload.run(kernel, -1, params)


@pytest.mark.parametrize('offload_method, binary', [
    ('native', 'dgemm_mic.x'),
    ('local', 'dgemm_cpu.x')
])
def test_run_hung_workload(
        mpss_connect_mock,
        offload_method,
        binary,
        monkeypatch,
        capsys):
    """validate that run() will kill the process corresponding to a kernel if
    it were hung or something went wrong and the kernel should be stopped"""
    monkeypatch.setattr(
        micp_kernels_dgemm.dgemm, 'path_aux_data',
        lambda self, __: 'libiomp5.so')
    monkeypatch.setattr(
        micp_kernels_dgemm.dgemm, 'path_dev_exec',
        lambda self, __: '/path/to/{0}'.format(binary))

    monkeypatch.setattr(os.path, 'exists', lambda __: True)
    monkeypatch.setattr(Pid, 'returncode', None)

    kernel =  micp_kernels_dgemm.dgemm()
    params = _init_kernel_params(kernel)
    offload = micp_offload.OffloadFactory().create(offload_method)

    with pytest.raises(micp_connect.CalledProcessError) as err:
        offload.run(kernel, -1, params)
        assert binary in str(err)

    stdout, __ = capsys.readouterr()
    assert KILLING_PROCESS in stdout
