# Copyright 2010-2017 Intel Corporation.
# 
# This library is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, version 2.1.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
# 
# Disclaimer: The codes contained in these modules may be specific
# to the Intel Software Development Platform codenamed Knights Ferry,
# and the Intel product codenamed Knights Corner, and are not backward
# compatible with other Intel products. Additionally, Intel will NOT
# support the codes or instruction set in future products.
# 
# Intel offers no warranty of any kind regarding the code. This code is
# licensed on an "AS IS" basis and Intel is not obligated to provide
# any support, assistance, installation, training, or other services
# of any kind. Intel is also not obligated to provide any updates,
# enhancements or extensions. Intel specifically disclaims any warranty
# of merchantability, non-infringement, fitness for any particular
# purpose, and any other warranty.
# 
# Further, Intel disclaims all liability of any kind, including but
# not limited to liability for infringement of any proprietary rights,
# relating to the use of the code, even if Intel is notified of the
# possibility of such liability. Except as expressly stated in an Intel
# license agreement provided with this code and agreed upon with Intel,
# no license, express or implied, by estoppel or otherwise, to any
# intellectual property rights is granted herein.

"""UT's for micp/micp/run.py"""

import os

import pytest

import micp.run as micp_run
import micp.info as micp_info
import micp.offload as micp_offload
import micp.stats as micp_stats
import micp.version as micp_version


EXPECTED_PICKLE = 'micp_run_stats_C0PRQ-7120-P-A-X-D_mpss-4.0_native-scif_optimal_mic0.pkl'
DUMMY_SINGLE_CSV_REPORT = 'dummy single csv performance result'
DUMMY_FULL_CSV_REPROT = 'dummy full csv performance results'
DUMMY_SINGLE_PLOT = 'ploting single performance results'
DUMMY_MULTIPLE_PLOT = 'ploting multiple performance results'

class InfoMock(object):
    """class to mock the behavior of micp.info.Info(),
    for validation purposes this mock returns data
    that corresponds to the SKU C0PRQ-7120"""
    def __init__(self):
        pass

    def set_device_index(self, __):
        pass

    def get_device_index(self):
        return 0

    def num_cores(self):
        return 61

    def mic_memory_size(self):
        return 16642998272

    def mic_sku(self):
        return 'C0PRQ-7120 P/A/X/D'

    def micperf_version(self):
        return '4.0'

    def is_processor_mcdram_available(self):
        return False

    def get_number_of_nodes_with_cpus(self):
        return 1

    def is_in_sub_numa_cluster_mode(self):
        return False

    def get_processor_codename(self):
        return "KNL"


class StatsMock(object):
    """class to mock the behavior of micp.stats.Stats()"""
    pass


def fake_run(__, kernel, device, paramList, devParamList=None, kernelStdOut=None):
    """this method 'runs' a performance benchmark and returns a list
    of Stats objects containing the 'results'"""
    print "running benchmark"
    return [StatsMock()]


@pytest.fixture()
def setup(monkeypatch, mock_network_functions):
    """fixture to setup test environment (do not run actual benchmarks,
    do not query hardware) required to execute test cases for run()"""
    monkeypatch.setattr(micp_info, 'Info', lambda: InfoMock())
    monkeypatch.setattr(micp_offload.Offload, 'run', fake_run)
    monkeypatch.setattr(micp_stats.StatsCollection, '__str__', lambda __: DUMMY_SINGLE_CSV_REPORT)
    monkeypatch.setattr(micp_stats.StatsCollection, 'csv_write', lambda arg0, arg1: DUMMY_FULL_CSV_REPROT)
    monkeypatch.setattr(micp_stats.StatsCollection, 'csv', lambda arg0, arg1: DUMMY_SINGLE_CSV_REPORT)

    # mock micp_stats.StatsCollection.plot() to print results to stdout
    def plot_results(output):
        print output
    monkeypatch.setattr(micp_stats.StatsCollection, 'plot',
                        lambda arg0, arg1: plot_results(DUMMY_SINGLE_PLOT))
    monkeypatch.setattr(micp_stats.StatsCollection, 'plot_all',
                        lambda arg0, arg1: plot_results(DUMMY_MULTIPLE_PLOT))


@pytest.mark.parametrize("kernel",
  ['all', 'sgemm', 'dgemm', 'linpack', 'stream', 'shoc_download', 'shoc_readback'])
def test_run_single_kernel(monkeypatch, setup, capsys, kernel):
    """happy path, validate micp.run.run() is capable of executing all the
    differnet kernels including 'all' (no pickle file for this test)"""

    # SHOC kernels not available for KNL SB
    if kernel.startswith('shoc') and micp_version.MIC_PERF_HOST_ARCH == 'x86_64_AVX512':
        pytest.skip()

    micp_run.run(kernelNames=kernel,
                 offMethod='native',
                 paramCat='optimal',
                 verbLevel=0,
                 sudo=True)

    stdout, stderr = capsys.readouterr()
    assert not stderr
    assert 'running benchmark' in stdout


@pytest.mark.parametrize("offload",
        ['all', 'auto', 'coi', 'myo', 'native', 'pragma', 'scif'])
def test_offload_methods(capsys, setup, offload):
    """validate micp.run.run() accepts all the
    valid offload methods including 'all'"""
    micp_run.run(kernelNames='sgemm',
                 offMethod=offload)

    stdout, stderr = capsys.readouterr()
    assert not stderr
    assert stdout


def test_available_kernels_help(capsys, setup):
    """validate micp.run.run() prints the supported kernels
    when the kernel help is requested"""

    # available kernels depends on the version KNC or KNL
    if micp_version.MIC_PERF_HOST_ARCH == 'x86_64':
        expected = '''Available kernels:
    dgemm
    hplinpack
    linpack
    sgemm
    shoc_download
    shoc_readback
    stream\n'''
    else:
        expected = '''Available kernels:
    dgemm
    fio
    hpcg
    hplinpack
    libxsmm_conv
    linpack
    mkl_conv
    sgemm
    stream\n'''

    micp_run.run(kernelNames='help')
    stdout, stderr = capsys.readouterr()
    assert not stderr
    assert stdout == expected


@pytest.mark.parametrize("kernel",
  ['sgemm', 'dgemm', 'linpack', 'stream', 'shoc_download', 'shoc_readback'])
def test_kernel_parameters_help(kernel, capsys, setup):
    """validate micp.run.run() returns the kernel parameter's help
    when the 'help' argument is passed"""

    # SHOC kernels not available for KNL SB
    if kernel.startswith('shoc') and micp_version.MIC_PERF_HOST_ARCH == 'x86_64_AVX512':
        pytest.skip()

    expected = 'Parameter help for kernel {0}'.format(kernel)
    micp_run.run(kernelNames=kernel,
                 paramCat='',
                 kernelArgs='--help')
    stdout, stderr = capsys.readouterr()
    assert not stderr
    assert expected in stdout


def test_kernel_output_file(monkeypatch, tmpdir, setup):
    """validate micp.run.run() creates a single pickle
    file when given an output directory"""
    output_dir = tmpdir.__str__()
    micp_run.run(kernelNames='sgemm', outDir=output_dir)

    output_files = os.listdir(output_dir)
    assert len(output_files) == 1
    assert EXPECTED_PICKLE in output_files


def test_run_verbosity1(capsys, setup):
    """validate when verbosity is set to 1 micp.run.run()
    prints a csv report to stdout"""

    # dummy results
    expected = DUMMY_SINGLE_CSV_REPORT
    micp_run.run(kernelNames='sgemm', verbLevel=1)

    stdout, stderr = capsys.readouterr()
    assert not stderr
    assert expected in stdout


@pytest.mark.parametrize("kernels, verbosity, expected",
        [('sgemm', 2, DUMMY_SINGLE_PLOT),
         ('sgemm', 3, DUMMY_SINGLE_PLOT),
         ('sgemm:dgemm', 3, DUMMY_MULTIPLE_PLOT)])
def test_run_verbosity_plots(capsys, setup, kernels, verbosity, expected):
    """validate when verbosity is equal to 2 or 3 and NO output dir is given
    micp.run.run() attempts to plot the results and display them on screen.
    Make sure run() doesn't plot the same results twice when a single kernel
    is given as input."""

    micp_run.run(kernelNames=kernels, offMethod='native', verbLevel=verbosity)
    stdout, stderr = capsys.readouterr()
    assert not stderr
    assert expected in stdout


@pytest.mark.parametrize("verbosity", [2, 3])
def test_run_verbosity_plots_exceptions(monkeypatch, capsys, setup, verbosity):
    """validate when verbosity is equal to 2 or 3 and NO output dir is given the
    application won't crash if micp.run.run() is unable to plot the results"""
    def raise_name_error(*args):
        raise NameError
    monkeypatch.setattr(micp_stats.StatsCollection, 'plot', raise_name_error)
    monkeypatch.setattr(micp_stats.StatsCollection, 'plot_all', raise_name_error)

    micp_run.run(kernelNames='sgemm', offMethod='native', verbLevel=verbosity)
    stdout, stderr = capsys.readouterr()
    assert stdout
    assert not stderr


def test_run_verbosity2_csv(tmpdir, setup):
    """validate when verbosity is equal to 2 and and an output dir is given
    micp.run.run() creates a csv file with a summary of the results"""

    # mock micp_stats.StatsCollection.plot() to print results to stdout
    output_dir = tmpdir.__str__()
    expected_csv = '{0}_all.csv'.format(EXPECTED_PICKLE[:-4])

    micp_run.run(kernelNames='sgemm',
                 paramCat='optimal',
                 verbLevel=2,
                 outDir=output_dir)

    output_files = os.listdir(output_dir)
    assert len(output_files) == 2
    assert EXPECTED_PICKLE in output_files
    assert expected_csv in output_files


def test_overlaping_parameters(setup, capsys):
    """validate when micp.run.run() receives both a valid parameters category
    and kernel paramers, run() alerts the user about the conflict and prints
    a warning to stderr"""
    expected = '''WARNING: paramCat and and kernel arguments both specified.
         Kernel arguments are ignored\n'''

    micp_run.run(kernelNames='sgemm',
                 paramCat='optimal',
                 kernelArgs='--f_first_matrix_size 1000')

    __, stderr = capsys.readouterr()
    assert stderr == expected



def test_regression_test(monkeypatch, capsys, setup):
    """validate when reference results and a margin are given to
    micp.micp.run() it rund a performance regression test"""
    def run_perf_test(*args):
        print 'running performance regression test...'
    monkeypatch.setattr(micp_stats.StatsCollection, 'perf_regression_test', run_perf_test)
    monkeypatch.setattr(micp_stats.StatsCollection, 'extend', lambda arg0, arg1: None)

    reference_results = StatsMock()
    micp_run.run(kernelNames='sgemm',
                 margin=0.04,
                 compResult=reference_results)

    stdout, stderr = capsys.readouterr()
    assert not stderr
    assert 'running performance regression test...' in stdout


def test_run_keyboard_interrupt(monkeypatch, tmpdir, capsys, setup):
    """validate micp.run.run() can handle a keyboard interruption,
    partial results (if any) should be saved into a pickle file
    then exception is re-raised."""

    def raise_keyboard_interrupt(kernel, device, paramList, devParamList=None, kernelStdOut=None):
        """raises a KeyboardInterrupt and set attribute partialResult"""
        an_interruption = KeyboardInterrupt()
        an_interruption.partialResult = 'could not complete exectution...'
        raise an_interruption

    monkeypatch.setattr(micp_offload.Offload, 'run', raise_keyboard_interrupt)

    output_dir = tmpdir.__str__()
    with pytest.raises(KeyboardInterrupt):
        micp_run.run(kernelNames='sgemm', outDir=output_dir)

    output_files = os.listdir(output_dir)
    assert len(output_files) == 1
    assert EXPECTED_PICKLE in output_files


