#!/usr/env python

MOCK_FUNCTION_CALLED = False


def mock_dummy_function(_):
    global MOCK_FUNCTION_CALLED
    MOCK_FUNCTION_CALLED = True
    return True

import sys
import contextlib
import pytest

# Workaround to avoid getting kicked if not running as root
# pytest are running as none root user
_sys_exit = sys.exit
sys.exit = mock_dummy_function
from .. import sys_diag

sys.exit = _sys_exit


DUMMY_TEST = [['dummy', "Dummy Test", mock_dummy_function, "Used for UT",
               'store_true', sys_diag.TestType.auto]
              ]

CMD_ARGS = [(["sys_diag.py", "--help"],
             """[cmd_args0] usage: sys_diag.py [-h] [-v] [-d {1,2,3,4}] [-o OUTPUT] [-i INTERVAL]
                   [-n NUMACTL] [-m] [-p] [-s] [-b] [-c] [-t] [-l] [-a]"""),
            ]


def mock_raw_input_return_one(r_input=""):
    """ Replaces raw_input, answering 1
    """
    return 1


def mock_raw_input_return_zero(r_input=""):
    """ Replaces raw_input, answering 0
    """
    return 0


def mock_raw_input_return_more_than_tests(r_input=""):
    """ Replaces raw_input, answering a big number
    """
    return sys.maxint


def test_run_test():
    """ Mock running a single SYS Diag test
    """
    global MOCK_FUNCTION_CALLED
    sys_diag.run_test("Dummy Test", mock_dummy_function, "")
    assert MOCK_FUNCTION_CALLED
    MOCK_FUNCTION_CALLED = False


def test_run_all_tests():
    """ Mock running all SYS Diag tests
    """
    global MOCK_FUNCTION_CALLED
    backup_all_tests = sys_diag.TESTS
    sys_diag.TESTS = DUMMY_TEST
    sys_diag.run_all_tests("")
    assert MOCK_FUNCTION_CALLED
    sys_diag.TESTS = backup_all_tests
    MOCK_FUNCTION_CALLED = False


def test_print_menu(capfd):
    """ Simply print the SYS Diag Menu
    """
    sys_diag.print_menu()
    out, err = capfd.readouterr()
    assert "End script" in out
    assert "Runs all tests sequentially" in out
    assert not err


def test_get_answer_one():
    """ The input from user is the same one received...
    """
    sys_diag.raw_input = mock_raw_input_return_one
    assert sys_diag.get_answer() == 1


def test_get_answer_zero():
    """ The input from user is the same one received...
    """
    sys_diag.raw_input = mock_raw_input_return_zero
    assert sys_diag.get_answer() == 0


@contextlib.contextmanager
def redirect_sys_argv(cmd_args):
    _argv = sys.argv[:]
    sys.argv = cmd_args
    yield
    sys.argv = _argv


@pytest.mark.parametrize("cmd_args, expected", CMD_ARGS)
def test_main(cmd_args, expected):
    """ Call sys_diag.main() for the help output
    """
    with redirect_sys_argv(cmd_args):
        with pytest.raises(SystemExit):
            assert sys_diag.main() == expected
