__author__ = 'foocampo'

import os
import pytest
import tempfile
import golden_output
from .. import sys_utils

MSR_MODULE = "msr"
DEBUG_LEVELS = [1, 2, 3, 4]
INTERVALS = ["10:00:01", "00:00:10", "01:01:01"]


def mock_proc_cpuinfo(tempfile):
    """ Mock for reading /proc/cpuinfo
    """
    with open(tempfile, 'r') as temp:
        pass


def mock_run(command, timeout=0):
    """ Mock run command
    """
    return "Success", "", 0


def mock_run_failed(command, timeout=0):
    """ Mock a failed run command
    """
    return "", "Failed", 1


# ToDo Still to work on how to mock the memkind library behavior
"""
class HbwMalloc():
    restype = ""
    def __init__(self, restype):
        self.restype = restype


class HbwFree():
    argtypes = ""
    def __init__(self, hbw_mem_ptr):
        self.argtypes = hbw_mem_ptr

    def __cmp__(self, other):
        pass


class MockMemkind():
    def __init__(self):
        self.hbw_free = HbwFree
        self.hbw_malloc = HbwMalloc

    def __nonzero__(self):
        return True

    def hbw_check_available(self):
        return 0


class MockEmptyMemkind(MockMemkind):
    def __nonzero__(self):
        return False


class MockCannotAllocateMemkind(MockMemkind):
    def hbw_malloc(self, hbw_reserve_size):
        return 0
"""


def test_check_log_dir():
    """  Verify the LOGS_DIR file can be written on.
    """
    backup_logs_dir = sys_utils.LOGS_DIR
    sys_utils.LOGS_DIR = tempfile.gettempdir()
    assert sys_utils.check_log_dir() == None
    sys_utils.LOGS_DIR = backup_logs_dir


def test_debug_log():
    """ Verify the message to print is returned
    sys_utils.debug_log(level, msg) will return msg if it is printed to screen.
    """
    msg = "Message to be printed"
    assert msg == sys_utils.debug_log(1, msg)


def test_debug_log_no_print():
    """ Verify the message to print is not returned
    """
    assert sys_utils.debug_log(sys_utils.debug_level + 1, "Message") is None


@pytest.mark.parametrize("level", DEBUG_LEVELS)
def test_set_debug_level(level):
    """ Verify the debug_level provided is actually assigned
    """
    assert sys_utils.set_debug_level(level) == level


def test_is_xeon_phi_200():
    """ Know if we are running on SYS
    """
    backup_cpuinfo = sys_utils.PROC_CPUINFO
    temp = tempfile.NamedTemporaryFile(delete=False)
    temp.write("model\t\t: 87")
    temp.close()
    sys_utils.PROC_CPUINFO = temp.name
    assert sys_utils.is_xeon_phi_200()
    os.unlink(temp.name)
    sys_utils.PROC_CPUINFO = backup_cpuinfo


def test_is_not_xeon_phi_200():
    """ Know if we are not running on SYS
    """
    backup_cpuinfo = sys_utils.PROC_CPUINFO
    temp = tempfile.NamedTemporaryFile(delete=False)
    temp.write("model\t\t: 88")
    temp.close()
    sys_utils.PROC_CPUINFO = temp.name
    assert not sys_utils.is_xeon_phi_200()
    os.unlink(temp.name)
    sys_utils.PROC_CPUINFO = backup_cpuinfo


def test_run_no_timeout():
    """ Simply run 'cd .' command
    """
    assert sys_utils.run("cd .")


def test_run_wih_timeout_no_kill():
    """ Simply run 'cd .' command and send a timeout of 3 seconds.
    It is not expected that the timeout occurs.
    """
    assert sys_utils.run("cd .", 3)


def test_terminate_process_no_sudo(monkeypatch):
    """ Terminate a process as not root
    """
    monkeypatch.setattr(sys_utils, "run", mock_run)
    assert sys_utils.terminate_process("9999") == 0


def test_terminate_process_sudo(monkeypatch):
    """ Terminate a process as root
    """
    monkeypatch.setattr(sys_utils, "run", mock_run)
    assert sys_utils.terminate_process("9999", sudo=True) == 0


def test_dependencies(monkeypatch):
    """ All dependencies of SYS Diag are installed
    """
    monkeypatch.setattr(sys_utils, "run", mock_run)
    assert sys_utils.dependencies()


def test_modprobe(monkeypatch):
    """ Able to load MSR module
    """
    monkeypatch.setattr(sys_utils, "run", mock_run)
    assert sys_utils.modprobe(MSR_MODULE) == 0


def test_modprobe_negative(monkeypatch):
    """ Unable to load MSR module
    """
    monkeypatch.setattr(sys_utils, "run", mock_run_failed)
    assert sys_utils.modprobe(MSR_MODULE) != 0


@pytest.mark.parametrize("interval", INTERVALS)
def test_to_secs(interval):
    """ Able to transform interval strings to secs
    """
    assert sys_utils.to_secs(interval)


def test_dependencies_negative(monkeypatch):
    """ Not all dependencies of SYS Diag are installed
    """
    monkeypatch.setattr(sys_utils, "run", mock_run_failed)
    assert not sys_utils.dependencies()
