#  Copyright 2017, Intel Corporation, All Rights Reserved.
#
# This software is supplied under the terms of a license
# agreement or nondisclosure agreement with Intel Corp.
# and may not be copied or disclosed except in accordance
# with the terms of that agreement.

import os
import math
import sys
import tempfile
import subprocess
import json
from distutils.spawn import find_executable

import micp.kernel as micp_kernel
import micp.info as micp_info
import micp.common as micp_common
import micp.params as micp_params

CONST_FIO_CONFIG_FILE = """[global]
filename={2}
iodepth=32
stonewall
direct=1
numjobs={0}
size={1}
thread
group_reporting
description={2}
[Read-4k-bw]
rw=read
bs=4k"""

CONST_NO_FIO_TEXT = """FIO was not found on this system.
please install it using a standard package manager
or visit https://github.com/axboe/fio
to download, build and install it directly.
Please make sure the directory containing the FIO
binary is added to the system PATH variable."""

CONST_FIO_CONFIG_FILE_NAME = 'fio.cfg'

CONST_FIO_PARAMS = {'numjobs':'10', 'size':'1MB', 'target_partition':''}

class fio(micp_kernel.Kernel):
    """Implements kernel interface for FIO benchmark"""

    def __init__(self):
        self.name = 'fio'
        self._working_directory = None
        self.score = None

        info = micp_info.Info()
        maxCount = info.num_cores()

        self.param_validator = micp_params.NO_VALIDATOR
        self._categoryParams = {}
        self._paramNames = CONST_FIO_PARAMS.keys()
        self._paramDefaults = CONST_FIO_PARAMS

        self._categoryParams['test'] = [' ']

        args_all = ''
        args_scaling = ''
        args_scaling_core = ''

        for key in CONST_FIO_PARAMS:
            args_all += '--{0} {1} '.format(key, CONST_FIO_PARAMS[key])
            args_scaling += '--{0} {1} '.format(key,
                ('{0}' if key == 'size' else CONST_FIO_PARAMS[key]))
            args_scaling_core += '--{0} {1} '.format(key,
                ('{0}' if key == 'numjobs' else CONST_FIO_PARAMS[key]))

        core_scale = range(1, maxCount + 1, 1)
        size_scale = ['4MB', '8MB', '16MB', '32MB', '64MB', '128MB']
        self._categoryParams['optimal'] = [ args_all ]
        self._categoryParams['scaling'] = [args_scaling.format(size)
                                           for size in size_scale]
        self._categoryParams['scaling_quick'] = self._categoryParams['scaling']

        self._categoryParams['optimal_quick'] = [ args_all ]
        self._categoryParams['scaling_core'] = \
            [args_scaling_core.format(core_count) for core_count in core_scale]

    def _do_unit_test(self):
        return True

    def offload_methods(self):
        return ['local']

    def path_host_exec(self, offload_method):
        # check if fio exists
        if offload_method is 'local':
            path = find_executable('fio')
            if path:
                return os.path.realpath(path)
            else:
                micp_common.mp_print(micp_common.CAT_ERROR, CONST_NO_FIO_TEXT)
        return None

    def path_dev_exec(self, offType):
        """returns None, Intel Xeon Phi Coprocessors not supported"""
        return None

    def param_type(self):
        """ FIO uses config file """
        return 'file'

    def requires_root_access(self):
        return True

    def get_fixed_args(self):
        return ['--output-format=json']

    def parse_desc(self, raw):
        try:
            rjson = json.loads(raw)
        except ValueError as e:
            raise micp_kernel.SelfCheckError(
                "JSON parse error. [{}] in:\n{}\n".format(e, raw))

        try:
            desc = ""
            desc += rjson["fio version"]
            desc += " " + rjson["jobs"][0]["jobname"]
            desc += "; total size: {} kB".format(
                rjson["jobs"][0]["read"]["io_bytes"])
            desc += "; device: " + rjson["jobs"][0]["desc"]
            self.score = rjson["jobs"][0]["read"]["bw"]
        except (ValueError, KeyError) as e:
            raise micp_kernel.SelfCheckError(
                "JSON parse error. [{}] in:\n{}\n".format(e, raw))

        return desc

    def parse_perf(self, raw):
        # parsed already in parse_desc, no point parsing twice
        if self.score is None:
            raise micp_kernel.SelfCheckError(
                "Score not found in JSON:\n{}\n".format(raw))

        result = {}
        result['Computation.Avg'] = \
            {'value':self.score, 'units':'kB/s', 'rollup':True}
        return result

    def independent_var(self, category):
        if category == 'scaling_core':
            return 'numjobs'
        return 'size'

    def param_file(self, param):
        numjobs_p = param.get_named('numjobs')
        size_p = param.get_named('size')
        target_partition_p = param.get_named('target_partition')

        if self._working_directory is None:
            self._working_directory = \
                tempfile.mkdtemp(prefix='micperf_fio_data_')

        if not target_partition_p:
            # by default run FIO on the same drive on which fio has
            # been installed get the name of partition by parsing fstat
            fio_path = self.path_host_exec('local')
            if not fio_path:
                raise micp_kernel.SelfCheckError("Internal error")
            fstat = os.stat(self.path_host_exec('local'))
            major = fstat.st_dev >> 8 & 0xFF
            minor = fstat.st_dev & 0xFF
            partiton_name = '/dev/'

            if major == 8:
                partiton_name += 'sd'
            else:
                partiton_name += 'hd'

            partiton_name += chr(ord('a') + (minor//16))
            partiton_name += str(minor%16)
        else:
            partiton_name = target_partition_p

        # simply use fixed body of conf just replace values
        config_file_content = CONST_FIO_CONFIG_FILE.format(
            numjobs_p, size_p, partiton_name, partiton_name)

        config_file_path = os.path.join(
            self._working_directory, CONST_FIO_CONFIG_FILE_NAME)

        with open(config_file_path, 'w') as fid:
            fid.write(config_file_content)

        return config_file_path
