/*
 * Copyright (C) 2012 Intel Corporation. All rights reserved.
 *
 * This software is supplied under the terms of a license
 * agreement or nondisclosure agreement with Intel Corp.
 * and may not be copied or disclosed except in accordance
 * with the terms of that agreement.
 */

/******************************************************************************
 *  $Header:  $
 *  $DateTime:  $
 *  $Author:  rcshipmx$
 *****************************************************************************/

#ifndef OS_GETOPT_H
#define OS_GETOPT_H

/*
 * Copyright (C) 2012 Intel Corporation. All rights reserved.
 *
 * This software is supplied under the terms of a license
 * agreement or nondisclosure agreement with Intel Corp.
 * and may not be copied or disclosed except in accordance
 * with the terms of that agreement.
 */

/******************************************************************************
 *  $Header:  $
 *  $DateTime:  $
 *  $Author:  rcshipmx$
 *****************************************************************************/

#ifndef OS_AGNOSTIC_H
#define OS_AGNOSTIC_H

/*
 * Copyright (C) 2010-2012 Intel Corporation. All rights reserved.
 *
 * This software is supplied under the terms of a license
 * agreement or nondisclosure agreement with Intel Corp.
 * and may not be copied or disclosed except in accordance
 * with the terms of that agreement.
 */

#ifndef _OSAL_H_
#define _OSAL_H_

//////////////////////////////////////////////////////////////////////////////
//  OSAL.h
//      Definitions to make code common to all operating systems.
//
//  Usage:
//      Use defines WINDOWS and LINUX to select an operating system.
//      Avoid local definitions of OS-specific code whenever possible.
//      Choose functions that begin with os_ to ensure OS-agnostic code;
//      many of these functions have a slightly redefined interface.
//
//  Function index:
//      void Sleep(ms)
//              Sleep the given number of milliseconds
//
//      bool os_fopen(FILE descriptor, name, mode)
//              Open a file descriptor for the given name and mode.
//              Return true if open succeeded.
//
//  C-string functions:
//      int os_vcprintf
//              Get the buffer size needed to write a printf-style string.
//
//      int os_vsnprintf(s, size, count, format, args)
//              Safe version of vsnprintf.
//
//      int os_strncpy(destination, size, source, num)
//      int os_sprintf(str, size, format, ...)
//      int os_vsprintf(str, size, args)
//      void *os_memcpy(dst, dstSize, src, maxCount)
//      void *os_memmove(dst, dstSize, src, maxCount)
//
//      void *VIRTUAL_ALLOC(size)
//      void VIRTUAL_FREE(address)
//
//  Where equivalent functions exist, prefer the Linux names.
//    Name to use         Windows equivalent
//      strncasecmp         strnicmp
//      popen               _popen
//      pclose              fclose
//
// Changed names; unchanged functionality
//////////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////////
// The preferred definitions to use if needed: LINUX or WINDOWS
//////////////////////////////////////////////////////////////////////////////

#if !defined(WINDOWS)
#if defined(_WIN32) || defined(_WIN64)
#define WINDOWS
#endif
#endif

#if !defined(LINUX) && defined(__linux__)
#define LINUX
#endif


//////////////////////////////////////////////////////////////////////////////
// Includes
//////////////////////////////////////////////////////////////////////////////

#include <string.h>
#include <stdio.h>
/*
 * Copyright (C) 2001-2012 Intel Corporation. All rights reserved.
 *
 * This software is supplied under the terms of a license
 * agreement or nondisclosure agreement with Intel Corp.
 * and may not be copied or disclosed except in accordance
 * with the terms of that agreement.
 */

#ifndef _TYPES_H
#define _TYPES_H

typedef unsigned short USHORT;
typedef unsigned char UCHAR;
#ifdef __linux__

#include <stdint.h>

typedef unsigned int DWORD;
#define stricmp      strcasecmp
#define strnicmp     strncasecmp
#define _stricmp     strcasecmp
#define _strnicmp    strncasecmp
#define getch        getchar

// Windows defines these, so only needed for linux
typedef uint8_t BYTE;
typedef uint16_t WORD;
typedef uint32_t ULONG;

#else // _WIN32
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#if _MSC_VER >= 1600
#include <stdint.h>
#else
// J.Potts: #error directive will not compile
//#error Unsupported version of Visual Studio! Must use VS2010 or greater.
#endif

typedef unsigned long DWORD;
#endif

typedef void *HANDLE;
typedef void *PVOID;
typedef uint64_t QWORD;
typedef uint64_t ULONG64;

#define ARG_USED(x)    while ((void)(x), 0)

#if !defined(__cplusplus)
typedef uint8_t bool;
#endif

typedef          int8_t I8;      //  8-bit signed integer
typedef          int64_t I64;    // 64-bit signed integer
typedef          uint64_t U64;   // 64-bit unsigned integer
typedef          int16_t I16;    // 16-bit signed integer
typedef          int32_t I32;    // 32-bit signed integer

typedef          uint8_t U8;     //  8-bit unsigned integer
typedef          uint16_t U16;   // 16-bit unsigned integer
typedef          uint32_t U32;   // 32-bit unsigned integer

typedef            float F32;    // 32-bit single precision floating point value
typedef            double F64;   // 64-bit double precision floating point value

#endif // _TYPES_H


#if defined(WINDOWS)
#include <errno.h>
#elif defined(LINUX)
#include <unistd.h>     // For getopt_long(), fsync(), etc.
#endif //__linux__


//////////////////////////////////////////////////////////////////////////////
// Not yet implemented
//////////////////////////////////////////////////////////////////////////////

#define os_sscanf1    sscanf
#define os_sscanf2    sscanf
#define os_strcat     strcat


//////////////////////////////////////////////////////////////////////////////
// Windows specifics
//////////////////////////////////////////////////////////////////////////////

#if defined(WINDOWS)

// Use Windows equivalent
#define popen          _popen
#define pclose         fclose
#define strncasecmp    _strnicmp
#define strtok_r       strtok_s

// Do not use these Windows values
#define __THROW        // Ignore this common modifier
#define ssize_t        SSIZE_T

// Redefined functions
#define os_fopen(fd, name,                                                 \
                 mode)                                (!fopen_s(&fd, name, \
                                                                mode))

// Windows needs this, but Linux does not
#define va_copy(ap, args)                             (ap) = (args)

// Use safe version (typically defined on Windows already)
#define os_vcprintf(fmt, ap)                          _vscprintf(fmt, ap)
#define os_vsnprintf(s, size, count, format, args)    _vsnprintf_s(s, size, \
                                                                   count,   \
                                                                   format, args)
#define os_strncpy(destination, size, source, num)    strncpy_s(destination,  \
                                                                size, source, \
                                                                num)
#define os_strncat(destination, size, source, num)    strncat_s(destination,  \
                                                                size, source, \
                                                                num)
#define os_sprintf(str, size, format, ...)            sprintf_s(str, size, \
                                                                format,    \
                                                                __VA_ARGS__)
#define os_vsprintf(str, size, args)                  vsprintf_s(str, size, \
                                                                 args)
#define os_memcpy(dst, dstSize, src, maxCount)        ::memcpy_s(dst, dstSize, \
                                                                 src, maxCount)
#define os_memmove(dst, dstSize, src, maxCount)       ::memmove_s(dst, dstSize, \
                                                                  src, maxCount)


#define VIRTUAL_ALLOC(size)                           VirtualAlloc(NULL, size,    \
                                                                   MEM_COMMIT,    \
                                                                   PAGE_READWRITE \
                                                                   |              \
                                                                   PAGE_NOCACHE)
#define VIRTUAL_FREE(address)                         VirtualFree(address, 0, \
                                                                  MEM_RELEASE)


//////////////////////////////////////////////////////////////////////////////
// Linux specifics
//////////////////////////////////////////////////////////////////////////////

#elif defined(LINUX)

// Redefined functions
#define os_fopen(fd, name, mode)                      (NULL != \
                                                       (fd = fopen(name, mode)))

#define os_vcprintf(fmt, ap)                          vsnprintf(NULL, 0, fmt, \
                                                                ap)
#define os_vsnprintf(s, size, count, format, args)    vsnprintf(s, count, \
                                                                format, args)
#define os_strncpy(destination, size, source, num)    strncpy(destination, \
                                                              source, num);
#define os_strncat(destination, size, source, num)    strncat(destination, \
                                                              source, num)
#define os_sprintf(str, size, format, ...)            sprintf(str, format, \
                                                              ## __VA_ARGS__)
#define os_vsprintf(str, size, args)                  vsprintf(str, args)
#define os_memcpy(dst, dstSize, src,                                    \
                  maxCount)                           (memcpy(dst, src, \
                                                              maxCount), 0)
#define os_memmove(dst, dstSize, src,                                    \
                   maxCount)                          (memmove(dst, src, \
                                                               maxCount), 0)

#define VIRTUAL_ALLOC(size)                           memalign(sysconf(           \
                                                                   _SC_PAGESIZE), \
                                                               size)
#define VIRTUAL_FREE(address)                         free(address)

#define Sleep(ms)                                     usleep((ms) * 1000)


//////////////////////////////////////////////////////////////////////////////
// Unsupported
//////////////////////////////////////////////////////////////////////////////

#else

#error Unsupported O/S! Not Linux or Windows


// Legacy code

// This is not safe: to make safe replace all uses of os_sprintf, os_strncpy and os_vsprintf
const size_t MAX_STRING_LEN = 1024 * 1024;

// [todo] Write a general replacement vcscanf with variable nmber of args
// [todo] This is not available on all systems

inline int os_sscanf1(const char *str, const char *format, void *p)
{
    return sscanf_s(str, format, p);
}

inline int os_sscanf2(const char *str, const char *format, void *p, void *q)
{
    return sscanf_s(str, format, p, q);
}

#endif


// Ensure needed value is defined to a reasonable value
#if !defined(PATH_MAX)
#define PATH_MAX    4096
#elif PATH_MAX > 4096
#undef PATH_MAX
#define PATH_MAX    4096
#endif


#endif // _OSAL_H_

#ifdef __cplusplus
extern "C"
{
#endif
//////////////////////////////////////////////////////////////////////////////
// List of common values:
//
// strchrnul()
// fileno()
//
//////////////////////////////////////////////////////////////////////////////

#ifdef UNICODE
#undef UNICODE
#endif

#ifdef _UNICODE
#undef _UNICODE
#endif


//////////////////////////////////////////////////////////////////////////////
// Windows definitions
//////////////////////////////////////////////////////////////////////////////


#if defined(WINDOWS)

// Windows-specific
#define _TCHAR    char

// Types

// String functions

#define fileno(f)    _fileno(f)


/// Return a pointer to the first occurrence of the character in the string, or the end of the string.
/// \attention Null is returned if null was passed for the string.
char *strchrnul(const char *s, int c);

/// Return the filename of a path, just like on Linux.
const char *basename(const char *name);

// OS-specific functions

// Flush data to disk
int fsync(int fd);

#endif  // WINDOWS
#ifdef __cplusplus
}
#endif
#endif  // OS_AGNOSTIC_H


#ifdef __cplusplus
extern "C"
{
#endif

#define NO_MORE_ARGUMENTS    (-1)
#define false                0
#define true                 1


//////////////////////////////////////////////////////////////////////////////
// Linux definitions
//////////////////////////////////////////////////////////////////////////////

#if defined(LINUX)

// For getopt_long(), fsync(), etc.
#include <getopt.h>

typedef struct option commandline_option_t;

# if defined(TESTING)
# define IMPLEMENT_GETOPT
# endif

#endif


//////////////////////////////////////////////////////////////////////////////
// Windows definitions
//////////////////////////////////////////////////////////////////////////////

#if defined(WINDOWS)

# define IMPLEMENT_GETOPT

extern char *optarg;
extern int optind, opterr, optopt;

enum GetOptArgs {
    no_argument,
    required_argument,
    optional_argument
};

typedef struct option {
    const char *name;
    int         has_arg;
    int *       flag;
    int         val;
} commandline_option_t;

// POSIX.11-standard routine, written for Windows, to get command line options
# define _getopt         getopt
# define _getopt_long    getopt_long

#endif


//////////////////////////////////////////////////////////////////////////////
// Either Windows or Linux (if testing) definitions
//////////////////////////////////////////////////////////////////////////////

#if defined(IMPLEMENT_GETOPT)

extern int getopt_nextchar;

int _getopt(int argc, char *const argv[], const char *optstring) __THROW;
int _getopt_long(int argc, char *const argv[], const char *optstring, const
                 struct option *longopts, int *longindex) __THROW;

#endif


#ifdef __cplusplus
}
#endif

// Guard
#endif
