#  Copyright 2012-2017, Intel Corporation, All Rights Reserved.
#
# This software is supplied under the terms of a license
# agreement or nondisclosure agreement with Intel Corp.
# and may not be copied or disclosed except in accordance
# with the terms of that agreement.
#
#  Author:  Christopher M. Cantalupo

import os

import micp.kernel as micp_kernel
import micp.info as micp_info
import micp.params as micp_params
import micp.common as micp_common

class shoc_readback(micp_kernel.Kernel):
    def __init__(self):
        self.name = 'shoc_readback'
        self.param_validator = micp_params.SHOC_VALIDATOR
        self._paramNames = ['target', 'passes', 'nopinned']

        info = micp_info.Info()
        devIdx = info.get_device_index()

        self._paramDefaults = {'target':str(devIdx),
                               'passes':'10',
                               'nopinned':None}
        self._categoryParams = {}
        self._categoryParams['optimal'] = [' ']
        self._categoryParams['scaling'] = [' ']
        self._categoryParams['test'] = [' ']
        self._categoryParams['optimal_quick'] = [' ']
        self._categoryParams['scaling_quick'] = [' ']
        self._categoryParams['scaling_core'] = [' ']

        self.paramDrop = {}
        self.paramMax = {}
        self.paramDrop['scif'] = 3
        self.paramMax['scif'] = 0
        self._set_defaults_to_optimal()

    def path_host_exec(self, offType):
        if offType == 'pragma' or offType == 'scif':
            if micp_common.is_platform_windows():
                binary_name = 'BusSpeedReadback_{0}.exe'.format(offType)
            else:
                binary_name = 'BusSpeedReadback_{0}'.format(offType)
            return self._path_exec(micp_kernel.LIBEXEC_HOST, binary_name)
        return None

    def path_dev_exec(self, offType):
        if offType == 'scif':
            return self._path_exec(micp_kernel.LIBEXEC_DEV, 'BusSpeed_mic')
        return None

    def independent_var(self, category):
        return 'size_kB'

    def environment_host(self):
        return dict([('MIC_USE_2MB_BUFFERS','32K'),
                     ('LD_LIBRARY_PATH', self.ld_library_path()),
                     ('MIC_LD_LIBRARY_PATH', self.mic_ld_library_path())])


    def environment_dev(self):
        return dict([('USE_2MB_BUFFERS','32K')])

    def _do_unit_test(self):
        return False

    def offload_methods(self):
         return ['scif', 'pragma']

    def parse_desc(self, raw):
        return ['SHOC Readback ' + line.split()[1]
                for line in raw.splitlines()
                if line.startswith('ReadbackSpeed')]


    def parse_perf(self, raw):
        result = []
        lines = [line for line in raw.splitlines()
                 if line.startswith('ReadbackSpeed')]
        for line in lines:
            stat = {}
            words = line.split()
            tag = 'TransferRate'
            data = {}
            data['size_kB'] = words[1].split('kB')[0]
            data['value'] = str(float(words[3])*(1024**3)*1e-9) # GiB/s -> GB/s
            data['units'] = 'GB/s'
            data['rollup'] = True
            stat[tag] = data
            result.append(stat)
        return result

    def param_type(self):
        return 'value'

    def internal_scaling(self):
        return True

    def device_param_name(self):
        return 'target'
