# Copyright 2010-2017 Intel Corporation.
# 
# This library is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, version 2.1.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
# 
# Disclaimer: The codes contained in these modules may be specific
# to the Intel Software Development Platform codenamed Knights Ferry,
# and the Intel product codenamed Knights Corner, and are not backward
# compatible with other Intel products. Additionally, Intel will NOT
# support the codes or instruction set in future products.
# 
# Intel offers no warranty of any kind regarding the code. This code is
# licensed on an "AS IS" basis and Intel is not obligated to provide
# any support, assistance, installation, training, or other services
# of any kind. Intel is also not obligated to provide any updates,
# enhancements or extensions. Intel specifically disclaims any warranty
# of merchantability, non-infringement, fitness for any particular
# purpose, and any other warranty.
# 
# Further, Intel disclaims all liability of any kind, including but
# not limited to liability for infringement of any proprietary rights,
# relating to the use of the code, even if Intel is notified of the
# possibility of such liability. Except as expressly stated in an Intel
# license agreement provided with this code and agreed upon with Intel,
# no license, express or implied, by estoppel or otherwise, to any
# intellectual property rights is granted herein.

"""Common fixtures to all UT's"""

import platform
import subprocess
import os
import pytest
import socket

import micp.common as micp_common
import micp.info as micp_info
import micp.connect as micp_connect
import micp.params as micp_params
import micp.stats as micp_stats

import mocks.popen_factory as popen_factory
import mocks.python_mocks as python_mocks
import mocks.data as mocks_data
import common_utils

CARD_DUMMY_IP_ADDRESS = '172.31.1.1'
CARD_USER = 'mic'

@pytest.fixture()
def running_on_linux(monkeypatch):
    """minimal fixture to simulate a Linux environment"""
    platform_mock = \
        python_mocks.platform(mocks_data.PYTHON_FOR_LINUX, mocks_data.VALID)
    monkeypatch.setattr(micp_common, 'is_platform_windows', lambda: False)
    monkeypatch.setattr(platform, 'uname', platform_mock.uname())
    monkeypatch.setattr(platform, 'dist', platform_mock.dist())


@pytest.fixture()
def running_on_windows(monkeypatch):
    """minimal fixture to simulate a Windows environment"""
    platform_mock = \
        python_mocks.platform(mocks_data.PYTHON_FOR_WINDOWS, mocks_data.VALID)
    monkeypatch.setattr(micp_common, 'is_platform_windows', lambda: True)
    monkeypatch.setattr(platform, 'uname', platform_mock.uname())
    monkeypatch.setattr(platform, 'dist', platform_mock.dist())


def _monkeypatch_python(monkeypatch, fixtures):
    """receives a monkeypatch object and a dictionary that maps the
    names of modules in python standard library to a fixture, these
    fixtures are used to 'monkeypatch' the python standard library"""

    os_fixture = fixtures['os']
    platform_fixture = fixtures['platform']
    subprocess_fixture = fixtures['subprocess']

    monkeypatch.setattr(
        subprocess, 'Popen',
        lambda cmd, stdout=None, stdin=None, stderr=None, bufsize=None, shell=None:
            popen_factory.create(subprocess_fixture, cmd))

    monkeypatch.setattr(
        os, 'listdir',
        lambda path:
            os_fixture.listdir(path))

    monkeypatch.setattr(
        os.path, 'join',
        lambda *tokens:
            os_fixture.path_join(tokens))

    monkeypatch.setattr(
        os.path, 'exists',
        lambda path:
            os_fixture.path_exists(path))

    monkeypatch.setattr(os, 'pathsep', os_fixture.pathsep)
    monkeypatch.setattr(os, 'environ', os_fixture.environ)
    monkeypatch.setattr(platform, 'uname', lambda: platform_fixture.uname())
    monkeypatch.setattr(platform, 'dist', lambda: platform_fixture.dist())


@pytest.fixture()
def mock_network_functions(monkeypatch):
    """mock network related settings used by micperf"""
    monkeypatch.setattr(micp_connect.MPSSConnect, 'get_user',
                        lambda __: CARD_USER)

    monkeypatch.setattr(socket, 'gethostbyname',
                        lambda __: CARD_DUMMY_IP_ADDRESS)


FULL_TEST_ENVIROMENTS = [
    (popen_factory.LINUX_HOST_KNC_MICINFO_KNC_CARD,
     mocks_data.PYTHON_FOR_LINUX, "linux"),
    (popen_factory.WIN_HOST_KNC_MICINFO_KNC_CARD,
     mocks_data.PYTHON_FOR_WINDOWS, "windows"),
    (popen_factory.LINUX_HOST_NO_MICINFO_KNX_CARD,
     mocks_data.PYTHON_FOR_LINUX, "linux"),
    (popen_factory.LINUX_HOST_KNL_MICINFO_KNC_CARD,
     mocks_data.PYTHON_FOR_LINUX, "linux"),
    (popen_factory.LINUX_HOST_KNL_MICINFO_KNL_CARD,
     mocks_data.PYTHON_FOR_LINUX, "linux")]
@pytest.fixture(params=FULL_TEST_ENVIROMENTS)
def _full_test_enviroment(monkeypatch, request, mock_network_functions):
    """provides mocks for full test environments: Linux/Windows,
    KNL/KNC micinfo, KNC/KNL card. This 'environments' are all
    that micperf requires to create InfoKNXXB objects which are
    used to store the hardware and software information of a
    system.

    TODO: Add missing windows cases:
      - WIN_HOST_KNL_MICINFO_KNC_CARD
      - WIN_HOST_KNL_MICINFO_KNL_CARD
    """
    python_fixtures = {}
    host_mock_name, python_mock_name, os_name = request.param
    os_mock = python_mocks.os(python_mock_name, mocks_data.VALID)
    platform_mock = python_mocks.platform(python_mock_name, mocks_data.VALID)

    python_fixtures['platform'] = platform_mock
    python_fixtures['os'] = os_mock
    python_fixtures['subprocess'] = host_mock_name
    _monkeypatch_python(monkeypatch, python_fixtures)

    monkeypatch.setattr(
        micp_common, 'is_platform_windows', lambda: os_name == "windows")


BASIC_TEST_ENVIROMENTS = [
    (popen_factory.LINUX_HOST_KNC_MICINFO_KNC_CARD,
     mocks_data.PYTHON_FOR_LINUX, "linux"),
    (popen_factory.WIN_HOST_KNC_MICINFO_KNC_CARD,
     mocks_data.PYTHON_FOR_WINDOWS, "windows")]
@pytest.fixture(params=BASIC_TEST_ENVIROMENTS)
def _basic_test_enviroment(monkeypatch, request, mock_network_functions):
    """
    Copy of _full_test_enviroment, the only difference is that os_enviroment_mock
    only mocks a single linux environment and a single windows environment
    this is used for test cases that should be tested on both operating systems
    but not in all the possible combinations.
    """
    python_fixtures = {}
    host_mock_name, python_mock_name, os_name = request.param
    os_mock = python_mocks.os(python_mock_name, mocks_data.VALID)
    platform_mock = python_mocks.platform(python_mock_name, mocks_data.VALID)

    python_fixtures['platform'] = platform_mock
    python_fixtures['os'] = os_mock
    python_fixtures['subprocess'] = host_mock_name
    _monkeypatch_python(monkeypatch, python_fixtures)

    monkeypatch.setattr(
        micp_common, 'is_platform_windows',
        lambda: os_name == "windows")


NEGATIVE_TEST_ENVIROMENTS = [
    (popen_factory.NEG_LINUX_HOST_KNC_MICINFO_KNC_CARD,
     mocks_data.PYTHON_FOR_LINUX, "linux")]
@pytest.fixture(params=NEGATIVE_TEST_ENVIROMENTS)
def _negative_basic_test_enviroment(monkeypatch, request):
    """Copy of _full_test_environment, the difference is
    _negative_basic_test_enviroment uses invalid data for
    the test environments this is useful for negative testing.
    """
    python_fixtures = {}
    host_mock_name, python_mock_name, os_name = request.param
    os_mock = python_mocks.os(python_mock_name, mocks_data.VALID)
    platform_mock = python_mocks.platform(python_mock_name, mocks_data.VALID)

    python_fixtures['platform'] = platform_mock
    python_fixtures['os'] = os_mock
    python_fixtures['subprocess'] = host_mock_name
    _monkeypatch_python(monkeypatch, python_fixtures)

    monkeypatch.setattr(
        micp_common, 'is_platform_windows',
        lambda: os_name == "windows")


@pytest.fixture()
def single_card_pci(monkeypatch):
    """force all methods that determine (by looking at the pci devices
    attached) the number of cards in the host to return 1"""
    monkeypatch.setattr(
        micp_info.InfoKNXXB, '_num_mics_pci',
        lambda self: 1)

    monkeypatch.setattr(
        micp_common, 'num_mics_pci',
        lambda: (micp_common.KNC, 1))


@pytest.fixture()
def init_incomplete_knc_object(monkeypatch, _full_test_enviroment, single_card_pci):
    """
    The creation of an InfoKNXXB object involves two functions:
    _init_command_dict() and _init_micinfo_dict(). This fixture
    sets the return value of init_micinfo_dict() to None and mocks
    some dependencies of init_command_dict() which isolates the
    core of init_command_dict() for testing purposes.

    IMPORTANT: objects created this way are not fully functional.
    """

    monkeypatch.setattr(
        micp_info.InfoKNXLB, '_init_micinfo_dict',
        lambda self: None)

    monkeypatch.setattr(
        micp_info.InfoKNXLB, '_get_command_list',
        lambda self: ['micinfo'])

    return micp_info.InfoKNXLB()


@pytest.fixture()
def negative_init_incomplete_knc_object(monkeypatch,
        _negative_basic_test_enviroment, single_card_pci):
    """copy of init_incomplete_knc_object, the difference is
    negative_init_incomplete_knc_object uses invalid data for
    the test environments which is useful for negative testing"""
    monkeypatch.setattr(
        micp_info.InfoKNXLB, '_init_micinfo_dict',
        lambda self: None)

    monkeypatch.setattr(
        micp_info.InfoKNXLB, '_get_command_list',
        lambda self: ['micinfo'])

    return micp_info.InfoKNXLB()


@pytest.fixture()
def init_full_knxlb_object(monkeypatch, _full_test_enviroment, single_card_pci):
    """
    The creation of an InfoKNXXB object involves two functions:
    _init_command_dict() and _init_micinfo_dict(). This fixture
    mocks only external dependencies for both methods, objects
    created with this mock are fully functional.
    Since this fixture uses _full_test_enviroment it will create
    InfoKNXXB object for different configurations e.g. Linux-KNC
    card-KNC micinfo, Windows-KNL micinfo-KNC card, etc. see
    _full_test_enviroment for details.
    """
    monkeypatch.setattr(
        micp_info.InfoKNXLB, '_get_command_list',
        lambda self: ['micinfo'])

    return micp_info.InfoKNXLB()


@pytest.fixture()
def init_basic_knxlb_object(monkeypatch, _basic_test_enviroment, single_card_pci):
    """
    Copy of init_full_knxlb_object() the only difference is that
    init_basic_knxlb_object() receives the parametrized mock
    basic_test_environment which creates a single test environment
    for Linux and a single test environment for windows.
    This is useful for test cases that should be executed on both
    Linux and Windows but not all the possible combinations.
    """
    monkeypatch.setattr(
        micp_info.InfoKNXLB, '_get_command_list',
        lambda self: ['micinfo'])

    return micp_info.InfoKNXLB()


@pytest.fixture()
def stats(request):
    """fixture to mock stats objects for Stats, StatsCollection
    and StatsCollectionStore tests"""
    perf = {'Computation.Avg':{'value':'1000', 'units':'GFlops', 'rollup':True}}
    params_object = micp_params.Params(common_utils.PARAM_CLI,
                                       common_utils.PARAM_NAMES)
    return micp_stats.Stats(params_object, common_utils.DESCRIPTION, perf)


@pytest.fixture(params=[
    {'paramCat':'optimal', 'offMethod':'native', 'devIdx':'0'},
    {'paramCat':'scaling', 'offMethod':'native', 'devIdx':'0'}
])
def stats_collection(init_basic_knxlb_object, request, stats):
    """fixture to mock stats objects for, StatsCollection
    and StatsCollectionStore tests"""
    run_args = request.param
    return micp_stats.StatsCollection(run_args)


@pytest.fixture(params=[
    {'paramCat':'optimal', 'offMethod':'native', 'devIdx':'0'},
    {'paramCat':'scaling', 'offMethod':'native', 'devIdx':'0'}
])
def stats_collection(init_basic_knxlb_object, request):
    """fixture to mock stats objects for StatsCollectionStore tests"""
    run_args = request.param
    return micp_stats.StatsCollection(run_args)
