# Copyright 2010-2017 Intel Corporation.
# 
# This library is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, version 2.1.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
# 
# Disclaimer: The codes contained in these modules may be specific
# to the Intel Software Development Platform codenamed Knights Ferry,
# and the Intel product codenamed Knights Corner, and are not backward
# compatible with other Intel products. Additionally, Intel will NOT
# support the codes or instruction set in future products.
# 
# Intel offers no warranty of any kind regarding the code. This code is
# licensed on an "AS IS" basis and Intel is not obligated to provide
# any support, assistance, installation, training, or other services
# of any kind. Intel is also not obligated to provide any updates,
# enhancements or extensions. Intel specifically disclaims any warranty
# of merchantability, non-infringement, fitness for any particular
# purpose, and any other warranty.
# 
# Further, Intel disclaims all liability of any kind, including but
# not limited to liability for infringement of any proprietary rights,
# relating to the use of the code, even if Intel is notified of the
# possibility of such liability. Except as expressly stated in an Intel
# license agreement provided with this code and agreed upon with Intel,
# no license, express or implied, by estoppel or otherwise, to any
# intellectual property rights is granted herein.

"""UT's for micp/common.py"""

import pytest
import sys
import subprocess

import micp.common as micp_common

import mocks.popen_factory as popen_factory


#=============================================================================
#                                Local Fixtures
#=============================================================================

@pytest.fixture()
def knl_sb_platform(monkeypatch):
    """mocks a KNL SB platform"""
    monkeypatch.setattr(
        subprocess, 'Popen',
        lambda cmd, stdout=None, stdin=None, stderr=None, bufsize=None, shell=None:
           popen_factory.create(popen_factory.SB_HOST_VALID_INFO, cmd)
    )


@pytest.fixture()
def spurious_knl_sb_platform(monkeypatch):
    """mocks a non KNL SB platform or a not fully configured KNL SB
    platform e.g. memkind not installed"""
    monkeypatch.setattr(
        subprocess, 'Popen',
        lambda cmd, stdout=None, stdin=None, stderr=None, bufsize=None, shell=None:
            popen_factory.create(popen_factory.SB_HOST_INVALID_INFO, cmd)
    )

#=============================================================================
#                                 Test Cases
#=============================================================================

def test_host_is_selfboot(running_on_linux, knl_sb_platform):
    """validate is_selfboot_platform() identifies a SB platform"""
    # reset static varible before running test case
    micp_common.is_selfboot_platform.result = None
    assert micp_common.is_selfboot_platform()


def test_host_is_not_selfboot(running_on_linux, spurious_knl_sb_platform):
    """validate is_selfboot_platform() identifies a non SB platform """
    # reset static varible before running test case
    micp_common.is_selfboot_platform.result = None
    assert not micp_common.is_selfboot_platform()


def test_host_is_selfboot_windows(running_on_windows):
    """KNL SB running windows is not supported at this time"""
    assert not micp_common.is_selfboot_platform()


@pytest.mark.parametrize("exit_code, message, expected_out, expected_err",
        [(-1, 'error', '', 'error'), (0, 'success', 'success\n', '')])
def test_exit_application(exit_code, message, expected_out, expected_err,
                          monkeypatch, capsys):
    """OS independent tests to validate applications (micprun, micpcsv, etc)
    can be terminated by calling micp_common.exit_application() and that exit
    messages are printed to stdout or stderr depending on the exit code"""

    # override sys.exit() to prevent pytest from exiting
    monkeypatch.setattr(sys, 'exit', lambda exit_code: None)

    # run exit_application() and capture stdout and stderr
    micp_common.exit_application(message, exit_code)
    out, err = capsys.readouterr()

    assert out == expected_out
    assert err == expected_err
