# Copyright 2010-2017 Intel Corporation.
# 
# This library is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, version 2.1.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
# 
# Disclaimer: The codes contained in these modules may be specific
# to the Intel Software Development Platform codenamed Knights Ferry,
# and the Intel product codenamed Knights Corner, and are not backward
# compatible with other Intel products. Additionally, Intel will NOT
# support the codes or instruction set in future products.
# 
# Intel offers no warranty of any kind regarding the code. This code is
# licensed on an "AS IS" basis and Intel is not obligated to provide
# any support, assistance, installation, training, or other services
# of any kind. Intel is also not obligated to provide any updates,
# enhancements or extensions. Intel specifically disclaims any warranty
# of merchantability, non-infringement, fitness for any particular
# purpose, and any other warranty.
# 
# Further, Intel disclaims all liability of any kind, including but
# not limited to liability for infringement of any proprietary rights,
# relating to the use of the code, even if Intel is notified of the
# possibility of such liability. Except as expressly stated in an Intel
# license agreement provided with this code and agreed upon with Intel,
# no license, express or implied, by estoppel or otherwise, to any
# intellectual property rights is granted herein.

"""UT's for micp/info.py"""

import pytest

import micp.info as micp_info
import micp.common as micp_common


# dummy values for test cases below
# they match the information in mocks/data.py
DUMMY_HOST_OS = 'Linux'
DUMMY_WIN_HOST_OS = 'Windows'
DUMMY_WIN_OS_VERSION = 'Microsoft Windows Server 2012 Standard'
DUMMY_HOST_NAME = 'localhost.localdomain'
DUMMY_HOST_KERNEL = '2.6.32-431.el6.x86_64'

LOCALHOST = -1
DUMMY_MEMSIZE = 33614520320
DUMMY_CPU_SPEED = '1200.0'
DUMMY_CPU_FAMILY = 'GenuineIntel Family 6 Model 45 Stepping 7'
DUMMY_CPU_CORES = 1

DUMMY_CARD = 'mic0'
DUMMY_MIC_CORES = 61
DUMMY_MIC_MEMORY_SIZE = 16642998272    # bytes == 15872 MB

MPSS_VERSION = '4.0'

#===============================================================================
#                            Auxiliary functions
#===============================================================================

def validate_key_in_info_dictionary(knx_info_object, key, sizeof_dictionary):
    """validates if 'key' is part of the _commandDict dictionary in the given
    knx_info_object, such dictionary is the 'heart' of the InfoKNXXB objects"""
    knx_info_dict = knx_info_object._commandDict
    assert len(knx_info_dict.keys()) == sizeof_dictionary
    assert key in knx_info_dict


def validate_function_result(result, out, err,
        expected_result, expected_out, expected_err):
    """compares the exit code, output message and error message returned by
    a function against the corresponding expected values. out* and err* are
    messages printed to stdout and stderr respectively."""
    assert result == expected_result
    assert out == expected_out
    assert err == expected_err


def micinfo_not_available(knxlb_info):
    """analyzes the private member _commandDict to determine
    if micinfo was found, returns verdict"""
    return 'command not found' in knxlb_info._commandDict['micinfo']

#===============================================================================
#                                     Tests
#===============================================================================


#-------------------------------------------------------------------------------
#                               InfoKNXLB Testing
#-------------------------------------------------------------------------------

# Tests for InfoKNXLB and InfoKNXXB helper methods
# These tests require a partial

def test_init_knc_command_list_happy_path(monkeypatch,
                                          init_incomplete_knc_object):
    """validate happy path in method _knc_command_list()"""
    monkeypatch.setattr(
        micp_info.InfoKNXLB, '_get_command_list',
        lambda self: ['micinfo']
    )
    knc_info_object = micp_info.InfoKNXLB()
    validate_key_in_info_dictionary(knc_info_object, 'micinfo', 1)


def test_init_knc_command_list_os_exception(monkeypatch,
                                            init_incomplete_knc_object):
    """validate execution path where a command failed and after catching
    an exception _knc_command_list() attempts to re-run the same command
    prepending '/sbin/' to the comand"""

    monkeypatch.setattr(
        micp_info.InfoKNXLB, '_get_command_list',
        lambda self: ['lspci']
    )
    knc_info_object = micp_info.InfoKNXLB()
    validate_key_in_info_dictionary(knc_info_object, '/sbin/lspci', 1)


def test_get_mpss_version_from_rpm(monkeypatch,
                                   init_incomplete_knc_object, capsys):
    """validate _get_mpss_version_from_rpm() can retrive
    the MPSS version from the rpm name """
    mpss_version = init_incomplete_knc_object._get_mpss_version_from_rpm()
    out, err = capsys.readouterr()
    validate_function_result(mpss_version, out, err, MPSS_VERSION, '', '')


def test_get_default_mpss_version_from_rpm(
        negative_init_incomplete_knc_object, capsys):
    """negative test case, validate MPSS version is set to 4.0 when
    micperf is unable to determine the version."""
    mpss_version = \
            negative_init_incomplete_knc_object._get_mpss_version_from_rpm()
    out, err = capsys.readouterr()
    expected_error = ('WARNING: unable to determine MPSS '
            'version from installed RPM\'s, setting to 4.0\n')
    validate_function_result(mpss_version, out, err,
                             MPSS_VERSION, '', expected_error)


@pytest.mark.parametrize("value, units, new_units, expected", [
    [32826680, 'b', micp_info.InfoKNXXB._MBYTES, 31],
    (32826680, 'B', micp_info.InfoKNXXB._BYTES, 32826680),
    (32826680, 'kb', micp_info.InfoKNXXB._MBYTES, 32057),
    (32826680, 'KB', micp_info.InfoKNXXB._BYTES, 33614520320),
    (32826680, 'MB', micp_info.InfoKNXXB._MBYTES, 32826680),
    (32826680, 'mb', micp_info.InfoKNXXB._BYTES, 34421268807680),
    (32826680, 'gb', micp_info.InfoKNXXB._BYTES, 35247379259064320),
    (32826680, 'GB', micp_info.InfoKNXXB._MBYTES, 33614520320),
])
def test_convert_to_xbytes(value, units, new_units, expected,
        init_incomplete_knc_object, capsys):
    """parametrized test to validate _convert_to_xbytes() can convert
    among the different units defined in the method"""
    result = \
        init_incomplete_knc_object._convert_to_xbytes(value, units, new_units)
    out, err = capsys.readouterr()
    validate_function_result(result, out, err, expected, '', '')

# TODO: Negative test cases for _convert_to_xbytes()


def test_get_memory_size_from_meminfo(init_incomplete_knc_object, capsys):
    """validate _get_memory_size_from_meminfo() can retrieve the memory
    size from PROCFS"""
    result = init_incomplete_knc_object._get_memory_size_from_meminfo(LOCALHOST)
    out, err = capsys.readouterr()
    validate_function_result(result, out, err, DUMMY_MEMSIZE, '', '')


def test_get_cpu_family_from_cpuinfo(init_incomplete_knc_object):
    """validate _get_cpu_family_from_cpuinfo() can retrive the
    CPU family from PROCFS"""
    result = init_incomplete_knc_object._get_cpu_family_from_cpuinfo(LOCALHOST)
    assert result == DUMMY_CPU_FAMILY


def test_get_cpu_mhz_from_cpuinfo(init_incomplete_knc_object):
    """validate _get_cpu_mhz_from_cpuinfo() can retrive the
    CPU frequency from PROCFS"""
    result = init_incomplete_knc_object._get_cpu_mhz_from_cpuinfo(LOCALHOST)
    assert result == DUMMY_CPU_SPEED


def test_get_num_cores_from_cpuinfo(init_incomplete_knc_object):
    """validate _get_num_cores_from_cpuinfo() can retrive the
    number of CPUs from PROCFS"""
    result = init_incomplete_knc_object._get_num_cores_from_cpuinfo(LOCALHOST)
    assert result == DUMMY_CPU_CORES


# Tests for host and card info
# These tests require fully functional InfoKNXLB objects

def test_host_info_exists(init_full_knxlb_object):
    """validate that host info exists"""
    knc_info = init_full_knxlb_object
    assert knc_info._HOST in knc_info._micinfoDict


def test_host_info_in_knc_micinfo_dict(init_full_knxlb_object):
    """validates that host info in a KNC object is populated correctly."""
    knc_info = init_full_knxlb_object
    host_info = knc_info._micinfoDict[knc_info._HOST]

    host_memory_in_mb = str(DUMMY_MEMSIZE/(1024**2)) + ' MB'
    assert host_info[knc_info._HOST_PHYSICAL_MEMORY] == host_memory_in_mb

    if micp_common.is_platform_windows():
        # Windows get the host os name from micinfo
        assert host_info['HOST OS'] == DUMMY_WIN_HOST_OS
        assert host_info[knc_info._HOST_OS_VERSION] == DUMMY_WIN_OS_VERSION
    else:
        assert host_info[knc_info._HOST_OS_NAME] == DUMMY_HOST_OS
        assert host_info[knc_info._HOST_OS_VERSION] == DUMMY_HOST_KERNEL
        assert host_info[knc_info._HOST_FAMILY] == DUMMY_CPU_FAMILY
        assert host_info[knc_info._HOST_CPU_SPEED] == DUMMY_CPU_SPEED

    assert host_info[knc_info._MIC_SOFTWARE_VERSION] == MPSS_VERSION


def test_card_info_exists(init_full_knxlb_object):
    """validate mic0's info exists"""
    knc_info = init_full_knxlb_object
    assert DUMMY_CARD in knc_info._micinfoDict


def test_micinfo_data_is_preprocessed(init_full_knxlb_object):
    """validates that selected fields in micinfo's output (used as keys
    in a dictionary) were replaced by new strings to standardized output
    for both KNC micinfo and KNL micinfo"""
    knc_info = init_full_knxlb_object
    card_info = knc_info._micinfoDict[DUMMY_CARD]

    # Information only available when micinfo is installed
    if micinfo_not_available(knc_info):
        pytest.skip()

    test_strings = (('MIC Processor', 'Coprocessor'),
                    ('MIC Silicon', 'Coprocessor'),
                    ('MIC Board', 'Coprocessor'),
                    ('coprocessor', 'Coprocessor'),
                    ('PCie Max payload size', 'PCIe Max payload size'),
                    ('PCie Max read req size', 'PCIe Max read req size'),
                    ('Subsystem ID', 'SubSystem ID'),
                    ('DRAM', 'GDDR'),
                    ('MCDRAM', 'GDDR'))

    card_info_keys = ' '.join(card_info.keys())

    for original_text, new_text in test_strings:
        assert original_text not in card_info_keys
        assert new_text in card_info_keys


def test_card_info_not_leading_0x(init_full_knxlb_object):
    """validates that _init_micinfo_dict() removes leading '0x'
    from selected fields"""
    knc_info = init_full_knxlb_object
    card_info = knc_info._micinfoDict[DUMMY_CARD]

    # Information only available when micinfo is installed
    if micinfo_not_available(knc_info):
        pytest.skip()

    assert not card_info[knc_info._MIC_DEVICE_ID].startswith('0x')
    assert not card_info[knc_info._MIC_VENDOR_ID].startswith('0x')
    assert not card_info[knc_info._MIC_SUBSYSTEM_ID].startswith('0x')


def test_card_info_number_of_cores(init_full_knxlb_object):
    """validates number of cores read from micinfo"""
    card_info = init_full_knxlb_object
    assert card_info.num_cores() == DUMMY_MIC_CORES


def test_card_memory_size(init_full_knxlb_object):
    """validates card's memory size read from micinfo"""
    card_info = init_full_knxlb_object
    assert card_info.mic_memory_size() == DUMMY_MIC_MEMORY_SIZE
