# Copyright 2010-2017 Intel Corporation.
# 
# This library is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, version 2.1.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
# 
# Disclaimer: The codes contained in these modules may be specific
# to the Intel Software Development Platform codenamed Knights Ferry,
# and the Intel product codenamed Knights Corner, and are not backward
# compatible with other Intel products. Additionally, Intel will NOT
# support the codes or instruction set in future products.
# 
# Intel offers no warranty of any kind regarding the code. This code is
# licensed on an "AS IS" basis and Intel is not obligated to provide
# any support, assistance, installation, training, or other services
# of any kind. Intel is also not obligated to provide any updates,
# enhancements or extensions. Intel specifically disclaims any warranty
# of merchantability, non-infringement, fitness for any particular
# purpose, and any other warranty.
# 
# Further, Intel disclaims all liability of any kind, including but
# not limited to liability for infringement of any proprietary rights,
# relating to the use of the code, even if Intel is notified of the
# possibility of such liability. Except as expressly stated in an Intel
# license agreement provided with this code and agreed upon with Intel,
# no license, express or implied, by estoppel or otherwise, to any
# intellectual property rights is granted herein.

"""UT's for micp/kernels/hplinpack.py"""

import micp.kernels.hplinpack as hplinpack
import micp.kernel as micp_kernel
import micp.params as micp_params
import micp.info as micp_info
import micp.common as micp_common
import pytest
import os
import __builtin__


@pytest.fixture
def hplinpack_kernel(init_basic_knxlb_object):
    """create a hplinpack() instance that can be used in all the tests below"""
    return hplinpack.hplinpack()

def test_path_host_exec_local_windows(monkeypatch, hplinpack_kernel):
    monkeypatch.setattr(hplinpack.micp_common, "is_platform_windows", lambda: True)
    with pytest.raises(OSError) as err_info:
        hplinpack_kernel.path_host_exec("local")
    assert err_info.value.message == "ERROR: Windows is not supported at this time"

def test_path_host_exec_local_no_windows_mklroot(monkeypatch, hplinpack_kernel):
    monkeypatch.setattr(hplinpack.micp_common, "is_platform_windows", lambda: False)
    monkeypatch.setattr(os, "environ", {"MKLROOT": "MKLROOT"})
    monkeypatch.setattr(hplinpack_kernel, "_search_path_to_file", lambda directory, binary_name: (directory, binary_name))
    res = hplinpack_kernel.path_host_exec("local")
    assert res == ("MKLROOT", "xhpl_intel64")

def test_path_host_exec_local_no_windows(monkeypatch, hplinpack_kernel):
    monkeypatch.setattr(hplinpack.micp_common, "is_platform_windows", lambda: False)
    monkeypatch.setattr(os, "environ", {})
    with pytest.raises(micp_kernel.NoExecutableError) as err_info:
        hplinpack_kernel.path_host_exec("local")
    assert err_info.value.message == "MKLROOT not in environment.  Source composer's compilervars.sh before running hplinpack"

def test_path_host_exec_no_local(hplinpack_kernel):
    assert hplinpack_kernel.path_host_exec("no_local") is None

def test_parse_desc_not_passed(hplinpack_kernel):
    with pytest.raises(micp_kernel.SelfCheckError) as err_info:
        hplinpack_kernel.parse_desc("NOT PASSED")
    assert err_info.value.message == "HPLinpack execution failed."

def test_parse_desc(hplinpack_kernel):
    desc = """"Some header
    N: 111
    NB : 121
    ...... PASSED"""
    res = hplinpack_kernel.parse_desc(desc)
    assert res == "HPLinpack problem size 111 block size 121"

def test_parse_perf(hplinpack_kernel):
    raw = """T/V                N    NB     P     Q               Time                 Gflops
    --------------------------------------------------------------------------------
    WC06C2C4       40000   336     1     1              35.40            111
    """
    res = hplinpack_kernel.parse_perf(raw)
    assert res == {"Computation.Avg": {'value':111, 'units':'GFlops', 'rollup':True}}

class MockFile():
    filename = ""
    mode = ""
    def write(self, contents):
        self.contents = contents

    def __enter__(self):
        return self

    def __exit__(self, param1, param2, param3):
        pass

def test_param_file(monkeypatch, hplinpack_kernel):
    params = "--problem_size 111 --block_size 222 --hpl_numthreads 333"
    paramsNames = ["problem_size", "block_size", "hpl_numthreads"]
    params_object = micp_params.Params(params, paramsNames)
    tmpdir = "/some_tmp_dir"
    mockfile = MockFile()
    def mock_open(filename, mode):
        mockfile.filename = filename
        mockfile.mode = mode
        return mockfile
    monkeypatch.setattr(hplinpack.tempfile, "mkdtemp", lambda: tmpdir)
    monkeypatch.setattr(__builtin__, "open", mock_open)
    config = hplinpack.HPLINPACK_CONFIG_FILE.format(problem_size=111, block_size=222, hpl_numthreads=333)
    config_file = hplinpack_kernel.param_file(params_object)
    assert config_file == os.path.join(tmpdir, "HPL.dat")
    assert mockfile.contents == config
    assert mockfile.filename == config_file
    assert mockfile.mode == "w"


@pytest.mark.parametrize("ddr_memory_size, expected_matrix_size", [
    (112520,  100000),       # memory size in MB
    (96*1024, 100000),
    (64*1024, 90000),
    (48*1024, 80000),
    (32*1024, 70000),
    (16*1024, 60000),
    (8*1024,  50000)
])
def test_get_max_matrix_size_on_processor(monkeypatch, hplinpack_kernel,
                                          ddr_memory_size, expected_matrix_size):
    """Validate hplinpack._get_max_matrix_size() returns the correct matrix
    size for the Xeon Phi processor. Max matrix size is limited by the amount
    of DDR available"""
    monkeypatch.setattr(micp_info.Info, "ddr_memory_size", lambda __: ddr_memory_size)
    monkeypatch.setattr(micp_common, "is_selfboot_platform", lambda: True)
    assert hplinpack_kernel._get_max_matrix_size() == expected_matrix_size


def test_get_max_matrix_size_on_processor_no_mem(monkeypatch, hplinpack_kernel):
    """validate hplinpack._get_max_matrix_size() raises an exception when there's
    not enough memory on this system to run HPLinpack """
    monkeypatch.setattr(micp_info.Info, "ddr_memory_size", lambda __: 1024)
    monkeypatch.setattr(micp_common, "is_selfboot_platform", lambda: True)
    with pytest.raises(ValueError):
        assert hplinpack_kernel._get_max_matrix_size()


def test_get_max_matrix_size_on_coprocessor(monkeypatch, hplinpack_kernel):
    """Validate hplinpack._get_max_matrix_size() returns the correct matrix
    size for the Xeon Phi Coprocessor. All coprocessors have 16GB of MCDRAM
    memory which limits the max matrix size to 40000"""
    monkeypatch.setattr(micp_common, "is_selfboot_platform", lambda: False)
    assert hplinpack_kernel._get_max_matrix_size() == 40000
