# Copyright 2010-2017 Intel Corporation.
# 
# This library is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, version 2.1.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
# 
# Disclaimer: The codes contained in these modules may be specific
# to the Intel Software Development Platform codenamed Knights Ferry,
# and the Intel product codenamed Knights Corner, and are not backward
# compatible with other Intel products. Additionally, Intel will NOT
# support the codes or instruction set in future products.
# 
# Intel offers no warranty of any kind regarding the code. This code is
# licensed on an "AS IS" basis and Intel is not obligated to provide
# any support, assistance, installation, training, or other services
# of any kind. Intel is also not obligated to provide any updates,
# enhancements or extensions. Intel specifically disclaims any warranty
# of merchantability, non-infringement, fitness for any particular
# purpose, and any other warranty.
# 
# Further, Intel disclaims all liability of any kind, including but
# not limited to liability for infringement of any proprietary rights,
# relating to the use of the code, even if Intel is notified of the
# possibility of such liability. Except as expressly stated in an Intel
# license agreement provided with this code and agreed upon with Intel,
# no license, express or implied, by estoppel or otherwise, to any
# intellectual property rights is granted herein.

"""UT's for micp/kernels/_shoc.py micp/kernels/shoc_download.py and
micp/kernels/shoc_raedback"""

import pytest

import micp.version as micp_version
import micp.kernels.shoc_download as micp_kernels_shoc_download
import micp.kernels.shoc_readback as micp_kernels_shoc_readback
import common_utils

# output has been slightly modified to ease UT development
SHOC_GENERIC_OUTPUT = """Running shoc_{name} --target 0 --passes 10
Please be patient, this may take a few minutes...


test    atts    units   median  mean    stddev  min     max     trial0  trial1  trial2  trial3  trial4  trial5  trial6  trial7  trial8  trial9
{results}Speed          1kB      GiB/sec 0.07193 0.07422 0.02909 0.02480 0.10792 0.02480 0.04956 0.05201 0.09884 0.10531 0.05276 0.05387 0.10718 0.10792 0.08999
{results}Speed          2kB      GiB/sec 0.13837 0.14685 0.05715 0.06273 0.22124 0.08827 0.10066 0.09969 0.20685 0.20437 0.06273 0.11118 0.22124 0.20795 0.16555
{results}Speed          4kB      GiB/sec 0.30482 0.29999 0.08905 0.18815 0.41469 0.18815 0.19745 0.19945 0.40121 0.41469 0.27353 0.22360 0.40738 0.35832 0.33610
{results}Time           1kB      ms      0.01415 0.01589 0.00864 0.00884 0.03846 0.03846 0.01924 0.01834 0.00965 0.00906 0.01808 0.01770 0.00890 0.00884 0.01060

Note: results marked with (*) had missing values such as
might occur with a mixture of architectural capabilities.

[ DESCRIPTION ] SHOC {results} 1kB
[ PERFORMANCE ] TransferRate 0.0772342494003 GB/s R
[ DESCRIPTION ] SHOC {results} 2kB
[ PERFORMANCE ] TransferRate 0.148573656187 GB/s R
[ DESCRIPTION ] SHOC {results} 4kB
"""

SHOC_DOWNLOAD_OUTPUT = SHOC_GENERIC_OUTPUT.format(name='download', results='Download')
SHOC_READBACK_OUTPUT = SHOC_GENERIC_OUTPUT.format(name='readback', results='Readback')


@pytest.fixture()
def shoc_download(init_basic_knxlb_object):
    """fixture to be used in the tests below"""
    return micp_kernels_shoc_download.shoc_download()


@pytest.fixture()
def shoc_readback(init_basic_knxlb_object):
    """fixture to be used in the tests below"""
    return micp_kernels_shoc_readback.shoc_readback()


def test_shoc_download_name(shoc_download):
    """validate the name of the kernel given by the constructor"""
    assert shoc_download.name == 'shoc_download'


def test_shoc_readback_name(shoc_readback):
    """validate the name of the kernel given by the constructor"""
    assert shoc_readback.name == 'shoc_readback'


def test_default_params(shoc_download):
    """validate the kernel default parameters (content of the private
    member _paramDefaults)"""
    expected = {'target' : '0',
                'passes' : '10',
                'nopinned' : None}
    assert shoc_download._paramDefaults == expected


def test_param_categories(shoc_download):
    """validate all param categories are empty since the shoc
    benchmarks don't support scaling"""
    expected = [' ']
    assert shoc_download._categoryParams['optimal'] == expected
    assert shoc_download._categoryParams['scaling'] == expected
    assert shoc_download._categoryParams['test'] == expected
    assert shoc_download._categoryParams['optimal_quick'] == expected
    assert shoc_download._categoryParams['scaling_quick'] == expected
    assert shoc_download._categoryParams['scaling_core'] == expected


def test_scif_parameters(shoc_download):
    """validate the values the constructor uses to initialize
    scif related parameters"""
    assert shoc_download.paramDrop['scif'] == 3
    assert not shoc_download.paramMax['scif']


def test_negative_host_path_exec(shoc_download):
    """validate path_host_exec() returns None when the offload
    method is not supported"""
    non_supported_methods = ('native', 'coi', 'myo', 'auto')
    for offload_method in non_supported_methods:
        assert shoc_download.path_host_exec(offload_method) is None


@pytest.mark.parametrize("offload_method, win_binary, linux_binary", [
    ('pragma', 'BusSpeedDownload_pragma.exe', 'BusSpeedDownload_pragma'),
    ('scif', 'BusSpeedDownload_scif.exe', 'BusSpeedDownload_scif')
])
def test_host_path_exec(shoc_download, offload_method, win_binary, linux_binary):
    """validate path_host_exec() returns the correct path
    depending on the offload method"""
    expected = common_utils.get_workload_binary_path(offload_method,
                                                     win_binary,
                                                     linux_binary)
    binary = shoc_download.path_host_exec(offload_method)
    assert binary == expected


def test_negative_dev_path_exec(shoc_download):
    """validate path_dev_exec() returns None when the offload
    method is not supported"""
    non_supported_methods = ('native', 'coi', 'myo', 'auto', 'pragma')
    for offload_method in non_supported_methods:
        assert shoc_download.path_dev_exec(offload_method) is None


def test_dev_path_exec(shoc_download):
    """validate dev_path_exec() returns the correct path
    depending on the offload method"""

    # skip test when card is not present
    if micp_version.MIC_PERF_CARD_ARCH == 'NONE':
        pytest.skip()

    # shoc_pragma is a made up offload method to ease UT development
    expected = common_utils.get_workload_binary_path('shoc_pragma',
                                                     'BusSpeed_mic',
                                                     'BusSpeed_mic')
    binary = shoc_download.path_dev_exec('scif')
    assert binary == expected


def test_independent_var(shoc_download):
    """validate shoc_ independent variable used to plot perfomance
    results is size_kB"""
    assert shoc_download.independent_var(None) == 'size_kB'


def test_enviroment_host(shoc_download):
    """validate the content of the dictionary that is used to set
    host enviroment variables for shoc_* benchmarks"""
    expected = {'MIC_USE_2MB_BUFFERS' : '32K',
            'LD_LIBRARY_PATH' : common_utils.get_host_library_path(),
            'MIC_LD_LIBRARY_PATH' : common_utils.get_dev_library_path()}
    assert shoc_download.environment_host() == expected


def test_enviroment_dev(shoc_download):
    """validate the content of the dictionary that is used to set card
    side enviroment variables for the execution of shoc kernels"""
    expected = {'USE_2MB_BUFFERS' : '32K'}
    assert shoc_download.environment_dev() == expected


def test_do_unit_test(shoc_download):
    """validate _do_unit_test() is set to False"""
    assert not shoc_download._do_unit_test()


def test_offload_methods(shoc_download):
    """validate offload_method() returns offload methods supported by
    shoc_* benchmarks"""
    expected = set(['scif', 'pragma'])
    assert set(shoc_download.offload_methods()) == expected


def test_parse_desc_shoc_download(shoc_download):
    """validate parse_desc() parses correctly the description in the
    output of shoc_download kernels"""
    transfer_sizes = (1, 2, 4)
    expected = ["SHOC Download {0}kB".format(size) for size in transfer_sizes]
    assert shoc_download.parse_desc(SHOC_DOWNLOAD_OUTPUT) == expected


def test_parse_perf_shoc_download(shoc_download):
    """validate parse_perf() parses correctly the performance results
    in the output of shoc_download"""
    expected = [{'TransferRate': {'units': 'GB/s', 'rollup': True, 'value': '0.0772342494003', 'size_kB': '1'}},
                {'TransferRate': {'units': 'GB/s', 'rollup': True, 'value': '0.148573656187', 'size_kB': '2'}},
                {'TransferRate': {'units': 'GB/s', 'rollup': True, 'value': '0.327297982792', 'size_kB': '4'}}]
    assert shoc_download.parse_perf(SHOC_DOWNLOAD_OUTPUT) == expected


def test_parse_desc_shoc_readback(shoc_readback):
    """validate parse_desc() parses correctly the description in the
    output of shoc_readback kernels"""
    transfer_sizes = (1, 2, 4)
    expected = ["SHOC Readback {0}kB".format(size) for size in transfer_sizes]
    assert shoc_readback.parse_desc(SHOC_READBACK_OUTPUT) == expected


def test_parse_perf_shoc_readback(shoc_readback):
    """validate parse_perf() parses correctly the performance results
    in the output of shoc_readback"""
    expected = [{'TransferRate': {'units': 'GB/s', 'rollup': True, 'value': '0.0772342494003', 'size_kB': '1'}},
                {'TransferRate': {'units': 'GB/s', 'rollup': True, 'value': '0.148573656187', 'size_kB': '2'}},
                {'TransferRate': {'units': 'GB/s', 'rollup': True, 'value': '0.327297982792', 'size_kB': '4'}}]
    assert shoc_readback.parse_perf(SHOC_READBACK_OUTPUT) == expected


def test_param_type(shoc_download):
    """validate param_type() returns 'value' as the param type for shoc"""
    assert shoc_download.param_type() == 'value'


def test_internal_scaling(shoc_download):
    """validate internal scaling is always set"""
    assert shoc_download.internal_scaling()


def test_device_param_name(shoc_download):
    """validate device_param_name() returns 'target' for shoc kernels"""
    assert shoc_download.device_param_name() == 'target'
