"""
This module provides storage and an interface for external clients to request
a dictionary to create mocks. Clients are responsible for the actual mock
implementation this module will only provide the requested dictionary given
a set of input parameter such as the version of micinfo, platform model, etc.

Output content for all the mocks (stdout, stderr, return code) is stored in
the following dictionaries:

    - _MICINFO: contains output for different combination of micinfo (KNC,
      KNL, no micinfo), OS (Linux & Windows) and cards (KNC & KNL).
    - _HOST_DATA: Contains two sub-dictionaries _X86_HOST_DATA and _SB_HOST_DATA
      where specific information for the corresponding platform is stored.
    - _PYTHON_DATA (which in turn contains _PYTHON_FOR_WINDOWS_DATA and
      _PYTHON_FOR_LINUX_DATA) use to mock the python standard library.

Any of the dictionaries above may contain VALID or INVALID data to create mocks
for positive and negative test cases respectively.

In its most basic form these dictionaries should map a command to a tuple of
the form (STDOUT, STDERR, RETURN_CODE). These dictionaries SHOULDN'T BE READ
DIRECTLY, instead mocks should use the public interface and constants defined
below.

IMPORTANT: The output for some commands has been modified to ease UT development
e.g. output has been trimmed or values changed refer to the comments below for
further details.
"""

import micp.version as micp_version

# directories where benchmarks will be installed
HOST_DIR = micp_version.MIC_PERF_HOST_ARCH
CARD_DIR = micp_version.MIC_PERF_CARD_ARCH

# keys to access stdout, stderr and exitcode
STDOUT = 0
STDERR = 1
EXITCODE = 2

# available combinations of KNX micinfo and KNX cards
# LINUX_KNC_MICINFO_KNL_CARD and WIN_KNL_MICINFO_KNC_CARD
# shouldn't be included because KNC micinfo won't be able
# to retrieve information from KNL hardware.
LINUX_KNC_MICINFO_KNC_CARD = 'linux_knc_micinfo_knc_card'
LINUX_KNL_MICINFO_KNC_CARD = 'linux_knl_micinfo_knc_card'
LINUX_KNL_MICINFO_KNL_CARD = 'linux_knl_micinfo_knl_card'
WIN_KNC_MICINFO_KNC_CARD = 'win_knc_micinfo_knc_card'
WIN_KNC_MICINFO_KNL_CARD = 'win_knc_micinfo_knl_card'
WIN_KNL_MICINFO_KNL_CARD = 'win_knl_micinfo_knl_card'
NO_MICINFO = 'no_micinfo'
EMPTY_MICINFO = 'empty_micinfo'

# flags to specify valid or invalid output
# useful to create positive/negative tests.
VALID = 'valid'
INVALID = 'invalid'

# flags to specify the plataform's model
X86_HOST = 'x86_host'
SB_HOST = 'sb_host'

# available mocks for the python standard library
PYTHON_FOR_WINDOWS = "python_for_windows"
PYTHON_FOR_LINUX = "python_for_linux"

XGEMM_OUTPUT = '''benchmarking: sgemm
timer       : native
num_threads : 0
min_niters  : 3
min_t       : 3.000000
first index : 15872
last  index : 15872
step        : 15872
fixed M     : -1
fixed N     : -1
fixed K     : -1
data transf.: maybe (depends on MKL AO setting)
MKL         : 2018.0.0 build 20170712 (Product)
processor   : Intel(R) Advanced Vector Extensions 512 (Intel(R) AVX-512) for Intel(R) Many Integrated Core Architecture (Intel(R) MIC Architecture) with support of AVX512_4FMAPS and AVX512_4VNNIW instruction groups
CPU freq.   : 1.59 (may float due to scaling)
# cores aval: 72
max threads : 288
threads used: 72 (autodetected)
threads/core: 1

#0: NN
testing XGEMM( 'N', 'N', n, n, ... )

          n        min        avg        max     stddev
      15872    1499.59    1500.19    1500.72  1.642e-03
*     15872    1499.59    1500.19    1500.72  1.642e-03
'''

LINPACK_OUTPUT = '''CPU frequency:    1.238 GHz
Number of CPUs: 1
Number of cores: 244
Number of threads: 244

Parameters are set to:

Number of tests: 1
Number of equations to solve (problem size) : 43008
Leading dimension of array                  : 43072
Number of trials to run                     : 3
Data alignment value (in Kbytes)            : 4

Maximum memory requested that can be used=14820390144, at the size=43008

=================== Timing linear equation system solver ===================

Size   LDA    Align. Time(s)    GFlops   Residual     Residual(norm) Check
43008  43072  4      56.472     939.1896 1.962490e-09 3.780653e-02   pass
43008  43072  4      59.295     894.4765 1.962490e-09 3.780653e-02   pass
43008  43072  4      56.688     935.6073 1.962490e-09 3.780653e-02   pass

Performance Summary (GFlops)

Size   LDA    Align.  Average  Maximal
43008  43072  4       923.0912 939.1896
'''

SHOC_OUTPUT = '''test    atts    units   median  mean    stddev  min     max     trial0  trial1  trial2  trial3  trial4  trial5  trial6  trial7  trial8  trial9
ReadbackSpeed          1kB      GiB/sec 0.66069 0.59749 0.13419 0.24952 0.67066 0.24952 0.64177 0.66458 0.66737 0.65680 0.64788 0.67019 0.67066 0.66925 0.4368
6
ReadbackSpeed          2kB      GiB/sec 1.17704 1.16743 0.05519 1.00971 1.20948 1.00971 1.19209 1.16586 1.16800 1.16373 1.20948 1.18249 1.20186 1.17159 1.2094
8
ReadbackSpeed          4kB      GiB/sec 2.00668 1.99030 0.05321 1.84018 2.03995 1.84018 1.97448 1.99513 2.02802 1.98579 2.00985 2.00352 2.01410 2.01197 2.0399
5
ReadbackSpeed          8kB      GiB/sec 3.04082 2.97449 0.15957 2.58975 3.10643 2.95255 3.08009 3.03476 3.06033 2.58975 3.04688 3.02154 3.10643 2.76127 3.0913
3
ReadbackSpeed         16kB      GiB/sec 4.18910 4.13674 0.11978 3.79950 4.22213 4.09632 4.12623 4.19082 3.79950 4.12511 4.18737 4.22213 4.22097 4.20700 4.1919
8
'''

_PICKLE_FILES = [
    'micp_run_stats_B1QS-5110P_mpss-3.5_native-scif-pragma_optimal_mic0.pkl',
    'micp_run_stats_B1QS-5110P_mpss-3.5_native-scif-pragma_scaling_mic0.pkl',
    'micp_run_stats_B1QS-7110-P-A-X_mpss-3.5_native-scif-pragma_optimal_mic0.pkl',
    'micp_run_stats_B1QS-7110-P-A-X_mpss-3.5_native-scif-pragma_scaling_mic0.pkl',
    'micp_run_stats_C0QS-3120-P-A_mpss-3.5_native-scif-pragma_optimal_mic0.pkl',
    'micp_run_stats_C0QS-3120-P-A_mpss-3.5_native-scif-pragma_scaling_mic0.pkl',
    'micp_run_stats_C0QS-5110P_mpss-3.5_native-scif-pragma_optimal_mic0.pkl',
    'micp_run_stats_C0QS-5110P_mpss-3.5_native-scif-pragma_scaling_mic0.pkl',
    'micp_run_stats_C0QS-7120-P-A-X-D_mpss-3.5_native-scif-pragma_optimal_mic0.pkl',
    'micp_run_stats_C0QS-7120-P-A-X-D_mpss-3.5_native-scif-pragma_scaling_mic0.pkl',
    'README.txt'
]

_MICINFO = {

    # Changes to KNL micinfo output to make it compatible with micperf:
    # - Total No. of Active Cores -> Total No of Active Cores
    # - Separate value from units e.g. 32057MB -> 32057 MB
    #
    # Per HSD5161970 is expected that the output will change.

    LINUX_KNL_MICINFO_KNC_CARD : ('''micinfo Utility Log
Created On      16/12/2014      15:54:47

System Info:
    HOST OS                        : Linux
    OS Version                     : 2.6.32-431.el6.x86_64
    Driver Version                 : 3.5-0.1
    MPSS Version                   : 4.0
    Host Physical Memory           : 32057 MB

Device No: 0, Device Name: mic0  (KNC)

Version Info:
    Flash Version                  : 2.1.02.0390
    SMC Firmware Version           : 1.16.5078
    SMC Boot Loader Version        : 1.8.4326
    Coprocessor OS Version         : 2.6.38.8+mpss3.5
    Device Serial Number           : ADKC34300202

Board Info:
    Vendor ID                      : 0x8086
    Device ID                      : 0x225c
    Subsystem ID                   : 0x7d99
    Coprocessor Stepping ID        : 0x2
    UUID                           : a6da4d009dd219379c68a505095c9ef7
    PCIe Width                     : 16
    PCIe Speed                     : 5000000000.00000 GT/s
    PCIe Ext Tag Field             : Disabled
    PCIe No Snoop                  : Enabled
    PCIe Relaxed Ordering          : Disabled
    PCIe Max payload size          : 256
    PCIe Max read req size         : 256
    Coprocessor Model              : 0x01
    Coprocessor Model Ext          : 0x00
    Coprocessor Type               : 0x00
    Coprocessor Family             : 0x0b
    Coprocessor Family Ext         : 0x00
    Coprocessor Stepping           : C0
    Board SKU                      : C0PRQ-7120 P/A/X/D
    ECC Mode                       : Enabled
    SMC HW Revision                : Product 300W Active CS
    PCIe Bus Information           : 0000:83:00.0
    Coprocessor SMBus Address      : 0x00000030

Core Info:
    Total No of Active Cores      : 61
    Threads per Core               : 4
    Voltage                        : 1046000.00000 uV
    Frequency                      : 1238095.00000 kHz

Thermal Info:
    Fan RPM                        : 2700
    Fan PWM                        : 50
    Die Temp                       : 47 C

Memory Info:
    DRAM Vendor                    : Samsung
    DRAM Version                   : 0x06
    DRAM Density                   : 4096.00000 MB
    DRAM Size                      : 15872.00000 MB
    DRAM Technology                : GDDR5
    DRAM Speed                     : 5.50000 GT/s
    DRAM Frequency                 : 2750000.00000 KHz
    DRAM Voltage                   : 1501000.00000 uV
''', '', 0),

    # Changes to KNL micinfo output to make it compatible with micperf:
    # - Total No. of Active Cores -> Total No of Active Cores
    # - Separate value from units e.g. 32057MB -> 32057 MB
    #
    # Per HSD5161970 is expected that the output will change.
    # It is also being assumed that when KNL micinfo displays
    # information about a KNL card it will only show MCDRAM fields
    # DRAM will be shown only for KNC cards.

    LINUX_KNL_MICINFO_KNL_CARD : ('''micinfo Utility Log
Created On      16/12/2014      15:54:47

System Info:
    HOST OS                        : Linux
    OS Version                     : 2.6.32-431.el6.x86_64
    Driver Version                 : 3.5-0.1
    MPSS Version                   : 4.0
    Host Physical Memory           : 32057 MB

Device No: 0, Device Name: mic0  (KNC)

Version Info:
    Flash Version                  : 2.1.02.0390
    SMC Firmware Version           : 1.16.5078
    SMC Boot Loader Version        : 1.8.4326
    Coprocessor OS Version         : 2.6.38.8+mpss3.5
    Device Serial Number           : ADKC34300202

Board Info:
    Vendor ID                      : 0x8086
    Device ID                      : 0x225c
    Subsystem ID                   : 0x7d99
    Coprocessor Stepping ID        : 0x2
    UUID                           : a6da4d009dd219379c68a505095c9ef7
    PCIe Width                     : 16
    PCIe Speed                     : 5000000000.00000 GT/s
    PCIe Ext Tag Field             : Disabled
    PCIe No Snoop                  : Enabled
    PCIe Relaxed Ordering          : Disabled
    PCIe Max payload size          : 256
    PCIe Max read req size         : 256
    Coprocessor Model              : 0x01
    Coprocessor Model Ext          : 0x00
    Coprocessor Type               : 0x00
    Coprocessor Family             : 0x0b
    Coprocessor Family Ext         : 0x00
    Coprocessor Stepping           : C0
    Board SKU                      : C0PRQ-7120 P/A/X/D
    ECC Mode                       : Enabled
    SMC HW Revision                : Product 300W Active CS
    PCIe Bus Information           : 0000:83:00.0
    Coprocessor SMBus Address      : 0x00000030

Core Info:
    Total No of Active Cores      : 61
    Threads per Core               : 4
    Voltage                        : 1046000.00000 uV
    Frequency                      : 1238095.00000 kHz

Thermal Info:
    Fan RPM                        : 2700
    Fan PWM                        : 50
    Die Temp                       : 47 C

Memory Info:
    MCDRAM Size              : 15872.00000 MB
    MCDRAM Configuration     : "TEST_STRING_KNL"
    MCDRAM - DRAM Extension  : "TEST_STRING_KNL"
    MCDRAM - Cache Extension : "TEST_STRING_KNL"
''', '', 0),


    LINUX_KNC_MICINFO_KNC_CARD : ('''MicInfo Utility Log
Created Tue Dec 16 15:58:46 2014


        System Info
                HOST OS                 : Linux
                OS Version              : 2.6.32-431.el6.x86_64
                Driver Version          : 3.5-0.1
                MPSS Version            : 4.0
                Host Physical Memory    : 32826 MB

Device No: 0, Device Name: mic0

        Version
                Flash Version            : 2.1.02.0390
                SMC Firmware Version     : 1.16.5078
                SMC Boot Loader Version  : 1.8.4326
                uOS Version              : 2.6.38.8+mpss3.5
                Device Serial Number     : ADKC34300202

        Board
                Vendor ID                : 0x8086
                Device ID                : 0x225c
                Subsystem ID             : 0x7d99
                Coprocessor Stepping ID  : 2
                PCIe Width               : x16
                PCIe Speed               : 5 GT/s
                PCIe Max payload size    : 256 bytes
                PCIe Max read req size   : 4096 bytes
                Coprocessor Model        : 0x01
                Coprocessor Model Ext    : 0x00
                Coprocessor Type         : 0x00
                Coprocessor Family       : 0x0b
                Coprocessor Family Ext   : 0x00
                Coprocessor Stepping     : C0
                Board SKU                : C0PRQ-7120 P/A/X/D
                ECC Mode                 : Enabled
                SMC HW Revision          : Product 300W Active CS

        Cores
                Total No of Active Cores : 61
                Voltage                  : 0 uV
                Frequency                : 1238095 kHz

        Thermal
                Fan Speed Control        : On
                Fan RPM                  : 900
                Fan PWM                  : 20
                Die Temp                 : 45 C

        GDDR
                GDDR Vendor              : Samsung
                GDDR Version             : 0x6
                GDDR Density             : 4096 Mb
                GDDR Size                : 15872 MB
                GDDR Technology          : GDDR5
                GDDR Speed               : 5.500000 GT/s
                GDDR Frequency           : 2750000 kHz
                GDDR Voltage             : 1501000 uV
''', '', 0),

    WIN_KNC_MICINFO_KNC_CARD : ('''MicInfo Utility Log


Created Mon Jan 05 10:20:06 2015








        System Info


               HOST OS                : Windows


               OS Version             : Microsoft Windows Server 2012 Standard


               Driver Version         : 3.4.34823.2


               MPSS Version           : 4.0


               Host Physical Memory   : 32057 MB





Device No: 0, Device Name: mic0





        Version


               Flash Version           : 2.1.02.0391


               SMC Firmware Version   : 1.17.6900


               SMC Boot Loader Version        : 1.8.4326


               uOS Version             : 2.6.38.8+mpss3.4.2


               Device Serial Number    : ADKC34600110





        Board


               Vendor ID               : 0x8086


               Device ID               : 0x225d


               Subsystem ID            : 0x3c98


               Coprocessor Stepping ID        : 2


               PCIe Width              : x16


               PCIe Speed              : 5 GT/s


               PCIe Max payload size  : 256 bytes


               PCIe Max read req size : 4096 bytes


               Coprocessor Model      : 0x01


               Coprocessor Model Ext  : 0x00


               Coprocessor Type       : 0x00


               Coprocessor Family     : 0x0b


               Coprocessor Family Ext : 0x00


               Coprocessor Stepping    : C0


               Board SKU               : C0PRQ-3120/3140 P/A


               ECC Mode                : Enabled


               SMC HW Revision         : Product 300W Active CS





        Cores


               Total No of Active Cores : 61


               Voltage                 : 957000 uV


               Frequency              : 1100000 kHz





        Thermal


               Fan Speed Control       : On


               Fan RPM                 : 2700


               Fan PWM                 : 50


               Die Temp               : 43 C





        GDDR


               GDDR Vendor            : Samsung


               GDDR Version           : 0x6


               GDDR Density           : 4096 Mb


               GDDR Size              : 15872 MB


               GDDR Technology        : GDDR5


               GDDR Speed             : 5.000000 GT/s


               GDDR Frequency         : 2500000 kHz


               GDDR Voltage           : 1501000 uV


''', '', 0),

    NO_MICINFO : ('', '-bash: micinfo: command not found\n', 127)
}


_SB_HOST_DATA = {
    VALID:{

        'cat /proc/cpuinfo' : ('''core id         : 0
core id         : 1
core id         : 2
core id         : 3
core id         : 4
core id         : 5
core id         : 6
core id         : 7
core id         : 0
core id         : 1
core id         : 2
core id         : 3
core id         : 4
core id         : 5
core id         : 6
core id         : 7
core id         : 0
core id         : 1
core id         : 2
core id         : 3
core id         : 4
core id         : 5
core id         : 6
core id         : 7
core id         : 0
core id         : 1
core id         : 2
core id         : 3
core id         : 4
core id         : 5
core id         : 6
core id         : 7

processor       : 31
vendor_id       : GenuineIntel
cpu family      : 6
model           : 87
model name      : Intel(R) Xeon Phi CPU E5-2680 0 @ 2.70GHz
stepping        : 7
cpu MHz         : 1200.000
cache size      : 20480 KB
physical id     : 1
siblings        : 16
core id         : 7
cpu cores       : 8
apicid          : 47
initial apicid  : 47
fpu             : yes
fpu_exception   : yes
cpuid level     : 13
wp              : yes
flags           : fpu vme de pse tsc msr pae mce cx8 apic sep mtrr pge mca cmov\
 pat pse36 clflush dts acpi mmx fxsr sse sse2 ss ht tm pbe syscall nx pdpe1gb\
 rdtscp lm constant_tsc arch_perfmon pebs bts rep_good xtopology nonstop_tsc\
 aperfmperf pni pclmulqdq dtes64 monitor ds_cpl vmx smx est tm2 ssse3 cx16 xtpr\
 pdcm pcid dca sse4_1 sse4_2 x2apic popcnt tsc_deadline_timer aes xsave avx\
 lahf_lm ida arat epb xsaveopt pln pts dts tpr_shadow vnmi flexpriority ept vpid
bogomips        : 5386.26
clflush size    : 64
cache_alignment : 64
address sizes   : 46 bits physical, 48 bits virtual
power management:
''', '', 0),

        'cat /proc/meminfo' : ('''MemTotal:       32826680 kB
MemFree:        20715716 kB
Buffers:          873484 kB
Cached:          3457468 kB
SwapCached:            0 kB
Active:          7363028 kB
Inactive:        3470788 kB
Active(anon):    4964824 kB
Inactive(anon):  1550844 kB
Active(file):    2398204 kB
Inactive(file):  1919944 kB
Unevictable:           0 kB
Mlocked:               0 kB
SwapTotal:       8191992 kB
SwapFree:        8191992 kB
Dirty:               112 kB
Writeback:             0 kB
AnonPages:       6502888 kB
Mapped:           170580 kB
Shmem:             12792 kB
Slab:             866160 kB
SReclaimable:     720816 kB
SUnreclaim:       145344 kB
KernelStack:        8360 kB
PageTables:        53784 kB
NFS_Unstable:          0 kB
Bounce:                0 kB
WritebackTmp:          0 kB
CommitLimit:    24605332 kB
Committed_AS:    7904296 kB
VmallocTotal:   34359738367 kB
VmallocUsed:    17166592 kB
VmallocChunk:   34325399036 kB
HardwareCorrupted:     0 kB
AnonHugePages:   3995648 kB
HugePages_Total:       0
HugePages_Free:        0
HugePages_Rsvd:        0
HugePages_Surp:        0
Hugepagesize:       2048 kB
DirectMap4k:      122880 kB
DirectMap2M:     5087232 kB
DirectMap1G:    28311552 kB
''', '', 0),

        'uname -o' : ('GNU/Linux', '', 0),

        'uname -r' : ('2.6.32-431.el6.x86_64', '', 0)
    },

    INVALID:{

        'cat /proc/cpuinfo' : ('''core id         : 0
core id         : 1
core id         : 2
core id         : 3
core id         : 4
core id         : 5
core id         : 6
core id         : 7
core id         : 0
core id         : 1
core id         : 2
core id         : 3
core id         : 4
core id         : 5
core id         : 6
core id         : 7
core id         : 0
core id         : 1
core id         : 2
core id         : 3
core id         : 4
core id         : 5
core id         : 6
core id         : 7
core id         : 0
core id         : 1
core id         : 2
core id         : 3
core id         : 4
core id         : 5
core id         : 6
core id         : 7

processor       : 31
vendor_id       : GenuineIntel
cpu family      : 6
model           : 45
model name      : Intel(R) Xeon(R) CPU E5-2680 0 @ 2.70GHz
stepping        : 7
cpu MHz         : 1200.000
cache size      : 20480 KB
physical id     : 1
siblings        : 16
core id         : 7
cpu cores       : 8
apicid          : 47
initial apicid  : 47
fpu             : yes
fpu_exception   : yes
cpuid level     : 13
wp              : yes
flags           : fpu vme de pse tsc msr pae mce cx8 apic sep mtrr pge mca cmov\
 pat pse36 clflush dts acpi mmx fxsr sse sse2 ss ht tm pbe syscall nx pdpe1gb\
 rdtscp lm constant_tsc arch_perfmon pebs bts rep_good xtopology nonstop_tsc\
 aperfmperf pni pclmulqdq dtes64 monitor ds_cpl vmx smx est tm2 ssse3 cx16 xtpr\
 pdcm pcid dca sse4_1 sse4_2 x2apic popcnt tsc_deadline_timer aes xsave avx\
 lahf_lm ida arat epb xsaveopt pln pts dts tpr_shadow vnmi flexpriority ept vpid
bogomips        : 5386.26
clflush size    : 64
cache_alignment : 64
address sizes   : 46 bits physical, 48 bits virtual
power management:
''', '', 0)

    }
}


_X86_HOST_DATA = {
    VALID:{

        # output has been trimmed since only the core ids and some
        # CPU specs are required.
        'cat /proc/cpuinfo' : ('''processor       : 31
vendor_id       : GenuineIntel
cpu family      : 6
model           : 45
model name      : Intel(R) Xeon CPU E5-2680 0 @ 2.70GHz
stepping        : 7
cpu MHz         : 1200.000
cache size      : 20480 KB
physical id     : 1
siblings        : 16
core id         : 7
cpu cores       : 8
apicid          : 47
initial apicid  : 47
fpu             : yes
fpu_exception   : yes
cpuid level     : 13
wp              : yes
flags           : fpu vme de pse tsc msr pae mce cx8 apic sep mtrr pge mca cmov\
 pat pse36 clflush dts acpi mmx fxsr sse sse2 ss ht tm pbe syscall nx pdpe1gb\
 rdtscp lm constant_tsc arch_perfmon pebs bts rep_good xtopology nonstop_tsc\
 aperfmperf pni pclmulqdq dtes64 monitor ds_cpl vmx smx est tm2 ssse3 cx16 xtpr\
 pdcm pcid dca sse4_1 sse4_2 x2apic popcnt tsc_deadline_timer aes xsave avx\
 lahf_lm ida arat epb xsaveopt pln pts dts tpr_shadow vnmi flexpriority ept vpid
bogomips        : 5386.26
clflush size    : 64
cache_alignment : 64
address sizes   : 46 bits physical, 48 bits virtual
power management:
''', '', 0),

        # output has been trimmed since only this portion is required
        '/sbin/lspci' : ('''80:04.1 System peripheral: Intel Corporation Xeon E5/Core i7 DMA Channel 1 (rev 07)
80:04.2 System peripheral: Intel Corporation Xeon E5/Core i7 DMA Channel 2 (rev 07)
80:04.3 System peripheral: Intel Corporation Xeon E5/Core i7 DMA Channel 3 (rev 07)
80:04.4 System peripheral: Intel Corporation Xeon E5/Core i7 DMA Channel 4 (rev 07)
80:04.5 System peripheral: Intel Corporation Xeon E5/Core i7 DMA Channel 5 (rev 07)
80:04.6 System peripheral: Intel Corporation Xeon E5/Core i7 DMA Channel 6 (rev 07)
80:04.7 System peripheral: Intel Corporation Xeon E5/Core i7 DMA Channel 7 (rev 07)
80:05.0 System peripheral: Intel Corporation Xeon E5/Core i7 Address Map, VTd_Misc, System Management (rev 07)
80:05.2 System peripheral: Intel Corporation Xeon E5/Core i7 Control Status and Global Errors (rev 07)
80:05.4 PIC: Intel Corporation Xeon E5/Core i7 I/O APIC (rev 07)
83:00.0 Co-processor: Intel Corporation Xeon Phi coprocessor SE10/7120 series (rev 20)
ff:08.0 System peripheral: Intel Corporation Xeon E5/Core i7 QPI Link 0 (rev 07)
ff:09.0 System peripheral: Intel Corporation Xeon E5/Core i7 QPI Link 1 (rev 07)
ff:0a.0 System peripheral: Intel Corporation Xeon E5/Core i7 Power Control Unit 0 (rev 07)
ff:0a.1 System peripheral: Intel Corporation Xeon E5/Core i7 Power Control Unit 1 (rev 07)
ff:0a.2 System peripheral: Intel Corporation Xeon E5/Core i7 Power Control Unit 2 (rev 07)
ff:0a.3 System peripheral: Intel Corporation Xeon E5/Core i7 Power Control Unit 3 (rev 07)
ff:0b.0 System peripheral: Intel Corporation Xeon E5/Core i7 Interrupt Control Registers (rev 07)
ff:0b.3 System peripheral: Intel Corporation Xeon E5/Core i7 Semaphore and Scratchpad Configuration Registers (rev 07)
ff:0c.0 System peripheral: Intel Corporation Xeon E5/Core i7 Unicast Register 0 (rev 07)
ff:0c.1 System peripheral: Intel Corporation Xeon E5/Core i7 Unicast Register 0 (rev 07)
''', '', 0),

        'rpm -qa --queryformat \'%{version}\' mpss-release' :
            ('4.0', '', 0),

        'rpm -qa --queryformat \'%{version}\' mpss-daemon' :
            ('4.0', '', 0),

        'rpm -qa --queryformat \'%{version}\' mpss-micperf' :
            ('4.0', '', 0),


        'cat /proc/meminfo' : ('''MemTotal:       32826680 kB
MemFree:        20715716 kB
Buffers:          873484 kB
Cached:          3457468 kB
SwapCached:            0 kB
Active:          7363028 kB
Inactive:        3470788 kB
Active(anon):    4964824 kB
Inactive(anon):  1550844 kB
Active(file):    2398204 kB
Inactive(file):  1919944 kB
Unevictable:           0 kB
Mlocked:               0 kB
SwapTotal:       8191992 kB
SwapFree:        8191992 kB
Dirty:               112 kB
Writeback:             0 kB
AnonPages:       6502888 kB
Mapped:           170580 kB
Shmem:             12792 kB
Slab:             866160 kB
SReclaimable:     720816 kB
SUnreclaim:       145344 kB
KernelStack:        8360 kB
PageTables:        53784 kB
NFS_Unstable:          0 kB
Bounce:                0 kB
WritebackTmp:          0 kB
CommitLimit:    24605332 kB
Committed_AS:    7904296 kB
VmallocTotal:   34359738367 kB
VmallocUsed:    17166592 kB
VmallocChunk:   34325399036 kB
HardwareCorrupted:     0 kB
AnonHugePages:   3995648 kB
HugePages_Total:       0
HugePages_Free:        0
HugePages_Rsvd:        0
HugePages_Surp:        0
Hugepagesize:       2048 kB
DirectMap4k:      122880 kB
DirectMap2M:     5087232 kB
DirectMap1G:    28311552 kB
''', '', 0),


    # output has been trimmed since only this portion is required
    '/sbin/lspci -d 8086: -n -m' : (
        '''80:04.6 "0880" "8086" "3c26" -r07 "8086" "3594"
80:04.7 "0880" "8086" "3c27" -r07 "8086" "3594"
80:05.0 "0880" "8086" "3c28" -r07 "8086" "3594"
80:05.2 "0880" "8086" "3c2a" -r07 "8086" "3594"
80:05.4 "0800" "8086" "3c2c" -r07 -p20 "8086" "3594"
83:00.0 "0b40" "8086" "225c" -r20 "8086" "7d99"
ff:08.0 "0880" "8086" "3c80" -r07 "8086" "3594"
ff:09.0 "0880" "8086" "3c90" -r07 "8086" "3594"
ff:0a.0 "0880" "8086" "3cc0" -r07 "8086" "3594"
ff:0a.1 "0880" "8086" "3cc1" -r07 "8086" "0000"
''', '', 0),

        # output has been trimmed since only the core ids and some
        # CPU specs are required.
        'ssh -l mic 172.31.1.1 cat /proc/cpuinfo' : ('''core id         : 60
core id         : 0
core id         : 0
core id         : 0
core id         : 0
core id         : 1
core id         : 1
core id         : 1
core id         : 1
core id         : 2
core id         : 2
core id         : 2
core id         : 2
core id         : 3
core id         : 3
core id         : 3
core id         : 3
core id         : 4
core id         : 4
core id         : 4
core id         : 4
core id         : 5
core id         : 5
core id         : 5
core id         : 5
core id         : 6
core id         : 6
core id         : 6
core id         : 6
core id         : 7
core id         : 7
core id         : 7
core id         : 7
core id         : 8
core id         : 8
core id         : 8
core id         : 8
core id         : 9
core id         : 9
core id         : 9
core id         : 9
core id         : 10
core id         : 10
core id         : 10
core id         : 10
core id         : 11
core id         : 11
core id         : 11
core id         : 11
core id         : 12
core id         : 12
core id         : 12
core id         : 12
core id         : 13
core id         : 13
core id         : 13
core id         : 13
core id         : 14
core id         : 14
core id         : 14
core id         : 14
core id         : 15
core id         : 15
core id         : 15
core id         : 15
core id         : 16
core id         : 16
core id         : 16
core id         : 16
core id         : 17
core id         : 17
core id         : 17
core id         : 17
core id         : 18
core id         : 18
core id         : 18
core id         : 18
core id         : 19
core id         : 19
core id         : 19
core id         : 19
core id         : 20
core id         : 20
core id         : 20
core id         : 20
core id         : 21
core id         : 21
core id         : 21
core id         : 21
core id         : 22
core id         : 22
core id         : 22
core id         : 22
core id         : 23
core id         : 23
core id         : 23
core id         : 23
core id         : 24
core id         : 24
core id         : 24
core id         : 24
core id         : 25
core id         : 25
core id         : 25
core id         : 25
core id         : 26
core id         : 26
core id         : 26
core id         : 26
core id         : 27
core id         : 27
core id         : 27
core id         : 27
core id         : 28
core id         : 28
core id         : 28
core id         : 28
core id         : 29
core id         : 29
core id         : 29
core id         : 29
core id         : 30
core id         : 30
core id         : 30
core id         : 30
core id         : 31
core id         : 31
core id         : 31
core id         : 31
core id         : 32
core id         : 32
core id         : 32
core id         : 32
core id         : 33
core id         : 33
core id         : 33
core id         : 33
core id         : 34
core id         : 34
core id         : 34
core id         : 34
core id         : 35
core id         : 35
core id         : 35
core id         : 35
core id         : 36
core id         : 36
core id         : 36
core id         : 36
core id         : 37
core id         : 37
core id         : 37
core id         : 37
core id         : 38
core id         : 38
core id         : 38
core id         : 38
core id         : 39
core id         : 39
core id         : 39
core id         : 39
core id         : 40
core id         : 40
core id         : 40
core id         : 40
core id         : 41
core id         : 41
core id         : 41
core id         : 41
core id         : 42
core id         : 42
core id         : 42
core id         : 42
core id         : 43
core id         : 43
core id         : 43
core id         : 43
core id         : 44
core id         : 44
core id         : 44
core id         : 44
core id         : 45
core id         : 45
core id         : 45
core id         : 45
core id         : 46
core id         : 46
core id         : 46
core id         : 46
core id         : 47
core id         : 47
core id         : 47
core id         : 47
core id         : 48
core id         : 48
core id         : 48
core id         : 48
core id         : 49
core id         : 49
core id         : 49
core id         : 49
core id         : 50
core id         : 50
core id         : 50
core id         : 50
core id         : 51
core id         : 51
core id         : 51
core id         : 51
core id         : 52
core id         : 52
core id         : 52
core id         : 52
core id         : 53
core id         : 53
core id         : 53
core id         : 53
core id         : 54
core id         : 54
core id         : 54
core id         : 54
core id         : 55
core id         : 55
core id         : 55
core id         : 55
core id         : 56
core id         : 56
core id         : 56
core id         : 56
core id         : 57
core id         : 57
core id         : 57
core id         : 57
core id         : 58
core id         : 58
core id         : 58
core id         : 58
core id         : 59
core id         : 59
core id         : 59
core id         : 59
core id         : 60
core id         : 60
core id         : 60

processor       : 243
vendor_id       : GenuineIntel
cpu family      : 11
model           : 1
model name      : 0b/01
stepping        : 2
cpu MHz         : 1238.094
cache size      : 512 KB
physical id     : 0
siblings        : 244
core id         : 60
cpu cores       : 61
apicid          : 243
initial apicid  : 243
fpu             : yes
fpu_exception   : yes
cpuid level     : 4
wp              : yes
flags           : fpu vme de pse tsc msr pae mce cx8 apic mtrr mca pat fxsr ht syscall nx lm nopl lahf_lm
bogomips        : 2484.11
clflush size    : 64
cache_alignment : 64
address sizes   : 40 bits physical, 48 bits virtual
power management:
''', '', 0),

        'ssh -l mic 172.31.1.1 cat /proc/meminfo' : (
'''MemTotal:       16252928 kB
MemFree:        15607488 kB
Buffers:               0 kB
Cached:            53920 kB
SwapCached:            0 kB
Active:            11664 kB
Inactive:          44016 kB
Active(anon):      11664 kB
Inactive(anon):    44016 kB
Active(file):          0 kB
Inactive(file):        0 kB
Unevictable:           0 kB
Mlocked:               0 kB
SwapTotal:             0 kB
SwapFree:              0 kB
Dirty:                 0 kB
Writeback:             0 kB
AnonPages:          1796 kB
Mapped:             2856 kB
Shmem:             53920 kB
Slab:              96544 kB
SReclaimable:      20664 kB
SUnreclaim:        75880 kB
KernelStack:       10184 kB
PageTables:          444 kB
NFS_Unstable:          0 kB
Bounce:                0 kB
WritebackTmp:          0 kB
CommitLimit:     7942740 kB
Committed_AS:      73632 kB
VmallocTotal:   34359738367 kB
VmallocUsed:       34256 kB
VmallocChunk:   34359700412 kB
AnonHugePages:         0 kB
HugePages_Total:       0
HugePages_Free:        0
HugePages_Rsvd:        0
HugePages_Surp:        0
Hugepagesize:       2048 kB
DirectMap4k:       10236 kB
DirectMap2M:    16224256 kB
''', '', 0),

    },

    INVALID:{
        'rpm -qa --queryformat \'%{version}\' mpss-release' : ('', '', 0),
        'rpm -qa --queryformat \'%{version}\' mpss-daemon' :  ('', '', 0),
        'rpm -qa --queryformat \'%{version}\' mpss-micperf' :  ('', '', 0),
    }
}

_PYTHON_FOR_LINUX_DATA = {
    VALID : {
        'uname':('Linux', 'localhost.localdomain', '2.6.32-431.el6.x86_64'),
        'pathsep':':',
        'join_with':'/',
        'environ':{'PATH':'/usr/local/bin:/usr/bin',
                   'MIC_LD_LIBRARY_PATH':'/lib/mic:/usr/lib/mic',
                   'LD_LIBRARY_PATH':'/lib/intel64:/usr/lib/intel64',
                   'MIC_PERF_EXEC':'/usr/libexec/micperf',
                   'MKLROOT':'/MKL',
                   'MIC_PERF_DATA':'/usr/share/micperf/micp'},
        'path_exists':set(('/usr/libexec/micperf/{0}/sgemm_cpu.x'.format(HOST_DIR),
                       '/usr/libexec/micperf/{0}/sgemm_ofl.x'.format(HOST_DIR),
                       '/usr/libexec/micperf/{0}/sgemm_mic.x'.format(CARD_DIR),
                       '/usr/libexec/micperf/{0}/dgemm_cpu.x'.format(HOST_DIR),
                       '/usr/libexec/micperf/{0}/sgemm_mcdram_cpu.x'.format(HOST_DIR),
                       '/usr/libexec/micperf/{0}/dgemm_ofl.x'.format(HOST_DIR),
                       '/usr/libexec/micperf/{0}/dgemm_mic.x'.format(CARD_DIR),
                       '/usr/libexec/micperf/{0}/stream_mic'.format(CARD_DIR),
                       '/usr/libexec/micperf/{0}/BusSpeed_mic'.format(CARD_DIR),
                       '/usr/libexec/micperf/{0}/stream'.format(HOST_DIR),
                       '/usr/libexec/micperf/{0}/stream_mpi'.format(HOST_DIR),
                       '/usr/libexec/micperf/{0}/xlinpack_xeon64'.format(HOST_DIR),
                       '/usr/libexec/micperf/{0}/BusSpeedDownload_pragma'.format(HOST_DIR),
                       '/usr/libexec/micperf/{0}/BusSpeedDownload_scif'.format(HOST_DIR),
                       '/usr/libexec/micperf/{0}/BusSpeedReadback_pragma'.format(HOST_DIR),
                       '/usr/libexec/micperf/{0}/BusSpeedReadback_scif'.format(HOST_DIR),
                       '/MKL/benchmarks/linpack/xlinpack_xeon64',
                       '/lib/mic/libiomp5.so',
                       '/lib/mic/libabc.so',
                       '/usr/lib/mic/libabc.so',
                       '/usr/libexec/micperf/{0}/DummyKernel'.format(HOST_DIR),
                       '/usr/libexec/micperf/{0}/DummyKernel'.format(CARD_DIR),
                       '/usr/libexec/micperf/x86_64/DummyKernel',
                       'MKLROOT/benchmarks/linpack/xlinpack_mic')),
        '/usr/share/micperf/micp' : _PICKLE_FILES
        }
}

_PYTHON_FOR_WINDOWS_DATA = {
    VALID :{
        'uname':('Windows', 'localhost.localdomain', '7'),
        'pathsep':';',
        'join_with':'\\',
        'environ':{'PATH':'C:\\windows\\system32;C:\\windows',
                   'MIC_LD_LIBRARY_PATH':'C:\\lib\\mic;C:\\Users\\lib\\mic',
                   'LD_LIBRARY_PATH':'C:\\lib\\intel64;C:\\Users\\lib\\intel64',
                   'INTEL_MPSS_HOST_SDK':'C:\\Program Files\\Intel\\MPSS\\sdk\\lib',
                   'MIC_PERF_EXEC':'C:\\MICPERF\\libexec',
                   'MKLROOT':'C:\\MKL',
                   'MIC_PERF_DATA':'C:\\MICPERF\\data'},
        'path_exists':set(('C:\\MICPERF\\libexec\\{0}\\sgemm_cpu.exe'.format(HOST_DIR),
                       'C:\\MICPERF\\libexec\\{0}\\sgemm_ofl.exe'.format(HOST_DIR),
                       'C:\\MICPERF\\libexec\\{0}\\sgemm_mic.x'.format(CARD_DIR),
                       'C:\\MICPERF\\libexec\\{0}\\dgemm_cpu.exe'.format(HOST_DIR),
                       'C:\\MICPERF\\libexec\\{0}\\dgemm_ofl.exe'.format(HOST_DIR),
                       'C:\\MICPERF\\libexec\\{0}\\dgemm_mic.x'.format(CARD_DIR),
                       'C:\\MICPERF\\libexec\\{0}\\stream_mic'.format(CARD_DIR),
                       'C:\\MICPERF\\libexec\\{0}\\BusSpeed_mic'.format(CARD_DIR),
                       'C:\\MICPERF\\libexec\\{0}\\stream'.format(HOST_DIR),
                       'C:\\MICPERF\\libexec\\{0}\\stream_mpi'.format(HOST_DIR),
                       'C:\\MICPERF\\libexec\\{0}\\BusSpeedDownload_pragma.exe'.format(HOST_DIR),
                       'C:\\MICPERF\\libexec\\{0}\\BusSpeedDownload_scif.exe'.format(HOST_DIR),
                       'C:\\MICPERF\\libexec\\{0}\\BusSpeedReadback_pragma.exe'.format(HOST_DIR),
                       'C:\\MICPERF\\libexec\\{0}\\BusSpeedReadback_scif.exe'.format(HOST_DIR),
                       'C:\\MKL\\benchmarks\\linpack\\linpack_xeon64.exe',
                       'C:\\lib\\mic\\libiomp5.so',
                       'C:\\lib\\mic\\libabc.so',
                       'C:\\Users\\lib\\mic\\libabc.so',
                       'C:\\MICPERF\\libexec\\{0}\\DummyKernel'.format(HOST_DIR),
                       'C:\\MICPERF\\libexec\\{0}\\DummyKernel'.format(CARD_DIR),
                       'C:\\MICPERF\\libexec\\x86_64\\DummyKernel',
                       'MKLROOT/benchmarks/linpack/xlinpack_mic')),
        'C:\\MICPERF\\data' : _PICKLE_FILES
        }
}

# additional dictionary to store all the python info
_PYTHON_DATA = {}
_PYTHON_DATA[PYTHON_FOR_WINDOWS] = _PYTHON_FOR_WINDOWS_DATA
_PYTHON_DATA[PYTHON_FOR_LINUX] = _PYTHON_FOR_LINUX_DATA

# additional dictonary to store all host info
_HOST_DATA = {}
_HOST_DATA[SB_HOST] = _SB_HOST_DATA
_HOST_DATA[X86_HOST] = _X86_HOST_DATA

def _get_host_data(target, output_type, micinfo_version):
    """returns a dictionary to mock CLI commands run on the host"""
    host_data = _HOST_DATA[target][output_type].copy()
    if micinfo_version != EMPTY_MICINFO:
        host_data['micinfo'] = _MICINFO[micinfo_version]
    return host_data

def _get_python_data(target, output_type):
    """returns a dictionary to mock the python standard library"""
    python_data = _PYTHON_DATA[target].copy()
    return python_data[output_type]

def get_mock_data(target, output_type, micinfo_version):
    """decides which kind of mock should be built given the target and
    delegates accordingly, returns dictionary.
    IMPORTANT: This is the only method that external modules should call."""
    if target == SB_HOST or target == X86_HOST:
        return _get_host_data(target, output_type, micinfo_version)
    elif target == PYTHON_FOR_WINDOWS or target == PYTHON_FOR_LINUX:
        return _get_python_data(target, output_type)
    else:
        raise IOError("ERROR: Invalid target '{0}'\n".format())
