# Copyright 2010-2017 Intel Corporation.
# 
# This library is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, version 2.1.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
# 
# Disclaimer: The codes contained in these modules may be specific
# to the Intel Software Development Platform codenamed Knights Ferry,
# and the Intel product codenamed Knights Corner, and are not backward
# compatible with other Intel products. Additionally, Intel will NOT
# support the codes or instruction set in future products.
# 
# Intel offers no warranty of any kind regarding the code. This code is
# licensed on an "AS IS" basis and Intel is not obligated to provide
# any support, assistance, installation, training, or other services
# of any kind. Intel is also not obligated to provide any updates,
# enhancements or extensions. Intel specifically disclaims any warranty
# of merchantability, non-infringement, fitness for any particular
# purpose, and any other warranty.
# 
# Further, Intel disclaims all liability of any kind, including but
# not limited to liability for infringement of any proprietary rights,
# relating to the use of the code, even if Intel is notified of the
# possibility of such liability. Except as expressly stated in an Intel
# license agreement provided with this code and agreed upon with Intel,
# no license, express or implied, by estoppel or otherwise, to any
# intellectual property rights is granted herein.

"""Mocks for objects and methods in the python standard library"""

import data

class Popen(object):
    """Popen mock"""
    def __init__(self, command, test_type, target, micinfo):
        """setup the mock's result depending on the desired target,
        output and micinfo version. command, test_type, target and micinfo
        should be initialized with the constants define in data.py"""
        self._target_name = target
        self._command = self._prepare_command(command)
        self._results = data.get_mock_data(target, test_type, micinfo)
        self.returncode = None

    def communicate(self):
        """returns the tuple (stdout, stderr) and sets
        the public variable returncode"""
        if self._command in self._results:
            self.returncode = self._results[self._command][data.EXITCODE]
            return (self._results[self._command][data.STDOUT],
                    self._results[self._command][data.STDERR])
        else:
            error_msg = "ERROR: Unknown key '{0}' for '{1}'\n"
            raise OSError(error_msg.format(self._command, self._target_name))

    @staticmethod
    def _prepare_command(command):
        """python's Popen() can take either a command as a list
        or string this method converts command to string"""
        if isinstance(command, str):
            return command
        if isinstance(command, list):
            return ' '.join(command)
        error_msg = 'Invalid parameter type: {0}'.format(str(type(command)))
        raise IOError(error_msg)


class platform(object):
    """platform mock"""
    def __init__(self, target, test_type):
        """The constants PYTHON_FOR_WINDOWS or PYTHON_FOR_LINUX which are
        defined in data.py should be used to initialize target and test_type"""
        self._results = data.get_mock_data(target,
                                            test_type, data.EMPTY_MICINFO)

    def uname(self):
        """mock returns the tuple (os_name, host_name, kernel_version)"""
        return self._results['uname']

    def dist(self):
        """mock returns the tuple (os_name, version, dummy)"""
        return ('rhel', '7.2', None)

class os(object):
    """os mocks"""

    pathsep = ''
    environ = {}

    def __init__(self, target, test_type):
        """The constants PYTHON_FOR_WINDOWS or PYTHON_FOR_LINUX which are
        defined in data.py should be used to initialize target and test_type"""
        self._results = data.get_mock_data(target,
                                            test_type, data.EMPTY_MICINFO)
        self.environ = self._results['environ']
        self.pathsep = self._results['pathsep']


    def path_exists(self, path):
        """os.path.exists() mock"""
        return path in self._results['path_exists']


    def path_join(self, *tokens):
        """os.path.join() mock"""
        join_with = self._results['join_with']
        list_of_tokens = list(*tokens)
        return join_with.join(list_of_tokens)


    def listdir(self, path):
        """os.listdir() mock"""
        return self._results[path]
