from .. import diag_memory
import ctypes
import pytest
import os

TEST_STRING = "Something_random"
NOT_AVAILABLE = "Not Available"
MEMORY_SIZE = 65536
MEMORY_FREQ = 1600

# Using a list to simplify
MCDRAM_INFO = [(0, "cache"),
               (16384, "flat"),
               (12288, "Hybrid 25/75 (cache)/(non-cache)"),
               (8192, "Hybrid 50/50 (cache)/(non-cache)"),
               ("default", "All2All/Quadrant")
               ]


class MockMemoryTopology():
    """ Mock class
    """

    def __init__(self, args):
        pass

    def get_total_ddr_memory(self):
        return "1024"

    def get_ddr_freq(self):
        return "10"

    def get_ddr_voltage(self):
        return "10"

    def get_ddr_access(self, ddr_size):
        return "Available"

    def get_total_mcd_memory(self):
        return "16384"

    def get_cache_mcd_memory(self):
        return "8192"

    def get_sys_mcd_memory(self):
        return "8192"

    def get_mcd_freq(self):
        return "10"

    def get_mcd_voltage(self):
        return "10"

    def get_mcd_access(self, mcd_size):
        return "Available"

    def get_cluster_mode(self):
        return "Quadrant"

    def get_memory_mode(self):
        return "Hybrid"

    def get_MCDRAM_cache_info(self):
        return "50% of MCDRAM used as Cache"


class MockMalloc():
    """ Mock class
    """
    restype = 0
    def __call__(self, data):
        pass


class MockFree():
    """ Mock class
    """
    argtypes = list()
    def __call__(self, data):
        pass


class MockMemkind():
    """ Mock class
    """
    hbw_malloc = MockMalloc()
    hbw_free   = MockFree()

    def hbw_check_available(self):
        return 0


class MockDMI():
    """ Mock class
    """
    def __init__(self):
        self.backup_dmi_dir = ""
        self.backup_dmi_filename = ""

    def mock_env(self):
        self.backup_dmi_dir = diag_memory.DMI_SYSFS_ROOT
        self.backup_dmi_filename = diag_memory.DMI_FILENAME
        diag_memory.RUNNING_UT = True
        diag_memory.DMI_SYSFS_ROOT = "./dmi-entries"
        diag_memory.DMI_FILENAME = diag_memory.DMI_SYSFS_ROOT + "/{0}/raw"

    def unmock_env(self):
        diag_memory.DMI_SYSFS_ROOT = self.backup_dmi_dir
        diag_memory.DMI_FILENAME = self.backup_dmi_filename


@pytest.yield_fixture(scope="function")
def mocked_dmi():
    mockDMI = MockDMI()
    mockDMI.mock_env()
    yield
    mockDMI.unmock_env()


def mock_run_ldconfig(command, timeout=0):
    """ Mock '/usr/sbin/ldconfig -p | grep libmemkind.so' command
    """
    return "libmemkind.so.0 (libc6,x86-64) => /lib64/libmemkind.so.0", "", 0


def mock_osopen_oserror(name, opt):
    """ Mock osopen to throw a OSError exception
    """
    raise OSError


def mock_osopen_nameerror(name, opt):
    """ Mock osopen to throw a NameError exception
    """
    raise NameError


def test_print_cluster_config(capfd):
    """ Not much to test here since simply prints the received value and a msg.
    :return:
    """
    mock_output = "Cluster Configuration is: {0}".format(TEST_STRING)
    diag_memory.print_cluster_config(TEST_STRING)
    out, err = capfd.readouterr()
    assert not err
    assert TEST_STRING in out


def test_print_memory_config(capfd):
    """ Not much to test here since simply prints the received value and a msg.
    :return:
    """
    mcdram_config, mcdram_cache = (TEST_STRING, TEST_STRING)
    diag_memory.print_memory_config(mcdram_config, mcdram_cache)
    out, err = capfd.readouterr()
    assert not err
    assert TEST_STRING in out


def test_print_memory_info(capfd):
    """ Since is just printing information and that information
    is validated somewhere else...
            Run it with real values
            Run it with "0" values
    :return:
    """
    diag_memory.print_memory_info("DDR", 1, 1, 1, 1, "RW")
    out, err = capfd.readouterr()
    assert not err
    assert NOT_AVAILABLE not in out


def test_print_memory_info_no_value(capfd):
    diag_memory.print_memory_info("MCDRAM", 0, 0, 0, 0, "Available", 0, 0)
    out, err = capfd.readouterr()
    assert not err
    assert NOT_AVAILABLE in out


def test_memory_info(capfd):
    """ Since diag_memory.py is just a wrapper of MemoryTopology
    run the function.
    :return:
    """
    backup_memory_topology = diag_memory.MemoryTopology
    diag_memory.MemoryTopology = MockMemoryTopology
    assert diag_memory.test_memory_info("") == 0
    diag_memory.MemoryTopology = backup_memory_topology


def test_ldconfig_error(capfd, mocked_dmi):
    diag_memory.test_memory_info(None)
    out, err = capfd.readouterr()
    assert "Unexpected" in err


def test_memking_load_error(capfd, monkeypatch, mocked_dmi):
    monkeypatch.setattr(diag_memory, "run", mock_run_ldconfig)
    diag_memory.test_memory_info(None)
    out, err = capfd.readouterr()
    assert "OSError" in err


def test_not_implemented_exception(capfd, monkeypatch, mocked_dmi):
    with pytest.raises(diag_memory.DMIException) as excinfo:
        parent = diag_memory.DMITable(16)
    assert 'NotImplementedError' in str(excinfo.value)

@pytest.mark.parametrize("error", [mock_osopen_oserror, mock_osopen_nameerror])
def test_dmiobject_error(error, capfd, monkeypatch, mocked_dmi):
    monkeypatch.setattr(os, "open", error)
    with pytest.raises(diag_memory.DMIException) as excinfo:
        parent = diag_memory.DMITable(16)
    assert 'Error processing DMI Type 16' in str(excinfo.value)


def test_dmitablefactory_fail(capfd):
    with pytest.raises(diag_memory.DMIException) as excinfo:
        fail_table = diag_memory.DMITableFactory.get_table(18)
    assert 'Table Type 18 is not supported!' in str(excinfo.value)


def test_original_output(capfd, monkeypatch, mocked_dmi):
    monkeypatch.setattr(diag_memory, "run", mock_run_ldconfig)
    monkeypatch.setattr(ctypes.cdll, "LoadLibrary", lambda __: MockMemkind())
    diag_memory.test_memory_info(None)
    out, err = capfd.readouterr()
    assert out
    assert not err
