from .. import diag_pcie
from .. import sys_utils
from golden_output import LSPCI_VV_OUTPUT


class MockPCIeInformation():
    def __init__(self, args):
        """ Example of PCIe devices parsed by lspci -vv
        :param args: Param used in PCIeInformation class
        """
        self.args = args
        self.pcie_devices = ["00:01.0",
                             "00:02.0",
                             ]

    def __len__(self):
        return len(self.pcie_devices)

    def get_pcie_dev(self, pcie):
        """ mock pcie devices returned
        Only the first pcie_device is assigned values.
        :return: mock pcie device
        :rtype: dict
        """
        pcie_dev = dict()
        if pcie == self.pcie_devices[0]:
            pcie_dev["Bridge"] = "80:02.0"
            pcie_dev["LnkWidth"] = "x16"
            pcie_dev["CapWidth"] = "x16"
            pcie_dev["LnkSpeed"] = "2.5GT/s"
            pcie_dev["CapSpeed"] = "8GT/s"
        return pcie_dev


class MockExceptionPCIeInformation(MockPCIeInformation):
    def __init__(self, args):
        """ A mock clas that raises an exception as soon is instantiated
        """
        self.pcie_devices = []
        raise RuntimeError("Something bad happened here.")


class MockEmptyPCIeInformation(MockPCIeInformation):
    def __init__(self, args):
        """ A mock class without PCIe devices
        """
        self.pcie_devices = []


class TestUTPcie():
    def test_pcie_info(self):
        """ Verify diag_pcie successfully returns 0
        """
        original_pcie_info = diag_pcie.PCIeInformation
        diag_pcie.PCIeInformation = MockPCIeInformation
        assert diag_pcie.test_pcie_info("") == 0
        diag_pcie.PCIeInformation = original_pcie_info

    def test_pcie_exception(self):
        """ Force a RunTimeError when getting PCIe device information
        """
        original_pcie_info = diag_pcie.PCIeInformation
        diag_pcie.PCIeInformation = MockExceptionPCIeInformation
        assert diag_pcie.test_pcie_info("") == -1
        diag_pcie.PCIeInformation = original_pcie_info


    def test_pcie_no_devices(self):
        """ Mock a system with no PCIe devices.
        """
        original_pcie_info = diag_pcie.PCIeInformation
        diag_pcie.PCIeInformation = MockEmptyPCIeInformation
        assert diag_pcie.test_pcie_info("") == -2
        diag_pcie.PCIeInformation = original_pcie_info

def mock_run_lscpi_verbose(command, timeout=0):
    """ Mock 'lspci -vv' command
    """
    return LSPCI_VV_OUTPUT, "", 0


def test_pcie_information_instance(monkeypatch):
    """ Able to instance PCIeInformation
    """
    monkeypatch.setattr(sys_utils, "run", mock_run_lscpi_verbose)
    assert len(diag_pcie.PCIeInformation('')) > 0

def test_pcie_information_get_pcie_dev_info(monkeypatch):
    """ Able to instance PCIeInformation  and get the device information
    """
    monkeypatch.setattr(sys_utils, "run", mock_run_lscpi_verbose)
    pcie_info = diag_pcie.PCIeInformation('')
    for item in pcie_info.pcie_devices:
        pcie_info.get_pcie_dev(item)
