/*
 * Copyright 2010-2017 Intel Corporation.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, version 2.1.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * Disclaimer: The codes contained in these modules may be specific
 * to the Intel Software Development Platform codenamed Knights Ferry,
 * and the Intel product codenamed Knights Corner, and are not backward
 * compatible with other Intel products. Additionally, Intel will NOT
 * support the codes or instruction set in future products.
 * 
 * Intel offers no warranty of any kind regarding the code. This code is
 * licensed on an "AS IS" basis and Intel is not obligated to provide
 * any support, assistance, installation, training, or other services
 * of any kind. Intel is also not obligated to provide any updates,
 * enhancements or extensions. Intel specifically disclaims any warranty
 * of merchantability, non-infringement, fitness for any particular
 * purpose, and any other warranty.
 * 
 * Further, Intel disclaims all liability of any kind, including but
 * not limited to liability for infringement of any proprietary rights,
 * relating to the use of the code, even if Intel is notified of the
 * possibility of such liability. Except as expressly stated in an Intel
 * license agreement provided with this code and agreed upon with Intel,
 * no license, express or implied, by estoppel or otherwise, to any
 * intellectual property rights is granted herein.
*/

#include <stdio.h>

    #include <unistd.h>
    #include <sched.h>
    #include <tr1/memory>
    #include <sys/mman.h>
    #include <sys/statvfs.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
    #include <linux/sysctl.h>
    #include <asm-generic/mman.h>

#include <common/COIMacros_common.h>
#include <internal/_Debug.h>
#include <internal/_DMA.h>
#include <internal/_Message.h>

#ifdef TRANSPORT_OFI
    #include <internal/_OFIComm.h>
#endif

#include <internal/_MemoryRegion.h>
#include <internal/_PthreadMutexAutoLock.h>
#include <internal/_Buffer.h>
#include <internal/_DependencyDag.h>
#include <internal/_Process.h>
#include <common/COIEvent_common.h>

#include <list>
#include <algorithm>

#include "buffer.h"
#include "buffernodes.h"

#if 0
    #define DPRINTF(...) printf(__VA_ARGS__)
#else
    #define DPRINTF(...)
#endif

// the physical local store on the sink
class local_store
{
public:
    local_store(_COIComm *comm, _COIComm **dma_comms,
                uint64_t dma_count, size_t len, bool TLB)
        :   m_dma_count(dma_count),
            m_local_handle(-1),
            m_length(len),
            m_fd(-1),
            m_file_address(0),
            hugeTLB(TLB)
    {
        m_cache_comm = comm;
        //Before we do anything we need to copy out the endpoints
        //from the incoming epd_dma's as they are pointer and not
        //gauranteed to be accesible beyond the lifespan of this constructor
        //and we need them for unregistering with scif
        for (uint64_t index = 0; index < m_dma_count; index++)
        {
            m_dma_comms[index] = dma_comms[index];
        }

        try
        {
            snprintf(m_filename, sizeof(m_filename) - 1, "%s.XXXXXX",
                     BUFFER_FILENAME_PREFIX);
            if (hugeTLB)
            {
                char name[PATH_MAX + 1] = { 0 };
                snprintf(name, PATH_MAX, "%s.XXXXXX", BUFFER_FILENAME_PREFIX);
                snprintf(m_filename, sizeof(m_filename) - 1, "%s/%s", _COISinkProcessCommon::GetHugeTLBfsPath(), name);
                m_filename[sizeof(m_filename) - 1] = '\0';
                m_fd = mkstemp(m_filename);
            }
            else
            {
                m_fd = mkstemp(m_filename);
            }

            if (ftruncate(m_fd, m_length) == -1)
            {
                throw COI_ERROR;
            }
            if (fsync(m_fd) == -1)
            {
                throw COI_ERROR;
            }
            m_file_address = (void *) - 1;
            struct statvfs info;
            if (-1 == statvfs(_COISinkProcessCommon::GetProcsPath(), &info))
            {
                perror("statvfs() error");
            }
            if (m_length > (info.f_blocks * info.f_bsize))
            {
                throw COI_OUT_OF_MEMORY;
            }

            m_file_address = mmap(0, m_length, PROT_READ | PROT_WRITE, MAP_SHARED, m_fd, 0);

            if (MAP_FAILED == m_file_address)
            {
                throw COI_OUT_OF_MEMORY;
            }
            else if (madvise(m_file_address, m_length, MADV_DONTFORK) != 0)
            {
                throw COI_ERROR;
            }

            COIRESULT reg_result = m_cache_comm->RegisterMemory(
                                       m_file_address, NULL, m_length, GetNextRegisteredOffsetHint(m_length),
                                       COI_COMM_READ | COI_COMM_WRITE, true, (uint64_t *)&m_local_handle);
            if (reg_result != COI_SUCCESS)
            {
                throw COI_OUT_OF_MEMORY;
            }

            // TODO clean this up, it's horribly messy to use ifdefs like that...
#ifdef TRANSPORT_OFI
            if (m_cache_comm->GetType() == COI_OFI_NODE)
            {
                ((_OFIComm *)m_cache_comm)->GetMRData(m_local_handle,
                                                      m_length,
                                                      &(m_ofi_data.sink_virt_address),
                                                      &(m_ofi_data.sink_mr_key));

            }
#endif

            for (uint64_t index = 0; index < m_dma_count; index++)
            {
                reg_result = m_dma_comms[index]->RegisterMemory(
                                 m_file_address, NULL, m_length, m_local_handle,
                                 COI_COMM_READ | COI_COMM_WRITE, true, (uint64_t *)&m_local_handle);
                // TODO clean this up, it's horribly messy to use ifdefs like that...
#ifdef TRANSPORT_OFI
                if (m_cache_comm->GetType() == COI_OFI_NODE)
                {
                    ((_OFIComm *)m_dma_comms[index])->GetMRData(m_local_handle,
                            m_length,
                            &(m_ofi_data.dma_virt_address[index]),
                            &(m_ofi_data.dma_mr_key[index]));
                }
#endif
            }
            if (reg_result != COI_SUCCESS)
            {
                throw COI_OUT_OF_MEMORY;
            }
        }
        catch (...)
        {
            cleanup();
            throw;
        }
    }

    void cleanup()
    {
        //free up memory registration and mmap stuff
        // TODO magic number because of COIComm's SCIF orientation
        if ((uint64_t) - 1 != m_local_handle)
        {
            for (uint64_t index = 0; index < m_dma_count; index++)
            {
                m_dma_comms[index]->UnRegisterMemory(m_local_handle, m_length);
            }
        }
        if ((void *) - 1 != m_file_address)
        {
            munmap(m_file_address, m_length);
        }
        if (-1 != m_fd)
        {
            close(m_fd);
            unlink(m_filename);
        }
    }

    virtual ~local_store()
    {
        cleanup();
    }

    COIRESULT Remap(void *virtual_address,
                    uint64_t physical_offset,
                    uint64_t length)
    {
        uint64_t store_begin = m_local_handle;
        uint64_t store_end = m_local_handle + m_length;
        uint64_t region_begin = physical_offset;
        uint64_t region_end = physical_offset + length;
        uint64_t mmap_begin = 0;
        uint64_t mmap_end = 0;
        uint64_t mmap_size = 0;
        uint64_t mmap_va = 0;
        bool     done = false;
        // find start of region in this local store
        // find end of region in this local store
        // calculate size

        // region starts after this local store
        if (region_begin > store_end)
        {
            return COI_ERROR;
        }

        // region starts inside of this local store
        else if (region_begin >= store_begin)
        {
            mmap_begin = region_begin;
        }

        // region starts before this local store
        else if (region_begin < store_begin)
        {
            mmap_begin = store_begin;
        }

        // region ends before this local store
        if (region_end < store_begin)
        {
            return COI_ERROR;
        }

        // region ends inside of this local store
        else if (region_end <= store_end)
        {
            done = true;
            mmap_end = region_end;
        }

        // region ends after of this local store
        else if (region_end > store_end)
        {
            mmap_end = store_end;
        }

        mmap_size = mmap_end - mmap_begin;
        mmap_va = (uint64_t)virtual_address + (mmap_begin - physical_offset);

        void *addr;

        addr = mmap((void *)mmap_va, mmap_size, PROT_READ | PROT_WRITE,
                    MAP_FIXED | MAP_SHARED,
                    m_fd, mmap_begin - m_local_handle);
        if (MAP_FAILED == addr)
        {
            return COI_ERROR;
        }
        else if (addr != (void *)mmap_va ||
                 madvise(addr, mmap_size, MADV_DONTFORK) != 0)
        {
            munmap(addr, length);
            return COI_ERROR;
        }

        // If the area mapped is smaller than the total region then this is
        // just a partial map.
        if (!done)
        {
            return COI_RETRY;
        }
        return COI_SUCCESS;
    }
    // TODO clean this up, it's horribly messy to use ifdefs like that...
#ifdef TRANSPORT_OFI
    local_store_ofi_data *ofi_data()
    {
        return &m_ofi_data;
    }
#endif

    uint64_t handle()
    {
        return m_local_handle;
    }

private:
    char        m_filename[PATH_MAX + 1];
    _COIComm   *m_dma_comms[COI_PROCESS_MAX_DMA_ENDPOINTS];
    uint64_t    m_dma_count;
    uint64_t    m_local_handle;
    size_t      m_length;
    int         m_fd;
    void       *m_file_address;
    _COIComm   *m_cache_comm;
    // TODO clean this up, it's horribly messy to use ifdefs like that...
#ifdef TRANSPORT_OFI
    local_store_ofi_data m_ofi_data;
#endif

public:
    bool        hugeTLB;
};

class COILocalMemoryStoreImpl
{
public:
    COILocalMemoryStoreImpl(_COIComm &comm);
    COILocalMemoryStoreImpl(_COIComm &comm, _COIComm **DMAcomm, uint64_t DMAcount);
    virtual ~COILocalMemoryStoreImpl();

    std::list<local_store *> m_local_stores;
    _COIComm               &m_comm;
    _COIComm              **m_DMAcomm;
    uint64_t                m_DMAcount;
};


COILocalMemoryStoreImpl::COILocalMemoryStoreImpl(
    _COIComm &comm, _COIComm **DMAcomm, uint64_t DMAcount)
    :   m_comm(comm),
        m_DMAcomm(DMAcomm),
        m_DMAcount(DMAcount)
{
}

COILocalMemoryStoreImpl::COILocalMemoryStoreImpl(_COIComm &comm)
    :   m_comm(comm)
{
}

COILocalMemoryStoreImpl::~COILocalMemoryStoreImpl()
{
    while (!m_local_stores.empty())
    {
        delete m_local_stores.front();
        m_local_stores.pop_front();
    }
}

COILocalMemoryStore::COILocalMemoryStore(_COIComm &comm)
{
    m_pImpl = new COILocalMemoryStoreImpl(comm);
}

COILocalMemoryStore::COILocalMemoryStore(_COIComm &comm, _COIComm **DMAcomm, uint64_t DMAcount)
{
    m_pImpl = new COILocalMemoryStoreImpl(comm, DMAcomm, DMAcount);
}

COILocalMemoryStore::COILocalMemoryStore(const COILocalMemoryStore &s)
{
    m_pImpl = NULL;
    UNREFERENCED_CONST_PARAM(&s);
    throw COI_NOT_SUPPORTED;
}

COILocalMemoryStore &COILocalMemoryStore::operator=(const COILocalMemoryStore &s)
{
    UNREFERENCED_CONST_PARAM(&s);
    m_pImpl = NULL;
    throw COI_NOT_SUPPORTED;
}

COILocalMemoryStore::~COILocalMemoryStore()
{
    if (m_pImpl)
    {
        delete m_pImpl;
        m_pImpl = NULL;
    }
}

// TODO clean this up, it's horribly messy to use ifdefs like that...
COIRESULT
COILocalMemoryStore::Create(size_t len, uint64_t &local_handle, bool Huge_TLB
#ifdef TRANSPORT_OFI
    , local_store_ofi_data *ofi_data
#endif
                           )
{
    try
    {
        local_store *tmp(new local_store(&m_pImpl->m_comm,
                                         m_pImpl->m_DMAcomm,
                                         m_pImpl->m_DMAcount,
                                         len, Huge_TLB ? true : false));
        m_pImpl->m_local_stores.push_back(tmp);
        local_handle = tmp->handle();
        // TODO clean this up, it's horribly messy to use ifdefs like that...
#ifdef TRANSPORT_OFI
        memcpy(ofi_data, tmp->ofi_data(), sizeof(local_store_ofi_data));
#endif
    }
    catch (COIRESULT &)
    {
        return COI_OUT_OF_MEMORY;
    }

    return COI_SUCCESS;
}

COIRESULT
COILocalMemoryStore::RemapVirtualToVirtual(
    void *virtual_address, uint64_t physical_offset,
    uint64_t length, uint8_t buf_type)
{
    // The data range being remapped may span multiple physical backing stores.
    // Ask each local store to map it's part of the range.
    COIRESULT result =  COI_ERROR;
    std::list<local_store *>::iterator it;
    for (it = m_pImpl->m_local_stores.begin();
            it != m_pImpl->m_local_stores.end(); ++it)
    {
        if (buf_type == 0x2)
        {
            if ((*it)->hugeTLB)
            {
                result = (*it)->Remap(virtual_address, physical_offset, length);
            }
        }
        else
        {
            result = (*it)->Remap(virtual_address, physical_offset, length);
        }
        if (COI_SUCCESS == result)
        {
            return result;
        }
    }
    return COI_ERROR;
}
