/*
 * Copyright 2010-2017 Intel Corporation.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, version 2.1.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * Disclaimer: The codes contained in these modules may be specific
 * to the Intel Software Development Platform codenamed Knights Ferry,
 * and the Intel product codenamed Knights Corner, and are not backward
 * compatible with other Intel products. Additionally, Intel will NOT
 * support the codes or instruction set in future products.
 * 
 * Intel offers no warranty of any kind regarding the code. This code is
 * licensed on an "AS IS" basis and Intel is not obligated to provide
 * any support, assistance, installation, training, or other services
 * of any kind. Intel is also not obligated to provide any updates,
 * enhancements or extensions. Intel specifically disclaims any warranty
 * of merchantability, non-infringement, fitness for any particular
 * purpose, and any other warranty.
 * 
 * Further, Intel disclaims all liability of any kind, including but
 * not limited to liability for infringement of any proprietary rights,
 * relating to the use of the code, even if Intel is notified of the
 * possibility of such liability. Except as expressly stated in an Intel
 * license agreement provided with this code and agreed upon with Intel,
 * no license, express or implied, by estoppel or otherwise, to any
 * intellectual property rights is granted herein.
*/

#define COI_LIBRARY_VERSION 2
#include <internal/_Event.h>
#include <internal/_Process.h>

extern "C" _COIStaticInit g_Initializer;

COIRESULT _SignalEvent(COIEVENT in_Event)
{
    COIRESULT result = COI_ERROR;
    _UserEventHandlerSink &handler = g_Initializer.GetProcess()->GetEventHandler();
    result = handler.SignalEvent(in_Event);
    return result;
}

// ************************Internal Load Library from Host to Sink***********************
COIRESULT _SinkLoadLibrary(const char *file_name,
                           const char *library_name,
                           const char *library_search_path,
                           uint32_t flags,
                           uint64_t *library)
{
    COILOG_FUNC_ENTER;

    COIRESULT operationResult = COI_ERROR;

    _UserEventHandlerSink &handler = g_Initializer.GetProcess()->GetEventHandler();
    if (g_Initializer.GetProcess())
    {
        operationResult = handler.LoadLibraryToSink(file_name, library_name, library_search_path, flags, library);
    }

    COILOG_FUNC_RETURN_RESULT(operationResult);
    return operationResult;
}

//Internal ReleaseRef that is called from the User api ReleaseRef from sink
COIRESULT
_SinkReleaseRef(void *host_buffer_address,
                uint64_t length,
                void *proc,
                void *pipeline,
                uint64_t count)
{

    COIRESULT result = COI_ERROR;

    _UserEventHandlerSink &handler
        = g_Initializer.GetProcess()->GetEventHandler();
    result = handler.SinkReleaseRef(host_buffer_address, length, proc, pipeline, count);


    return result;
}

_UserEventHandlerSink::_UserEventHandlerSink()
    : m_evtComm(NULL),
      m_evtListnr(NULL)
{
}

_UserEventHandlerSink::~_UserEventHandlerSink()
{
    if (m_evtComm)
    {
        m_evtComm->Disconnect();
        delete m_evtComm;
        m_evtComm = NULL;
    }

    if (m_evtListnr)
    {
        delete m_evtListnr;
        m_evtListnr = NULL;
    }
}

void _UserEventHandlerSink::GetConnectionInfo(_COICommInfo *connection_info)
{
    m_evtListnr->GetConnectionInfo(connection_info);
}

int _UserEventHandlerSink::WaitForConnect()
{
    int status = -1;
    status = m_evtListnr->WaitForConnect(*m_evtComm);
    if (status != COI_SUCCESS)
    {
        COILOG_ERROR("Error connecting to the sink Comm. status = %d\n",
                     status);
    }
    return status;
}

void _UserEventHandlerSink::Shutdown()
{
    COILOG_FUNC_ENTER;

    COIEventMessage_t message;
    COIEventMessage_t::SHUTDOWN_T *shutdown;
    message.SetPayload(shutdown);
    if (m_evtComm->SendAtomic(message) != COI_SUCCESS)
    {
        COILOG_ERROR("Error Sending shutdown Message to Source");

    }
    COILOG_FUNC_EXIT;
}

COIRESULT _UserEventHandlerSink::SignalEvent(COIEVENT in_Event)
{
    COILOG_FUNC_ENTER;

    COIRESULT result = COI_ERROR;

    COIEventMessage_t message;

    COIEventMessage_t::SIGNALED_T *args;
    message.SetPayload(args);
    args->event = in_Event;
    //Send a message back to event receiver thread With the event contents
    result = m_evtComm->SendAtomic(message);

    COILOG_FUNC_RETURN_RESULT(result);
}

//Sends the ReleaseRef message to the host, this message does not require
//an ack from the host since the release ref itself signals
//the buffer events to start
COIRESULT
_UserEventHandlerSink::SinkReleaseRef(void *host_buffer_address,
                                      uint64_t length,
                                      void *proc,
                                      void *pipeline,
                                      uint64_t count)
{
    COILOG_FUNC_ENTER;

    COIRESULT result = COI_ERROR;
    COIEventMessage_t message;

    COIEventMessage_t::RELEASE_REF_T *args;
    message.SetPayload(args, 0);

    args->buffer = host_buffer_address;
    args->length = length;
    args->proc = proc;
    args->count = count;
    args->pipeline = pipeline;

    //Send a message back to event receiver thread With the buffer to decrement the reference count
    result = m_evtComm->SendAtomic(message);
    COILOG_FUNC_RETURN_RESULT(result);
}

// Sends the LoadLibrary message to the host
COIRESULT
_UserEventHandlerSink::LoadLibraryToSink(const char *inFileName,
        const char *inLibraryName,
        const char *inLibrarySearchPath,
        uint32_t inFlags,
        uint64_t *library)
{
    COIRESULT operationResult = COI_ERROR;
    COIEventMessage_t message;
    COIEventMessage_t::SINK_LOAD_LIBRARY_T *sinkLoadLibraryMsgPayload;
    uint32_t totalSize = 0, fileNameSize = 1, libraryNameSize = 1, searchPathSize = 1;

    //this is one thing that we can verify on Sink
    if (inFileName == NULL)
    {
        return COI_INVALID_POINTER;
    }

    // Calculate space needed for the arguments
    fileNameSize = (uint32_t)strnlen(inFileName, COI_MAX_FILE_NAME_LENGTH) + 1;
    if (inLibraryName != NULL)
    {
        libraryNameSize = (uint32_t)strnlen(inLibraryName, COI_MAX_FILE_NAME_LENGTH) + 1;
    }
    if (inLibrarySearchPath != NULL)
    {
        searchPathSize = (uint32_t)strnlen(inLibrarySearchPath, PATH_MAX) + 1;
    }
    totalSize = fileNameSize + libraryNameSize + searchPathSize + sizeof(inFlags);

    // Alloc space for the arguments
    message.SetPayload(sinkLoadLibraryMsgPayload, totalSize);

    // Map args to pointers
    char *fileName = sinkLoadLibraryMsgPayload->data;
    char *libraryName = sinkLoadLibraryMsgPayload->data + fileNameSize;
    char *librarySearchPath = sinkLoadLibraryMsgPayload->data +
                              fileNameSize +
                              libraryNameSize;

    // Fill args
    sinkLoadLibraryMsgPayload->flags = inFlags;

    libraryName[0] = '\0';
    librarySearchPath[0] = '\0';

    strncpy(fileName, inFileName, fileNameSize);
    if (inLibraryName)
    {
        strncpy(libraryName, inLibraryName, libraryNameSize);
    }
    if (inLibrarySearchPath)
    {
        strncpy(librarySearchPath, inLibrarySearchPath, searchPathSize);
    }

    // Send a message to source to ask to load library for you
    if (m_evtComm->SendAtomic(message) != COI_SUCCESS)
    {
        return COI_ERROR;
    }

    // Wait for the result (and hope for the best)
    COIProcessMessage_t resultMessage;
    if (m_evtComm->ReceiveUnsafe(resultMessage) != COI_SUCCESS)
    {
        return COI_ERROR;
    }

    COIProcessMessage_t::SINK_LOAD_LIBRARY_RES_T *resultPayload = resultMessage.GetPayload();
    *library = resultPayload->handle;
    operationResult = (COIRESULT)resultPayload->result;

    return operationResult;
}
