#!/bin/bash
#
# Intel VCA Software Stack (VCASS)
#
# Copyright(c) 2017 Intel Corporation.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License, version 2, as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# The full GNU General Public License is included in this distribution in
# the file called "COPYING".
#
# Intel VCA Scripts.
#
set -eu

readonly SCRIPT_DIR="$( cd "$(dirname "$0")" && pwd )"	# $0 works for more shells than  ${BASH_SOURCE[0]}
. "${SCRIPT_DIR}/library_image_creation.sh"

ARCHIVE_FILE=""
AUTHOR=""
CHROOT_DIR=""
CUSTOM_FILE_DIR=""
DESCRIPTION=""
KERNEL_VER=""
KS_FILE=""
PKG_LIST=""
PREADD_SCRIPT=""
POSTADD_SCRIPT=""
_REPO_VCA=""
_REPO_EXTRAS=""
_REPO_OS=""

# TODO: --post and --post should probably also get the following params: DESCR OS, KERNEL
show_help(){
	# Keep parameters alphabetically sorted in help and in the 'case' switch which parses them.
	# Prefer single-character options, i.e. do not introduce long options without an apparent reason.
	echo "Usage: $0 [OTHER OPTIONS] [-p <package_list>]
Creates a regular software archive or a bootstrap archive. The packages for the software archive can come from a local set of ${_CONST_PKG_EXT} package files, or from named packages that will be automatially downloaded from one of the official public repositories.
Creation of the regular software archive involves adding the packages into a temporary chroot environment containing a fully functional apt-get and dpkg, which mimics the target environment. This ensures that any dependencies or installation errors are caught at the stage of software archive creation.
Every time when creating an archive, all archives to be added later in the process of creating the root file system must be re-created if they already exist. This is to ensure that they have package dependency information not earlier than the packages in this archive.
Options:
-a <path>	The pathname of the archive file to be created. The archive file will contain downloaded *.deb package files, apt database and dpkg database, and information on essential (not: dependant) files for which this archive has been created.
-c <author>	The string indentifying the author of this archive. This parameter is mandatory.
-d <descr>	Download original Ubuntu bootstrap for the OS version description <descr> from a public repository or create a boostrap CentOS OS. Example description strings are: \"CENTOS\", \"UBUNTU 16.04\", \"UBUNTU 16.04.3\", \"UBUNTU 16.10\" but beware that it is worthless do distinguish between Ubuntu point releases for LTS (Long Term Support) versions. Each point release is merely a snapshot of updated packages in the LTS version at that time - the public repositories do not foresee to install other releases than 'base LTS' (e.g. 16.04) or 'last point release' (currently: 16.04.5 for the 16.04).
-e <path>	The path to a repository containing optional software packages to be added to the CentOS root filesystem (neglected when creating Ubuntu root filesystem).
-f <path>	The pathname of the kickstart file used for creation of CentOS root filesystem (neglected when creating Ubuntu root filesystem).
-g <path>	Include the indicated custom file or directory tree into the archive. This file/directory will be included into the archive without any other processing during creation. When applying the archive, this file/directory will get extracted, but not otherwse processed except by the <pre-add_script> or <post-add_script>, if so provided.
-h		Displays this help page.
-k <version>	Target kernel version, used to fake commands if the kernel version in the build environment is different.
-o <path>	The path to the OS repository containing OS official distribution software packages to be added to the CentOS root filesystem (neglected when creating Ubuntu root filesystem).
-p <package_list>	List of essential packages for which this archive is created. Dependant packages do not need to be specified, as they will be automatically found, downloaded, and installed into the chroot environment. This must always be the last option specified.
--post <post-add_script>
--pre <pre-add_script>	The name of the shell script to be run after/before adding packages from this archive. The scritp will be renamed to ${_CONST_ADD_SCRIPTS["POST"]}/${_CONST_ADD_SCRIPTS["PRE"]} in the archive and will be interpreted by /bin/bash with the following command-line parameters:
	* A string indicating wheter the current execution environment is the build environment (BUILD) or  the target environment (TARGET).
	* A string indicating whether the script is called as a pre-add (PRE) or a post-add (POST) script.
	* The <root_dir>-based path to the unpacked archive.
	* The fully-qualified path to this ($0) build script's directory, potentially containing shell script libraries with usable functions.
	* The fully qualified path to the root of the rootfs filesystem (<root_dir>) being created at the time the packages from the archive are added.
The --post and --pre parameters are optional.
-r <root_dir>	The root directory of the chroot environment containing fully functional package management tools. The archive will be created only from new packages selected and installed in this environment, so it should already contain all packages which are required, but which are NOT to be included in the resulting archive. In case of creating the bootstrap, this is used as a temporary bootstrap directory.
-v <path>		The path to the VCA build repository containing VCA software packages to be added to the CentOS root filesystem (neglected when creating Ubuntu root filesystem).

Options '-d' and '-p' are mutually exclusive.
Options '-d' and '-g' are mutually exclusive.

Option '-p' must be the last option, if used.
"
}

parse_parameters(){
	while [ $# -gt 0 ] ; do
		case "$1" in
			-a)
				ARCHIVE_FILE="${2:-""}"
				shift; shift;;
			-b)
				PREADD_SCRIPT="${2:-""}"
				shift; shift;;
			-c)
				AUTHOR="${2:-""}"
				shift; shift;;
			-d)
				DESCRIPTION="${2:-""}"
				shift; shift;;
			-e)
				_REPO_EXTRAS="${2:-""}"
				shift; shift;;
			-f)
				KS_FILE="${2:-""}"
				shift; shift;;
			-g)
				CUSTOM_FILE_DIR="${2:-""}"
				shift; shift;;
			-k)
				KERNEL_VER="${2:-""}"
				shift; shift;;
			-h)
				show_help
				exit 0;;
			-o)
				_REPO_OS="${2:-""}"
				shift; shift;;
			-p)
				shift || die "Argument for -p missing!"
				PKG_LIST="$@"
				return 0;;	# this must be the last parameter
			--post)
				POSTADD_SCRIPT="${2:-""}"
				shift; shift;;
			--pre)
				PREADD_SCRIPT="${2:-""}"
				shift; shift;;
			-r)
				CHROOT_DIR="${2:-""}"
				shift; shift;;
			-v)
				_REPO_VCA="${2:-""}"
				shift; shift;;
			*)
				show_help && die "Unknown parameter '$1'"
		esac
	done
}

# output to stdout the currently installed kernel version, or an empty string:
get_kernel_ver(){
	local _CHROOT_DIR="$1"
	ls "${CHROOT_DIR}"/lib/modules 2>/dev/null
}

check_parameters () {
	[ -z "${ARCHIVE_FILE}" ] && show_help && die "Name of the archive file to be created is not given "
	[[ -n "${CHROOT_DIR}" && -n "${DESCRIPTION}" ]] && show_help && die "Root directory of the operating environment ${CHROOT_DIR} has no meaning when creating the boostrap archive"
	[[ -z "${CHROOT_DIR}" && -z "${DESCRIPTION}" ]] && show_help && die "Give the root directory of the operating environment for a regular archive or the OS version description for a bootstrap archive"
	[ -z "${AUTHOR}" ]	&& show_help && die "Author string not given ( '-p' must be the last option, if used)"
	[ -z "${PKG_LIST}" ] && [ -z "${DESCRIPTION}" ] && [ -z "${CUSTOM_FILE_DIR}" ] && show_help && die "No version <descr> to download official bootstrap given. No list of packages to be included into the archive given. No custom files to be included into the archive given. ( '-p' must be the last option, if used)"
	[ -n "${DESCRIPTION}" ] && [ -n "${CUSTOM_FILE_DIR}" ] && show_help && die "Custom file/dir can not be added to a bootstrap archive"

	[ -n "${ARCHIVE_FILE}" ] && [ -s "${ARCHIVE_FILE}" ] && die "Archive file ${ARCHIVE_FILE} already exists"
	[ -n "${CHROOT_DIR}" ]	&& [ ! -d "${CHROOT_DIR}" ]	&& die "Root directory of the operating environment ${CHROOT_DIR} does not exist"
	[ -n "${CUSTOM_FILE_DIR}" ]	&& [ ! -e "${CUSTOM_FILE_DIR}" ] && die "Custom file/directory ${CUSTOM_FILE_DIR} does not exist"
	[ -n "${PREADD_SCRIPT}"  ]	&& [ ! -f "${PREADD_SCRIPT}"  ]	&& die "Pre-add script ${PREADD_SCRIPT} does not exist"
	[ -n "${POSTADD_SCRIPT}" ]	&& [ ! -f "${POSTADD_SCRIPT}" ]	&& die "Post-add script ${POSTADD_SCRIPT} does not exist"

	if [[ "${DESCRIPTION}" == CENTOS* ]] ; then
		[ -z "${KS_FILE}" ]		&& die "Kickstart file not given for ${DESCRIPTION}"
		[ -z "${_REPO_OS}" ]	&& die "OS package repository path not given for ${DESCRIPTION}"
		[ -z "${_REPO_VCA}" ]	&& die "VCA package repository path not given for ${DESCRIPTION}"
	fi
	if [ -n "${CHROOT_DIR}" ] ; then
		# So this is not the bootstrap. Is kernel already installed?
		local _INSTALLED_KERNEL_VER="$(get_kernel_ver "${CHROOT_DIR}")"
		# exactly one kernel versions must be known:
		if [ -n "${_INSTALLED_KERNEL_VER}" ] ; then
			[ -n "${KERNEL_VER}" ] && [ "${KERNEL_VER}" != "${_INSTALLED_KERNEL_VER}" ] && die "Target kernel version mismatch: command line says ${KERNEL_VER}, the environment in ${CHROOT_DIR} says ${_INSTALLED_KERNEL_VER}"
			[ -z "${KERNEL_VER}" ] && KERNEL_VER="${_INSTALLED_KERNEL_VER}"
		else
			[ -z "${KERNEL_VER}" ] && die "Target kernel version not given"
		fi
	fi

	return 0
}

# Downloads packages and refreshes apt-get database and dpkg database
get_packages(){
	local _CHROOT_DIR="$1"
	shift
	local _PKG_LIST="$@"

	local _HARMLESS="^Get:\|^Hit:\|^Preparing to unpack \|^Unpacking \|^Processing triggers for \|^Setting up \|^Selecting previously unselected package\|^(Reading database ... \|^E: Getting name for slave of master fd "
	# to avoid the message: "debconf: delaying package configuration, since apt-utils is not installed"
	add_packages "${_CHROOT_DIR}" \
				"${_HARMLESS}"	\
				UPDATE	\
				NOCLEAN	\
				apt-utils

	# to fill the apt-get cache with curretly downloaded *.deb package files:
	add_packages "${_CHROOT_DIR}" \
				"${_HARMLESS}"	\
				UPDATE	\
				CLEAN	\
				"${_PKG_LIST}"
}

create_UBUNTU_bootstrap(){
	local _CHROOT_DIR="$1"
	
	local _UBUNTU_VERSION="${DESCRIPTION#* }"	#get a DESCRIPTION substring, from the end until space
	local _CODENAME="$(get_codename "${_UBUNTU_VERSION}")"
	# debootstrap variants: --variant=minbase|buildd|fakechroot|scratchbox or just the Debian base when called without --variant=
	local _VARIANT="minbase"
	# Consider getting the newest debootstrap from http://archive.ubuntu.com/ubuntu/pool/main/d/debootstrap/?C=M;O=D  + ar  debootstrap_1.0.*_all.deb + tar xf data.tar.gz.
	# Calling debootstrap requires root privileges. Consider fakeroot fakeroot /polystrap: https://unix.stackexchange.com/a/214830
	debootstrap --arch=amd64 --variant "${_VARIANT}" "${_CODENAME}" "${_CHROOT_DIR}" http://archive.ubuntu.com/ubuntu/
}

# returns path to the _IMG_CREATOR_ROOTFS_IMG image on which _CHROOT_DIR is mounted 
# _IMG_CREATOR_ROOTFS_IMG is to be umounted & deleted by the caller when _CHROOT_DIR no longer needed
create_CENTOS_bootstrap(){
	local _CHROOT_DIR="$1"
	local _KS_FILE="$2"
	local _REPO_OS="$3"
	local _REPO_VCA="$4"
	local _REPO_EXTRAS="$5"

	# modify a copy of the original _KS_FILE:
	local _KS_TMP_FILE="$(mktemp --tmpdir ks_file.XXXXXX)"
	cp -f "${_KS_FILE}" "${_KS_TMP_FILE}" || die "Could not copy ${_KS_FILE} to ${_KS_TMP_FILE}"
	sed -i "s|VCA_OS_REPO|${_REPO_OS}|g"			"${_KS_TMP_FILE}"
	sed -i "s|VCA_BUILD_REPO|${_REPO_VCA}|g"		"${_KS_TMP_FILE}"
	sed -i "s|VCA_EXTRAS_REPO|${_REPO_EXTRAS}|g"	"${_KS_TMP_FILE}"
	
	# Create rootfs image
	local _IMG_CREATOR_TMP_DIR="$(mktemp --directory --tmpdir imgage_creator_tmp_dir.XXXXXX)"
	local _IMG_CREATOR_ROOTFS_IMG="$(mktemp --tmpdir image_creator_rootfs_image.XXXXXX)"
	# the image-creator only accepts _IMG_CREATOR_ROOTFS_IMG without path, and creates it in its current working directory:
	(
		cd "$(dirname "${_IMG_CREATOR_ROOTFS_IMG}")"
		image-creator --config "${_KS_TMP_FILE}" --name "$(basename "${_IMG_CREATOR_ROOTFS_IMG}")" --tmpdir "${_IMG_CREATOR_TMP_DIR}" >&2 || die "Failed to create image with image-creator"
	) || die "Image creator failed - correct the KickStart file (${_KS_FILE})?"

	# mount the rootfs image & cleanup:
	# the image-creator unnecessarily appends a hardcoded ".img" to the name of _IMG_CREATOR_ROOTFS_IMG
	_IMG_CREATOR_ROOTFS_IMG="${_IMG_CREATOR_ROOTFS_IMG}".img
	mount "${_IMG_CREATOR_ROOTFS_IMG}" "${_CHROOT_DIR}"
	# find "${_CHROOT_DIR}"/etc/sysconfig/network-scripts/ -name ifcfg-\* ! -name ifcfg-lo -exec /bin/rm "{}" \;
	/bin/rm "${_KS_TMP_FILE}"
	# according to stdout/stderr mesages from the above call to image-creator, _IMG_CREATOR_TMP_DIR is already unmounted here. But trying to removing the ${_IMG_CREATOR_TMP_DIR}/*/install_root/sys dir induces many error messages. Supressing this, as _IMG_CREATOR_TMP_DIR is anyway re-created with random name at every run:
	/bin/rm -rf "${_IMG_CREATOR_TMP_DIR}" > /dev/null 2>&1
	echo "${_IMG_CREATOR_ROOTFS_IMG}"	# to be deleted by the caller
	return 0
}

create_archive(){
	parse_parameters "$@"
	check_parameters

	if [ -n "${DESCRIPTION}" ] ; then	# start with bootstrap if its download requested
		CHROOT_DIR="$(mktemp --directory --tmpdir root_dir.XXXXXX)"
		CHROOT_DIR="${CHROOT_DIR}/bootstrap"	# to ensure a nice dir name in the archive
		# TODO: a dirty hack to make the CHROOT_DIR available to create_sw_archives.sh
		# A better way would be to allow for non-empty CHROOT_DIR when DESCRIPTION is non-empty and only set the CHROOT_DIR here if CHROOT_DIR is empty.
		# Easy, so far. But the above will change the 'nice dir name in the archive': "${CHROOT_DIR}/bootstrap
		# Also, up to now CHROOT_DIR used to be removed at the end of this function...
		echo "${CHROOT_DIR}" > /tmp/create_sw_archives.information.on.rootfs
		mkdir -p "${CHROOT_DIR}"
		CUSTOM_FILE_DIR="${CHROOT_DIR}"
		fakeAndRegisterCommand /bin/uname <<EOF >/dev/null
		case "\$1" in
			-r)
				echo "${KERNEL_VER}"
				;;
			*)
				"$(getFakedCommandSavedName /bin/uname)" \$@
				;;
		esac
EOF
		local _ROOTFS_IMAGE_FILE=""
		local _STATUS=""
		case "${DESCRIPTION}" in
			UBUNTU*)
				create_UBUNTU_bootstrap "${CHROOT_DIR}"
				;;
			CENTOS)
				# The caller is responsible for removing _ROOTFS_IMAGE_FILE file, so getting the name of the file:
				_ROOTFS_IMAGE_FILE="$(create_CENTOS_bootstrap "${CHROOT_DIR}" "${KS_FILE}" "${_REPO_OS}" "${_REPO_VCA}" "${_REPO_EXTRAS}" )" 
				[ -n "${_ROOTFS_IMAGE_FILE}" ] # to force _STATUS below
				;;
			*)
				# cannot break here as environment has to be restored first
				echo "Unknown description: ${DESCRIPTION}" >&2
				false
		esac  || _STATUS=$?
		
		restoreFakedRegisteredCommands
		[ -n "${_STATUS}" ] && die "Could not download or initialize the bootstrap. Check the distribution description: ${DESCRIPTION}."
	fi
 
	# Packages in a local directory (*.deb files) can require downloading dependencies from the Internet repository.
	# so always calling get_packages in the hope that apt-get will resolve this
	local _PKG_APT_DIR="${CHROOT_DIR}"/var/cache/apt/	# directory of packages downloaded by apt_get (in the subdirectory archives)
	local _PKG_DEB_DIR="$(mktemp --directory --tmpdir="${CHROOT_DIR}"/tmp  package_file_dir.XXXXXX )"	# directory of package files provided by the user (in the subdirectory archives)
	mkdir "${_PKG_DEB_DIR}"/archives

	# transfer package files (if any) to a directory accessible in ${CHROOT_DIR}; changing their names on the package list accordingly:
	local _NEW_PKG_LIST=""
	local _PKG
	for _PKG in ${PKG_LIST} ; do
		# TODO: this will accept names of non-existing files (even when given with paths) as package names
		# The distinction between package name and file name shoud be done on the basis of path preceding the name. A path must contain a slash '/'
		# This is a place to check for existence of such files. What about unexpanded '*' or other wildards?
		if [ -f "${_PKG}" ] ; then
			local _PKG_FILENAME="$(basename "${_PKG}")"

			cp "${_PKG}" "${_PKG_DEB_DIR}"
			_NEW_PKG_LIST="${_NEW_PKG_LIST} ${_PKG_DEB_DIR}/${_PKG_FILENAME}"
		else
			_NEW_PKG_LIST="${_NEW_PKG_LIST} ${_PKG}"
		fi
	done

	# prepare a virtual structure identical to the archive format.
	# Relative paths should still be OK as links, as long as the CWD stays unchanged
	# TODO: Architectural question: at this point only CUSTOM_FILE_DIR and PRE/POSTADD scripts are know.
	# What if someone wants to e.g. tweak the package files? read the _ARCHIVE_INFO_FILE_NAME? All this is possible during create_rootfs
	# TODO: decide if _VIRTUAL_ARCHIVE can be inside CHROOT_DIR also during create_rootfs, and also when bootstrap is extracted in create_rootfs
	# TODO: does CUSTOM_FILE_DIR sometimes contains the ${_CUSTOM} ?
	# The file/directory structure of an archive is as follows:
	# 	Max 2 directories:
	#		/${_CUSTOM_DIR}/* 		# max one entry; this entry may contain bootstrap or something else
	#		/archives/*.deb	# more than 0 *.deb files
	#	An ${_APT_DB_FILENAME} if /archives/ exist
	#	Max 2 named script files:
	#		${_ADD_SCRIPTS["PRE"]}		# the PREADD script name
	#		${_ADD_SCRIPTS["POST"]}		# the POSTADD script name
	#	Always an ${_ARCHIVE_INFO_FILE_NAME}
	local _VIRTUAL_ARCHIVE="$(mktemp --directory --tmpdir="${CHROOT_DIR}" virtual_archive.XXXXXX )"
	[ ! -d "${_VIRTUAL_ARCHIVE}" ] && die "Could not create virtual archive ${_VIRTUAL_ARCHIVE}"
	if [ -n "${CUSTOM_FILE_DIR}" ] ; then
		if [ -d "${CUSTOM_FILE_DIR}" ] ; then
			mkdir -p "${_VIRTUAL_ARCHIVE}/${_CONST_CUSTOM_DIR}"
			mount --bind -o ro "${CUSTOM_FILE_DIR}" "${_VIRTUAL_ARCHIVE}/${_CONST_CUSTOM_DIR}"
		else #so this is just one file
			ln -s "${CUSTOM_FILE_DIR}"	"${_VIRTUAL_ARCHIVE}/${_CONST_CUSTOM_DIR}"
		fi
	fi
	# TO DO: before adding packages no package dir exists; hard to overcome as packages are not downloaded yet:
	# [ ... ] && ln -s "${_PKG_SRC_DIR}" "${_VIRTUAL_ARCHIVE}"
	# TO DO: before adding packages the locatoin of _APT_DB_FILENAME not known yet:
	# [ ... ] && ln -s "${_APT_DB_FILENAME}" "${_VIRTUAL_ARCHIVE}"
	# TODO: should not use the internal library variable _CONST_ADD_SCRIPTS below. Move creating virtual archive to the library:
	[ -n "${PREADD_SCRIPT}"  ] && ln -s "${PREADD_SCRIPT}"  "${_VIRTUAL_ARCHIVE}/${_CONST_ADD_SCRIPTS[PRE]}"
	[ -n "${POSTADD_SCRIPT}" ] && ln -s "${POSTADD_SCRIPT}" "${_VIRTUAL_ARCHIVE}/${_CONST_ADD_SCRIPTS[POST]}"
	create_info_file "${_VIRTUAL_ARCHIVE}" "${DESCRIPTION}"  "${KERNEL_VER}" "${AUTHOR}" "${PKG_LIST}"

	# The PRE/POST scripts should be run even if there are no packages to be added:
	# TODO: Why passing PRE/POST script directories only (not names, which is correct as scripts have already be renamed to standard names), if directory ${_VIRTUAL_ARCHIVE} is known anyway?
	pre_add_packages BUILD "${_VIRTUAL_ARCHIVE}" "${SCRIPT_DIR}" "${CHROOT_DIR}" "$(dirname "${_VIRTUAL_ARCHIVE}/${_CONST_ADD_SCRIPTS[PRE]}")"  "${KERNEL_VER}" "${DESCRIPTION}"

	if [ -n "${DESCRIPTION}" ] ; then	# continue with bootstrap package upgrade
		# do a blind upgrade to the newest distribution if this is not the base (e.g. 16.04.3 will get upgraded to 16.04.400 if such version exists)
		local _DOTS_NUMBER="$(echo "${DESCRIPTION}" | awk -F. '{ print NF-1 }')"
		# a base release (like: 16.04) has only one dot in version decription:
		[ "${_DOTS_NUMBER}" -gt 1 ] && get_packages "${CHROOT_DIR}" DIST_UPGRADE
	fi

	local _PKG_SRC_DIR=""	# an empty string indicates that no packages are to be added to the archive
	if [ -n "${_NEW_PKG_LIST}" ] ; then
		get_packages "${CHROOT_DIR}" "${_NEW_PKG_LIST}"
		# now uniting packages from potential two places: _PKG_DEB_DIR and - possible fresher - _PKG_APT_DIR
		# This is sub-optimal: e.g. if any package from the originally provided *.deb files has ben overriden by a newer downloaded one, both versions will be stored in the archive
		local _PKG_SRC_DIR="$(mktemp --directory --tmpdir  package_source_dir.XXXXXX )" # the directory being the source of packages for assemble_archive:
		mkdir "${_PKG_SRC_DIR}"/archives
		# TODO: make a 'isEmptyDir' function
		# one of these may be empty:
		[ -n "$(find "${_PKG_DEB_DIR}"/archives/ -maxdepth 1 -mindepth 1)" ] && cp "${_PKG_DEB_DIR}"/archives/* "${_PKG_SRC_DIR}"/archives
		[ -n "$(find "${_PKG_APT_DIR}"/archives/ -maxdepth 1 -mindepth 1)" ] && cp "${_PKG_APT_DIR}"/archives/* "${_PKG_SRC_DIR}"/archives
	fi
	# The PRE/POST scripts should be run even if there are no packages to be added:

	post_add_packages BUILD "${_VIRTUAL_ARCHIVE}" "${SCRIPT_DIR}" "${CHROOT_DIR}" "$(dirname "${_VIRTUAL_ARCHIVE}/${_CONST_ADD_SCRIPTS[POST]}")"

	# TODO: _VIRTUAL_ARCHIVE should not be included in archive. But it can not be removed yet if it contains the info file. Should _VIRTUAL_ARCHIVE be in CHROOT really when creating the bootstrap archive?
	assemble_archive "${ARCHIVE_FILE}" "${_PKG_SRC_DIR}" "${CHROOT_DIR}" "${_VIRTUAL_ARCHIVE}" "${CUSTOM_FILE_DIR}" "${PREADD_SCRIPT}" "${POSTADD_SCRIPT}" || die "Could not assemble archive ${ARCHIVE_FILE}"
	[ -n "${_PKG_SRC_DIR}" ] && /bin/rm -r "${_PKG_SRC_DIR}"
	/bin/rm -r "${_PKG_DEB_DIR}"
	[ -d "${_VIRTUAL_ARCHIVE}/${_CONST_CUSTOM_DIR}" ] && umount "${_VIRTUAL_ARCHIVE}/${_CONST_CUSTOM_DIR}"
	/bin/rm -r "${_VIRTUAL_ARCHIVE}"
	if [ -n "${DESCRIPTION}" ] ; then
		# TODO: investigate if the "${_ROOTFS_IMAGE_FILE}" should be removed here. Does not seem correct (perhaps only works so far, because the file stays opened by the caller?)
		[ -n "${_ROOTFS_IMAGE_FILE}" ] && umount "${_ROOTFS_IMAGE_FILE}" && /bin/rm "${_ROOTFS_IMAGE_FILE}"
		# TODO: (cont.) a dirty hack to make the CHROOT_DIR available to create_sw_archives.sh: Commenting out the below
		#/bin/rm -r "${CHROOT_DIR}" || die "Could not remove rootfs in ${CHROOT_DIR}"
	fi
	return 0
}

echo "Called as: $0 $@" >&2
create_archive "$@"  && echo "Finished: $0 $@" >&2