#!/bin/bash
#
# Intel VCA Software Stack (VCASS)
#
# Copyright(c) 2017 Intel Corporation.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License, version 2, as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# The full GNU General Public License is included in this distribution in
# the file called "COPYING".
#
# Intel VCA Scripts.
#
set -eu

readonly SCRIPT_DIR="$( cd "$(dirname "$0")" && pwd )"	# $0 works for more shells than  ${BASH_SOURCE[0]}
. "${SCRIPT_DIR}/library_image_creation.sh"

ARCHIVE_FILES_ARRAY=()	# array of archives with software packages to be added to the bootstrap; to be set from the parameters
BOOTSTRAP_FILE=""	# the archive containing the bootstrap
DESCRIPTION=""		# OS description
# TODO: what should happen if an archive (e.g. with VCA software) added in the middle of other archives installs a kernel with different version?
KER_VER=""			# kernel version
OUTPUT_DIR=""		# the directory housing the final environment; to be set from the parameters

show_help() {
	# Keep parameters alphabetically sorted in help and in the 'case' switch which parses them.
	# Prefer single-character options, i.e. do not introduce long options without an apparent reason.
	echo "Usage: $0 [OPTIONS]
Creates a root file system utilizing regular VCA software archives and/or a bootstrap VCA archive.
This script requires root privileges and creates an operating environment that contains complete file and direcotry tree of an operating system.
The scripts main input is:
- The location of (possibly multiple) archives created with create_archive.sh. The archives can contain additional software packages to be added to the bootstrap, including the VCA modules package and the VCA kernel packages. The archives can also contain a custom file/directory with other data necessary for installation, and the pre- and post-installation scripts. A specific example of the custom directory is the bootstrap, i.e. a minimal operating environment, sufficient to add further software packages and to prepare the necessary configuration. The files from the other archive(s) are aplied to the unarchived bootstrap to form the final operating environment.
- The destination directory for the final operating environment.

Options:
-a, --archive <path>	The location of the archive to be added to the bootstrap. This option can be used multiple times. Archives are processed in the order in which they appear on the command line. To allow to satisfy the dependencies for packages within every archive, all dependent packages must be included in the same archive, or in one of the archives applied before.
-b, --bootstrap	<path>	The location of the bootstrap archive. This archive will be aplied first. This parameter is mandatory.
-d, --descr <description>	The OS version description, to fake the target environment.
-k, --kernel <version>	The kernel version, to fake the target environment.
-h, --help	Show this help screen.
-o, --out-dir <path>	The destination directory for the final operating environment. This parameter is mandatory.
"
}

parse_parameters(){
	while [ $# -gt 0 ] ; do
		case "$1" in
			-a|--archive)
				ARCHIVE_FILES_ARRAY+=("${2:-""}")
				shift; shift;;
			-b|--bootstrap-dir)
				BOOTSTRAP_FILE="${2:-""}"
				shift; shift;;
			-d|--description)
				DESCRIPTION="${2:-""}"
				shift; shift;;
			-k|--kernel)
				KER_VER="${2:-""}"
				shift; shift;;
			-h|--help)
				show_help
				exit 0;;
			-o|--out-dir)
				OUTPUT_DIR="${2:-""}"
				shift; shift;;
			*)
				show_help && die "Unknown parameter '$1'"
		esac
	done
}

check_parameters () {
	[ -z "${BOOTSTRAP_FILE}" ] && show_help && die "No bootstrap file given"
	[ -n "${BOOTSTRAP_FILE}" ] && [ ! -f "${BOOTSTRAP_FILE}" ] && show_help && die "No bootstrap file ${BOOTSTRAP_FILE}"
	local _ARCHIVE
	if [ ${#ARCHIVE_FILES_ARRAY[@]} -ne 0 ] ; then
		for _ARCHIVE in "${ARCHIVE_FILES_ARRAY[@]}" ; do
			[ ! -f "${_ARCHIVE}" ] && show_help && die "Archive file ${_ARCHIVE} not found"
		done
	fi
	[ -z "${OUTPUT_DIR}" ] && show_help && die "Indicating the output directory is mandatory"
	[ -n "${OUTPUT_DIR}" ] && [ -n "$(find "${OUTPUT_DIR}" -mindepth 1 -maxdepth 1 2>/dev/null )" ]  && show_help && die "Output directory ${OUTPUT_DIR} already exists and is not empty"
	[ -z "${DESCRIPTION}" ] && show_help && die "No OS description given"
	[ -z "${KER_VER}" ] && show_help && die "No kernel version given"
	return 0
}

add_archive(){
	local _ARCHIVE="$1"
	local _UNTAR_DIR="$2"
	local _ARCHIVE_TYPE="$3"	# value BOOTSTRAP, or other (e.g. NotBOOTSTRAP)

	if [ "${_ARCHIVE_TYPE}" == BOOTSTRAP ] ; then
		echo "*** Extracting bootstrap archive ${_ARCHIVE} into ${OUTPUT_DIR}" >&2
	else
		echo "*** Applying archive ${_ARCHIVE} to the environment in ${OUTPUT_DIR}" >&2
	fi
	extract_archive "${_ARCHIVE}" "${OUTPUT_DIR}/${_UNTAR_DIR}" || die "Could not extract archive ${_ARCHIVE} to ${OUTPUT_DIR}/${_UNTAR_DIR}"

	# Until moving the bootstrap to its final place, the ${OUTPUT_DIR} is temporarily ${_OUTPUT_DIR}
	# _OUTPUT_DIR=${OUTPUT_DIR}/<untar tmp dir>/{_CUSTOM}/<bootstrap dir name>
	local _OUTPUT_DIR=""
	if [ "${_ARCHIVE_TYPE}" == BOOTSTRAP ] ; then
		#_OUTPUT_DIR="$(echo "${OUTPUT_DIR}"/*/*/*)"
		_OUTPUT_DIR="${OUTPUT_DIR}/${_UNTAR_DIR}/$(get_archive_element_path "${OUTPUT_DIR}/${_UNTAR_DIR}" CUSTOM_DIR)"
		# TODO: try to beutify this later, when "../../../" are resolved
		#_UNTAR_DIR="$(readlink --canonicalize-existing "../../../${_UNTAR_DIR}")"
		_UNTAR_DIR="../../${_UNTAR_DIR}"	# relative to _OUTPUT_DIR
	else
		_OUTPUT_DIR="${OUTPUT_DIR}"
	fi

	# The PRE/POST scripts should be run even if there are no packages to be added:
	pre_add_packages TARGET  "${_OUTPUT_DIR}/${_UNTAR_DIR}" "${SCRIPT_DIR}" "${_OUTPUT_DIR}" "${_OUTPUT_DIR}/${_UNTAR_DIR}" "${KER_VER}" "${DESCRIPTION}"

	# existence of apt-get in a correclty build archive indicates that some packages are to be added
	if [ -d "${_OUTPUT_DIR}/${_UNTAR_DIR}/var/lib/apt" ] ; then
		# prepare apt-get package dependency database:
		/bin/rm -fr "${_OUTPUT_DIR}/var/lib/apt"
		mv "${_OUTPUT_DIR}/${_UNTAR_DIR}/var/lib/apt" "${_OUTPUT_DIR}/var/lib/apt"

		# disable apt-get contacting any other package databases than the local one
		local _DISABLED_APT_SRC_LIST="$(mktemp --tmpdir="${_OUTPUT_DIR}/etc/apt/" sources.list.XXXXXX)"
		mv "${_OUTPUT_DIR}/etc/apt/sources.list" "${_DISABLED_APT_SRC_LIST}"

		# add packages:
		# *.deb must be given with path to avoid treating as regex
		local _HARMLESS="^Get:.*${_UNTAR_DIR}\|^Preparing to unpack \|^Unpacking \|^Processing triggers for \|^Setting up \|^Selecting previously unselected package\|^(Reading database ... \|^E: Getting name for slave of master fd "
		pushd "${_OUTPUT_DIR}"		> /dev/null	# to get relative package paths below
		add_packages "${_OUTPUT_DIR}" \
				"${_HARMLESS}"	\
				NOUPDATE	\
				CLEAN	\
				"${_UNTAR_DIR}/archives/*"
		popd				> /dev/null

		mv "${_DISABLED_APT_SRC_LIST}" "${_OUTPUT_DIR}/etc/apt/sources.list"
	fi

	# The PRE/POST scripts should be run even if there are no packages to be added:
	post_add_packages TARGET  "${_OUTPUT_DIR}/${_UNTAR_DIR}" "${SCRIPT_DIR}" "${_OUTPUT_DIR}" "${_OUTPUT_DIR}/${_UNTAR_DIR}" "${KER_VER}" "${DESCRIPTION}"

	# if this is a bootstrap archive, move the directory up.
	[ "${_ARCHIVE_TYPE}" == BOOTSTRAP ] && move_bootstrap "${OUTPUT_DIR}"
	return 0
}

add_archive_list (){
	local _ARCHIVE_FILES_ARRAY=("$@")

	local _ARCHIVE=""
	for _ARCHIVE in "${_ARCHIVE_FILES_ARRAY[@]}" ; do
		# Make directory in the directory /tmp in chroot:
		local _UNTAR_DIR="$(cd "${OUTPUT_DIR}" ; mktemp --directory --tmpdir=. tmp/untar.XXXXXX)"
		add_archive "${_ARCHIVE}" "${_UNTAR_DIR}" NotBOOTSTRAP
		/bin/rm -fr "${OUTPUT_DIR}/${_UNTAR_DIR}"
	done
}

create_rootfs() {
	parse_parameters "$@"
	check_parameters

	mkdir -p "${OUTPUT_DIR}"
	# make directory in the root of the new system
	local _UNTAR_DIR="$(mktemp --directory --tmpdir="${OUTPUT_DIR}" untar.XXXXXX)"

	add_archive "${BOOTSTRAP_FILE}" "$(basename "${_UNTAR_DIR}")" BOOTSTRAP
	/bin/rm -r "${_UNTAR_DIR}"

	[ ${#ARCHIVE_FILES_ARRAY[@]} -ne 0 ] && add_archive_list "${ARCHIVE_FILES_ARRAY[@]}"
	return 0
}

echo "Called as: $0 $@" >&2
create_rootfs "$@" && echo "Finished: $0 $@" >&2
