#!/bin/bash
# Makes the indicated software archive (SA) - one of: bootstrap, base, vca, mss, kvmgt.
# Also makes all the SAs which follow the indicated SA in the process of creation of a VCA bootable image.

set  -euEo pipefail
ARCHIVES=""		# to be appended later
ARCHIVE_PATH_IN=""	# to be set from the command line parameters
ARCHIVE_PATH_OUT="."	# path to archives to be created
DESCR=""	# to be set from the command line parameters
KERNEL_VER=""	# to be set from the command line parameters
readonly SCRIPT_DIR="$( cd "$(dirname "$0")" && pwd )"	# $0 works for more shells than  ${BASH_SOURCE[0]}
PATH="${SCRIPT_DIR}":${PATH}	# to incude the path where this script resides
PKGS_DIR_VCA=""	# optional, to be set from the command line parameters
PKGS_DIR_MSS=""	# optional, to be set from the command line parameters
TARGET=""	# to be set from the command line parameters

die(){
	local EXIT_CODE=$(( $? == 0 ? 99 : $? ))
	echo "$@" >&2
	exit ${EXIT_CODE}
}

in_set(){
	local _VALUE="$1";
	local _SET="$2"

	for _V in ${_SET} ; do
		[ "${_V}" == "${_VALUE}" ]  && return 0
	done
	return 1
}

show_help() {
	# Keep parameters alphabetically sorted in help and in the 'case' switch which parses them.
	# Prefer single-character options, i.e. do not introduce long options without an apparent reason.
	echo "Usage: $0 [OPTIONS]
Makes the indicated software archive (SA). Also makes all the SAs which follow the indicated SA in the process of creation of a VCA bootable image.

Options:
-d, --descr <descrptn>	OS version description, for which the SA is prepared.
-h, --help		Show this help screen.
-i, --in-dir <dir>	The input directory containing existing predecessor SAs which will be used during creation the requested SA.
-k, --kernel <version>	Kernel version, for which the SA is prepared.
-m, --mss <dir>	Name of directory containing the MSS installation files. Needed when requesting the creation of the mss SA, or any of its predecessors in the process of creation of a VCA bootable image.
-o, --out-dir <dir>	The name of the destination directory for the SAs created. By default the current directory is assumed.
-t, --target <name>	The name of the SA to be created. Additionally, all the SAs will be created which follow the indicated SA in the process of creation of a VCA bootable image. Supported names:
	bootstrap, base, vca, mss, kvmgt.
-v, --vca <dir>	Name of directory containing the VCA *.deb files. Needed when requesting the creation of the vca SA, or any of its predecessors in the process of creation of a VCA bootable image.

Examples:
To create the following SAs: bootstrap, base, vca, mss, and kvmgt in one shot (this means not using any existing SAs) in /tmp/b:
$0 -d \"UBUNTU 16.04\" -o /tmp/b -k 4.4.0-1.2.1.182.vca -t bootstrap -v /tmp/vca_pkgs_dir -m /tmp/mss_instal_dir
To create the base SA (using existing bootstrap SA from /tmp/b), and thus to create vca, mss, and kvmgt SAa as well in /tmp/o:
$0 -d \"UBUNTU 16.04\" -o /tmp/o -k 4.4.98-1.2.1.217.vca -t base -v /tmp/v -m /tmp/m/custom -i /tmp/b
To create only the kvmgt SA in /tmp/o (using existing bootstrap, base, and vca SAs from /tmp/b):
$0 -d \"UBUNTU 16.04\" -o /tmp/o -k 4.4.98-1.2.1.217.vca -t kvmgt -v /tmp/v -m /tmp/m/custom -i /tmp/b
"
}

parse_parameters(){
	while [ $# -gt 0 ] ; do
		case "$1" in
			-d|--descr)
				DESCR="${2:-""}"
				shift; shift;;
			-h|--help)
				show_help
				exit 0;;
			-i|--in-dir)
				ARCHIVE_PATH_IN="${2:-""}"
				shift; shift;;
			-k|--kernel)
				KERNEL_VER="${2:-""}"
				shift; shift;;
			-m|--mss)
				PKGS_DIR_MSS="${2:-""}"
				shift; shift;;
			-o|--out-dir)
				ARCHIVE_PATH_OUT="${2:-""}"
				shift; shift;;
			-t|--target)
				TARGET="${2:-""}"
				case "${TARGET}" in
					bootstrap|base|vca|mss|kvmgt)
					;;
					*)
						show_help && die "Unknown target ${TARGET}"
				esac
				shift; shift;;
			-v|--vca)
				PKGS_DIR_VCA="${2:-""}"
				shift; shift;;
			*)
				show_help && die "Unknown parameter '$1'"
		esac
	done
}

check_parameters(){
	[ -z "${DESCR}" ] 	&& show_help && die "OS version description not given"
	[ -z "${KERNEL_VER}" ]	&& show_help && die "Kernel version not given"
	[ -z "${TARGET}" ]	&& show_help && die "Target software archive name not given"
	[ "${TARGET}" != bootstrap ] && [ -z "${ARCHIVE_PATH_IN}" ] && show_help && die "The input directory with existing predecessor SAs not given"

	in_set "${TARGET}" "bootstrap base vca" && [ -z "${PKGS_DIR_VCA}" ]	&& show_help && die "Name of directory containing the VCA *.deb files not given"
	in_set "${TARGET}" "bootstrap base vca mss" && [ -z "${PKGS_DIR_MSS}" ]	&& show_help && die "Name of directory containing the MSS installation files not given"
	return 0
}

main(){
	parse_parameters "$@"
	check_parameters

	ROOTFS="$(mktemp --tmpdir --directory rootfs.XXX)"

	# create rootfs from existing SAs in an ordered manner:
	# This rootfs is the build environment for creating the desired target SA
	if [ "${TARGET}" !=  bootstrap ] ; then
		# make rootfs with bootstrap
		ARCHIVES="${ARCHIVES} -b ${ARCHIVE_PATH_IN}"/deb.tar
		if [ "${TARGET}" !=  base ] ; then
			# make rootfs with base
			ARCHIVES="${ARCHIVES} -a ${ARCHIVE_PATH_IN}"/base.tar
			if [ "${TARGET}" !=  vca ] ; then
				# make rootfs with vca
				ARCHIVES="${ARCHIVES} -a ${ARCHIVE_PATH_IN}"/vca.tar
			fi
		fi

		create_rootfs.sh -k "${KERNEL_VER}" -d "${DESCR}" -o "${ROOTFS}" ${ARCHIVES}
	fi

	CREATE_ARCHIVE_BASE="create_archive.sh -c '$(hostname):$0,pwd=$(pwd)' -k ${KERNEL_VER}"
	CREATE_ARCHIVE_CMD='${CREATE_ARCHIVE_BASE} -r "${ROOTFS}"' #do not expand variables yet; ROOTFS may change
	# create next SAs in an ordered manner:
	case "${TARGET}" in
		bootstrap)
			${CREATE_ARCHIVE_BASE} -a "${ARCHIVE_PATH_OUT}"/deb.tar	--pre "${SCRIPT_DIR}"/archive_scripts/bootstrap/ubuntu_preadd.sh -d "${DESCR}"
			# A dirty hack to get the newly created rootfs from the call to CREATE_ARCHIVE_BASE:
			ROOTFS="$(cat /tmp/create_sw_archives.information.on.rootfs)"
		;&	# fall trough
		base)
			eval "${CREATE_ARCHIVE_CMD}" -a "${ARCHIVE_PATH_OUT}"/base.tar	--post "${SCRIPT_DIR}"/archive_scripts/base/postadd.sh -p $(awk -F# '{ print $1 }' "${SCRIPT_DIR}"/archive_scripts/base/ubuntu_pkgs.list)
			# grep ^[^#] ubuntu_pkgs.list;  egrep -v "^\s*(#|$)" ubuntu_pkgs.list; cat ubuntu_pkgs.list | cut -f1 -d"#"
		;&	# fall trough
		vca)
			eval "${CREATE_ARCHIVE_CMD}" -a "${ARCHIVE_PATH_OUT}"/vca.tar --post "${SCRIPT_DIR}"/archive_scripts/vca/postadd.sh -g "${PKGS_DIR_VCA}"
		;&	# fall trough
		mss)
			eval "${CREATE_ARCHIVE_CMD}" -a "${ARCHIVE_PATH_OUT}"/mss.tar	--post "${SCRIPT_DIR}"/archive_scripts/mss/ubuntu_postadd.sh -g "${PKGS_DIR_MSS}" -p $(awk -F# '{ print $1 }' "${SCRIPT_DIR}"/archive_scripts/mss/ubuntu_pkgs.list)
		;;&	# fall trough but check the label
		bootstrap|base|vca|kvmgt)	# in every case except mss
			eval "${CREATE_ARCHIVE_CMD}" -a "${ARCHIVE_PATH_OUT}"/kvmgt.tar	-p $(awk -F# '{ print $1 }' "${SCRIPT_DIR}"/archive_scripts/kvmgt/ubuntu_pkgs.list)
		;;	# do not fall trough
	esac

	#just in failure case could anything remain mounted
	umount "${ROOTFS}"/virtual_archive.*/custom 2>/dev/null
	/bin/rm -fr "${ROOTFS}"
}

echo "Called as: $0 $@" >&2
main "$@" && echo "Finished: $0 $@" >&2