# library of shell functions; to be sourced
set -eu

# Library's constants and internal variables. The variables not to be modified out of the library:
declare -gr -A _CONST_ADD_SCRIPTS=( ["PRE"]="preadd.sh" ["POST"]="postadd.sh" )	# Bash v.4 supports dictionaries. Mapping of installation steps to script file name:
readonly _CONST_APT_DB_FILENAME=apt.tar.gz
readonly _CONST_ARCHIVE_INFO_FILE_NAME=arch_info.txt
readonly _CONST_CUSTOM_DIR=custom
declare -gr -A _CONST_DISTROS=( ["16.04"]="xenial" ["16.04.3"]="xenial" ["16.04.4"]="xenial" ["16.10"]="yakkety")	# Bash v.4 supports dictionaries. Mapping of description to distribution name:
readonly _CONST_FAKE_EXT=ORG	# can not contains characters special to regex (used in sed)
_FAKED_COMMANDS_REGISTER=() 	# an array of faked commands' original and faked names; to be reverted to their original state
readonly _CONST_PKG_EXT=deb			# the file name extension for the software packages

die() {
        local EXIT_CODE=$(( $? == 0 ? 99 : $? ))
        echo "ERROR:"$@ >&2
        exit ${EXIT_CODE}
}

# Prepares a chroot environemt wich mounted /dev, /sys and proc,
# which it tries to unmount cleanly in case of failure (and success).
# Executes commands from stdin in the chrooted environment.
# returns the exit code of chroot.
# To get chroot fail at any error, pass your commands with 'set -e'
# Typical call:
#	chroot /tmp/dir /bin/bash << <a HERE-document with commands to be executed, if more than one>
do_chroot() {
	local _DIR="$1"
	shift
	local _CMD="$@"
	local _MSG="chroot to ${_DIR} with command: ${_CMD}"

	#echo "Prepraring ${_MSG}" >&2
	# fakeroot needs libraries
	# see Linux namespaces in https://boxbase.org/entries/2015/oct/5/linux-namespaces/ , http://man7.org/linux/man-pages/man1/unshare.1.html
	# read-only mounting should be enough for many cases
	sudo mount -o ro -t proc none "${_DIR}/proc"	|| die "Could not mount ${_DIR}/proc"
	sudo mount -o ro -t sysfs /sys "${_DIR}/sys" || die "Could not mount ${_DIR}/sys"
	# /dev must be RW, as Ubuntu needs write access during dist-upgrade, probably running 'MAKEDEV generic' as 88 devices are created. OTOH CentOS needs /dev mounted as it only populates a handful of devices: fd  full  kmsg  null  ptmx  pts  random  shm  stderr  stdin  stdout  tty  urandom  zero and we do not include MAKEDEV package to run 'MAKEDEV generic'. All in one: mounting rw:
	sudo mount --bind /dev "${_DIR}/dev"	|| die "Could not mount ${_DIR}/dev"	# bind just causes re-mounting, not re-creation of the filesystem
	sudo mount -o newinstance -t devpts devpts "${_DIR}/dev/pts" || die "Could not mount ${_DIR}/dev/pts" # to replace with another one the dpkg error: E: Can not write log (Is /dev/pts mounted?) - posix_openpt (2: No such file or directory)
	sudo mount --bind "${_DIR}/dev/ptmx" "${_DIR}/dev/pts/ptmx"	|| die "Could not mount ${_DIR}/dev/pts/ptmx"
	# TODO: try mounting in this way: https://github.com/spotify/linux/blob/master/Documentation/filesystems/devpts.txt

	local _STATUS=0
	# if using sudo, -E is necessary if e.g. environmental variables for proxiea are to be passed
	sudo -E chroot "${_DIR}" ${_CMD} || _STATUS=$?	# a trick to save exit code in case this is executed from a script with "-e"

	# a dare attempt to kill any processes occupying the chroot.
	# there should be none, and if the out-of-chroot systemd or rsyslogd ocupy it, they are unkillable anyway
	# local _IS_BUSY="$(fuser -m "${_DIR}/proc" 2>/dev/null )"
	# if [ -n "${_IS_BUSY}" ] ; then
		# echo "The chroot ${_DIR} is used by the following processes. Killing before trying to unmount:" >&2
		# ps -p $(fuser -m "${_DIR}/proc" 2>/dev/null) | awk '{ printf "\t(%5d) %s\n", $1, $5 }' >&2
		# exit 1
		# # killing systemd does not hurt Ubuntu:
		# fuser -m "${_DIR}/proc" 2>/dev/null | xargs kill -9
	# fi
	sudo umount -lf "${_DIR}/dev/pts/ptmx"	|| die "Could not umount ${_DIR}/dev/pts/ptmx"
	sudo umount -lf "${_DIR}/dev/pts"|| die "Could not umount ${_DIR}/dev/pts"
	sudo umount -lf "${_DIR}/dev"	|| die "Could not umount ${_DIR}/dev"
	sudo umount -lf "${_DIR}/sys"	|| die "Could not umount ${_DIR}/sys"
	sudo umount -lf "${_DIR}/proc"	|| die "Could not umount ${_DIR}/proc"

	#echo "End of ${_MSG}" >&2
	return ${_STATUS}	# late return of chroot exit status
}

create_info_file(){
	local _ARCHIVE_INFO_DIR="$1"	; shift
	local _DESCR="$1"	; shift
	local _KERNEL_VER="$1"	; shift
	local _AUTHOR="$1"	; shift
	local _PKG_LIST="$@"

	# add the info file to the final archive. It is assumed to be run in chrooted environment, so to collect information on the TARGET environment:
	local _TARGET_DISTRO="$(cat /etc/*release 2>/dev/null | grep "DISTRIB_DESCRIPTION\|PRETTY_NAME")"
	which lsb_release > /dev/null 2>&1 && _TARGET_DISTRO="$(lsb_release -a 2>/dev/null)"

	echo "
This archive can contain a bootstrap with already installed software packages and/or package files and other files to be used during the application of this archive.
Packages usually have dependencies on another packages. Such dependant packages (if any) were automatically selected and the corresponding package files have been included in this archive. These dependant packages are guaranteed to match the needs of the essential packages at the time of selection, and to fit with the packages already pre-existing in the build chroot environment.
The apt-get db included in the archive with the packages is up-to-date to the point to reflect the dependencies for all the packages included in this archive, and for the packages already pre-existing in the build chroot environment.
When adding any future packages in the production environment, the apt-get db should be updated first (apt-get update) as future packages may depend on newer version of the packages that are in this archive.

Essential packages: $(echo "${_PKG_LIST}" | tr '\n' ' ')
Creation date: $(date 2>/dev/null)
Author: ${_AUTHOR}
Build system: $(uname -a 2>/dev/null)
Build system distribution: $(echo "${_TARGET_DISTRO}" | tr '\n' ',')
Target kernel: ${_KERNEL_VER}
Target system distribution: ${_DESCR}" \
		> "${_ARCHIVE_INFO_DIR}/${_CONST_ARCHIVE_INFO_FILE_NAME}"
}

# The file/directory structure of an archive is as follows:
# 	Max 2 directories:
#		/${_CONST_CUSTOM_DIR}/* 		# max one entry; this entry may contain bootstrap or something else
#		/archives/*.deb	# more than 0 *.deb files
#	An ${_CONST_APT_DB_FILENAME} if /archives/ exist
#	Max 2 named script files:
#		${_CONST_ADD_SCRIPTS["PRE"]}		# the PREADD script name
#		${_CONST_ADD_SCRIPTS["POST"]}		# the POSTADD script name
#	Always an ${_CONST_ARCHIVE_INFO_FILE_NAME}
# TODO: the role of this script is to only use prepared components, including an already-existing archive info file
assemble_archive (){
	local _ARCHIVE_FILE="$1"	; shift
	local _PKG_SRC_DIR="$1"		; shift
	local _PKG_DB_ROOT_DIR="$1"	; shift
	local _ARCHIVE_INFO_DIR="$1"; shift
	local _CUSTOM_FILE_DIR="$1"	; shift	# "" means no custom file
	local _PREADD_SCRIPT="$1"	; shift	# "" means no script
	local _POSTADD_SCRIPT="$1"	; shift	# "" means no script

	touch "${_ARCHIVE_FILE}"	# to add files/dirs later with 'tar rf'
	if [ -n "${_PKG_SRC_DIR}" ] ; then
		# add the *.deb files to the final archive:
		tar rf "${_ARCHIVE_FILE}" -C "${_PKG_SRC_DIR}" archives --exclude=lock --exclude=partial
		# add the apt-get db to the final archive:
		local _DB_TAR_TMP_DIR="$(mktemp --directory )"	# uses directory ${TMPDIR:-/tmp} (i.e. /tmp is the value by default)
		#tar czf "${_DB_TAR_TMP_DIR}/dpkg.tar.gz" -C "${_PKG_DB_ROOT_DIR}" var/lib/dpkg # dpkg db; compressed approx 6x
		tar czf "${_DB_TAR_TMP_DIR}/${_CONST_APT_DB_FILENAME}"  -C "${_PKG_DB_ROOT_DIR}" var/lib/apt  # apt-get db; compressed approx 6x
		tar rf "${_ARCHIVE_FILE}" -C "${_DB_TAR_TMP_DIR}" "${_CONST_APT_DB_FILENAME}"	# dpkg.tar.gz
		rm -fr "${_DB_TAR_TMP_DIR}"
	fi

	# add the _CUSTOM_FILE_DIR to the ${_CONST_CUSTOM_DIR} in the final archive:
	if [ -n "${_CUSTOM_FILE_DIR}" ] ; then
		# ensure by regex that the ${_CUSTOM_FILE_DIR} seems to reside in ${_CONST_CUSTOM_DIR}
		# regex is: insert ${_CONST_CUSTOM_DIR}/ at the beginning of each filename, excluding soft link tagets (-S)
		local _CUSTOM_PATH="$(dirname "${_CUSTOM_FILE_DIR}")"
		local _CUSTOM_NAME="$(basename "${_CUSTOM_FILE_DIR}")"	# this can be a directory or a file
		tar --transform "s,^${_CUSTOM_NAME%/},${_CONST_CUSTOM_DIR%/}/,S" -rf "${_ARCHIVE_FILE}" -C "${_CUSTOM_PATH}" "${_CUSTOM_NAME}"
	fi

	# add the PREADD/POSTADD scripts to the final archive only if they exist:
	local _SCRIPT_DIR="$(mktemp --directory )"	# uses directory ${TMPDIR:-/tmp} (i.e. /tmp is the value by default)
	if [ -n "${_PREADD_SCRIPT}"  ] ; then
		cp "${_PREADD_SCRIPT}"  "${_SCRIPT_DIR}/${_CONST_ADD_SCRIPTS["PRE"]}"
		tar rf "${_ARCHIVE_FILE}" -C "${_SCRIPT_DIR}" "${_CONST_ADD_SCRIPTS["PRE"]}"
	fi
	if [ -n "${_POSTADD_SCRIPT}" ] ; then
		cp "${_POSTADD_SCRIPT}" "${_SCRIPT_DIR}/${_CONST_ADD_SCRIPTS["POST"]}"
		tar rf "${_ARCHIVE_FILE}" -C "${_SCRIPT_DIR}" "${_CONST_ADD_SCRIPTS["POST"]}"
	fi
	/bin/rm -r "${_SCRIPT_DIR}"

	tar rf "${_ARCHIVE_FILE}" -C "${_ARCHIVE_INFO_DIR}" "${_CONST_ARCHIVE_INFO_FILE_NAME}"
}

# extract & check the file/directory structure
# return non-zero if verification unsuccesfull, zero otherwise
# The file/directory structure of an archive is as follows:
# 	Max 2 directories:
#		/${_CONST_CUSTOM_DIR}/* 		# max one entry; this entry may contain bootstrap or something else
#		/archives/*.deb	# more than 0 *.deb files
#	An ${_CONST_APT_DB_FILENAME} if /archives/ exist
#	Max 2 named script files:
#		${_CONST_ADD_SCRIPTS["PRE"]}		# the PREADD script name
#		${_CONST_ADD_SCRIPTS["POST"]}		# the POSTADD script name
#	Always an ${_CONST_ARCHIVE_INFO_FILE_NAME}
extract_archive () {
	local _TAR="$1"
	local _DST="$2"

	tar xf "${_TAR}" -C "${_DST}" || die "Could not untar ${_TAR} to ${_DST}"

	# check ${_CONST_CUSTOM_DIR} and archives
	local _UNEXPECTED_DIRS="$(find "${_DST}" -mindepth 1 -maxdepth 1 -type d ! -name ${_CONST_CUSTOM_DIR} ! -name archives)"
	[ -n "${_UNEXPECTED_DIRS}" ] && echo "Unexpected entries in archive ${_TAR}: ${_UNEXPECTED_DIRS}" >&2 && return 1

	# check the ${_CONST_CUSTOM_DIR}/
	[ -d "${_DST}/${_CONST_CUSTOM_DIR}" ] && [ $(count_entries_in_dir "${_DST}/${_CONST_CUSTOM_DIR}" "*") -ne 1 ] && echo "NOTICE: More than one custom file/directory entry in ${_TAR}" >&2

	# check the archives/
	[ -d "${_DST}/archives" ] && [ ! -s "${_DST}/${_CONST_APT_DB_FILENAME}" ] && echo "Archive directory without apt-get database "${_CONST_APT_DB_FILENAME}" in archive ${_TAR} (warning only)" >&2 && return 0
	[ ! -d "${_DST}/archives" ] && [ -s "${_DST}/${_CONST_APT_DB_FILENAME}" ] && echo "An apt-get database "${_CONST_APT_DB_FILENAME}" without the archive directory in archive ${_TAR}" >&2 && return 1
	if [ -d "${_DST}/archives" ] ; then
		local _PKG_ENTRIES="$(count_entries_in_dir "${_DST}/archives" "*.${_CONST_PKG_EXT}")"
		[ ${_PKG_ENTRIES} -lt 1 ] && echo "Directory ${_DST}/archives exists but no package files in archive ${_TAR}" >&2 && return 1
		local _OTHER_ENTRIES="$(count_entries_in_dir "${_DST}/archives" "*")"
		let _OTHER_ENTRIES=$((${_OTHER_ENTRIES}-${_PKG_ENTRIES}))
		[ ${_OTHER_ENTRIES} -gt 0 ] && echo "Non-package files the directory ${_DST}/archives of the archive ${_TAR}: $(ls ${_DST}/archives/*.${_CONST_PKG_EXT} | fgrep -v .deb )" >&2 && return 1
	fi

	# Count the number of regular files, excluding the named ones
	local _UNEXPECTED_FILES="$(find "${_DST}" -mindepth 1 -maxdepth 1 -type f ! -name "${_CONST_ADD_SCRIPTS["PRE"]}" ! -name "${_CONST_ADD_SCRIPTS["POST"]}" ! -name "${_CONST_APT_DB_FILENAME}" ! -name "${_CONST_ARCHIVE_INFO_FILE_NAME}" )"
	[ -n "${_UNEXPECTED_FILES}" ] && echo "Unexpected files in archive ${_TAR}: ${_UNEXPECTED_FILES}" >&2 && return 1

	[ ! -s "${_DST}/${_CONST_ARCHIVE_INFO_FILE_NAME}" ] && echo "Missing or empty ${_DST}/${_CONST_ARCHIVE_INFO_FILE_NAME} file in ${_TAR}" >&2 && return 1

	# check the apt-get db
	if [ -f "${_DST}/${_CONST_APT_DB_FILENAME}" ] ; then
		tar xzf "${_DST}/${_CONST_APT_DB_FILENAME}" -C "${_DST}"
		[ ! -d  "${_DST}/var/lib/apt" ] && echo "Missing apt-get db under ${_DST}" >&2 && return 1
		# expecting 4 subdirectories like: keyrings/  lists/  mirrors/  periodic/
		_NUM_OF_DIRS=$(ls -d "${_DST}/var/lib/apt/"*/				2>/dev/null | wc -l  )
		[[ ${_NUM_OF_DIRS} -lt 4  || ${_NUM_OF_DIRS} -gt 6 ]] && echo "Incorrect apt-get db structure under ${_DST} (${_NUM_OF_DIRS} subdirectories)" >&2 && return 1
		# As per https://askubuntu.com/a/28375 only *Release and *Packages files are needed; but I have observed: 
		# -name \*_Packages -o -name \*_InRelease -or -name \*_Release:
		# -name \*_Packages -o -name \*_InRelease -or -name \*_i18n_Translation-\*:
		local _NUM_OF_DB_FILES=$(find "${_DST}/var/lib/apt/lists/" -maxdepth 1 -size +0 \
			\( -name \*_Release -o -name \*_InRelease -o -name \*_Packages \)			2>/dev/null | wc -l )
		[ ${_NUM_OF_DB_FILES} -lt 2 ] && echo "Missing or empty apt-get db files under ${_DST}/var/lib/apt/lists" >&2 && return 1
	fi
	return 0
}

# Move the extracted bootstrap tree up into the ${_OUTPUT_DIR}
# leaving any entries in the destination root ${_OUTPUT_DIR}
# (the only expectd one is the untared archvive during bootstrap install)
# The bootstrap tree is initially in: ${_OUTPUT_DIR}/<untar temp name>/custom/<bootstrap dir name>
move_bootstrap(){
	local _OUTPUT_DIR="$1"

	local _UNTAR_TMP_DIR="$(cd "${_OUTPUT_DIR}"; ls)"

	# move up:
	cd "${_OUTPUT_DIR}"
	mv "${_UNTAR_TMP_DIR}/${_CONST_CUSTOM_DIR}"/* . || die "Could not move ${_UNTAR_TMP_DIR}/${_CONST_CUSTOM_DIR}/* to ${_OUTPUT_DIR}"
	cd - > /dev/null
}

count_entries_in_dir(){
	local _DIR="${1%/}"	# strip trailing '/', if any
	local _PATTERN="$2"

	find "${_DIR}" -mindepth 1 -maxdepth 1 -name "${_PATTERN}"  | wc -l
}

# Extracts and verifies the content of archive/bootstrap _ARCHIVE_FILE
# Prints to stdout the archive information if _PRINT_INFO_FILE is set to any value
# Returns status of the extraction
# The file/directory structure of an archive is as follows:
# 	Max 2 directories:
#		/${_CONST_CUSTOM_DIR}/* 		# max one entry; this entry may contain bootstrap or something else
#		/archives/*.deb	# more than 0 *.deb files
#	An ${_CONST_APT_DB_FILENAME} if /archives/ exist
#	Max 2 named script files:
#		${_CONST_ADD_SCRIPTS["PRE"]}		# the PREADD script name
#		${_CONST_ADD_SCRIPTS["POST"]}		# the POSTADD script name
#	Always an ${_CONST_ARCHIVE_INFO_FILE_NAME}
verify_archive(){
	local _ARCHIVE_FILE="$1"
	local _PRINT_INFO="${2:-FALSE}"

	local _VERIF_DIR="$(mktemp -d)"
	local _STATUS=0
	extract_archive "${_ARCHIVE_FILE}" "${_VERIF_DIR}" || _STATUS=$?

	if [ "${_PRINT_INFO}" != FALSE ] ; then
		[ -s "${_VERIF_DIR}/${_CONST_ARCHIVE_INFO_FILE_NAME}" ] &&
			cat "${_VERIF_DIR}/${_CONST_ARCHIVE_INFO_FILE_NAME}"
		local _CUST_ENTRY_NAME="$(ls ${_VERIF_DIR}/${_CONST_CUSTOM_DIR} 2>/dev/null)"
		printf "Custom file/directory: ${_CUST_ENTRY_NAME:-NONE}" >&2 || true
		if [ -z "${_CUST_ENTRY_NAME}" ] ; then
			echo "" >&2		# a newline
		else
			# the /tmp direcoty is used during extraction od archives:
			if [[	-d "${_VERIF_DIR}/${_CONST_CUSTOM_DIR}/${_CUST_ENTRY_NAME}"/bin	&& \
					-d "${_VERIF_DIR}/${_CONST_CUSTOM_DIR}/${_CUST_ENTRY_NAME}"/lib	&& \
					-d "${_VERIF_DIR}/${_CONST_CUSTOM_DIR}/${_CUST_ENTRY_NAME}"/tmp	&& \
					-d "${_VERIF_DIR}/${_CONST_CUSTOM_DIR}/${_CUST_ENTRY_NAME}"/etc ]] ; then
				echo "	(looks like a proper bootstrap)" >&2
			else
				echo "	(doesn't look like a bootstrap)" >&2
			fi
		fi
		local _PKG_FILE_COUNT="$(ls "${_VERIF_DIR}/archives" 2>/dev/null | wc -l )" || true
		echo "${_CONST_PKG_EXT} package files ("${_PKG_FILE_COUNT}"): $(ls "${_VERIF_DIR}/archives" 2>/dev/null | tr '\n' ' ' )" >&2 || true
		local _FILE
		_FILE="$(cd "${_VERIF_DIR}"; ls "${_CONST_ADD_SCRIPTS["PRE"]}" 2>/dev/null)" || true
		echo "PREADD script:	${_FILE:-NONE}" >&2
		_FILE="$(cd "${_VERIF_DIR}"; ls "${_CONST_ADD_SCRIPTS["POST"]}" 2>/dev/null)" || true
		echo "POSTADD script:	${_FILE:-NONE}" >&2
	fi
	echo "" >&2
	if [ ${_STATUS} -eq 0 ] ; then
		echo "Archive ${_ARCHIVE_FILE} verified succesfully" >&2
	else
		echo "Archive ${_ARCHIVE_FILE} verification failed." >&2
	fi
	/bin/rm -r "${_VERIF_DIR}"
	return ${_STATUS}
}

# adds packages given by ${_PKG_LIST} by filename or by package name into the environment at the root directory at ${_CHROOT_DIR}
# synchronizes apt-get database with Internet, if requested with ${_UPDATE_APT_GET}
# cleans apt-get cache before adding packages, if requested with ${_CLEAN_APT_GET}
# supresses ${_HARMLESS_MSGS} in lenghty apt-get/dpkg stdout
add_packages(){
	local _CHROOT_DIR="$1";		shift
	local _HARMLESS_MSGS="$1";	shift	# values: NOSUPRESS or a regex
	local _UPDATE_APT_GET="$1";	shift	# values: UPDATE or other (e.g. NOUPDATE)
	local _CLEAN_APT_GET="$1";	shift	# values: CLEAN or other (e.g. NOCLEAN)
	local _PKG_LIST="$@"				# values: DIST_UPGRADE or a list of packages

	do_chroot "${_CHROOT_DIR}" /bin/bash << EOF || die "Could not add package(s): ${_PKG_LIST} in chroot ${_CHROOT_DIR}"
		set -eo pipefail	# to fail pipes when grepping the output
		# prevent automatic start of services (from initd, but systemd should handle this as well):
		echo -e "#!/bin/sh\nexit 101" > /usr/sbin/policy-rc.d
		chmod a+x /usr/sbin/policy-rc.d
		export LC_ALL=C		# to avoid apt-get complainign on locales

		if [ "${_HARMLESS_MSGS}" == NOSUPRESS ] ; then
			GREP=":"
			MSG=""
		else
			GREP="grep -v \"${_HARMLESS_MSGS}\""
			MSG="
Supressing the following apt-get messages considered harmless: ${_HARMLESS_MSGS}"
		fi
		if [ "${_UPDATE_APT_GET}" == UPDATE ] ; then
			echo "Updating apt-get local database to install ${_PKG_LIST}\${MSG}" >&2
			apt-get update 2>&1 | eval "\${GREP}"
		fi

		[ "${_CLEAN_APT_GET}" == CLEAN ] && apt-get clean	# to clean the apt-get cache of any previously downloaded *.deb package files

		echo "Installing ${_PKG_LIST} \${MSG}" >&2
		if [ "${_PKG_LIST}" == DIST_UPGRADE ] ; then
			# piping to GREP will not mask failures, because -o pipefail is set above
			DEBIAN_FRONTEND=noninteractive RUNLEVEL=1 apt-get -y dist-upgrade \
				| eval "\${GREP}"
		else
			# not piping to GREP to not mask failures:
			DEBIAN_FRONTEND=noninteractive RUNLEVEL=1 apt-get install -y --no-install-recommends ${_PKG_LIST} 2>&1 \
				| eval "\${GREP}"
		fi
		rm /usr/sbin/policy-rc.d
EOF
}

get_codename(){
	local _DESCRIPTION="$1"

	local _DIST=""
	local _CORRECT_DIST=FALSE
	for _DIST in "${!_CONST_DISTROS[@]}"; do
		[ "${_DIST}" == "${_DESCRIPTION}" ] && _CORRECT_DIST=TRUE
	done
	[ "${_CORRECT_DIST}" == FALSE ] && die "Unsupported description given: ${_DESCRIPTION}. Supported descriptions: ${!_CONST_DISTROS[@]}"

	echo "${_CONST_DISTROS["${_DESCRIPTION}"]}"
}

get_from_archive_info_file(){
	local _ARCHIVE_INFO_FILE_PATH="$1"
	local _LABEL="$2"

	_LABEL="${_LABEL}: "	# this part will be deleted by sed below
	# print the rest of the line starting right after ^${_LABEL}
	# sed: replace ${_LABEL} with an empty string, and print (p) the result
	grep "^${_LABEL}" "${_ARCHIVE_INFO_FILE_PATH}/${_CONST_ARCHIVE_INFO_FILE_NAME}" |	sed -n -e "s/^${_LABEL}//p"
	return 0
}

# Ensure file/dir _DST_NAME does not exists
# If it does, move existing file(s) by appending _EXT to each, recursively
# The function accepts that _DST_NAME may end with _EXT
reserve_name(){
	local _DST_NAME="$1"
	local _EXT="$2"

	local _DST_BCKP="${_DST_NAME}"
	if [ ! -e "${_DST_BCKP}" ] ; then
		echo "NOTICE: File/directory ${_DST_BCKP} does not exists, no need to free the name" >&2
	else
		# find the shortest, yet unused, backup name based on _DST_NAME
		while [ -e "${_DST_BCKP}${_EXT}" ] ; do
			_DST_BCKP="${_DST_BCKP}${_EXT}"
		done
		# Now it is sure ${_DST_BCKP}.${_EXT} does not exist
		# rename all existing backup names to free the name _DST_NAME
		while [[ "${_DST_BCKP}" != "${_DST_NAME}" ]] ; do
			# TODO: use sudo or fakeroot?
			mv "${_DST_BCKP}" "${_DST_BCKP}${_EXT}"
			_DST_BCKP="${_DST_BCKP%${_EXT}}"		# cut off the trailing _EXT
		done
		# TODO: use sudo or fakeroot?
		mv "${_DST_NAME}" "${_DST_NAME}${_EXT}"
	fi
	# _DST_NAME name is unused now. Its content will be overriden if it exists and a file/dir is copied/renamed to it
}

# Revert the effects of reserve_name() by moving the content of _DST_NAME backup files.
# Recreates the name and contents of _DST_NAME and of its backup files, recursively (by removing the appendix _EXT  appended to backup files of _DST_NAME)
# The _DST_NAME will be created or its content will get overwritten, if its backup exists
release_name(){
	local _DST_NAME="$1"
	local _EXT="$2"

	# move all potentially remaining backup versions by cutting one level of _EXT
	[ ! -e "${_DST_NAME}${_EXT}" ] && echo "NOTICE: release of ${_DST_NAME} requested, but no backup exists" >&2
	[ -d "${_DST_NAME}" ] && [ -e "${_DST_NAME}${_EXT}" ] && /bin/rm -fr "${_DST_NAME}"	# moving below will not work apropriately if _DST_NAME directoy exists
	while [ -e "${_DST_NAME}${_EXT}" ] ; do 
		# TODO: use sudo or fakeroot?
		mv "${_DST_NAME}${_EXT}" "${_DST_NAME}"
		_DST_NAME="${_DST_NAME}${_EXT}"
	done
}

# Create the faked _COMMAND  name for its  given _VERSION using the appropriate format
createFakedCommandName(){
	local _COMMAND="$1"
	local _VERSION="$2"	# VERSION start with 1
	awk "BEGIN {printf(\"${_COMMAND}.%03d.${_CONST_FAKE_EXT}\", ${_VERSION})}"
}

# returns the name of the original, unfaked _COMMAND in the format it gets when faked
# Does not modify anything, does not check _COMMAND existence, does not check if the _COMMAND has been already faked indeed
getFakedCommandSavedName(){
	local _COMMAND="$1"
	createFakedCommandName "${_COMMAND}" 1
}

# find the faked command with the highest version, i.e. probably the one recently faked and return the next one
getFakedCommandBackupName(){
	local _COMMAND="$1"
	local _CMD_DIR="$(dirname "${_COMMAND}")"
	local _CMD_NAME="$(basename "${_COMMAND}")"
	
	local _CURRENT_HIGHEST="$(ls "${_CMD_DIR}/${_CMD_NAME}".???."${_CONST_FAKE_EXT}" 2>/dev/null | sort -r | head -1 | sed -e "s|.*${_CMD_NAME}\.\(.*\)\.${_CONST_FAKE_EXT}.*|\1|")"
	createFakedCommandName "${_COMMAND}" $((_CURRENT_HIGHEST + 1))
}

# renaming commands would be weak as their behaviour may depend on arg[0]
# manipulating PATH would be weak as executable path may be incluided in one of foreign executables
# Returns the backup name of the command (which can not be the original command at this point)
fakeAndRegisterCommand(){
	local _COMMAND="$1"

	[ ! -f "${_COMMAND}" ] && die "Command ${_COMMAND} does not exist, can not fake it !"
	local _FAKED_ORIGINAL_NAME="$(getFakedCommandSavedName "${_COMMAND}")"
	[ -f "${_FAKED_ORIGINAL_NAME}" ] && echo "NOTICE: Command ${_COMMAND} already faked, original command exists as ${_FAKED_ORIGINAL_NAME}" >&2

	# Save command permissions:
	local _CMD_PERMISSIONS="$(mktemp)"
	getfacl --absolute-names "${_COMMAND}" > "${_CMD_PERMISSIONS}"
	local _COMMAND_BACKUP="$(getFakedCommandBackupName "${_COMMAND}")"
	mv "${_COMMAND}" "${_COMMAND_BACKUP}"
	echo "#!/bin/bash
#
# Faked by $(hostname):$0 on $(date)
# Original command still exists as ${_FAKED_ORIGINAL_NAME}
# Previous (if any) faked version saved as ${_COMMAND_BACKUP}
"> "${_COMMAND}"
	cat - >> "${_COMMAND}"	# append stdin to ${_COMMAND}

	#restore command permissions:
	setfacl --restore="${_CMD_PERMISSIONS}"
	/bin/rm "${_CMD_PERMISSIONS}"
	# remove duplicate slashes, as a duplicate slash is used as field separator in the array below. Yes, thus loosing the triple slashes semantics.
	_COMMAND="$(tr -s / <<< "${_COMMAND}")" 
	_COMMAND_BACKUP="$(tr -s / <<< "${_COMMAND_BACKUP}")" 
	
	_FAKED_COMMANDS_REGISTER+=("${_COMMAND}"//"${_COMMAND_BACKUP}" )
	echo "${_COMMAND_BACKUP}"
}

# Restoring registered faked commands, in reverse order.
restoreFakedRegisteredCommands(){
	local _LAST_ONLY="${1:-""}"		# set to e.g. LASTONLY to only restore the last one
	
	local _END_IDX=0
	[ -n "${_LAST_ONLY}" ] && _END_IDX=$((${#_FAKED_COMMANDS_REGISTER[@]} - 1))
	# echo "Restoring registered faked commands, in reverse order. The register contains: ${_FAKED_COMMANDS_REGISTER[@]}" >&2
	for (( IDX=${#_FAKED_COMMANDS_REGISTER[@]}-1 ; IDX>=${_END_IDX}	; IDX-- )) ; do
		local _COMMAND_ORIGN="$(awk -F// '{print $1}' <<<"${_FAKED_COMMANDS_REGISTER[IDX]}")"
		local _COMMAND_SAVED="$(awk -F// '{print $2}' <<<"${_FAKED_COMMANDS_REGISTER[IDX]}")"
		mv "${_COMMAND_SAVED}" "${_COMMAND_ORIGN}"
	done
	if [ -n "${_LAST_ONLY}" ] ; then
		unset '_FAKED_COMMANDS_REGISTER[${#_FAKED_COMMANDS_REGISTER[@]}-1]' # remove the last element; the quotes prevent wildcard expansion in filenames
	else
		_FAKED_COMMANDS_REGISTER=()	#remove all elements by resetting the array
	fi
}

# Returns an integer:
# 00b -saved faked command does not exists, not registered
# 01b -saved faked command does not exists, but registered (should never happen!)
# 10b -saved faked command does exists, but unregistered (can happen if faking was done in another script run)
# 11b -saved faked command does exists, and registered
# In summary: value >1 means command is to be unFaked
isFakedOrRegisteredCommand(){
	die "Call to deprecated isFakedOrRegisteredCommand()"	# TODO: this needs replacement/removal since changing _FAKED_COMMANDS_REGISTER to array of pairs
	local _COMMAND="$1"

	local _CODE=0
	local _REGISTERED=""
	for _REGISTERED in "${_FAKED_COMMANDS_REGISTER[@]}"; do
		[ "${_REGISTERED}" == "${_COMMAND}" ] && _CODE=1 && break
	done
	[ -f "${_COMMAND}.${_CONST_FAKE_EXT}" ] && _CODE=$((_CODE+2))
	return ${_CODE}
}

# Run the  pre-add or post-add script with parameters.
run(){
	local _STEP="$1"
	local _PHASE="$2"
	local _ARCHIVE_DIR="$3"		# directory to the archive; during archive creation parts may be located far from the rest of other archive components so are softlinked to _ARCHIVE_DIR
	local _SCRIPTS_DIR="$4"		# path to the build scripts directory
	local _CHROOT_DIR="$5"
	local _SCRIPT_PATH="$6"		# path to the script to be run withouth script file name, eg. ${_OUTPUT_DIR}/${_UNTAR_DIR}

	local _SCRIPT="${_SCRIPT_PATH}/${_CONST_ADD_SCRIPTS["${_STEP}"]}"

	if [ -f "${_SCRIPT}" ] ; then
		echo "Running the ${_SCRIPT} script, phase ${_PHASE}, step ${_STEP} " >&2
		/bin/bash -e "${_SCRIPT}" "${_STEP}" "${_PHASE}" "${_ARCHIVE_DIR}" "${_SCRIPTS_DIR}" "${_CHROOT_DIR}" || die "Phase ${_PHASE} step ${_STEP} script ${_SCRIPT} failed"
	else
		return 0
	fi
}

# Prepare the environment and run the PRE-add script with parameters.
pre_add_packages(){
	local _PHASE="$1"
	local _ARCHIVE_DIR="$2"		# directory to the _archive; during archive creation parts may be located far from the rest of other archive components so are softlinkked to _VIRTUAL_ARCHIVE
	local _SCRIPTS_DIR="$3"			# path to the build scripts directory
	local _CHROOT_DIR="$4"
	local _SCRIPT_PATH="$5"			# path to the script to be run withouth script file name, eg. ${_OUTPUT_DIR}/${_UNTAR_DIR}
	local _KERNEL_VER="$6"
	local _DISTRIBUTION="$7"

	fakeAndRegisterCommand "${_CHROOT_DIR}"/bin/uname <<EOF >/dev/null
	case "\$1" in
		-r)
			echo "${_KERNEL_VER}"
			;;
		*)
			"$(getFakedCommandSavedName /bin/uname)" \$@
			;;
	esac
EOF
	# [ -x "${_CHROOT_DIR}"/usr/bin/lsb_release ] && fakeAndRegisterCommand "${_CHROOT_DIR}"/usr/bin/lsb_release <<EOF >/dev/null
	# case "\$1" in
		# -ds)
			# echo \"Ubuntu ${_DISTRIBUTION}\"
			# ;;
		# *)
			# "$(getFakedCommandSavedName /usr/bin/lsb_release)" \$@
			# ;;
	# esac
# EOF

	run PRE "${_PHASE}" "${_ARCHIVE_DIR}" "${_SCRIPTS_DIR}" "${_CHROOT_DIR}" "${_SCRIPT_PATH}"
}

# Run the POST-add script with parameters and restore the environment
post_add_packages(){
	local _PHASE="$1"
	local _ARCHIVE_DIR="$2"	# directory to the _archive; during archive creation parts may be located far from the rest of other archive components so are softlinkked to _VIRTUAL_ARCHIVE
	local _SCRIPTS_DIR="$3"			# path to the build scripts directory
	local _CHROOT_DIR="$4"
	local _SCRIPT_PATH="$5"			# path to the script to be run withouth script file name, eg. ${_OUTPUT_DIR}/${_UNTAR_DIR}

	run POST "${_PHASE}" "${_ARCHIVE_DIR}" "${_SCRIPTS_DIR}" "${_CHROOT_DIR}" "${_SCRIPT_PATH}"

	restoreFakedRegisteredCommands
}

# returns a dir full of links named appropriately
# missing archive elements not linked
# The caller is responsible for managing the created ${_VIRTUAL_ARCHIVE_DIR}
# TODO: consider replacing soft links with bind mount not only for _CUSTOM_FILE_DIR
# TODO: who umounts?
create_virtual_archive(){
	local _PACKAGE_DIR="$1"			# calling with "" means: none
	local _LOCAL_APT_DB_NAME="$2"	# calling with "" means: none
	local _CUSTOM_FILE_DIR="$3"		# calling with "" means: none
	local _PREADD_SCRIPT="$4"		# calling with "" means: none
	local _POSTADD_SCRIPT="$5"		# calling with "" means: none
	local _INFO_FILE="$6"			# calling with "" means: none

	local _VIRTUAL_ARCHIVE_DIR="$(mktemp --directory --tmpdir archive.XXXXXX)"
	[ -n "${_PACKAGE_DIR}" ] && mkdir -p "${_VIRTUAL_ARCHIVE_DIR}"/archives && mount --bind -o ro  "${_PACKAGE_DIR}" "${_VIRTUAL_ARCHIVE_DIR}"/archives
	[ -n "${_LOCAL_APT_DB_NAME}" ]	&& ln -s "${_LOCAL_APT_DB_NAME}" "${_VIRTUAL_ARCHIVE_DIR}/${_APT_DB_NAME}"
	[ -n "${_PREADD_SCRIPT}" ]		&& ln -s "${_PREADD_SCRIPT}"	"${_VIRTUAL_ARCHIVE_DIR}/${_CONST_ADD_SCRIPTS["PRE"]}"
	[ -n "${_POSTADD_SCRIPT}" ]		&& ln -s "${_POSTADD_SCRIPT}"	"${_VIRTUAL_ARCHIVE_DIR}/${_CONST_ADD_SCRIPTS["POST"]}"
	[ -n "${_INFO_FILE}" ]			&& ln -s "${_INFO_FILE}"	"${_VIRTUAL_ARCHIVE_DIR}/${_CONST_ARCHIVE_INFO_FILE_NAME}"
	if [ -n "${_CUSTOM_FILE_DIR}" ] ; then
		if [ -d "${_CUSTOM_FILE_DIR}" ] ; then
			mkdir -p "${_VIRTUAL_ARCHIVE_DIR}/${_CONST_CUSTOM_DIR}"
			mount --bind -o ro "${_CUSTOM_FILE_DIR}" "${_VIRTUAL_ARCHIVE_DIR}/${_CONST_CUSTOM_DIR}"
		else #so this is just one file
			ln -s "${_CUSTOM_FILE_DIR}"	"${_VIRTUAL_ARCHIVE_DIR}/${_CONST_CUSTOM_DIR}"
		fi
	fi

	echo "${_VIRTUAL_ARCHIVE_DIR}"
}

# gets a dir with well-known names of links to its components
# verifies this virtual archive silently
# creates archive blindly, after check
# TODO: join with assemble_archive
# TODO: remove '/archives' SEMANTICALLY from any form of _PKG_SRC_DIR/archives; internally use ${_CONST_...} to refer to the directory with *.deb's
# TODO: verification of (non)existing elements of the ufuture archive should be done once in the script
# TODO: _CONST_CUSTOM_DIR is currently unnecessarily appended twice: in create_archive_from_links and during archive creation (requires qorksround in kvmgt/qemu postadd script)
# TOTO: use this function in create_archive.sh
create_archive_from_links(){
	local _VIRTUAL_ARCHIVE_DIR="$1"
	local _ARCHIVE_FILE="$2"

	local _PKG_SRC_DIR=""	# TODO: This is a lie (in general)
	local _ARCHIVE_INFO_DIR="$(readlink --canonicalize-existing "${_VIRTUAL_ARCHIVE_DIR}")"
	local _PKG_DB_ROOT_DIR=""
	local _CUSTOM_FILE_DIR=""
	[ -e "${_VIRTUAL_ARCHIVE_DIR}/${_CONST_CUSTOM_DIR}" ] && _CUSTOM_FILE_DIR="$(readlink --canonicalize-existing "${_VIRTUAL_ARCHIVE_DIR}/${_CONST_CUSTOM_DIR}")"
	local _PREADD_SCRIPT=""
	[ -f "${_VIRTUAL_ARCHIVE_DIR}/${_CONST_ADD_SCRIPTS["PRE"]}"  ] && _PREADD_SCRIPT="$(readlink --canonicalize-existing "${_VIRTUAL_ARCHIVE_DIR}/${_CONST_ADD_SCRIPTS["PRE"]}")"
	local _POSTADD_SCRIPT=""
	[ -f "${_VIRTUAL_ARCHIVE_DIR}/${_CONST_ADD_SCRIPTS["POST"]}" ] && _POSTADD_SCRIPT="$(readlink --canonicalize-existing "${_VIRTUAL_ARCHIVE_DIR}/${_CONST_ADD_SCRIPTS["POST"]}")"

	assemble_archive "${_ARCHIVE_FILE}" "${_PKG_SRC_DIR}" "${_PKG_DB_ROOT_DIR}" "${_ARCHIVE_INFO_DIR}" "${_CUSTOM_FILE_DIR}" "${_PREADD_SCRIPT}" "${_POSTADD_SCRIPT}"  || die "Could not assemble archive ${_ARCHIVE_FILE}"

	[ -d "${_VIRTUAL_ARCHIVE_DIR}"/archives ] 				&& umount "${_VIRTUAL_ARCHIVE_DIR}"/archives
	[ -d "${_VIRTUAL_ARCHIVE_DIR}/${_CONST_CUSTOM_DIR}" ]	&& umount "${_VIRTUAL_ARCHIVE_DIR}/${_CONST_CUSTOM_DIR}"
	/bin/rm -r "${_VIRTUAL_ARCHIVE_DIR}"
}

#returns path relative to _ARCHIVE_DIR
get_archive_element_path(){
	local _ARCHIVE_DIR="$1"
	local _ARCHIVE_ELEM="$2"	# one of CUSTOM_DIR,
	
	case "${_ARCHIVE_ELEM}" in
		CUSTOM_DIR)
			echo "${_CONST_CUSTOM_DIR}"
			;;
		*)
			die "Unknown name of archive element: ${_ARCHIVE_ELEM}"
	esac
}

# call dpkg to add packages, supressing its messages except "^Selecting "
# Selecting previously unselected package linux-firmware-image-4.10.0-1.2.1.138.vca.
# (Reading database ... 11356 files and directories currently installed.)
# Preparing to unpack .../linux-firmware-image-4.10.0-1.2.1.138.vca_1.0_amd64.deb ...
# Unpacking linux-firmware-image-4.10.0-1.2.1.138.vca (1.0) ...
# Selecting previously unselected package linux-headers-4.10.0-1.2.1.138.vca.
# Preparing to unpack .../linux-headers-4.10.0-1.2.1.138.vca_1.0_amd64.deb ...
# Unpacking linux-headers-4.10.0-1.2.1.138.vca (1.0) ...
# Selecting previously unselected package linux-image-4.10.0-1.2.1.138.vca.
# Preparing to unpack .../linux-image-4.10.0-1.2.1.138.vca_1.0_amd64.deb ...
# Unpacking linux-image-4.10.0-1.2.1.138.vca (1.0) ...
# Selecting previously unselected package linux-libc-dev.
# Preparing to unpack .../linux-libc-dev_1.0_amd64.deb ...
# Unpacking linux-libc-dev (1.0) ...
# Selecting previously unselected package vcass-modules.
# Preparing to unpack .../vcass-modules_2.1.91-1_amd64.deb ...
# Unpacking vcass-modules (2.1.91-1) ...
# Setting up linux-firmware-image-4.10.0-1.2.1.138.vca (1.0) ...
# Setting up linux-headers-4.10.0-1.2.1.138.vca (1.0) ...
# Setting up linux-image-4.10.0-1.2.1.138.vca (1.0) ...
# Setting up linux-libc-dev (1.0) ...
# Setting up vcass-modules (2.1.91-1) ...
do_dpkg(){
	local -r _CONST_HARMLESS=""
}