/*
 * Intel VCA Software Stack (VCASS)
 *
 * Copyright(c) 2015-2017 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * The full GNU General Public License is included in this distribution in
 * the file called "COPYING".
 *
 * Intel VCA User Space Tools.
 */

#include "helper_funcs.h"

#include <stdio.h>
#include <stdarg.h>
#include <stdint.h>
#include <stdlib.h>

#include <assert.h>
#include <dirent.h>
#include <iostream>
#include <sstream>
#include <iomanip>
#include <vector>
#include <map>
#include <algorithm>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <string.h>
#include <typeinfo>

#include <boost/scope_exit.hpp>
#include <boost/filesystem.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/thread/thread.hpp>
#include <boost/interprocess/sync/file_lock.hpp>
#include <boost/interprocess/sync/scoped_lock.hpp>

#include <vca_mgr_ioctl.h>
#include <vca_mgr_extd_ioctl.h>
#include <vca_csm_ioctl.h>

#include "vca_defs.h"
#include "vcactrl.h"
#include "vca_config_parser.h"
#include "vca_eeprom_update.h"
#include "vca_blockio_ctl.h"
#include "vca_devices.h"
#include "log_args.h"

#define VCASYSFSDIR						"/sys/class/vca"
#define LINK_DOWN_STATE					"link_down"
#define LINK_DOWN_STATE_TRIES			60
#define LINK_DOWN_RESET_TRIES			3
#define WAIT_CMD_WA_ATTEMPTS			10	// this is a WA for states which requires a little bit longer to return with error
#define POWER_BUTTON_TIMEOUT			10
#define NODE_POWERDOWN_TIMEOUT			5000
#define GOLD_BIOS_LOAD_TIMEOUT			200000
#define MIN_MEM_FREE_OF_CACHE_CARD_SIDE "524288"

#define HANDSHAKE_RESET_TRIES		3
#define AFTER_HANDSHAKE_RESET_TRIES	3

#define BOARD_ID_INVALID 0xff

#define LINK_DOWN_WORKAROUND

using namespace Printer;

/* global error from vcactrl */
uint8_t command_err = 0;

static void print_help();

extern "C" void common_log(const char *format, ...) {
	va_list args;
	va_start(args, format);
	vprintf(format, args);
	va_end(args);
}

const char * get_vca_lbp_param_str(vca_lbp_param val)
{
	switch (val) {
	case VCA_LBP_PARAM_i7_IRQ_TIMEOUT_MS:
		return "VCA_LBP_PARAM_i7_IRQ_TIMEOUT_MS";
	case VCA_LBP_PARAM_i7_ALLOC_TIMEOUT_MS:
		return "VCA_LBP_PARAM_i7_ALLOC_TIMEOUT_MS";
	case VCA_LBP_PARAM_i7_CMD_TIMEOUT_MS:
		return "VCA_LBP_PARAM_i7_CMD_TIMEOUT_MS";
	case VCA_LBP_PARAM_i7_MAC_WRITE_TIMEOUT_MS:
		return "VCA_LBP_PARAM_i7_MAC_WRITE_TIMEOUT_MS";
	case VCA_LBP_PARAM_SIZE:
	default:
		LOG_ERROR("value not in vca_lbp_retval enum: %d\n", (int)val);
		return "NOT_VALID vca_lbp_param";
	}
}

const char *get_vca_lbp_retval_str(vca_lbp_retval val)
{
	switch (val) {
	case LBP_STATE_OK:
		return "LBP_STATE_OK";
	case LBP_SPAD_i7_WRONG_STATE:
		return "LBP_SPAD_i7_WRONG_STATE";
	case LBP_IRQ_TIMEOUT:
		return "LBP_IRQ_TIMEOUT";
	case LBP_ALLOC_TIMEOUT:
		return "LBP_ALLOC_TIMEOUT";
	case LBP_CMD_TIMEOUT:
		return "LBP_CMD_TIMEOUT";
	case LBP_BAD_PARAMETER_VALUE:
		return "LBP_BAD_PARAMETER_VALUE";
	case LBP_UNKNOWN_PARAMETER:
		return "LBP_UNKNOWN_PARAMETER";
	case LBP_INTERNAL_ERROR:
		return "LBP_INTERNAL_ERROR";
	case LBP_PROTOCOL_VERSION_MISMATCH:
		return "LBP_PROTOCOL_VERSION_MISMATCH";
	case LBP_WAIT_INTERRUPTED:
		return "LBP_WAIT_INTERRUPTED";
	case LBP_SIZE:
	default:
		LOG_ERROR("value not in vca_lbp_retval enum: %d\n", (int)val);
		return "NOT_VALID vca_lbp_retval";
	}
}

const char *get_plx_eep_retval_str(plx_eep_retval val)
{
	switch (val) {
	case PLX_EEP_STATUS_OK:
		return "PLX_EEP_STATUS_OK";
	case PLX_EEP_INTERNAL_ERROR:
		return "PLX_EEP_INTERNAL_ERROR";
	case PLX_EEP_TIMEOUT:
		return "PLX_EEP_TIMEOUT";
	case PLX_EEP_SIZE:
	default:
		LOG_ERROR("value not in plx_eep_retval enum: %d\n", (int)val);
		return "NOT_VALID plx_eep_retval";
	}
}

using namespace vca_config_parser;

struct commands {
	enum cmd {
		status = 0,
		reset,
		wait,
		wait_bios,
		boot,
		reboot,
		update_BIOS,
		recover_BIOS,
		update_MAC_ADDR,
		update_EEPROM,
		clear_SMB_event_log,
		script,
		config_show,
		config,
		config_use,
		config_default,
		temp,
		ICMP_watchdog,
		network,
		info,
		info_hw,		/* this command will stay to keep backward compatibility */
		info_system,		/* this command will stay to keep backward compatibility */
		pwrbtn_short,
		pwrbtn_long,
		help,
		blockio,
		os_shutdown,
		get_bios_cfg,
		set_bios_cfg,
		SIZE			/* for iteration over enum, contains size of enum*/
	};
};

class subcmds {
public:
	struct cmp_string {
	public:
		bool operator()(const char* lhs, const char* rhs) const {
			return strcmp(lhs, rhs) < 0;
		}
	};
	typedef std::map<const char*, const char*, cmp_string> subcmd;
	typedef std::map<const char*, const char*>::const_iterator iter;
public:
	subcmd _subcmds;

	subcmds() {}
	subcmds(subcmd subcmds) : _subcmds(subcmds) {}
	subcmds(const subcmds&);
	subcmds & operator=(const subcmds&);

	void add_subcmd(const char* name, const char* desc) {
		_subcmds.insert(std::pair<const char*, const char*>(name, desc));
	}
	~subcmds() {
		for (subcmd::iterator it = _subcmds.begin(); it != _subcmds.end();) {
			delete it->second;
			_subcmds.erase(it++);
		}
	}
};

struct debug_commands {
	enum {
		boot_USB = 0,
		set_SMB_id,
		gold,
		SIZE /* for iteration over enum, contains size of enum*/
	};
};

class thread_manager {
	std::vector<boost::thread*> threads;
public:
	void create_thread(void *function(void*), void* data) {
		threads.push_back(new boost::thread(function, data));
	}
	~thread_manager() {
		for (size_t i = 0; i < threads.size(); i++)
			threads[i]->join();

		for(size_t i = 0; i < threads.size(); i++)
			delete threads[i];

		threads.clear();
	}
};

class file_manager {
public:
	struct managed_file {
		char* content;
		size_t size;
		managed_file(char * content, size_t size): content(content), size(size){}
		~managed_file() { delete[] content; }
	private:
		managed_file(const managed_file&);
		managed_file & operator=(const managed_file&);
	};
private:
	struct cmp_string {
	public:
		bool operator()(const char* lhs, const char* rhs) const {
				return strcmp(lhs, rhs) < 0;
		   }
	};
	typedef std::map<const char*, managed_file*, cmp_string> file_map;
	file_map files;
	boost::mutex files_lock_mtx;

public:
	file_manager() {}
	managed_file *get_file_content(const char* filename) {
		boost::lock_guard<boost::mutex> guard(files_lock_mtx);

		file_map::iterator it = files.find(filename);
		if(it != files.end())
			return it->second;

		close_on_exit fd(open_path(filename, O_RDONLY));
		if (fd.fd == FAIL) {
			LOG_WARN("%s open %s\n", strerror(errno), filename);
			return NULL;
		}

		size_t f_sz = get_file_size(filename);
		char *buffer = new char[f_sz];
		if (read(fd, buffer, f_sz) < 0) {
			LOG_ERROR("could not read from %s\n", filename);
			delete[] buffer;
			return NULL;
		}

		managed_file * f = new managed_file(buffer, f_sz);
		files.insert(std::pair<const char*, managed_file*>(filename, f));
		return f;
	}
	~file_manager() {
		for(file_map::iterator it = files.begin(); it != files.end();) {
			delete it->second;
			files.erase(it++);
		}
	}
};

file_manager file_mgr;
vca_config config(VCA_CONFIG_PATH);

bool load_vca_config()
{
	try {
		if (!config.get_vca_config_from_file()) {
			LOG_ERROR("could not parse vca configuration file!\n");
			return false;
		}
	}
	catch (std::exception &ex) {
		LOG_ERROR("Exception loading vca configuration file: %s", ex.what());
		return false;
	}

	return true;
}

class args_holder {
private:
	std::vector<const data_field*> _holder;
	const char *_cmd_name;
	bool force;
	bool skip_modprobe_check;
public:
	args_holder() : _cmd_name(""), force(false), skip_modprobe_check(false) {}

	void add_arg(const char *name, const char *value) {
		data_field *df = new data_field(data_field::string, name, value);
		_holder.push_back(df);
	}
	void add_arg(const char *name, unsigned int value) {
		data_field *df = new data_field(data_field::number, name, int_to_string(value).c_str());
		_holder.push_back(df);
	}
	std::string get_arg(const char *name) {
		for (std::vector<const data_field*>::const_iterator it = _holder.begin(); it != _holder.end(); it++)
			if (strcmp((*it)->get_name(), name) == 0) {
				if (*it)
					return (*it)->get_string();
			}

		return "";
	}
	void set_cmd_name(const char *cmd_name) {
		_cmd_name = cmd_name;
	}
	const char *get_cmd_name() {
		return _cmd_name;
	}
	bool process_execution_flag(const char *arg) {
		if (strcmp(arg, "-v") == 0) {
			verbose = DEBUG_INFO;
		} else if (strcmp(arg, "-vv") == 0) {
			verbose = FULL_INFO;
		} else if (strcmp(arg, "--force") == 0) {
			force = true;
		} else if (strcmp(arg, "--skip-modprobe-check") == 0) {
			skip_modprobe_check = true;
		} else {
			return false;
		}
		return true;
	}
	bool is_force_cmd_enabled() {
		return force;
	}
	bool is_modprobe_check_skipped() {
		return skip_modprobe_check;
	}
	int size() const {
		return _holder.size();
	}
	const data_field* operator()(const char* name) const {
		for (std::vector<const data_field*>::const_iterator it = _holder.begin(); it != _holder.end(); it++)
			if (strcmp((*it)->get_name(), name) == 0)
				return *it;
		return NULL;
	}
	~args_holder() {
		for (size_t i = 0; i < _holder.size(); i++)
			delete _holder[i];
	}
};

typedef int(*parse_arg_func)(const char * arg, args_holder & holder);

enum parsing_output {
	PARSED = 0,
	NOT_PARSED,
	PARSING_FAILED
};

struct caller_data {
	const int card_id;
	const int cpu_id;
	int caller_id;	/* to identify the number of call, usually only need
			to check if it is the first call( caller_id == 0 )*/
	args_holder &args;
	caller_data(int card_id, int cpu_id, args_holder &holder)
		:card_id(card_id), cpu_id(cpu_id), caller_id(0), args(holder){}
	caller_data(int card_id, args_holder &holder)
		:card_id(card_id), cpu_id(0), caller_id(0), args(holder) {}

	const char *get_cmd_name() {
		return args.get_cmd_name();
	}
	void LOG_CPU(e_verbose verbose_level, const char *format, ...) const {
		if(verbose_level <= verbose) {
			char buffer[BUFFER_SIZE];
			va_list _args;
			va_start(_args, format);
			vsnprintf(buffer, sizeof(buffer), format, _args);
			va_end(_args);
			LOG(verbose_level, "Card: %d Cpu: %d - %s", card_id, cpu_id, buffer);
		}
	}
	void LOG_CPU_WARN(const char * format, ...) const {
		char buffer[BUFFER_SIZE];
		va_list _args;
		va_start(_args, format);
		vsnprintf(buffer, sizeof(buffer), format, _args);
		va_end(_args);
		LOG_WARN("Card: %d Cpu: %d - %s", card_id, cpu_id, buffer);
	}
	void LOG_CPU_ERROR(const char * format, ...) const {
		char buffer[BUFFER_SIZE];
		va_list _args;
		va_start(_args, format);
		vsnprintf(buffer, sizeof(buffer), format, _args);
		va_end(_args);
		LOG_ERROR("Card: %d Cpu: %d - %s", card_id, cpu_id, buffer);
	}
};

typedef bool (*function)(caller_data data);

class function_caller {
protected:
	function f;
	unsigned int calls_count;
public:
	function_caller(function f): f(f), calls_count(0) {}
	virtual void call(caller_data data) = 0;
	virtual ~function_caller(){}
};

class sequential_caller: public function_caller {
public:
	sequential_caller(function f): function_caller(f) {}
	void call(caller_data data) {
		try {
			data.caller_id = calls_count++;
			bool success = f(data);
			if (!success)
				command_err = EAGAIN;
		}
		catch (boost::interprocess::interprocess_exception &ex) {
			data.LOG_CPU_ERROR("Exception (interprocess_exception) encountered while executing "
				"%s command: %s\n", data.get_cmd_name(), ex.what());
			command_err = EAGAIN;
		}
		catch (std::exception &ex) {
			data.LOG_CPU_ERROR("Exception encountered while executing "
				"%s command: %s\n", data.get_cmd_name(), ex.what());
			command_err = EAGAIN;
		}
	}
};

class threaded_caller: public function_caller {
private:
	struct thread_data {
		function f;
		caller_data data;
		thread_data(function f, caller_data data):
			f(f), data(data){}
	};
	static void *thread_func(void *arg) {
		thread_data *d = (thread_data*)arg;
		try {
			bool success = d->f(d->data);
			if (!success)
				command_err = EAGAIN;
			delete d;
		}
		catch (boost::interprocess::interprocess_exception &ex) {
			d->data.LOG_CPU_ERROR("Exception (interprocess_exception) encountered while executing "
				"%s command: %s\n", d->data.get_cmd_name(), ex.what());
			command_err = EAGAIN;
		}
		catch (std::exception &ex) {
			d->data.LOG_CPU_ERROR("Exception encountered while executing "
				"%s command: %s\n", d->data.get_cmd_name(), ex.what());
			command_err = EAGAIN;
		}
		return NULL;
	}
	static thread_manager * _thread_mgr;
public:
	static void set_thread_manager(thread_manager * tm) {
		_thread_mgr = tm;
	}
	threaded_caller(function f): function_caller(f) {}
	void call(caller_data data) {
		if (_thread_mgr) {
			data.caller_id = calls_count++;
			_thread_mgr->create_thread(thread_func, new thread_data(f, data));
		}

		else
			data.LOG_CPU_ERROR("Thread manager not set for threaded_caller!\n");
	}
};

thread_manager *threaded_caller::_thread_mgr = NULL;


static bool validate_node_range(caller_data &d)
{
	if (d.card_id < 0 || d.card_id >= MAX_CARDS) {
		LOG_ERROR("Wrong card id! (got %d, expected from 0 to %d)\n",
			d.card_id, MAX_CARDS - 1);
		return false;
	}
	if (d.cpu_id < 0 || d.cpu_id >= MAX_CPU) {
		LOG_ERROR("Wrong cpu id! (got %d, expected from 0 to %d)\n",
			d.card_id, MAX_CPU - 1);
		return false;
	}

	return true;
}

static std::string get_node_lock_file_name(caller_data &d)
{
	return std::string(VCACTL_NODE_LOCK_PATH) + int_to_string(d.card_id) + int_to_string(d.cpu_id);
}


struct cmd_desc;
typedef parsing_output(*arg_parser)(const char *arg, args_holder &holder);
std::string get_card_gen(const unsigned card_id);
std::string get_eeprom_version(unsigned card_id);

struct cmd_desc {
	const char * name;
	const char * desc;
	function_caller * caller;
	arg_parser args[6];
	subcmds *subcmd;
	size_t args_size;

	cmd_desc(const char* name, const char* desc, function_caller * caller,
		arg_parser arg0 = NULL,
		arg_parser arg1 = NULL,
		arg_parser arg2 = NULL,
		arg_parser arg3 = NULL,
		arg_parser arg4 = NULL,
		arg_parser arg5 = NULL) :
		name(name), desc(desc), caller(caller), subcmd(NULL), args_size(0) {
		if (arg0)
			args[args_size++] = arg0;
		else return;
		if (arg1)
			args[args_size++] = arg1;
		else return;
		if (arg2)
			args[args_size++] = arg2;
		else return;
		if (arg3)
			args[args_size++] = arg3;
		else return;
		if (arg4)
			args[args_size++] = arg4;
		else return;
		if (arg5)
			args[args_size++] = arg5;
	}
	cmd_desc(const char* name, const char* desc, function_caller * caller, subcmds *sub,
		arg_parser arg0 = NULL,
		arg_parser arg1 = NULL,
		arg_parser arg2 = NULL,
		arg_parser arg3 = NULL,
		arg_parser arg4 = NULL,
		arg_parser arg5 = NULL) :
		name(name), desc(desc), caller(caller), subcmd(sub), args_size(0) {
		if (arg0)
			args[args_size++] = arg0;
		else return;
		if (arg1)
			args[args_size++] = arg1;
		else return;
		if (arg2)
			args[args_size++] = arg2;
		else return;
		if (arg3)
			args[args_size++] = arg3;
		else return;
		if (arg4)
			args[args_size++] = arg4;
		else return;
		if (arg5)
			args[args_size++] = arg5;
	}
	bool parse_args(char *argv[], args_holder &holder) const {
		for (const arg_parser* parser = args, *const end = args + args_size; parser < end; ++parser)
			switch((*parser)(*argv, holder)) {
			case PARSING_FAILED:
				LOG_DEBUG("%s\tPARSING_FAILED by %p\n", (*argv)?:"", parser);
				return false;
			case PARSED:
				LOG_DEBUG("%s\tPARSED by %p\n", (*argv)?:"", parser);
				if (*argv) ++argv; // Stay in NULL, because probably can be next parser with optional values
				continue;
			case NOT_PARSED:
				LOG_DEBUG("%s\tNOT_PARSED by %p\n", (*argv)?:"", parser);
				continue;
			}
		if (!*argv) {
			LOG_DEBUG("Parsed\n");
			return true;
		}
		LOG_ERROR("Unrecognized parameter %s\n", *argv);
		LOG_INFO("use: vcactl help\n");
		return false;
	}
};

std::string get_cpu_state(caller_data d);
bool try_link_up(const caller_data & d, int n, bool need_reset);
std::string get_cpu_os(caller_data d);
std::string get_bios_version(caller_data d, close_on_exit& cpu_fd);
bool is_mv_bios_version_correct(caller_data d);
bool is_os_windows(std::string os);

std::string get_vca_sysfs_dir(const caller_data &d)
{
	return std::string(VCASYSFSDIR) + "/vca" + int_to_string(d.card_id) + int_to_string(d.cpu_id) + "/";
}

std::string read_cpu_sysfs(const caller_data & d, const char *entry)
{
	std::string ret = "";
	std::string filename = get_vca_sysfs_dir(d) + entry;
	char value[PAGE_SIZE];

	close_on_exit fd(open_path(filename.c_str(), O_RDONLY));
	if (fd.fd == FAIL)
		return ret;

	int len = read(fd, value, sizeof(value) -1);
	if (len < 0) {
		d.LOG_CPU_ERROR("Failed to read sysfs entry %s\n",
			filename.c_str());
		return ret;
	}
	if (len == 0)
		return ret;

	if (value[len - 1] == '\n')
		value[len - 1] = '\0';
	else
		value[len] = '\0';

	ret = value;
	return ret;
}

const char *get_vca_ioctl_name(unsigned long ioctl_cmd)
{
	switch(ioctl_cmd) {
	case VCA_READ_CARD_TYPE:
		return "VCA_READ_CARD_TYPE";
	case VCA_READ_CPU_NUM:
		return "VCA_READ_CPU_NUM";
	case VCA_RESET:
		return "VCA_RESET";
	case VCA_POWER_BUTTON:
		return "VCA_POWER_BUTTON";
	case VCA_SET_SMB_ID:
		return "VCA_SET_SMB_ID";
	case VCA_UPDATE_EEPROM:
		return "VCA_UPDATE_EEPROM";
	case VCA_UPDATE_SECONDARY_EEPROM:
		return "VCA_UPDATE_SECONDARY_EEPROM";
	case VCA_READ_MODULES_BUILD:
		return "VCA_READ_MODULES_BUILD";
	case VCA_READ_BOARD_ID:
		return "VCA_READ_BOARD_ID";
	case VCA_READ_EEPROM_CRC:
		return "VCA_READ_EEPROM_CRC";
	case VCA_ENABLE_GOLD_BIOS:
		return "VCA_ENABLE_GOLD_BIOS";
	case VCA_DISABLE_GOLD_BIOS:
		return "VCA_DISABLE_GOLD_BIOS";
	default:
		LOG_ERROR("vca ioctl command name not found!\n");
		return "";
	};
}

inline bool check_plx_eep_state_ok(plx_eep_retval state, unsigned long ioctl_cmd)
{
	if (state != PLX_EEP_STATUS_OK) {
		LOG_ERROR("EEP %s returned with %s\n",
			get_vca_ioctl_name(ioctl_cmd), get_plx_eep_retval_str(state));
		return false;
	}
	return true;
}

bool vca_ioctl(filehandle_t fd, unsigned long ioctl_cmd, void *arg)
{
	int rc = ioctl(fd, ioctl_cmd, arg);
	if (rc != SUCCESS) {
		LOG_ERROR("%s failed: %s!\n", get_vca_ioctl_name(ioctl_cmd), strerror(errno));
		return false;
	}
	return true;
}

int vca_plx_eep_ioctl_with_bin(filehandle_t fd, char *img, size_t bin_size)
{
	int ret = SUCCESS;
	const unsigned long ioctl_cmd = VCA_UPDATE_EEPROM;
	vca_eeprom_desc *desc = (vca_eeprom_desc *)malloc(sizeof(*desc) + bin_size);
	if (!desc)
		return -ENOMEM;

	desc->ret = PLX_EEP_INTERNAL_ERROR;
	desc->buf_size = bin_size;
	memcpy(desc->buf, img, bin_size);

	if (!vca_ioctl(fd, ioctl_cmd, desc)) {
		ret = -EPERM;
	} else {
		if (!check_plx_eep_state_ok(desc->ret, ioctl_cmd))
			ret = FAIL;
	}

	free(desc);
	return ret;
}

int vca_plx_eep_ioctl_with_bin_mgr_extd(filehandle_t fd, char *img, size_t bin_size)
{
	int ret = SUCCESS;
	const unsigned long ioctl_cmd = VCA_UPDATE_SECONDARY_EEPROM;
	vca_secondary_eeprom_desc *desc =
			(vca_secondary_eeprom_desc *)malloc(sizeof(*desc) + bin_size);
	if (!desc)
		return -ENOMEM;

	desc->ret = PLX_EEP_INTERNAL_ERROR;
	desc->buf_size = bin_size;
	memcpy(desc->buf, img, bin_size);

	if (!vca_ioctl(fd, ioctl_cmd, desc)) {
		ret = -EPERM;
	} else {
		if (!check_plx_eep_state_ok(desc->ret, ioctl_cmd))
			ret = FAIL;
	}

	free(desc);
	return ret;
}

enum vca_card_type get_card_type(int card_id)
{
	close_on_exit card_fd(open_card_fd(card_id));
	if (card_fd < 0)
		return VCA_UNKNOWN;
	vca_card_type type;
	if (!vca_ioctl(card_fd, VCA_READ_CARD_TYPE, &type))
		return VCA_UNKNOWN;
	return type;
}

std::string get_modules_version(int card_id)
{
	close_on_exit card_fd(open_card_fd(card_id));
	if (card_fd < 0)
		return "Build unknown";
	char build_info[SMALL_OUTPUT_SIZE] = {0};
	if (!vca_ioctl(card_fd, VCA_READ_MODULES_BUILD, &build_info))
		return "Build unknown";
	return build_info;
}
int get_board_id(int card_id)
{
	filehandle_t card_extd_fd = open_extd_card_fd(card_id);
	close_on_exit fd_extd(card_extd_fd);
	 if (fd_extd < 0) {
		LOG_ERROR("Could not open card_id: %d file descriptor!", card_id);
		return BOARD_ID_INVALID;
	}
	int boardId = 0;
	if (!vca_ioctl(fd_extd, VCA_READ_BOARD_ID, &boardId))
		return BOARD_ID_INVALID;
	return boardId;
}

const char* get_csm_ioctl_name(unsigned long ioctl_cmd)
{
	switch(ioctl_cmd) {
	case LBP_HANDSHAKE:
		return "LBP_HANDSHAKE";
	case LBP_BOOT_RAMDISK:
		return "LBP_BOOT_RAMDISK";
	case LBP_BOOT_FROM_USB:
		return "LBP_BOOT_FROM_USB";
	case LBP_FLASH_BIOS:
		return "LBP_FLASH_BIOS";
	case LBP_FLASH_FIRMWARE:
		return "LBP_FLASH_FIRMWARE";
	case LBP_SET_PARAMETER:
		return "LBP_SET_PARAMETER";
	case CSM_START:
		return "CSM_START";
	case CSM_STOP:
		return "CSM_STOP";
	case LBP_GET_MAC_ADDR:
		return "LBP_GET_MAC_ADDR";
	case LBP_UPDATE_MAC_ADDR:
		return "LBP_UPDATE_MAC_ADDR";
	case LBP_SET_TIME:
		return "LBP_SET_TIME";
	case LBP_GET_BIOS_PARAM:
		return "LBP_GET_BIOS_PARAM";
	case LBP_SET_BIOS_PARAM:
		return "LBP_SET_BIOS_PARAM";
	case VCA_AGENT_COMMAND:
		return "VCA_AGENT_COMMAND";
	case LBP_CLEAR_SMB_EVENT_LOG:
		return "LBP_CLEAR_SMB_EVENT_LOG";
	case VCA_READ_EEPROM_CRC:
		return "VCA_READ_EEPROM_CRC";
	case LBP_BOOT_BLKDISK:
		return "LBP_BOOT_LBPDISK";
	default:
		LOG_ERROR("csm ioctl command name not found!\n");
		return "";
	};
}

bool csm_ioctl(filehandle_t fd, unsigned long ioctl_cmd, void* arg)
{
	if(ioctl(fd, ioctl_cmd, arg)) {
		LOG_ERROR("%s failed!\n", get_csm_ioctl_name(ioctl_cmd));
		return false;
	}
	return true;
}

bool is_link_up(const caller_data & d)
{
	return read_cpu_sysfs(d, "link_state") == "up";
}

bool is_power_off(const caller_data & d)
{
	return read_cpu_sysfs(d, "state") == VCA_POWER_DOWN_TEXT;
}

bool is_bios_up_or_ready(const caller_data & d)
{
	if (read_cpu_sysfs(d, "state") == "bios_up" ||
	    read_cpu_sysfs(d, "state") == "bios_ready")
		return true;
	else
		return false;
}

bool check_for_link_up(const caller_data & d, unsigned int timeout)
{
	boost::posix_time::ptime start = get_time();

	while(get_passed_time_ms(start) < timeout) {
		if (is_link_up(d))
			return true;
		sleep(1);
	}
	return false;
}

bool start_csm(filehandle_t cpu_fd, const caller_data & d)
{
	d.LOG_CPU(DEBUG_INFO, "Starting csm!\n");
	int ret;
	if(!csm_ioctl(cpu_fd, CSM_START, &ret))
		return false;
	if(ret) {
		if(ret == -EEXIST)
			return true;

		d.LOG_CPU_ERROR("Could not start CSM: %d\n", ret);
		return false;
	}
	return true;
}

bool stop_csm(filehandle_t cpu_fd, const caller_data & d)
{
	d.LOG_CPU(DEBUG_INFO, "Stopping CSM!\n");
	if (!csm_ioctl(cpu_fd, CSM_STOP, NULL))
		return false;
	return true;
}

bool get_mac_addr(filehandle_t cpu_fd, const caller_data & d, char mac_addr[6])
{
	d.LOG_CPU(DEBUG_INFO, "GETTING MAC ADDRESS!\n");
	vca_csm_ioctl_mac_addr_desc desc;
	if(!csm_ioctl(cpu_fd, LBP_GET_MAC_ADDR, &desc))
		return false;
	int mac_size = 6;
	memcpy((void*)mac_addr, (void*)desc.mac_addr, mac_size);
	return true;
}

inline bool check_lbp_state_ok(vca_lbp_retval state, unsigned long ioctl_cmd, const caller_data & d)
{
	if(state != LBP_STATE_OK) {
		if (state == LBP_PROTOCOL_VERSION_MISMATCH) {
			d.LOG_CPU_ERROR("\tToo old version of BIOS which does not support command ' %s BIOS '"
				". Update BIOS at node\n", d.args.get_cmd_name());
			command_err = EPERM;
			return false;
		}
		else if (state == LBP_BIOS_INFO_CACHE_EMPTY) {
			d.LOG_CPU_ERROR("\tCannot read BIOS info. Try again or if you will see this error"
				" repeatedly then try to update bios at node\n");
			command_err = EAGAIN;
			return false;
		}
		else {
			d.LOG_CPU_ERROR("%s returned with %s\n", get_csm_ioctl_name(ioctl_cmd), get_vca_lbp_retval_str(state));
			return false;
		}
	}
	return true;
}

bool set_time(filehandle_t cpu_fd, const caller_data & d)
{
	d.LOG_CPU(DEBUG_INFO, "SETTING TIME!!\n");
	vca_lbp_retval ret;
	if(!csm_ioctl(cpu_fd, LBP_SET_TIME, &ret))
		return false;

	return check_lbp_state_ok(ret, LBP_SET_TIME, d);
}

bool wait(caller_data d)
{
	unsigned int timeout = config.get_global_field(global_fields::wait_cmd_timeout_s).get_number() * 1000;
	boost::posix_time::ptime start = get_time();

	std::string state;
	int wait_wa_counter = 0;
	while(get_passed_time_ms(start) < timeout) {
		state = get_cpu_state(d);
		if(state == get_vca_state_string(VCA_NET_DEV_READY)) {
			d.LOG_CPU(VERBOSE_DEFAULT, "Net device ready!\n");
			return true;
		}
		else if ((state == get_vca_state_string(VCA_NET_DEV_DOWN) ||
				 state == get_vca_state_string(VCA_DHCP_ERROR)) &&
				 wait_wa_counter < WAIT_CMD_WA_ATTEMPTS) {
			wait_wa_counter++;
			// this is a WA for 'net_device_down' existance of after first Windows OS run (there is automatically reboot)
			// 'dhcp_error' state also need more time to change to 'net_device_ready'
		}
		else if (state == get_vca_state_string(VCA_DRV_PROBE_ERROR) ||
			 state == get_vca_state_string(VCA_DHCP_ERROR) ||
			 state == get_vca_state_string(VCA_NFS_MOUNT_ERROR) ||
			 state == get_vca_state_string(VCA_NET_DEV_DOWN)) {
			d.LOG_CPU_ERROR("error state detected: %s\n", state.c_str());
			return false;
		}
		sleep(1);
	}
	d.LOG_CPU_ERROR("wait TIMEOUT!\n");
	return false;
}

bool devices_ready(args_holder &holder, int available_nodes)
{
	int card_id = NO_CARD, cpu_id = NO_CPU;

	const data_field *card_id_f = holder(CARD_ID_ARG);
	const data_field *cpu_id_f = holder(CPU_ID_ARG);
	if (card_id_f) {
		card_id = card_id_f->get_number();
	}
	if (cpu_id_f) {
		cpu_id = cpu_id_f->get_number();
	}

	if (card_id == NO_CARD) {
		LOG_FULL("Waiting for devices");
	}
	else if (cpu_id == NO_CPU) {
		LOG_FULL("Waiting for card %d", card_id);
	}
	else
		LOG_FULL("Waiting for card %d cpu %d", card_id, cpu_id);

	int last_ready_nodes = 0, ready_nodes = 0;
	boost::posix_time::ptime start = get_time();
	do {
		if (ready_nodes > last_ready_nodes)
			start = get_time();
		last_ready_nodes = ready_nodes;

		ready_nodes = count_ready_nodes();
		if (ready_nodes == FAIL) {
			LOG_ERROR("Count ready devices failed!\n");
			return false;
		}
		if (available_nodes == ready_nodes) {
			LOG_FULL(" DONE!\n");
			return true;
		}

		if (card_id != NO_CARD && cpu_id == NO_CPU) {
			int cpu_count = 0;
			for (int i = 0; i < MAX_CPU; i++) {
				if (is_node_ready(card_id, i))
					++cpu_count;
			}
			if (cpu_count == MAX_CPU) {
				LOG_FULL(" DONE!\n");
				return true;
			}
		}
		else if (card_id != NO_CARD) {
			if (is_node_ready(card_id, cpu_id)) {
				LOG_FULL(" DONE!\n");
				return true;
			}
		}
		LOG_FULL(".");
		sleep(1);
	} while (get_passed_time_ms(start) < MODPROBE_TIMEOUT_MS);

	if (verbose == VERBOSE_DEFAULT || verbose == DEBUG_INFO) {
		if (card_id == NO_CARD)
			LOG_ERROR("Command vcactl %s FAILED due to modprobe timeout for some of nodes!\n", holder.get_cmd_name());
		else if (cpu_id == NO_CPU)
			LOG_ERROR("Command vcactl %s FAILED due to modprobe timeout for card %d!\n", holder.get_cmd_name(), card_id);
		else
			LOG_ERROR("Command vcactl %s FAILED due to modprobe timeout for card %d cpu %d!\n", holder.get_cmd_name(), card_id, cpu_id);
	}

	LOG_FULL(" FAILED: Timeout!\n");
	return false;
}

bool wait_bios(caller_data d)
{
	unsigned int normal_timeout = config.get_global_field(global_fields::wait_bios_cmd_timeout_s).get_number() * 1000;
	unsigned int flashing_timeout = config.get_global_field(global_fields::wait_bios_cmd_flashing_s).get_number() * 1000;
	unsigned int timeout = normal_timeout;
	unsigned int flashing_timeout_inc = 0;
	boost::posix_time::ptime start = get_time();
	if (flashing_timeout > normal_timeout)
		flashing_timeout_inc = flashing_timeout - normal_timeout;

	std::string state;
	int wait_wa_counter = 0;
	while(get_passed_time_ms(start) < timeout) {
		if (!try_link_up(d, LINK_DOWN_STATE_TRIES, false)) {
			d.LOG_CPU(DEBUG_INFO, "link_down occurs longer than %d sec!\n",
				(LINK_DOWN_STATE_TRIES * config.get_global_field(global_fields::link_up_timeout_ms).get_number()) / 1000);
			break;
		}

		state = get_cpu_state(d);
		if (state == get_vca_state_string(VCA_FLASHING)) {
			timeout += flashing_timeout_inc;
			flashing_timeout_inc = 0;
		}

		else if (state == get_vca_state_string(VCA_NET_DEV_DOWN)) {
			if (wait_wa_counter < WAIT_CMD_WA_ATTEMPTS)
				timeout += 1000;
			wait_wa_counter++;
			// this is a WA for longer existance of 'net_device_down' after powering down Windows OS
		}

		else if(state != get_vca_state_string(VCA_BIOS_DOWN) &&
			state != get_vca_state_string(VCA_FLASHING) &&
			state != get_vca_state_string(VCA_RESETTING) &&
			state != LINK_DOWN_STATE &&
			state != VCA_POWER_DOWN_TEXT &&
			state != VCA_POWERING_DOWN_TEXT) {
				d.LOG_CPU(VERBOSE_DEFAULT, "BIOS is up and running!\n");
			return true;
		}
		sleep(1);
	}
	d.LOG_CPU_ERROR("wait-BIOS TIMEOUT!\n");
	return false;
}

bool reset(caller_data d)
{
	d.LOG_CPU(DEBUG_INFO, "Resetting!\n");
	struct vca_ioctl_desc desc;
	desc.cpu_id = d.cpu_id;
	close_on_exit fd(open_card_fd(d.card_id));
	if (fd.fd == FAIL)
		return false;

	close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));
	if (cpu_fd.fd == FAIL)
		return false;

	stop_csm(cpu_fd, d);

	return vca_ioctl(fd, VCA_RESET, &desc);
}


bool is_blk_disk_active(caller_data d, filehandle_t blk_dev_fd, unsigned int
			blk_dev_id)
{
	if (is_blk_disk_opened(blk_dev_fd, blk_dev_id) &&
		!is_bios_up_or_ready(d) && get_cpu_state(d) != get_vca_state_string(VCA_BIOS_DOWN)) {
		return true;
	}
	else
		return false;
}

bool check_if_blkdev_is_not_active(caller_data &d)
{
	close_on_exit blk_dev_fd = open_blk_fd(d.card_id, d.cpu_id);
	if (blk_dev_fd.fd == FAIL)
		return false;

	for (int i = 0; i < MAX_BLK_DEVS; i++) {
		if (is_blk_disk_rw(blk_dev_fd.fd, i) &&
			is_blk_disk_active(d, blk_dev_fd.fd, i)) {
			return false;
		}
	}
	return true;
}

bool check_if_vcablk0_is_not_active(caller_data &d)
{
	close_on_exit blk_dev_fd = open_blk_fd(d.card_id, d.cpu_id);
	if (blk_dev_fd.fd == FAIL)
		return false;

	if (is_blk_disk_rw(blk_dev_fd.fd, 0) &&
		is_blk_disk_active(d, blk_dev_fd.fd, 0))
		return false;
	return true;
}

bool _reset(caller_data d)
{
	if(d.args.is_force_cmd_enabled())
		return reset(d);

	if (check_if_blkdev_is_not_active(d))
		return reset(d);
	else {
		d.LOG_CPU_ERROR("VCA Card %d CPU %d has an active "\
			"blockio device in RW mode\n", d.card_id,
			d.cpu_id);
		d.LOG_CPU(VERBOSE_DEFAULT, "Resetting such a "\
			"device may lead to data corruption. Reset the "\
			"node in a graceful way (from OS level), or "\
			"invoke reset with --force flag\n");
		return false;
	}
}

int check_power_button_state(caller_data d)
{
	struct vca_ioctl_desc desc;
	desc.cpu_id = d.cpu_id;
	close_on_exit fd(open_card_fd(d.card_id));
	if (fd.fd == FAIL)
		return -ENOENT;


	if (!vca_ioctl(fd, VCA_CHECK_POWER_BUTTON, &desc))
		return -EAGAIN;

	return desc.ret;
}

bool wait_power_button_release(caller_data d)
{
	d.LOG_CPU(DEBUG_INFO, "Power button wait for end!\n");
	int time_max = POWER_BUTTON_TIMEOUT;
	while (check_power_button_state(d) > 0 && --time_max > 0)
		sleep(1);

	return (time_max != 0);
}

static bool press_power_button(caller_data d, bool hold)
{
	struct vca_ioctl_desc desc;
	desc.cpu_id = d.cpu_id;
	desc.hold = hold;
	close_on_exit fd(open_card_fd(d.card_id));
	if (fd.fd == FAIL)
		return false;

	if (!vca_ioctl(fd, VCA_POWER_BUTTON, &desc))
		return false;

	if (hold){
		close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));
		csm_ioctl(cpu_fd, VCA_WRITE_SPAD_POWER_OFF, NULL);
	}

	if (!wait_power_button_release(d)) {
		d.LOG_CPU_ERROR("Failed when waiting for power button to be released.\n");
		return false;
	}

	if (hold) {
		close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));
		if (!csm_ioctl(cpu_fd, VCA_WRITE_SPAD_POWER_OFF, NULL)) {
			d.LOG_CPU_ERROR("Cannot write 'power_off' state to BIOS SPADs.\n");
			return false;
		}
	}

	return true;
}

bool _toggle_power_button(caller_data d)
{
	bool pwr_up = false;
	std::string state = get_cpu_state(d);
	d.LOG_CPU(DEBUG_INFO, "Toggling power button!\n");

	if (state == get_vca_state_string(VCA_POWER_DOWN)) {
		pwr_up = true;
	}

	if(!press_power_button(d, false))
		return false;
	if (pwr_up) {
		close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));

		if(cpu_fd.fd == FAIL)
			return false;

		return csm_ioctl(cpu_fd, VCA_WRITE_SPAD_POWER_BUTTON, NULL);
	}
	return true;
}

bool check_if_power_button_supported(caller_data d) {
	enum vca_card_type card_type = get_card_type(d.card_id);
	if (card_type == VCA_UNKNOWN) {
		d.LOG_CPU_ERROR("Card generation unknown, command %s not supported\n", d.get_cmd_name());
		return false;
	}
	else if (card_type & VCA_VV ) {
		d.LOG_CPU_ERROR("Card GEN1, command %s not supported\n", d.get_cmd_name());
		return false;
	}
	else
		return true;
}

bool pwrbtn_boot(caller_data d);
bool toggle_power_button(caller_data d)
{
	if (!check_if_power_button_supported(d))
		return false;

	bool pwr_up = false;
	std::string state = get_cpu_state(d);

	if (state == get_vca_state_string(VCA_POWER_DOWN)) {
		sleep(TIME_TO_POWER_DOWN_NODE_S); // to make sure that very quick 'pwrbtn-short' after reaching 'power_off' state will work as design
		pwr_up = true;
	}

	if (!_toggle_power_button(d))
		return false;
	if (!pwr_up)
		return true;

	return pwrbtn_boot(d);
}

bool hold_power_button(caller_data d)
{
	if (!check_if_power_button_supported(d))
		return false;

	d.LOG_CPU(DEBUG_INFO, "Holding power button!\n");
	close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));
	if (cpu_fd.fd == FAIL)
		return false;

	stop_csm(cpu_fd, d);
	return press_power_button(d, true);
}

bool set_SMB_id(caller_data d)
{
	LOG_DEBUG("Card: %d - Setting SMB id!\n", d.card_id);
	close_on_exit fd(open_card_fd(d.card_id));
	if (fd.fd == FAIL)
		return false;

	const data_field *smb_id = d.args(SMB_ID_ARG);
	if (!smb_id)
		return false;

	unsigned int id = smb_id->get_number();

	return vca_ioctl(fd, VCA_SET_SMB_ID, &id);
}

bool ICMP_watchdog(caller_data d)
{
	const data_field *trigger_f = d.args(TRIGGER_ARG);
	if (!trigger_f)
		return false;

	unsigned int trigger = trigger_f->get_number();
	std::string ip = "";

	const data_field *ip_f = d.args(IP_ADDR_ARG);
	if (ip_f)
		ip = ip_f->get_cstring();

	std::string watchdog_cmd = ECHO " " VCA_PING_DAEMON_CMD " "
		+ int_to_string(trigger) + " "
		+ int_to_string(d.card_id) + " "
		+ int_to_string(d.cpu_id) + " "
		+ ip + " > "
		MSG_FIFO_FILE;

	if (run_cmd(watchdog_cmd.c_str()) == FAIL) {
		LOG_ERROR("Cannot execute: %s\n", watchdog_cmd.c_str());
		return false;
	}

	return true;
}

bool set_lbp_param(filehandle_t cpu_fd, vca_lbp_param param, unsigned int value, const caller_data & d)
{
	d.LOG_CPU(FULL_INFO, "SET LBP PARAM: %s with %d\n", get_vca_lbp_param_str(param), value);
	vca_csm_ioctl_param_desc desc;
	desc.param = param;
	desc.value = value;
	if(!csm_ioctl(cpu_fd, LBP_SET_PARAMETER, &desc))
		return false;

	return check_lbp_state_ok(desc.ret, LBP_SET_PARAMETER, d);
}


/**
Checks whether cpu is in "link_up" state or not.

@param d Data about parameters of excecuted command.
@param n Number of iteration of tries reading sysfs file (there is 2 sec of each try).
@param need_reset Determine if reset is needed after link state will be down
@return Returns true if link_state is equal to "up"
        or false if within n seconds link_state will be still "down".
*/
bool try_link_up(const caller_data & d, int n, bool need_reset)
{
	d.LOG_CPU(FULL_INFO, "Checking for link up!\n");
	for(int i = 0; i < n; i++) {
		if(check_for_link_up(d, config.get_global_field(global_fields::link_up_timeout_ms).get_number()))
			return true;

		if (need_reset)
			reset(d);
	}
	return false;
}

bool handshake(filehandle_t cpu_fd, const caller_data & d)
{
	vca_lbp_retval ret;
	if(!csm_ioctl(cpu_fd, LBP_HANDSHAKE, &ret))
		return false;

	return check_lbp_state_ok(ret, LBP_HANDSHAKE, d);
}

bool try_handshake(filehandle_t cpu_fd, const caller_data & d)
{
	d.LOG_CPU(DEBUG_INFO, "Trying handshake!\n");
	for(int i = 0; i < HANDSHAKE_RESET_TRIES; i++) {
		if(handshake(cpu_fd, d))
			return true;

		reset(d);

		if (!try_link_up(d, LINK_DOWN_RESET_TRIES, true))
			return false;
	}
	d.LOG_CPU_ERROR("Failed to handshake after %d tries.\n", HANDSHAKE_RESET_TRIES);

	return false;
}

bool csm_ioctl_with_img(filehandle_t cpu_fd, unsigned long ioctl_cmd, const caller_data & d, void *img, size_t img_size)
{
	d.LOG_CPU(DEBUG_INFO, "%s\n", get_csm_ioctl_name(ioctl_cmd));
	vca_csm_ioctl_mem_desc desc;
	desc.mem = img;
	desc.mem_size = img_size;
	if(!csm_ioctl(cpu_fd, ioctl_cmd, &desc))
		return false;

	return check_lbp_state_ok(desc.ret, ioctl_cmd, d);
}

bool try_ioctl_with_img(filehandle_t cpu_fd, unsigned long ioctl_cmd, const caller_data & d, void *img, size_t img_size)
{
	for(int i = 0; i < AFTER_HANDSHAKE_RESET_TRIES; i++) {
		if (csm_ioctl_with_img(cpu_fd, ioctl_cmd, d, img, img_size))
			return true;

		reset(d);

		if(!try_handshake(cpu_fd, d))
			return false;
	}
	d.LOG_CPU_ERROR("Failed to execute IOCTL (%s) after %d tries.\n",
		get_csm_ioctl_name(ioctl_cmd), AFTER_HANDSHAKE_RESET_TRIES);

	return false;
}

bool csm_ioctl_with_blk(filehandle_t cpu_fd, unsigned long ioctl_cmd, const caller_data & d)
{
	d.LOG_CPU(DEBUG_INFO, "%s\n", get_csm_ioctl_name(ioctl_cmd));
	enum vca_lbp_retval ret = LBP_STATE_OK;
	if(!csm_ioctl(cpu_fd, ioctl_cmd, &ret))
		return false;

	return check_lbp_state_ok(ret, ioctl_cmd, d);
}

bool try_ioctl_with_blk(filehandle_t cpu_fd, unsigned long ioctl_cmd, const caller_data & d)
{
	for(int i = 0; i < AFTER_HANDSHAKE_RESET_TRIES; i++) {
		if (csm_ioctl_with_blk(cpu_fd, ioctl_cmd, d))
			return true;

		reset(d);

		if(!try_handshake(cpu_fd, d))
			return false;
	}
	return true;
}

bool prepare_cpu(filehandle_t cpu_fd, const caller_data &d)
{
	d.LOG_CPU(DEBUG_INFO, "PREPARING CARD!\n");

	if (!set_lbp_param(cpu_fd, VCA_LBP_PARAM_i7_IRQ_TIMEOUT_MS,
		config.get_global_field(global_fields::handshake_irq_timeout_ms).get_number(), d))
			return false;

	if (!set_lbp_param(cpu_fd, VCA_LBP_PARAM_i7_ALLOC_TIMEOUT_MS,
		config.get_global_field(global_fields::alloc_timeout_ms).get_number(), d))
			return false;

	if (!set_lbp_param(cpu_fd, VCA_LBP_PARAM_i7_CMD_TIMEOUT_MS,
		config.get_global_field(global_fields::cmd_timeout_ms).get_number(), d))
			return false;

	if (!set_lbp_param(cpu_fd, VCA_LBP_PARAM_i7_MAC_WRITE_TIMEOUT_MS,
		config.get_global_field(global_fields::mac_write_timeout_ms).get_number(), d))
			return false;

	if (!try_link_up(d, LINK_DOWN_RESET_TRIES, true))
		return false;

	if (get_vca_state_string(VCA_BIOS_READY) != get_cpu_state(d)) {
		if (!try_handshake(cpu_fd, d))
			return false;
	}
	else
		LOG_DEBUG("Skipping handshake, because node is already in bios_ready state\n");

	d.LOG_CPU(DEBUG_INFO, "Preparing card SUCCESS!\n");
	return true;
}

inline const char * get_cpu_field_if_not_empty(const caller_data & d, cpu_fields::_enum field, bool silent=false)
{
	const data_field & data = config.get_cpu_field(d.card_id, d.cpu_id, field);
	const char * value = data.get_cstring();
	if(strcmp(value, "") == 0) {
		if(!silent)
			d.LOG_CPU_ERROR("%s in config file is EMPTY!\n", data.get_name());
		return NULL;
	}
	return value;
}

static std::string get_hostip(const caller_data & d)
{
	const char *ip = get_cpu_field_if_not_empty(d, cpu_fields::ip);
	char dev_buffer[BUFFER_SIZE];
	if (!ip)
		return "";
	/* In case of IP assigned directly to node, use host side IP of virtual
	 * interface */
	if (is_ip_address(ip)) {
		const char *hostip = get_cpu_field_if_not_empty(d, cpu_fields::host_ip);
		/* if hostip == NULL log message will be printed in function
		 * 'get_cpu_field_if_not_empty' */
		if (!hostip)
			return "";
		if (!is_ip_address(hostip)) {
			d.LOG_CPU_ERROR("Card's node has IP statically assigned,"
					" while host side of the interface doesn't"
					" have a proper address configured: %s\n", hostip);
			return "";
		}
		return hostip;
	}
	if (strncasecmp(DHCP_STRING, ip, strlen(DHCP_STRING))) {
		d.LOG_CPU_ERROR("Card doesn't have IP or dhcp assigned: %s\n", ip);
		return "";
	}
	/* In case of node's IP taken from dhcp, must check if node's virtual
	 * interface is assinged to bridge. If so, must use bridge's IP */
	const char *bridge = get_cpu_field_if_not_empty(d, cpu_fields::bridge_interface);
	if (bridge) {
		size_t brigde_len = strlen(bridge);
		if (brigde_len + 1 > sizeof(dev_buffer)) {
			d.LOG_CPU_ERROR("Bridge device name too long: %d, max %d\n",
					brigde_len, sizeof(dev_buffer));
			return "";
		}
		STRCPY_S(dev_buffer, bridge, sizeof(dev_buffer));
	} else {
		/* In case of lack of bridging, use address assigned to default
		 * host interface. To do so, we need name of that interface. */
		const char *cmd = "ip route | awk '/^default / { print $5 }'";
		if (run_cmd_with_output(cmd, dev_buffer, sizeof(dev_buffer)) == FAIL) {
			LOG_ERROR("Cannot execute: %s\n", cmd);
			return "";
		}

		if (!dev_buffer[0]) {
			d.LOG_CPU_ERROR("Cannot obtain default network interface\n");
			return "";
		}
	}

	char command[BUFFER_SIZE];
	const int ret = snprintf(command, sizeof(command),
			"ip addr show dev %s | awk -F'[ /]+' '$2 == \"inet\" { print $3 }'",
			dev_buffer);
	if (ret == FAIL || ret >= (int) sizeof(command)) {
		d.LOG_CPU_ERROR("Command string to obtain host IP on interface %s too long: %s\n",
				dev_buffer, command);
		return "";
	}

	if (run_cmd_with_output(command, dev_buffer, sizeof(dev_buffer)) == FAIL) {
		LOG_ERROR("Cannot execute: %s\n", command);
		return "";
	}
	if (!is_ip_address(dev_buffer)) {
		d.LOG_CPU_ERROR("Read address isn't proper IP: %s\n", dev_buffer);
		return "";
	}

	return dev_buffer;
}

static std::string get_nodename(const caller_data & d)
{
	return std::string("\%s");
}

static std::string get_nodeip(const caller_data & d)
{
	return std::string("\%i");
}

static void strip_quotes(std::string& phrase)
{
	replace_all(phrase, "\"", "");
	replace_all(phrase, "\'", "");
}

bool parse_nfs_path(const caller_data & d, std::string& path)
{

	if(path.empty())
		return true;

	if(path.find("%hostname") != std::string::npos)
	{
		char hostname[BUFFER_SIZE] = "";
		if (gethostname(hostname, BUFFER_SIZE-1) == FAIL) {
			if (errno == ENAMETOOLONG)
				LOG_ERROR("hostname does not fit into %d bytes\n", BUFFER_SIZE-1);
			LOG_ERROR("Could not get hostname\n");
			return false;
		}
		replace_all(path, "%hostname", hostname);
	}

	if(path.find("%hostip") != std::string::npos)
	{
		std::string hostip = get_hostip(d);
		if (hostip.empty())
			return false;
		replace_all(path, "%hostip", hostip);
	}

	if(path.find("%nodename") != std::string::npos)
	{
		std::string nodename = get_nodename(d);
		if (nodename.empty())
			return false;
		replace_all(path, "%nodename", nodename);
	}

	if(path.find("%nodeip") != std::string::npos)
	{
		std::string nodeip = get_nodeip(d);
		if (nodeip.empty())
			return false;
		replace_all(path, "%nodeip", nodeip);
	}

	strip_quotes(path);

	return true;
}

bool pass_script(const caller_data & d, char mac_addr[6])
{
	std::string card_script;
	const char *node_name;
	std::string nfs_path, nfs_server;
	std::string net_cfg_cmd, net_cfg_path, sys_cfg_cmd;

	// gathering data
	const char* script_path = get_cpu_field_if_not_empty(d, cpu_fields::script_path, true);
	if (script_path && !file_exists(script_path))
		return false;

	const char *ip = get_cpu_field_if_not_empty(d, cpu_fields::ip);
	if (!ip)
		return false;

	const char *mask = get_cpu_field_if_not_empty(d, cpu_fields::mask);
	if (!mask || !is_unsigned_number(mask))
		return false;

	// to validate mask number range
	int imask = atoi(mask);
	if (imask < 0 || imask > 32)
		return false;

	const char *gateway = get_cpu_field_if_not_empty(d, cpu_fields::gateway);
	if (!gateway || !is_ip_address(gateway))
		return false;

	node_name = get_cpu_field_if_not_empty(d, cpu_fields::node_name);

	const char *nfs_server_data = get_cpu_field_if_not_empty(d, cpu_fields::nfs_server);
	if (nfs_server_data) {
		nfs_server = nfs_server_data;
		strip_quotes(nfs_server);
	}

	const char *nfs_path_data = get_cpu_field_if_not_empty(d, cpu_fields::nfs_path);
	if (nfs_path_data)
		nfs_path = nfs_path_data;

	if (!parse_nfs_path(d, nfs_path)) {
		d.LOG_CPU_ERROR("Problems during processing NFS path %s\n", nfs_path.c_str());
	}

	char buf[256];
	snprintf(buf, 18,"%x:%x:%x:%x:%x:%x",
		(unsigned char)mac_addr[0], (unsigned char)mac_addr[1], (unsigned char)mac_addr[2],
		(unsigned char)mac_addr[3], (unsigned char)mac_addr[4], (unsigned char)mac_addr[5]);

	// passing network configuration for Windows

	net_cfg_path = std::string(VCASYSFSDIR) + "/vca"
		+ int_to_string(d.card_id) + int_to_string(d.cpu_id) + "/net_config_windows";

	card_script.append("ip=" + char_to_string((char*)ip) + "\n");
	card_script.append("mask=" + char_to_string((char*)mask) + "\n");
	card_script.append("gateway=" + char_to_string((char*)gateway) + "\n");
	card_script.append("mtu=" + std::string(MTU_VALUE));

	net_cfg_cmd = std::string(STDBUF " -o 1024 " ECHO " \"")
		+ card_script + "\" > " + net_cfg_path;

	if (run_cmd(net_cfg_cmd.c_str()) == FAIL) {
		LOG_ERROR("Cannot execute: %s\n", net_cfg_cmd.c_str());
		return false;
	}

	// passing network configuration for Linux

	if (strncasecmp(DHCP_STRING, ip, strlen(DHCP_STRING)) == 0) {
		card_script = std::string("#!/bin/bash\n")
			+ IP " link set " INTERFACE_NAME " mtu " MTU_VALUE "\n"
			+ IP " link set dev " INTERFACE_NAME " address " + buf + "\n"
			+ IP " link set dev " INTERFACE_NAME " up\n"
			+ ECHO " \'" + nfs_server + "\' > /etc/nfs_server\n"
			+ ECHO " \'" + nfs_path + "\' > /etc/nfs_path\n"
			+ ECHO " " VCA_DHCP_IN_PROGRESS_TEXT
			+ " > /sys/class/vca/vca/state\n"
			+ "if [ ! -d /var/lib/dhclient ] ; then "
			+ MKDIR " -p /var/lib/dhclient ; fi\n"
			+ DHCLIENT + " " + INTERFACE_NAME + "\n"
			+ "if [ -s /var/lib/dhclient/dhclient.leases ]; then "
			+ ECHO " " VCA_DHCP_DONE_TEXT
			+ " > /sys/class/vca/vca/state; else "
			+ ECHO " " VCA_DHCP_ERROR_TEXT
			+ " > /sys/class/vca/vca/state; fi";
	}
	else if (is_ip_address(ip)) {
		card_script = std::string("#!/bin/bash\n")
			+ IP " link set " INTERFACE_NAME " mtu " MTU_VALUE "\n"
			+ IP " addr add " + ip + "/" + int_to_string(imask) + " dev " INTERFACE_NAME "\n"
			+ IP " link set dev " INTERFACE_NAME " address " + buf + "\n"
			+ IP " link set dev " INTERFACE_NAME " up\n"
			+ IP " route add " + gateway + " dev " INTERFACE_NAME "\n"
			+ IP " route add default via " + gateway + "\n"
			+ ECHO " \'" + nfs_server + "\' > /etc/nfs_server\n"
			+ ECHO " \'" + nfs_path + "\' > /etc/nfs_path";
	}
	else {
		d.LOG_CPU_ERROR("Wrong network configuration.");
		return false;
	}

	if (node_name) {
		card_script.append("\n" ECHO " \'");
		card_script.append(node_name);
		card_script.append("\' > /proc/sys/kernel/hostname \n");
	}

	net_cfg_path = std::string(VCASYSFSDIR) + "/vca"
		+ int_to_string(d.card_id) + int_to_string(d.cpu_id) + "/net_config";

	net_cfg_cmd = std::string(STDBUF " -o 1024 " ECHO " \"")
		+ card_script + "\" > " + net_cfg_path;

	if (run_cmd(net_cfg_cmd.c_str()) == FAIL) {
		LOG_ERROR("Cannot execute: %s\n", net_cfg_cmd.c_str());
		return false;
	}

	const char *va_free_mem_enable = get_cpu_field_if_not_empty(d, cpu_fields::va_free_mem);
	if (va_free_mem_enable) {
		std::string va_free_mem_bit = std::string(va_free_mem_enable);
		if (va_free_mem_bit == "1" && script_path) {

			sys_cfg_cmd = CAT " " + std::string(script_path) + " > " + TMP_BUFFER_FILE;
			if (run_cmd(sys_cfg_cmd.c_str()) == FAIL) {
				LOG_ERROR("Cannot execute: %s\n", sys_cfg_cmd.c_str());
				return false;
			}

			sys_cfg_cmd = ECHO " sysctl -w vm.min_free_kbytes="MIN_MEM_FREE_OF_CACHE_CARD_SIDE
				" >> " TMP_BUFFER_FILE;
			if (run_cmd(sys_cfg_cmd.c_str()) == FAIL) {
				LOG_ERROR("Cannot execute: %s\n", sys_cfg_cmd.c_str());
				return false;
			}

			sys_cfg_cmd = OVERWRITE " "  TMP_BUFFER_FILE " " VCASYSFSDIR "/vca" +
				int_to_string(d.card_id) + int_to_string(d.cpu_id) + "/sys_config "
				DOUBLE_AMPERSAND " " RM_FORCE " " TMP_BUFFER_FILE;
		}
		else if (va_free_mem_bit == "1") {
			sys_cfg_cmd = ECHO " sysctl -w vm.min_free_kbytes="MIN_MEM_FREE_OF_CACHE_CARD_SIDE
				" > " VCASYSFSDIR "/vca" + int_to_string(d.card_id) + int_to_string(d.cpu_id)
				+ "/sys_config";
		}
	}
	else {
		if (script_path) {
			sys_cfg_cmd = CAT " " + std::string(script_path) + " > " + VCASYSFSDIR + "/vca" +
				int_to_string(d.card_id) + int_to_string(d.cpu_id) + "/sys_config";
		}
		else {
			sys_cfg_cmd = ECHO " \"\" > " VCASYSFSDIR "/vca" +
				int_to_string(d.card_id) + int_to_string(d.cpu_id) + "/sys_config";
		}
	}

	if (run_cmd(sys_cfg_cmd.c_str()) == FAIL)
		LOG_ERROR("Cannot execute: %s\n", sys_cfg_cmd.c_str());

	return true;
}

const char * try_get_path(const caller_data & d, const data_field & data)
{
	const data_field * file_path_f = d.args(FILE_PATH_ARG);
	if (file_path_f)
		return file_path_f->get_cstring();
	else {
		const char * default_path = data.get_cstring();
		if(strcmp(default_path, "") == 0) {
			d.LOG_CPU_ERROR("%s not set! Please run: vcactl config <card_id> <cpu_id> %s <path_to_file>\n",
				data.get_name(), data.get_name());
			return NULL;
		}

		if (!strcmp(default_path, BLOCKIO_BOOT_DEV_NAME)) {
			return default_path;
		} else {
			if(file_exists(default_path))
				return default_path;
			else {
				d.LOG_CPU_ERROR("%s file %s does not exist!\n", data.get_name(),
						default_path);
				return NULL;
			}
		}
	}
}

std::string get_cpu_state(caller_data d)
{
	if (is_power_off(d) || is_link_up(d))
		return read_cpu_sysfs(d, "state");
	else
		return LINK_DOWN_STATE;
}

std::map<unsigned int, std::string> get_card_state(caller_data d)
{
	std::map<unsigned int, std::string> card;
	for (unsigned int i = 0; i < MAX_CPU; i++) {
		caller_data nodes = caller_data(d.card_id, (int)i, d.args);
		card[i] = get_cpu_state(nodes);
	}

	return card;
}

std::string get_cpu_os(caller_data d)
{
	if (is_link_up(d))
		return read_cpu_sysfs(d, "os_type");
	else
		return VCA_OS_TYPE_UNKNOWN_TEXT;
}

bool is_os_windows(std::string os)
{
	return (os == "windows");
}

void get_bios_flags_state(caller_data d, std::string &state)
{
	if (is_bios_up_or_ready(d))
		state = read_cpu_sysfs(d, "bios_flags");
}

void check_rcvy_jumper_state(caller_data d)
{
	std::string state;
	get_bios_flags_state(d, state);
	if (state == VCA_JUMPER_OPEN)
		d.LOG_CPU(VERBOSE_DEFAULT, "Warning: Jumper selects GOLD BIOS. Next reboot will boot it.\n");
}

static const char* update_status_with_bios_flags(caller_data d)
{
	std::string state;
	get_bios_flags_state(d, state);
	if (state == VCA_JUMPER_OPEN)
		return ", gold_bios";
	return "";
}

static const char* check_dma_state(caller_data d, std::string dma_hang_path)
{
	if (file_exists(dma_hang_path.c_str())) {
		std::string dma_state = read_cpu_sysfs(d, VCA_DMA_PATH);
		d.LOG_CPU(DEBUG_INFO, "Check dma_state: %s\n", dma_state.c_str());
		if (dma_state == VCA_DMA_HANG)
			return ", dma_hang";
	}
	return "";
}

static const char* update_power_button_status(caller_data d)
{
	d.LOG_CPU(DEBUG_INFO, "Check power button pressed\n");
	int state = check_power_button_state(d);
	if (state > 0) {
		return ", pwr_btn_press";
	} else if (state < 0) {
		return ", pwr_btn_unknown";
	}
	return "";
}

bool help(caller_data d)
{
	if (d.caller_id == 0) {
		print_help();
	}
	return true;
}

bool status(caller_data d)
{
	std::string state;
	state = get_cpu_state(d);
	std::string dma_hang_path = get_vca_sysfs_dir(d) + VCA_DMA_PATH;

	if(state.empty()) {
		d.LOG_CPU_ERROR("Could not read state!\n");
		return false;
	}
	else {
		d.LOG_CPU(DEBUG_INFO, "Check path DMA HANG: %s\n", dma_hang_path.c_str());
		printf("Card: %d Cpu: %d STATE: %s%s%s%s\n", d.card_id, d.cpu_id, state.c_str(),
			check_dma_state(d, dma_hang_path),
			update_status_with_bios_flags(d),
			update_power_button_status(d));
	}
	return true;
}

bool try_set_new_bios_params(filehandle_t cpu_fd, const caller_data & d)
{
	vca_csm_ioctl_bios_param_desc desc;
	desc.param = VCA_LBP_BIOS_PARAM_CPU_MAX_FREQ_NON_TURBO;
	unsigned int value = config.get_cpu_field(d.card_id, d.cpu_id,
		cpu_fields::cpu_max_freq_non_turbo).get_number();
	desc.value.value = value;

	d.LOG_CPU(DEBUG_INFO, "GETTING BIOS PARAM!\n");
	if(!csm_ioctl(cpu_fd, LBP_GET_BIOS_PARAM, &desc))
		return false;

	if(desc.ret == LBP_PROTOCOL_VERSION_MISMATCH)
		return false;

	if(!check_lbp_state_ok(desc.ret, LBP_GET_BIOS_PARAM, d))
		return false;

	d.LOG_CPU(DEBUG_INFO, "GOT BIOS PARAM OK!\n");

	if(value != desc.value.value) {
		d.LOG_CPU(DEBUG_INFO, "BIOS PARAM DIFFER FROM XML! GOT: %d XML: %d\n",
			desc.value, value);
		desc.value.value = value;
		d.LOG_CPU(DEBUG_INFO, "SETTING BIOS PARAM!\n");
		if(!csm_ioctl(cpu_fd, LBP_SET_BIOS_PARAM, &desc))
			return false;

		if(!check_lbp_state_ok(desc.ret, LBP_SET_BIOS_PARAM, d))
			return false;

		d.LOG_CPU(DEBUG_INFO, "BIOS PARAM SET OK!\n");
	} else
		return false;

	return true;
}

bool enable_blk_dev(caller_data d, unsigned int blk_dev_id)
{
	return config.save_blk_field(d.card_id, d.cpu_id, blk_dev_id, BLK_CONFIG_ENABLED_STR, "1");
}

bool disable_blk_dev(caller_data d, unsigned int blk_dev_id)
{
	return config.save_blk_field(d.card_id, d.cpu_id, blk_dev_id, BLK_CONFIG_ENABLED_STR, "0");
}

bool is_blk_enabled(int card_id, int cpu_id, unsigned int blk_dev_id)
{
	return !strcmp(config.get_blk_field(card_id, cpu_id, blk_dev_id, blk_dev_fields::enabled).get_cstring(), "1");
}

bool is_any_blkdev_enabled(caller_data d)
{
	config.get_vca_config_from_file();
	for (int i = 0; i < MAX_BLK_DEVS; i++) {
		if (is_blk_enabled(d.card_id, d.cpu_id, i)) {
			d.LOG_CPU_ERROR("VCA has an active blockio device in RW mode. You need to close it before config-default\n");
			return true;
		}
	}
	return false;
}

static bool is_blk_ramdisk_size_mb_ok(size_t size_mb)
{
	size_t max_size_t = std::numeric_limits<size_t>::max();
	size_t max_ramdisk_size_mb = B_TO_MB(max_size_t);
	return (size_mb > 0 && size_mb <= max_ramdisk_size_mb);
}

bool get_blk_config_field(caller_data d, unsigned int blk_dev_id, struct vca_blk_dev_info &blk_dev_info)
{
	std::string mode = config.get_blk_field(d.card_id, d.cpu_id, blk_dev_id, blk_dev_fields::mode).get_string();

	blk_dev_info.blk_dev_id = blk_dev_id;
	blk_dev_info.type = VCABLK_DISK_TYPE_FILE;
	blk_dev_info.size_mb = 0;
	blk_dev_info.mode = BLK_MODE_RW;
	blk_dev_info.path = config.get_blk_field(d.card_id, d.cpu_id, blk_dev_id, blk_dev_fields::path).get_string();

	if (mode == BLOCKIO_MODE_RAMDISK) {
		blk_dev_info.type = VCABLK_DISK_TYPE_MEMORY;
		blk_dev_info.size_mb = config.get_blk_field(d.card_id, d.cpu_id, blk_dev_id, blk_dev_fields::ramdisk_size_mb).get_number();

		if (!is_blk_ramdisk_size_mb_ok(blk_dev_info.size_mb)) {
			LOG_ERROR("Invalid block device ramdisk-size. Have to be more than 0.\n");
			return false;
		}
	}
	else if (mode == BLOCKIO_MODE_RO || mode == BLOCKIO_MODE_RW) {
		if (!file_exists(blk_dev_info.path.c_str())) {
			d.LOG_CPU_ERROR("File (%s) used from config to create block device %s not exist!\n",
				blk_dev_info.path.c_str(), get_block_dev_name_from_id(blk_dev_id).c_str());
			return false;
		}

		if (mode == BLOCKIO_MODE_RO) {
			blk_dev_info.mode = BLK_MODE_RO;
		}
	}
	else {
		d.LOG_CPU_ERROR("Incorrect block device mode (%s) in config file for block device %s!\n",
				mode.c_str(), get_block_dev_name_from_id(blk_dev_id).c_str());
		return false;
	}

	blk_dev_info.enabled = is_blk_enabled(d.card_id, d.cpu_id, blk_dev_id);

	return true;
}

bool check_blk_path(caller_data d, int cards_num);
bool open_enabled_blk_devs(caller_data d)
{
	close_on_exit blk_dev_fd = open_blk_fd(d.card_id, d.cpu_id);
	if (blk_dev_fd.fd == FAIL)
		return false;

	struct vca_blk_dev_info blk_dev_info;

	if (!check_blk_path(d, get_cards_num()))
		return false;

	for (int blk_dev_id = 0; blk_dev_id < MAX_BLK_DEVS; blk_dev_id++) {
		if (config.blk_dev_exist(d.card_id, d.cpu_id, blk_dev_id) && is_blk_enabled(d.card_id, d.cpu_id, blk_dev_id)) {
			if (check_blk_disk_exist(blk_dev_fd.fd, blk_dev_id)) {
				d.LOG_CPU(DEBUG_INFO, "Block device %s already exist.\n", get_block_dev_name_from_id(blk_dev_id).c_str());
			}
			else {
				if (!get_blk_config_field(d, blk_dev_id, blk_dev_info)) {
					d.LOG_CPU_ERROR("Cannot get blk_dev_info from config\n");
					return false;
				}

				if (open_blk_dev(blk_dev_fd.fd, blk_dev_info) != SUCCESS) {
					d.LOG_CPU_ERROR("Open block device %s failed!\n", get_block_dev_name_from_id(blk_dev_id).c_str());
					return false;
				}
			}
		}
	}
	return true;
}

bool is_network_manager_running()
{
	char buf[4] = "";
	// wc used to suppress nonzero exit code from pgrep
	run_cmd_with_output("pgrep NetworkManager | wc -l", buf, sizeof(buf));
	return buf[0] != '0';
}

static bool is_mv_bios_version_2_0(std::string current_bios_version, std::string bios_version_2_0)
{
	// based on BIOS versioning model since beginning of this project we are sure that comparing strings is enough
	return current_bios_version >= bios_version_2_0;
}

bool is_mv_bios_version_correct(caller_data d)
{
	close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));
	if (cpu_fd.fd == FAIL)
		return false;

	if (!is_mv_bios_version_2_0(get_bios_version(d, cpu_fd), BIOS_MV_2_0_VERSION))
		return false;

	return true;
}

bool boot(caller_data d)
{
	using namespace boost::interprocess;

	d.LOG_CPU(DEBUG_INFO, "Running \"boot\" function!\n");

	if (!validate_node_range(d)) {
		d.LOG_CPU_ERROR("Invalid caller data\n");
		return false;
	}

	if (!d.args.is_force_cmd_enabled() && is_network_manager_running()) {
		d.LOG_CPU_ERROR("You're trying to boot while NetworkManager is running.\n"
						"You should stop it before ( systemctl stop NetworkManager )\n"
						"or use --force flag to ignore this warning.\n");
		return false;
	}

	std::string node_lock_file_name = get_node_lock_file_name(d);

	if (!file_create(node_lock_file_name.c_str(), O_WRONLY))
		return false;

	file_lock f_lock(node_lock_file_name.c_str());

	// this scoped_lock ensures that unlock() method will be executed
	// on every exit point from this function
	scoped_lock<file_lock> e_lock(f_lock, try_to_lock_type());

	if (!f_lock.try_lock()) {
		d.LOG_CPU_ERROR("Boot command already running! "
			"Cannot execute it twice at the same time.\n");
		return false;
	}

	const char *relative_path = NULL;

	const data_field * param_name = d.args(FORCE_GET_LAST_OS_IMAGE);

	if (param_name){
		relative_path = config.get_cpu_field(d.card_id,
			d.cpu_id, cpu_fields::last_img_path).get_cstring();
		d.LOG_CPU(DEBUG_INFO, "Booting from last-os-image field.\n");
	}
	else{
		relative_path = try_get_path(d,
			config.get_cpu_field(d.card_id, d.cpu_id, cpu_fields::img_path));
	}

	if (!relative_path)
		return false;

	char boot_path[PATH_MAX + 1];
	file_manager::managed_file *file_lbp = NULL;

	if (!strcmp(relative_path, BLOCKIO_BOOT_DEV_NAME)) {
		d.LOG_CPU(DEBUG_INFO, "BOOT BLOCK DEVICE!\n");
		STRCPY_S(boot_path, relative_path, sizeof(boot_path));
	} else {
		if (!realpath(relative_path, boot_path)) {
			d.LOG_CPU_ERROR("Cannot canonicalize OS image path (got %s): %s!\n",
					relative_path, strerror(errno));
			return false;
		}

		d.LOG_CPU(DEBUG_INFO, "LOADING FILE INTO RAM!\n");
		file_lbp = file_mgr.get_file_content(boot_path);
		if (!file_lbp) {
			d.LOG_CPU_ERROR("Could not open file: %s\n", boot_path);
			return false;
		}
	}

	close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));
	if (cpu_fd.fd == FAIL)
		return false;

	char mac_addr[6];
	enum vca_card_type type = get_card_type(d.card_id);
	if (type & VCA_PRODUCTION) {
		std::string state;
		state = get_cpu_state(d);
		if (state == get_vca_state_string(VCA_NET_DEV_READY)) {
			d.LOG_CPU(VERBOSE_DEFAULT, "OS ALREADY UP!\n");
			return false;
		}

		if (state != get_vca_state_string(VCA_BIOS_UP) && state != get_vca_state_string(VCA_BIOS_READY) && state != get_vca_state_string(VCA_AFTER_REBOOT)) {
			d.LOG_CPU_ERROR("Card needs to be in \"%s\", \"%s\" or \"%s\" state!\n",
				get_vca_state_string(VCA_BIOS_UP), get_vca_state_string(VCA_BIOS_READY), get_vca_state_string(VCA_AFTER_REBOOT));
			return false;
		}

		if (!open_enabled_blk_devs(d)) {
			d.LOG_CPU_ERROR("Cannot open block devices.\n");
			return false;
		}

		if (!file_lbp) {
			close_on_exit blk_dev_fd = open_blk_fd(d.card_id, d.cpu_id);
			if (blk_dev_fd.fd == FAIL)
				return false;

			if (!check_blk_disk_exist(blk_dev_fd.fd, 0)) {
				d.LOG_CPU_ERROR("Block device vcablk0 need be in inactive/active state!\n");
				return false;
			}
		}

		/* Make sure vop device is removed. It is possible that virtio device is removed
		 * while old vop device is still registered. This happens when card reset
		 * was triggered by the card rather than via vcactl */
		if (!stop_csm(cpu_fd, d)) {
			d.LOG_CPU_ERROR("Could not stop csm!\n");
			return false;
		}

		if (!prepare_cpu(cpu_fd, d)) {
			d.LOG_CPU_ERROR("Could not prepare cpu!\n");
			return false;
		}

		if (try_set_new_bios_params(cpu_fd, d)) {
			reset(d);
			d.LOG_CPU(DEBUG_INFO, "WAITING FOR BIOS UP!\n");
			if (!wait_bios(d))
				return false;
			d.LOG_CPU(DEBUG_INFO, "PREPARING CPU ONCE AGAIN WITH NEW BIOS SETTING!\n");
			if (!prepare_cpu(cpu_fd, d))
				return false;
		}

		if (!get_mac_addr(cpu_fd, d, mac_addr)) {
			d.LOG_CPU_ERROR("Could not get mac address!\n");
			return false;
		}

		if (type == VCA_MV && !is_mv_bios_version_correct(d)) {
			if (d.args.is_force_cmd_enabled()) {
				d.LOG_CPU_WARN("OS booted with wrong BIOS version. You are doing it on your own risk!\n");
			}
			else {
				d.LOG_CPU_ERROR("Booting aborted! Invalid VCA2 BIOS version."
					" Please update to at least \"" BIOS_MV_2_0_VERSION "\" version or newer one.\n");
				d.LOG_CPU(VERBOSE_DEFAULT, "Use '--force' execution mode if you are sure what you try to do.\n");
				return false;
			}
		}

		if (!set_time(cpu_fd, d)){
			d.LOG_CPU_ERROR("Could not set time!\n");
			return false;
		}

		bool boot_res = false;
		if (!file_lbp) {
			d.LOG_CPU(DEBUG_INFO, "TRYING TO BOOT VCABLK0!\n");
			boot_res = try_ioctl_with_blk(cpu_fd, LBP_BOOT_BLKDISK, d);
		} else {
			d.LOG_CPU(DEBUG_INFO, "TRYING TO BOOT LBP!\n");
			boot_res = try_ioctl_with_img(cpu_fd, LBP_BOOT_RAMDISK, d, (void*)file_lbp->content, file_lbp->size);
		}

		if (boot_res) {
			d.LOG_CPU(DEBUG_INFO, "EFI OS LOADER COMPLETED!\n");
			start_csm(cpu_fd, d);
		}
		else {
			d.LOG_CPU_ERROR("Boot failed!\n");
			return false;
		}
	} else {
		start_csm(cpu_fd, d);
	}

	if (pass_script(d, mac_addr))
		return config.save_cpu_field(d.card_id, d.cpu_id, "last-os-image", boot_path);

	return false;
}

bool boot_USB(caller_data d)
{
	close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));
	if (cpu_fd.fd == FAIL)
		return false;
	config.save_cpu_field(d.card_id, d.cpu_id, "last-os-image", "");
	d.LOG_CPU(DEBUG_INFO, "Trying boot from USB!\n");
	enum vca_card_type type = get_card_type(d.card_id);
	char mac_addr[6];
	if(type & VCA_PRODUCTION) {

		std::string state;
		state = get_cpu_state(d);
		if(state == get_vca_state_string(VCA_NET_DEV_READY)) {
			d.LOG_CPU(VERBOSE_DEFAULT, "OS ALREADY UP!\n");
			return false;
		}

		if (!open_enabled_blk_devs(d)) {
			d.LOG_CPU_ERROR("Cannot open block devices.\n");
			return false;
		}

		/* Make sure vop device is removed. It is possible that virtio device is removed
		 * while old vop device is still registered. This happens when card reset
		 * was triggered by the card rather than via vcactl */
		stop_csm(cpu_fd, d);

		if (!prepare_cpu(cpu_fd, d))
			return false;

		if (!get_mac_addr(cpu_fd, d, mac_addr))
			return false;

		if (!set_time(cpu_fd, d))
			return false;

		vca_lbp_retval ret;
		if(!csm_ioctl(cpu_fd, LBP_BOOT_FROM_USB, &ret))
			return false;

		if(!check_lbp_state_ok(ret, LBP_BOOT_FROM_USB, d))
			return false;
	}
	start_csm(cpu_fd, d);
	return pass_script(d, mac_addr);
}

bool pwrbtn_boot(caller_data d)
{
	const char *img_path = config.get_cpu_field(d.card_id, d.cpu_id,
				cpu_fields::img_path).get_cstring();

	if (!config.get_global_field(global_fields::auto_boot).get_number()) {
		d.LOG_CPU(DEBUG_INFO, "No autoboot after power button. Autoboot disabled.\n");
		return true;
	}

	if (strcmp(img_path, "") == 0) {
		d.LOG_CPU(DEBUG_INFO, "No autoboot after power button. OS image shall be configured.\n");
		return true;
	}

	char actual_path[PATH_MAX + 1];

	if (strcmp(img_path, BLOCKIO_BOOT_DEV_NAME) != 0) {
		if (!realpath(img_path, actual_path))
			d.LOG_CPU_ERROR("Cannot canonicalize OS image path (got %s): %s!\n",
				img_path, strerror(errno));

		if (!file_exists(actual_path)) {
			d.LOG_CPU_ERROR("file %s does not exist!\n", actual_path);
			return true;
		}
	}
	else
		STRCPY_S(actual_path, img_path, sizeof(actual_path));

	std::string pwrbtn_boot_cmd = ECHO " " VCA_PWRBTN_BOOT_CMD " "
		+ int_to_string(d.card_id) + " "
		+ int_to_string(d.cpu_id) + " "
		+ actual_path + " > "
		MSG_FIFO_FILE;

	if (run_cmd(pwrbtn_boot_cmd.c_str()) == FAIL) {
		LOG_ERROR("Cannot execute: %s\n", pwrbtn_boot_cmd.c_str());
		return false;
	}

	return true;
}

bool boot_gold_bios(caller_data d)
{
	close_on_exit fd(open_card_fd(d.card_id));
	if (fd.fd == FAIL)
		return false;

	struct vca_ioctl_desc desc;
	desc.cpu_id = d.cpu_id;
	desc.hold = false;
	if (!vca_ioctl(fd, VCA_ENABLE_GOLD_BIOS, &desc)) {
		d.LOG_CPU_ERROR("Could not initiate BIOS update\n");
		return false;
	}
	//powerdown node to boot it back in golden bios
	if (!hold_power_button(d)) {
		d.LOG_CPU_ERROR("Could not toggle power button\n");
		return false;
	}
#ifndef LINK_DOWN_WORKAROUND
	bool link_up = true;
	while (get_passed_time_ms(start) < NODE_POWERDOWN_TIMEOUT) {
		if (!is_link_up(d)) {
			link_up = false;
			break;
		}
		sleep(1);
	}
	if (link_up) {
		d.LOG_CPU_ERROR("Node powerdown error!\n");
		return false;
	}
#endif
	//boot up node, now with gold bios
	if (!_toggle_power_button(d)) {
		d.LOG_CPU_ERROR("Could not toggle power button\n");
		return false;
	}
	return true;
}

bool setup_standard_bios_update(caller_data d)
{
	std::string state;
	state = get_cpu_state(d);
	if (state != get_vca_state_string(VCA_BIOS_UP) && state != get_vca_state_string(VCA_BIOS_READY)) {
		d.LOG_CPU_ERROR("Card needs to be in \"%s\" or \"%s\" state!\n",
			get_vca_state_string(VCA_BIOS_UP), get_vca_state_string(VCA_BIOS_READY));
	return false;
	}

	check_rcvy_jumper_state(d);

	close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));
	if (cpu_fd.fd == FAIL)
		return false;
	if (!prepare_cpu(cpu_fd, d)) {
		d.LOG_CPU_ERROR("Could not prepare cpu!\n");
		return false;
	}
	return true;
}

bool restore_cpu_after_bios_update(caller_data d)
{
	close_on_exit fd(open_card_fd(d.card_id));
	if (fd.fd == FAIL)
		return false;
	struct vca_ioctl_desc desc;
	desc.cpu_id = d.cpu_id;
	desc.hold = false;
	if (!vca_ioctl(fd, VCA_DISABLE_GOLD_BIOS, &desc)) {
		d.LOG_CPU_ERROR("Could not disable gold bios\n");
		return false;
	}
	if (!hold_power_button(d))
		return false;
	if (!_toggle_power_button(d))
		return false;
	return true;
}

bool setup_bios_recovery(caller_data d)
{
	if (!(boot_gold_bios(d)))
		return false;

	close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));
	if (cpu_fd.fd == FAIL)
		return false;

	bool state = false;
	boost::posix_time::ptime start = get_time();
	do {
		vca_lbp_retval ret;
		if (!csm_ioctl(cpu_fd, LBP_HANDSHAKE, &ret))
			return false;
		if (ret != LBP_STATE_OK) {
			state = false;
		} else {
			state = true;
		}
		if (get_passed_time_ms(start) > GOLD_BIOS_LOAD_TIMEOUT) {
			d.LOG_CPU_ERROR("GOLD BIOS load timeout!\n");
			restore_cpu_after_bios_update(d);
		}
	} while (!state);
	return true;
}

bool get_bios_cfg_for(caller_data &d, filehandle_t fd, vca_lbp_bios_param param, unsigned long long &result)
{
	// TODO: refactor state checking to common function for both {get,set}_bios_cfg
	// TODO: use this function also for get_bios_cfg

	vca_csm_ioctl_bios_param_desc desc;
	desc.param = param;

	if (!csm_ioctl(fd, LBP_GET_BIOS_PARAM, &desc)) {
		return false;
	}
	if (!check_lbp_state_ok(desc.ret, LBP_GET_BIOS_PARAM, d)) {
		return false;
	}
	result = desc.value.value;
	return true;
}

bool set_bios_cfg_for(caller_data &d, filehandle_t fd, vca_csm_ioctl_bios_param_desc desc)
{
	return csm_ioctl(fd, LBP_SET_BIOS_PARAM, &desc) && check_lbp_state_ok(desc.ret, LBP_SET_BIOS_PARAM, d);
}

bool set_bios_cfg_for(caller_data &d, filehandle_t fd, vca_lbp_bios_param param, unsigned long long value)
{
	vca_csm_ioctl_bios_param_desc desc;
	desc.param = param;
	desc.value.value = value;
	return set_bios_cfg_for(d, fd, desc);
}

bool update_bios(caller_data d)
{
	std::string cmd_name = d.get_cmd_name();

	enum vca_card_type type = get_card_type(d.card_id);
	if (type != VCA_MV && cmd_name == RECOVER_BIOS_CMD) {
		LOG_INFO("Recovery BIOS update is supported only on second generation VCA\n");
		return false;
	}

	if (type & VCA_PRODUCTION && (cmd_name != RECOVER_BIOS_CMD)) {
		if (!setup_standard_bios_update(d))
			return false;
	}

	std::string file_path = "";

	const data_field *file_path_f = d.args(FILE_PATH_ARG);
	if (file_path_f)
		file_path = file_path_f->get_cstring();
	else
		return false;

	file_manager::managed_file *file = file_mgr.get_file_content(file_path.c_str());
	if (!file) {
		d.LOG_CPU_ERROR("Could not open file: %s\n", file_path.c_str());
		return false;
	}

	if (cmd_name == RECOVER_BIOS_CMD) {
		if(!setup_bios_recovery(d))
			return false;
	}

	d.LOG_CPU(VERBOSE_DEFAULT, "BIOS UPDATE STARTED. DO NOT POWERDOWN SYSTEM\n");
	fflush(stdout);

	close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));
	if (cpu_fd.fd == FAIL)
		return false;

	unsigned long long sgx_value;
	if (is_mv_bios_version_2_0(get_bios_version(d, cpu_fd), BIOS_MV_2_0_VERSION_SGX) && get_bios_cfg_for(d, cpu_fd, VCA_LBP_PARAM_SGX, sgx_value) && sgx_value) {
		if (!set_bios_cfg_for(d, cpu_fd, VCA_LBP_PARAM_SGX, SGX_DISABLED))
			d.LOG_CPU_WARN("Attempt to disable SGX in BIOS configuration failed. Update will continue anyway.\n");
		else
			d.LOG_CPU(DEBUG_INFO, "SGX was disabled. BIOS UPDATE CONTINUED\n");
	}

	if (type & VCA_PRODUCTION) {
		/* reset needed until bios will support unmapping file! */
		if (try_ioctl_with_img(cpu_fd, LBP_FLASH_BIOS, d, (void*)file->content, file->size)) {
			d.LOG_CPU(VERBOSE_DEFAULT, "UPDATE BIOS SUCCESSFUL\n");
			if ((cmd_name == RECOVER_BIOS_CMD)) {
				if (!restore_cpu_after_bios_update(d))
					return false;
			}
			else {
				if (type == VCA_MV) {
					press_power_button(d, true);
					sleep(3); // waiting 3 seconds to make sure that CPU power go down
					press_power_button(d, false);
					d.LOG_CPU(VERBOSE_DEFAULT, "Node will be power down and up automatically to make the change active.\n"
						"Please wait for 'bios_up' to start work with the node.\n");
				}
				else
					reset(d);
			}
		}
		else {
			d.LOG_CPU_ERROR("Update bios failed!\n");
			return false;
		}
	} else {
		d.LOG_CPU_ERROR("This type of card does not support update_bios!\n");
		return false;
	}
	return true;
}

bool gold_control(caller_data d)
{
	enum vca_card_type type = get_card_type(d.card_id);
	if (type != VCA_MV) {
		LOG_INFO("Golden BIOS controls available only on second generation VCA\n");
		return false;
	}

	close_on_exit fd(open_card_fd(d.card_id));
	if (fd.fd == FAIL)
		return false;
	struct vca_ioctl_desc desc;
	desc.cpu_id = d.cpu_id;
	desc.hold = false;

	const std::string subcmd = d.args.get_arg(SUBCMD);
	if (subcmd == "on") {
		if (!vca_ioctl(fd, VCA_ENABLE_GOLD_BIOS, &desc)) {
			d.LOG_CPU_ERROR("Could not enable golden bios\n");
			return false;
		}
	} else if (subcmd == "off") {
		if (!vca_ioctl(fd, VCA_DISABLE_GOLD_BIOS, &desc)) {
			d.LOG_CPU_ERROR("Could not disable golden bios\n");
			return false;
		}
	} else {
		LOG_ERROR("unknown subcommand %s\n", subcmd.c_str());
		return false;
	}

	return true;
}

bool update_img_file_with_mac(const char * mac)
{
	FILE *fp = fopen(VCA_UPDATE_MAC_SET_MAC_PATH, "w");
	if (!fp) {
		LOG_ERROR("could not create/write to %s\n", VCA_UPDATE_MAC_SET_MAC_PATH);
		return false;
	}

	std::string file_content = std::string("set EEMAC ") + mac[0] + mac[1] +
		mac[3] + mac[4] + mac[6] + mac[7] + mac[9] + mac[10] + mac[12] + mac[13] +
		mac[15] + mac[16];

	fprintf(fp, "%s", file_content.c_str());
	fclose(fp);

	if (!file_exists(VCA_UPDATE_MAC_IMG_PATH)) {
		LOG_ERROR("File %s does not exist!\n", VCA_UPDATE_MAC_IMG_PATH);
		return false;
	}

	/* Added option '-D o' (Overwrites primary names by default) instead of using 'mdel'.
	 * 'mdel' tried to delete not existing file which caused (expected) error */
	std::string copy_cmd(std::string("mcopy -D o -i") + VCA_UPDATE_MAC_IMG_PATH
			     + " " + VCA_UPDATE_MAC_SET_MAC_PATH + " ::EFI/BOOT/setmacev.nsh");

	if (run_cmd(copy_cmd.c_str()) == FAIL) {
		LOG_ERROR("Cannot execute: %s\n", copy_cmd.c_str());
		return false;
	}

	return true;
}

bool update_mac_addr(caller_data d)
{
	if (!is_root()) {
		LOG_ERROR("Need to be root to perform this command!\n");
		return false;
	}

	close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));
	if (cpu_fd.fd == FAIL)
		return false;

	enum vca_card_type type = get_card_type(d.card_id);
	if(type & VCA_PRODUCTION) {

		std::string state;
		state = get_cpu_state(d);
		if(state != get_vca_state_string(VCA_BIOS_UP) && state != get_vca_state_string(VCA_BIOS_READY)) {
			d.LOG_CPU_ERROR("Card needs to be in \"%s\" or \"%s\" state!\n",
				get_vca_state_string(VCA_BIOS_UP), get_vca_state_string(VCA_BIOS_READY));
			return false;
		}

		d.LOG_CPU(DEBUG_INFO, "Updating img file with mac!\n");
		const data_field * mac_addr_f = d.args(MAC_ADDR_ARG);
		if (!mac_addr_f)
			return false;

		if (!update_img_file_with_mac(mac_addr_f->get_cstring()))
			return false;

		if (!prepare_cpu(cpu_fd, d)) {
			d.LOG_CPU_ERROR("Could not prepare cpu!\n");
			return false;
		}

		close_on_exit fd(open_path(VCA_UPDATE_MAC_IMG_PATH));
		if (fd.fd == FAIL)
			return false;

		size_t f_sz = get_file_size(VCA_UPDATE_MAC_IMG_PATH);
		if (f_sz > IMG_MAX_SIZE)
			return false;

		char *buffer = new char[f_sz];
		if (buffer == NULL)
			return false;

		if (read(fd, buffer, f_sz) < 0) {
			d.LOG_CPU_ERROR("could not read from %s\n", VCA_UPDATE_MAC_IMG_PATH);
			delete[] buffer;
			return false;
		}

		/* reset needed until bios will support unmapping file! */
		if(try_ioctl_with_img(cpu_fd, LBP_UPDATE_MAC_ADDR, d, (void*)buffer, f_sz)) {
			d.LOG_CPU(DEBUG_INFO, "UPDATE MAC SUCCESSFUL!\n");
			reset(d);
		}
		else {
			d.LOG_CPU_ERROR("Update mac failed!\n");
			delete[] buffer;
			return false;
		}
	} else {
		d.LOG_CPU_ERROR("This type of card does not support update_mac!\n");
		return false;
	}
	return true;
}

static std::string crc_pair_str(vca::crc_pair crcs)
{
	std::stringstream message;
	message << "CRC1: 0x";
	message << std::setfill('0') << std::setw(8);
	message << std::hex << crcs.first;
	message << " CRC2: 0x";
	message << std::setfill('0') << std::setw(8);
	message << std::hex << crcs.second;
	return message.str();
}


bool is_eeprom_compatible(vca::EepromEntry entry, unsigned card_id)
{
	enum vca_card_type type = get_card_type(card_id);
	return entry.card_gen == type;
}

uint32_t read_eeprom_crc(std::vector<char> const& eeprom)
{
	unsigned size = eeprom.size();
	if (size < 4 )
		throw std::bad_cast();

	uint32_t crc = (eeprom[size-4])     & 0x000000ff;
	crc = crc + ((eeprom[size-3] << 8)  & 0x0000ffff);
	crc = crc + ((eeprom[size-2] << 16) & 0x00ffffff);
	crc = crc + (eeprom[size-1] << 24);
	return crc;
}


bool check_if_eeproms_crc_is_allowed(const vca::crc_pair crcs)
{
	std::vector<vca::crc_pair> valid_eeprom_pairs;

	for (unsigned i = 0; i < vca::num_known_eeproms; ++i)
		valid_eeprom_pairs.push_back(vca::known_eeproms[i].crcs);

	return std::find(valid_eeprom_pairs.begin(),
			 valid_eeprom_pairs.end(),
			 crcs)
		   != valid_eeprom_pairs.end();
}

bool check_if_card_state_is_ok_to_update_eeprom(caller_data d)
{
	std::map<unsigned int, std::string> card = get_card_state(d);
	std::string state;
	for (unsigned int i = 0; i < card.size(); i++) {
		state = card[i];
		LOG_DEBUG("state %s for node %d\n", state.c_str(), i);
		if (state == LINK_DOWN_STATE || state == get_vca_state_string(VCA_BIOS_DOWN) ||
		    state == get_vca_state_string(VCA_BOOTING) || state == get_vca_state_string(VCA_FLASHING) ||
		    state == get_vca_state_string(VCA_RESETTING) || state == get_vca_state_string(VCA_BUSY) ||
		    state == get_vca_state_string(VCA_DONE) || state == get_vca_state_string(VCA_ERROR) ||
		    state == get_vca_state_string(VCA_DRV_PROBE_ERROR)) {
			return false;
		}
	}
	return true;
}

bool update_eeprom(caller_data d)
{
	vca::eeprom_binaries eeprom_bin;
	struct vca::EepromEntry valid_entry;

	if (!is_root()) {
		LOG_ERROR("Need to be root to perform this command!\n");
		return false;
	}

	if (!check_if_card_state_is_ok_to_update_eeprom(d)) {
		if (!d.args.is_force_cmd_enabled()) {
			LOG_ERROR("Cannot update EEPROM for card %d. Wrong state of one of the nodes!\n", d.card_id);
			LOG_INFO("Use '--force' execution mode if you are sure what you try to do.\n");
			return false;
		}
	}

	try {
		const data_field *file_path = d.args(FILE_PATH_ARG);
		vca::EepromFile file(file_path->get_cstring());
		if (!file.is_open()) {
			return false;
		}

		std::vector<vca::EepromEntry> valid_entries = file.get_entries(is_eeprom_compatible, d.card_id);
		if (valid_entries.size() != 1) {
			LOG_INFO("Found %lu matching eeprom binary entries for card %d in input file, but exactly 1 is required.\n",
				(unsigned long) valid_entries.size(), d.card_id);
			return false;
		}
		valid_entry = valid_entries[0];
		eeprom_bin = file.get_binary_eeproms(valid_entry);
	} catch (std::ifstream::failure &e) {
		LOG_ERROR("%s\n", e.what());
		return false;
	} catch (std::bad_cast &e) {
		LOG_ERROR("%s\n", e.what());
		return false;
	} catch (std::runtime_error &e) {
		LOG_ERROR("%s\n", e.what());
		return false;
	} catch ( ... ) {
		LOG_ERROR("EEPROM update aborted!\n");
		return false;
	}

	vca::crc_pair crcs = std::make_pair(
				read_eeprom_crc(eeprom_bin.first),
				read_eeprom_crc(eeprom_bin.second));
	if (!check_if_eeproms_crc_is_allowed(crcs)) {
		std::stringstream message;
		message << "EEPROM pair is unknown: " << crc_pair_str(crcs) << ".";
		if (!d.args.is_force_cmd_enabled()) {
			message << " Update aborted.\n";
			LOG_ERROR("%s", message.str().c_str());
			return false;
		} else {
			message << std::endl;
			LOG_WARN("%s", message.str().c_str());
		}
	}

	LOG_INFO("Update EEPROM process started (for card %d). "
		 "Do not power down system!\n", d.card_id);

	filehandle_t card_fd, card_extd_fd;
	card_fd = open_card_fd(d.card_id);
	close_on_exit fd(card_fd);
	if (!fd) {
		return false;
	}

	card_extd_fd = open_extd_card_fd(d.card_id);
	close_on_exit fd_extd(card_extd_fd);
	if (!fd_extd) {
		return false;
	}

	if (vca_plx_eep_ioctl_with_bin(fd, &eeprom_bin.first.front(),
				       valid_entry.first_eeprom_size) == SUCCESS) {
		LOG_INFO("Update EEPROM for first PCIe switch successful!\n");
	}
	else {
		LOG_ERROR("Update EEPROM for first PCIe switch failed!\n");
		return false;
	}

	if (vca_plx_eep_ioctl_with_bin_mgr_extd(fd_extd, &eeprom_bin.second.front(),
						valid_entry.second_eeprom_size) == SUCCESS) {
		LOG_INFO("Update EEPROM for second PCIe switch successful!\n");
	}
	else {
		LOG_ERROR("Update EEPROM for second PCIe switch failed!\n");
		return false;
	}

	LOG_INFO("Update EEPROM successful (for card %d). "
		"Reboot system is required to reload EEPROM.\n", d.card_id);

	return true;
}

bool clear_smb_event_log(caller_data d)
{
	close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));
	if (cpu_fd.fd == FAIL)
		return false;

	enum vca_card_type type = get_card_type(d.card_id);
	if (type & VCA_PRODUCTION) {

		std::string state;
		state = get_cpu_state(d);
		if (state != get_vca_state_string(VCA_BIOS_UP) && state != get_vca_state_string(VCA_BIOS_READY)) {
			d.LOG_CPU_ERROR("Card needs to be in \"%s\" or \"%s\" state!\n",
				get_vca_state_string(VCA_BIOS_UP), get_vca_state_string(VCA_BIOS_READY));
			return false;
		}

		close_on_exit fd(open_path(VCA_CLEAR_SMB_EVENT_LOG_PATH));
		if (fd.fd == FAIL)
			return false;

		size_t f_sz = get_file_size(VCA_CLEAR_SMB_EVENT_LOG_PATH);
		if (f_sz > IMG_MAX_SIZE)
			return false;

		char *buffer = new char[f_sz];
		if (buffer == NULL)
			return false;

		if (read(fd, buffer, f_sz) < 0) {
			d.LOG_CPU_ERROR("could not read from %s\n", VCA_CLEAR_SMB_EVENT_LOG_PATH);
			delete[] buffer;
			return false;
		}

		if (!prepare_cpu(cpu_fd, d)) {
			d.LOG_CPU(DEBUG_INFO, "Preparing card failed!\n");
			delete[] buffer;
			return false;
		}

		/* reset needed until bios will support unmapping file! */
		if (try_ioctl_with_img(cpu_fd, LBP_CLEAR_SMB_EVENT_LOG, d, (void*)buffer, f_sz)) {
			d.LOG_CPU(DEBUG_INFO, "Clear SMB event log successful!\n");
			reset(d);
		}
		else {
			d.LOG_CPU_ERROR("Clear SMB event log failed!\n");
			delete[] buffer;
			return false;
		}
	}
	else {
		d.LOG_CPU_ERROR("This type of card does not support clear-SMB-event-log!\n");
		return false;
	}
	return true;
}

bool script(caller_data d)
{
	const data_field * file_path = d.args(FILE_PATH_ARG);
	bool ret;

	if (file_path) {
		ret = config.save_cpu_field(d.card_id, d.cpu_id,
			config.get_cpu_field(d.card_id, d.cpu_id,
			cpu_fields::script_path).get_name(),
			file_path->get_cstring());
	}
	else {
		ret = config.save_cpu_field(d.card_id, d.cpu_id,
			config.get_cpu_field(d.card_id, d.cpu_id,
			cpu_fields::script_path).get_name(),
			"");
	}

	return ret;
}

void print_padding(int padding_level)
{
	while (padding_level) {
		printf(" ");
		padding_level--;
	}
}

void print_field(const data_field & data, int spaces) {
	print_padding(spaces);
	if(data.get_type() == data_field::number)
		printf("%s: %d\n", data.get_name(), data.get_number());
	else
		printf("%s: %s\n", data.get_name(), data.get_cstring());
}

void print_blk_config_fields(caller_data d, int padding)
{
	for (int j = 0; j < MAX_BLK_DEVS; j++) {
		if (config.blk_dev_exist(d.card_id, d.cpu_id, j)) {
			print_padding(padding);
			printf("%s%d:\n", BLK_CONFIG_NAME, j);
			for (int k = 0; k < blk_dev_fields::ENUM_SIZE; k++) {
				const data_field blk_data = config.get_blk_field(d.card_id, d.cpu_id, j, (blk_dev_fields::_enum)k);
				print_field(blk_data, padding + 2);
			}
		}
	}
}

bool config_show(caller_data d)
{
	const data_field * param_name = d.args(CONFIG_PARAM_NAME);
	if (param_name) {
		if (d.caller_id == 0) {
			if (config.is_field_global(param_name->get_cstring())) {
				const data_field * const field = config.get_global_field(param_name->get_cstring());
				if (!field)
					return false;
				printf("%s\n", field->get_cstring());
			}
		}
		if (config.is_field_cpu(param_name->get_cstring())) {
			const data_field * const field = config.get_cpu_field(d.card_id, d.cpu_id, param_name->get_cstring());
			if (!field)
				return false;

			if (strcmp(param_name->get_cstring(), config.get_cpu_field(d.card_id, d.cpu_id, cpu_fields::block_devs).get_name()) == 0) {
				printf("card %d cpu %d: \n", d.card_id, d.cpu_id);
				print_blk_config_fields(d, 2);
				return true;
			}

			const data_field * strip_param = d.args(STRIP_PARAM);
			if (!strip_param) {
				printf("card %d cpu %d: \n", d.card_id, d.cpu_id);
				print_field(*field, 2);
			}
			else
				printf("%s\n", field->get_cstring());

		}
		// for vcablkN fields
		else {
			for (int j = 0; j < MAX_BLK_DEVS; j++) {
				if (config.blk_dev_exist(d.card_id, d.cpu_id, j)) {
					if (strcmp(param_name->get_cstring(), get_block_dev_name_from_id(j).c_str()) == 0) {
						printf("card %d cpu %d: \n", d.card_id, d.cpu_id);
						printf("  %s:\n", get_block_dev_name_from_id(j).c_str());
						for (int k = 0; k < blk_dev_fields::ENUM_SIZE; k++) {
							const data_field blk_data = config.get_blk_field(d.card_id, d.cpu_id, j, (blk_dev_fields::_enum)k);
							print_field(blk_data, 4);
						}
					}
				}
			}
		}
	}
	else {
		if(d.caller_id == 0) {
			printf("Global configuration:\n");
			for(int i = 0; i < global_fields::ENUM_SIZE; i++)
				print_field(config.get_global_field((global_fields::_enum)i), 2);
		}

		printf("card %d cpu %d: \n", d.card_id, d.cpu_id);
		for (int i = 0; i < cpu_fields::ENUM_SIZE; i++) {
			const data_field cpu_data = config.get_cpu_field(d.card_id, d.cpu_id, (cpu_fields::_enum)i);
			if (cpu_data.get_name() != config.get_cpu_field(d.card_id, d.cpu_id, cpu_fields::block_devs).get_name()) {
				print_field(cpu_data, 2);
			}
			else {
				printf("  %s:\n", cpu_data.get_name());
				print_blk_config_fields(d, 4);
			}
		}
	}
	return true;
}

bool config_change(caller_data d)
{
	const data_field * param_name = d.args(CONFIG_PARAM_NAME);
	const data_field * param_val = d.args(CONFIG_PARAM_VALUE);
	const data_field * blk_id_f = d.args(BLOCKIO_ID_ARG);
	const data_field * card_id_f = d.args(CARD_ID_ARG);
	const data_field * cpu_id_f = d.args(CPU_ID_ARG);
	bool ret;

	if (d.caller_id == 0) {
		if (!config.contains_field(param_name->get_cstring())) {
			LOG_ERROR("invalid parameter name: %s\n", param_name->get_cstring());
			return false;
		}
		if (config.is_field_global(param_name->get_cstring())) {

			ret = config.save_global_field(param_name->get_cstring(),
							param_val->get_cstring());

			return ret;
		}
	}
	if (strcmp(param_name->get_cstring(), config.get_cpu_field(d.card_id, d.cpu_id,
		cpu_fields::ip).get_name()) == 0) {
		if (strcmp(param_val->get_cstring(), DHCP_STRING)) {
			if (!card_id_f || !cpu_id_f) {
				LOG_ERROR("card_id and cpu_id is required to change config field \"ip\"!\n");
				return false;
			}
			if (!is_ip_address(param_val->get_cstring())) {
				LOG_ERROR("Ip address incorrect!\n");
				return false;
			}
		}
	}
	if (config.is_field_cpu(param_name->get_cstring())) {
		ret = config.save_cpu_field(d.card_id, d.cpu_id,
					     param_name->get_cstring(), param_val->get_cstring());
		return ret;
	}
	if (config.is_field_blk(param_name->get_cstring()) && blk_id_f) {
		if (!config.blk_dev_exist(d.card_id, d.cpu_id, blk_id_f->get_number()))
			config.add_blk_dev_fields(d.card_id, d.cpu_id, blk_id_f->get_number());

		return config.save_blk_field(d.card_id, d.cpu_id, blk_id_f->get_number(),
			param_name->get_cstring(), param_val->get_cstring());
	}

	return true;
}

bool config_use(caller_data d)
{
	if (d.caller_id == 0) {
		std::string realod_config_cmd = ECHO " " VCA_CONFIG_USE_CMD " > " MSG_FIFO_FILE;

		if (run_cmd(realod_config_cmd.c_str()) == FAIL) {
			LOG_ERROR("Cannot execute: %s\n", realod_config_cmd.c_str());
			return false;
		}
	}
	return true;
}

bool config_default(caller_data d)
{
	if (command_err != EAGAIN || !open_blk_fd(d.card_id, d.cpu_id) || is_any_blkdev_enabled(d))
		return false;

	if (d.caller_id < (count_available_nodes() - 1))
		return true;

	return config.save_default_config();
}

bool vca_agent_command(caller_data d, std::string &output, vca_agent_cmd cmd)
{
	bool success = false;
	char * new_str = NULL;

	close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));
	if (cpu_fd.fd == FAIL)
		return false;

	std::string state;
	state = get_cpu_state(d);
	if(state != get_vca_state_string(VCA_OS_READY)
	   && state != get_vca_state_string(VCA_NET_DEV_READY)
	   && state != get_vca_state_string(VCA_DHCP_ERROR)
	   && state != get_vca_state_string(VCA_NET_DEV_NO_IP)
	   && state != get_vca_state_string(VCA_DHCP_IN_PROGRESS)
	   && state != get_vca_state_string(VCA_DHCP_DONE)
	   && state != get_vca_state_string(VCA_NET_DEV_DOWN)) {
		d.LOG_CPU_ERROR("Card needs to be in \"%s\", \"%s\", \"%s\", \"%s\", \"%s\" or \"%s\" state!\n",
				get_vca_state_string(VCA_OS_READY),
				get_vca_state_string(VCA_NET_DEV_READY),
				get_vca_state_string(VCA_DHCP_ERROR),
				get_vca_state_string(VCA_NET_DEV_NO_IP),
				get_vca_state_string(VCA_DHCP_IN_PROGRESS),
				get_vca_state_string(VCA_DHCP_DONE),
				get_vca_state_string(VCA_NET_DEV_DOWN));
		return false;
	}

	vca_csm_ioctl_agent_cmd *data = (vca_csm_ioctl_agent_cmd *)malloc(sizeof(*data) + PAGE_SIZE);
	if (!data)
		return false;

	memset(data, 0, sizeof(*data) + PAGE_SIZE);

	*(char *)data->buf = cmd;
	data->buf_size = PAGE_SIZE;

	if(!csm_ioctl(cpu_fd, VCA_AGENT_COMMAND, data))
		goto free;

	if (!check_lbp_state_ok(data->ret, VCA_AGENT_COMMAND, d))
		goto free;

	if(data->buf_size >= PAGE_SIZE) {
		d.LOG_CPU_ERROR("IOCTL returned with corrupted values!\n");
		goto free;
	}

	if (data->buf_size == 0) {
		output = "";
		success = true;
		goto free;
	}

	new_str = (char*)data->buf;
	if(new_str[data->buf_size-1] == '\n')
		new_str[data->buf_size-1] = '\0';
	new_str[PAGE_SIZE - 1] = '\0';

	output = new_str;

	success = true;
free:
	free(data);
	return success;
}

bool read_temp(caller_data d)
{
    bool ret;
    std::string output = "";

    ret = vca_agent_command(d, output, VCA_AGENT_SENSORS);

    if (is_os_windows(get_cpu_os(d)) && ret)
    {
        std::istringstream iss_output(output);
        std::string temp_cores[5];

        int i = 0;
        while (std::getline(iss_output, temp_cores[i]))
        {
            i++;
            if (i > 4)
                break;
        }
        for (i = 0; i < 4; i++) {
            temp_cores[i].insert(0, "+");
            if (temp_cores[i].size() < 4) { temp_cores[i].insert(0, " "); } //unification of windows and linux messages
                                                                            //additional space before 2-digit temperature reading
        }
        printf("Card %d Cpu %d\nCore 0:        %s.0\u00B0C  (high = +%s.0\u00B0C,"
            " crit = +%s.0\u00B0C)\nCore 1:        %s.0\u00B0C  (high = +%s.0\u00B0C,"
            " crit = +%s.0\u00B0C)\nCore 2:        %s.0\u00B0C  (high = +%s.0\u00B0C,"
            " crit = +%s.0\u00B0C)\nCore 3:        %s.0\u00B0C  (high = +%s.0\u00B0C,"
            " crit = +%s.0\u00B0C)\n",
            d.card_id, d.cpu_id, temp_cores[0].c_str(), temp_cores[4].c_str(),
            temp_cores[4].c_str(), temp_cores[1].c_str(), temp_cores[4].c_str(),
            temp_cores[4].c_str(), temp_cores[2].c_str(), temp_cores[4].c_str(),
            temp_cores[4].c_str(), temp_cores[3].c_str(), temp_cores[4].c_str(),
            temp_cores[4].c_str());
    }
    else
    {
        if (ret){
            for (int i = 0; i < 3; i++)
                output.erase(0, output.find("\n") + 1);     //unification of windows and linux messages
                                                            //first 3 lines removal for linux standard temperature report
            printf("Card %d Cpu %d:\n%s\n", d.card_id, d.cpu_id, output.c_str());
        }

        else
            LOG_ERROR("Execute 'temp' command failed!\n");

    }
    return ret;
}

std::string extract_network_data(std::string output, std::string kind)
{
	std::string begin_pattern;
	std::string end_pattern;

	if (kind == NETWORK_SUBCMD_IP) {
		begin_pattern = "inet ";
		end_pattern = "/";
	}
	else if (kind == NETWORK_SUBCMD_IP6) {
		begin_pattern = "inet6 ";
		end_pattern = "/";
	}
	else if (kind == NETWORK_SUBCMD_MAC) {
		begin_pattern = "ether ";
		end_pattern = " brd";
	}

	int begin_pattern_pos = output.find(begin_pattern);
	if (begin_pattern_pos == -1) {
		LOG_ERROR("%s not found", kind.c_str());
		return "";
	}
	// tmp_output needed because there is more than one end_pattern in original output
	std::string tmp_output = output.substr(begin_pattern_pos, output.size() - begin_pattern_pos);
	int end_pattern_pos = tmp_output.find(end_pattern);
	if (end_pattern_pos == -1) {
		LOG_ERROR("%s not found", kind.c_str());
		return "";
	}

	return tmp_output.substr(begin_pattern.size(), end_pattern_pos - begin_pattern.size());
}

void parse_network_info(caller_data d, std::string subcmd, std::string &output)
{
	if (d.args.size() < 3)
		printf("Card %d Cpu %d:\n", d.card_id, d.cpu_id);

	if (subcmd == NETWORK_SUBCMD_ALL || subcmd == NETWORK_SUBCMD_STATS) {
		std::string pattern = ": ";
		int shift = output.find_first_of(pattern);
		printf("%s\n", output.substr(shift + pattern.size(), output.size() - (shift + pattern.size())).c_str());
	}
	else if (subcmd == NETWORK_SUBCMD_IP || subcmd == NETWORK_SUBCMD_IP6 || subcmd == NETWORK_SUBCMD_MAC)
		printf("%s\n", extract_network_data(output, subcmd).c_str());
	else if (subcmd == NETWORK_SUBCMD_RENEW || subcmd == NETWORK_SUBCMD_VM_MAC)
		printf("%s\n", output.c_str());
	else
		printf("Wrong command: %s\n", subcmd.c_str());
}

void parse_network_info_windows(caller_data d, std::string subcmd, std::string &output)
{
	if (d.args.size() < 3)
		printf("Card %d Cpu %d:\n", d.card_id, d.cpu_id);

	std::istringstream input(output);
	std::string line;

	if (subcmd == NETWORK_SUBCMD_ALL || subcmd == NETWORK_SUBCMD_STATS ||
	    subcmd == NETWORK_SUBCMD_RENEW || subcmd == NETWORK_SUBCMD_VM_MAC ||
	    subcmd == NETWORK_SUBCMD_IP6) {
		printf("%s\n", output.c_str());
	}
	else if (subcmd == NETWORK_SUBCMD_IP) {
		while (std::getline(input, line))
		{
			if (is_ip_address(line.c_str()))
				printf("%s\n", line.c_str());
		}
	}
	else if	(subcmd == NETWORK_SUBCMD_MAC) {
		while (std::getline(input, line))
		{
			if (line.length() == 17)
				printf("%s\n", line.c_str());
			// TODO add better mac validation
		}
	}
	else
		printf("Wrong command: %s\n", subcmd.c_str());
}

bool get_network_info(caller_data d)
{
	bool ret;
	std::string output = "";
	std::string subcmd = "";

	const data_field *subcmd_f = d.args(SUBCMD);
	if (subcmd_f)
		subcmd = subcmd_f->get_cstring();

	if (subcmd == NETWORK_SUBCMD_STATS)
		ret = vca_agent_command(d, output, VCA_AGENT_IP_STATS);
	else if (subcmd == NETWORK_SUBCMD_RENEW) {
		const char *ip = get_cpu_field_if_not_empty(d, cpu_fields::ip);
		if (!ip)
			return false;
		if (strncasecmp(DHCP_STRING, ip, strlen(DHCP_STRING))) {
			LOG_ERROR("Cannot renew DHCP lease on node with static IP assigned\n");
			return false;
		}
		ret = vca_agent_command(d, output, VCA_AGENT_RENEW);
	}
	else if (subcmd == NETWORK_SUBCMD_VM_MAC) {
		ret = vca_agent_command(d, output, VCA_AGENT_VM_MAC);
	}
	else
		ret = vca_agent_command(d, output, VCA_AGENT_IP);

	if (ret) {
		if (is_os_windows(get_cpu_os(d)))
			parse_network_info_windows(d, subcmd, output);
		else
			parse_network_info(d, subcmd, output);
	}
	else
		LOG_ERROR("Execute " NETWORK_CMD " command failed!\n");

	return ret;
}

bool print_hw_info(caller_data callerData)
{
	std::string card_generation = get_card_gen(callerData.card_id);
	std::string eeprom_version = get_eeprom_version(callerData.card_id);

	printf("Card %d: %s, EEPROM version: %s\n", callerData.card_id, card_generation.c_str(), eeprom_version.c_str());

	return true;
}

bool print_system_info(caller_data callerData)
{
	std::string appsVersion = "build: " BUILDNO " build on: " BUILDDATE;
	printf("Apps version: %s\n", appsVersion.c_str());

	std::string modulesVersion = get_modules_version(0);
	printf("Modules build: %s\n", modulesVersion.c_str());

	return true;
}

void print_node_os_info(caller_data d)
{
	if (d.args.size() < 3)
		printf("Card %d Cpu %d:\n", d.card_id, d.cpu_id);

	std::string os_type = get_cpu_os(d);

	if (!os_type.empty())
		printf("%s\n", os_type.c_str());
	else
		printf(VCA_OS_TYPE_UNKNOWN_TEXT"\n");
}

bool print_node_bios_date_info(caller_data d, close_on_exit& cpu_fd)
{
	vca_csm_ioctl_bios_param_desc buffer;
	buffer.param = VCA_LBP_BIOS_PARAM_BUILD_DATE;
	if (csm_ioctl(cpu_fd, LBP_GET_BIOS_PARAM, &buffer))
		if (LBP_STATE_OK == buffer.ret)	{
			printf("\tRelease Date:\t%04u.%02u.%02u %02u:%02u:%02u\n",
				   (unsigned)buffer.value.time.year,
				   (unsigned)buffer.value.time.month,
				   (unsigned)buffer.value.time.day,
				   (unsigned)buffer.value.time.hour,
				   (unsigned)buffer.value.time.minutes,
				   (unsigned)buffer.value.time.seconds);
			return true;
		}
		else if (LBP_PROTOCOL_VERSION_MISMATCH == buffer.ret) {
			d.LOG_CPU_ERROR("\tToo old version of BIOS which does not support command ' %s BIOS '"
				". Update BIOS at node\n", d.get_cmd_name());
			command_err = EPERM;
			return false;
		}
		else if (LBP_BIOS_INFO_CACHE_EMPTY == buffer.ret) {
			d.LOG_CPU_ERROR("\tCannot read BIOS info. Try again or if you will see this error"
				" repeatedly then try to update bios at node\n");
			command_err = EAGAIN;
			return false;
		}
		else d.LOG_CPU_ERROR("Error %u\n", buffer.ret);
	else command_err = 1;
	return false;
}

std::string get_bios_version(caller_data d, close_on_exit& cpu_fd)
{
	vca_csm_ioctl_bios_param_desc buffer;
	buffer.param = VCA_LBP_BIOS_PARAM_VERSION;

	if (csm_ioctl(cpu_fd, LBP_GET_BIOS_PARAM, &buffer)) {
		if (LBP_STATE_OK == buffer.ret) {
			char command[BUFFER_SIZE];
			if (snprintf(command, sizeof(command), "%.8s", buffer.value.version)) {
				return command;
			}
		}
	}
	return BIOS_VERSION_ERROR;
}

bool print_node_bios_version_info(caller_data d, close_on_exit& cpu_fd)
{
	vca_csm_ioctl_bios_param_desc buffer;
	buffer.param = VCA_LBP_BIOS_PARAM_VERSION;
	if (csm_ioctl(cpu_fd, LBP_GET_BIOS_PARAM, &buffer))
		if (LBP_STATE_OK == buffer.ret)	{
			printf("\tVersion:\t%.8s\n", buffer.value.version);
			return true;
		}
		else if (LBP_PROTOCOL_VERSION_MISMATCH == buffer.ret) {
			d.LOG_CPU_ERROR("\tToo old version of BIOS which does not support command ' %s BIOS '"
				". Update BIOS at node\n", d.get_cmd_name());
			command_err = EPERM;
			return false;
		}
		else if (LBP_BIOS_INFO_CACHE_EMPTY == buffer.ret) {
			d.LOG_CPU_ERROR("\tCannot read BIOS info. Try again or if you will see this error"
				" repeatedly then try to update bios at node\n");
			command_err = EAGAIN;
			return false;
		}
		else d.LOG_CPU_ERROR("Error %u\n", buffer.ret);
	else command_err = 1;
	return false;
}


bool print_node_bios_info(caller_data d)
{
	if (d.args.size() < 3)
		printf("Card %d Cpu %d:\n", d.card_id, d.cpu_id);
	close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));
	if (cpu_fd) {
		if (get_cpu_state(d) == VCA_BIOS_UP_TEXT)
			if(!try_handshake(cpu_fd, d))
				return false;
		if (print_node_bios_version_info(d, cpu_fd) && print_node_bios_date_info(d, cpu_fd))
			return true;
		else
			return false;
	}
	return false;
}


bool print_info_cmd_data(caller_data d)
{
	std::string subcmd = d.args.get_arg(SUBCMD);

	if (subcmd == INFO_SUBCMD_HW)
		print_hw_info(d);
	else if (subcmd == INFO_SUBCMD_SYSTEM) {
		print_system_info(d);
	}
	else if (subcmd == INFO_SUBCMD_NODE_OS)
		print_node_os_info(d);
	else if (subcmd == INFO_SUBCMD_NODE_BIOS)
		print_node_bios_info( d);
	else {
		d.LOG_CPU_ERROR("Unknown subcommand: %s", subcmd.c_str());
		return false;
	}

	return true;
}

static void print_blk_dev_description(unsigned int card_id, unsigned int cpu_id, std::string name, enum blk_state state, std::string mode, std::string size_mb, std::string file_path)
{
	if (mode == "") mode = "-";
	if (size_mb == "") size_mb = "-";
	if (file_path == "") file_path = "-";

	LOG_INFO("%5u %5u %9s %9s %8s %9s %s\n", card_id, cpu_id, name.c_str(), get_blk_state_cstr(state), mode.c_str(), size_mb.c_str(), file_path.c_str());
}

static void print_blk_dev_header()
{
	LOG_INFO("%5s %5s %9s %9s %8s %9s %2s\n", "Card", "Cpu", "Name", "State", "Mode", "Size(MB)", "FilePath");
	LOG_INFO("------------------------------------------------------------\n");
}

bool reboot(caller_data d)
{
	bool ret;
	std::string output = "";

	if (get_cpu_state(d) == get_vca_state_string(VCA_NET_DEV_UP))
		d.LOG_CPU_ERROR("It's impossible to use %s command in %s state, because in this"
		" state VcaAgent doesn't work properly.\n", d.get_cmd_name(), get_vca_state_string(VCA_NET_DEV_UP));
	else {
		ret = vca_agent_command(d, output, VCA_AGENT_OS_REBOOT);

		if (ret)
			d.LOG_CPU(VERBOSE_DEFAULT, output.append("\n").c_str());
		else
			d.LOG_CPU_ERROR("Execute '%s' command failed!\n", d.get_cmd_name());
		return ret;
	}
	return false;
}

bool os_shutdown(caller_data d)
{
        bool ret;
        std::string output = "";

	if (get_card_type(d.card_id) == VCA_MV && get_cpu_state(d) == get_vca_state_string(VCA_NET_DEV_UP))
		d.LOG_CPU_ERROR("It's impossible to use %s command in %s state, because in this"
		" state VcaAgent doesn't work properly. Please use 'pwrbtn-short' command!\n", d.get_cmd_name(),
		get_vca_state_string(VCA_NET_DEV_UP));
	else {
		ret = vca_agent_command(d, output, VCA_AGENT_OS_SHUTDOWN);

		if (ret)
			d.LOG_CPU(VERBOSE_DEFAULT, output.append("\n").c_str());
		else
			d.LOG_CPU_ERROR("Execute '%s' command failed!\n", d.get_cmd_name());

		return ret;
	}
	return false;
}

struct ApertureSize {
	const char *as_string;
	unsigned long long encoded_for_bios;
} ApertureSizes[] = {
	{  "128", 0x00 },
	{  "256", 0x01 },
	{  "512", 0x03 },
	{ "1024", 0x07 },
	{ "2048", 0x0f },
	{ "4096", 0x1f },
	{ NULL, 0 } // guard, keep last
};
bool aperture_from_string(const char *str, unsigned long long *encoded_for_bios) {
	for (int i = 0; ApertureSizes[i].as_string; ++i) {
		if (!strcmp(ApertureSizes[i].as_string, str)) {
			if (encoded_for_bios) {
				*encoded_for_bios = ApertureSizes[i].encoded_for_bios;
			}
			return true;
		}
	}
	return false;
}
const char *aperture_to_string(unsigned long long encoded_for_bios) {
	for (int i = 0; ApertureSizes[i].as_string; ++i) {
		if (encoded_for_bios == ApertureSizes[i].encoded_for_bios) {
			return ApertureSizes[i].as_string;
		}
	}
	return "unknown";
}

struct TdpSize {
	const char *as_string;
	unsigned long long encoded_for_bios;
} TdpSizes[] = {
	{ "0", 0 },
	{ "1", 1 },
	{ "2", 2 },
	{ "3", 3 },
	{ "4", 4 },
	{ "5", 5 },
	{ "6", 6 },
	{ "7", 7 },
	{ "8", 8 },
	{ "9", 9 },
	{ "10", 10 },
	{ "11", 11 },
	{ NULL, 0 } // guard, keep last
};

bool tdp_from_string(const char *str, unsigned long long *encoded_for_bios) {
	for (int i = 0; TdpSizes[i].as_string; ++i) {
		if (!strcmp(TdpSizes[i].as_string, str)) {
			if (encoded_for_bios) {
				*encoded_for_bios = TdpSizes[i].encoded_for_bios;
			}
			return true;
		}
	}
	return false;
}

const char *tdp_to_string(unsigned long long encoded_for_bios) {
	for (int i = 0; TdpSizes[i].as_string; ++i) {
		if (encoded_for_bios == TdpSizes[i].encoded_for_bios) {
			return TdpSizes[i].as_string;
		}
	}
	return "unknown";
}

bool is_bios_cfg_option(const char *cfg)
{
	return (!strcmp(cfg, BIOS_CFG_SGX) ||
		    !strcmp(cfg, BIOS_CFG_GPU_APERTURE) ||
			!strcmp(cfg, BIOS_CFG_TDP));
}

bool get_bios_cfg(caller_data d)
{
	bool is_gen1_card = get_card_type(d.card_id) & VCA_VV;
	close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));
	std::string cfg = d.args.get_arg(BIOS_CFG_NAME_ARG);
	std::string state = get_cpu_state(d);
	vca_csm_ioctl_bios_param_desc desc;
	
	// TODO: add better decode/encode BIOS cfg options

	const int bios_cfg_quantity = 3;
	vca_lbp_bios_param cfg_cmd[bios_cfg_quantity] = { VCA_LBP_PARAM_SGX, VCA_LBP_PARAM_GPU_APERTURE, VCA_LBP_PARAM_TDP };
	int cfg_enabled[bios_cfg_quantity] = { 0, 0, 0};
	int cfg_output[bios_cfg_quantity] = {-1, -1, -1};
	std::string aperture_output;

	if (cfg == BIOS_CFG_SGX) {
		cfg_enabled[0] = 1;
		if (is_gen1_card) {
			d.LOG_CPU_WARN("SGX is not supported on VCA GEN1 card.\n");
			return false;
		}
	}
	else if (cfg == BIOS_CFG_GPU_APERTURE) {
		cfg_enabled[1] = 1;
	}
	else if (cfg == BIOS_CFG_TDP) {
		cfg_enabled[2] = 1;
		if (is_gen1_card) {
			d.LOG_CPU_WARN("Changing TDP is not supported on VCA GEN1 card.\n");
			return false;
		}
	}
	else if (cfg == "") { // for all cfg
		cfg_enabled[0] = !is_gen1_card;
		cfg_enabled[1] = 1;
		cfg_enabled[2] = !is_gen1_card;
	}

	if (state == get_vca_state_string(VCA_BIOS_UP)) {
		if (!set_lbp_param(cpu_fd, VCA_LBP_PARAM_i7_IRQ_TIMEOUT_MS,
			config.get_global_field(global_fields::handshake_irq_timeout_ms).get_number(), d))
			return false;

		if (!try_handshake(cpu_fd, d))
			return false;
	}
	else if (state == get_vca_state_string(VCA_BIOS_DOWN) ||
		 state == get_vca_state_string(VCA_ERROR) ||
		 state == get_vca_state_string(VCA_POWERING_DOWN) ||
		 state == get_vca_state_string(VCA_POWER_DOWN) ||
		 state == LINK_DOWN_STATE) {
		d.LOG_CPU_ERROR("BIOS configuration could not be changed on node in %s state!\n", state.c_str());
		return false;
	}

	for (int i = 0; i < bios_cfg_quantity; i++) {
		if (cfg_enabled[i]) {
			desc.param = cfg_cmd[i];
			if (!csm_ioctl(cpu_fd, LBP_GET_BIOS_PARAM, &desc))
				return false;

			if (!check_lbp_state_ok(desc.ret, LBP_GET_BIOS_PARAM, d))
				return false;

			cfg_output[i] = desc.value.value;
		}
	}
	std::string cmd_output = "BIOS configuration:\n";
	if (cfg_enabled[0]) { // sgx
		cmd_output += "SGX:\t\t";
		cmd_output += cfg_output[0] ? "enabled\n" : "disabled\n";
	}
	if (cfg_enabled[1]) { // gpu aperture
		cmd_output += "GPU_APERTURE:\t";
		cmd_output += aperture_to_string(cfg_output[1]);
		cmd_output += "\n";
	}
	if (cfg_enabled[2]) { // tdp
		cmd_output += "TDP:\t\tBase +";
		cmd_output += tdp_to_string(cfg_output[2]);
		cmd_output += " W\n";
	}
	d.LOG_CPU(VERBOSE_DEFAULT, "%s", cmd_output.c_str());

	return true;
}

// return true if sgx==enabled and aperture_size==aperture_val is supported conf.
static bool is_sgx_supported_for_aperture(unsigned long long aperture_val) {
	return aperture_val <= 0x03; // less than or equal to 512MB
	// NOTE: when changing this function, grep also for
	// 'Invalid BIOS configuration requested' in this file
}
static bool is_aperture_size_valid_on_gen1_card(unsigned long long aperture_val) {
	return aperture_val <= 0x07; // less than or equal to 1024MB
	// NOTE: when changing this function, grep also for
	// 'greater than 1024MB on VCA GEN1 card' in this file
}

bool prepare_SPAD_to_ready(caller_data d, filehandle_t cpu_fd)
{
	if (!set_lbp_param(cpu_fd, VCA_LBP_PARAM_i7_IRQ_TIMEOUT_MS,
		config.get_global_field(global_fields::handshake_irq_timeout_ms).get_number(), d))
		return false;

	return try_handshake(cpu_fd, d);
}

bool set_bios_cfg(caller_data d)
{
	bool is_gen1_card = get_card_type(d.card_id) & VCA_VV;
	std::string cfg = d.args.get_arg(BIOS_CFG_NAME_ARG);
	std::string value = d.args.get_arg(BIOS_CFG_VALUE_ARG);
	std::string status = get_cpu_state(d);
	close_on_exit cpu_fd(open_cpu_fd(d.card_id, d.cpu_id));
	vca_csm_ioctl_bios_param_desc desc;
	bool needs_compat_check = 0;

	if (cfg == BIOS_CFG_SGX) {
		bool enable = value == "enable";
		if (is_gen1_card) {
			d.LOG_CPU_ERROR("SGX is not supported on VCA GEN1 card.\n");
			return false;
		}
		desc.param = VCA_LBP_PARAM_SGX;
		desc.value.value = enable;
		needs_compat_check = enable;
	}
	else if (cfg == BIOS_CFG_GPU_APERTURE) {
		desc.param = VCA_LBP_PARAM_GPU_APERTURE;
		if (!aperture_from_string(value.c_str(), &desc.value.value)) {
			d.LOG_CPU_ERROR("invalid value for GPU_APERTURE: `%s'\n", value.c_str());
			return false;
		}
		needs_compat_check = !is_gen1_card && !is_sgx_supported_for_aperture(desc.value.value);
		if (is_gen1_card && !is_aperture_size_valid_on_gen1_card(desc.value.value)) {
			d.LOG_CPU_ERROR("GPU_APERTURE could not be greater than 1024MB on VCA GEN1 card.\n");
			return false;
		}
	}
	else if (cfg == BIOS_CFG_TDP) {
		desc.param = VCA_LBP_PARAM_TDP;
		if (is_gen1_card) {
			d.LOG_CPU_ERROR("VCA GEN1 card doesn't support changing TDP value.\n");
			return false;
		}
		if (!tdp_from_string(value.c_str(), &desc.value.value)) {
			d.LOG_CPU_ERROR("Value (%s) to change TDP is not correct. See 'vcactl help' for reference.\n",
				value.c_str());
			return false;
		}
	} else {
		return false;
	}

	bool status_is_bios_up = status == get_vca_state_string(VCA_BIOS_UP);
	bool status_is_bios_ready = status == get_vca_state_string(VCA_BIOS_READY);
	if (!status_is_bios_up && !status_is_bios_ready) {
		d.LOG_CPU_ERROR("Node needs to be in \"%s\" or \"%s\" state!\n",
			get_vca_state_string(VCA_BIOS_UP), get_vca_state_string(VCA_BIOS_READY));
		return false;
	}

	if (status_is_bios_up) {
		if (!prepare_SPAD_to_ready(d, cpu_fd))
			return false;
	}

	bool param_is_sgx = desc.param == VCA_LBP_PARAM_SGX;
	int max_attempts = param_is_sgx ? 3 : 1;
	if (needs_compat_check) {
		vca_lbp_bios_param other_param = param_is_sgx ? VCA_LBP_PARAM_GPU_APERTURE : VCA_LBP_PARAM_SGX;
		unsigned long long other_param_value;
		if (!get_bios_cfg_for(d, cpu_fd, other_param, other_param_value)) {
			d.LOG_CPU_ERROR("Prerequisite checking failed. Aborting.\n");
			return false;
		}

		if (!(!other_param_value || (param_is_sgx && is_sgx_supported_for_aperture(other_param_value)))) {
			d.LOG_CPU_ERROR("Invalid BIOS configuration requested. SGX could be enabled only with aperture size <= 512MB.\n");
			return false;
		}
	}

	for (int i = 0; i < max_attempts; i++)
	{
		set_bios_cfg_for(d, cpu_fd, desc);
		d.LOG_CPU(VERBOSE_DEFAULT, "BIOS configuration changed. Node will be restarted...\n");
		if (cfg != BIOS_CFG_TDP && !is_gen1_card) {
			press_power_button(d, true);
			sleep(TIME_TO_POWER_DOWN_NODE_S); // waiting to make sure that CPU power go down
			press_power_button(d, false);
			d.LOG_CPU(VERBOSE_DEFAULT, "Node will be power down and up to make the change active.\n"
				"Please wait for 'bios_up' to start work with the node.\n");
		}
		else {
			reset(d);
			d.LOG_CPU(VERBOSE_DEFAULT, "Node will be reset to make the change active.\n"
				"Please wait for 'bios_up' to start work with the node.\n");
		}

		if (!wait_bios(d))
			return false;

		if (!prepare_SPAD_to_ready(d, cpu_fd))
			return false;

		unsigned long long new_value;
		if (get_bios_cfg_for(d, cpu_fd, desc.param, new_value) && new_value == desc.value.value)
			return true;

		if (i + 1 != max_attempts)
			d.LOG_CPU(VERBOSE_DEFAULT, "Retrying to set param in BIOS\n");
		else
			d.LOG_CPU_ERROR("Failed to set param in BIOS. Exiting...\n");
	}

	return false;
}

bool print_blk_list(caller_data d, filehandle_t blk_dev_fd, unsigned int blk_dev_id)
{
	std::string name = get_block_dev_name_from_id(blk_dev_id);
	std::string mode;
	std::string file_path;
	std::string size_mb;

	enum blk_state state = BLK_STATE_DISABLED;

	if (check_blk_disk_exist(blk_dev_fd, blk_dev_id)) {
		struct vcablk_disk_info_desc info;
		info.disk_id = blk_dev_id;
		info.exist = 0;

		int rc = ioctl(blk_dev_fd, VCA_BLK_GET_DISK_INFO, &info);
		if (rc < 0) {
			LOG_ERROR("Ioctl VCA_BLK_GET_DISK_INFO error: %s\n", strerror(errno));
			return false;
		}

		file_path = "";
		if (info.type == VCABLK_DISK_TYPE_MEMORY) {
			mode = BLOCKIO_MODE_RAMDISK;
		}
		else {
			file_path = info.file_path;
			mode = get_mode_string(info.mode);
		}

		size_mb = int_to_string(B_TO_MB(info.size));

		if (is_blk_disk_active(d, blk_dev_fd, blk_dev_id))
			state = BLK_STATE_ACTIVE;
		else
			state = BLK_STATE_INACTIVE;

	}
	else if (config.blk_dev_exist(d.card_id, d.cpu_id, blk_dev_id)) {
		std::string s_mode = config.get_blk_field(d.card_id, d.cpu_id, blk_dev_id, blk_dev_fields::mode).get_string();
		std::string s_path = config.get_blk_field(d.card_id, d.cpu_id, blk_dev_id, blk_dev_fields::path).get_string();
		size_mb = config.get_blk_field(d.card_id, d.cpu_id, blk_dev_id, blk_dev_fields::ramdisk_size_mb).get_string();

		file_path = "";
		bool ramdisk = (s_mode == BLOCKIO_MODE_RAMDISK);
		if (ramdisk) {
			mode = BLOCKIO_MODE_RAMDISK;
		} else {
			file_path = s_path;
			mode = s_mode;
		}

		if (is_blk_enabled(d.card_id, d.cpu_id, blk_dev_id))
			state = BLK_STATE_ENABLED;
		else
			state = BLK_STATE_DISABLED;
	} else
		return true;

	print_blk_dev_description(d.card_id, d.cpu_id, name, state, mode, size_mb, file_path);
	return true;
}

bool blockio_ctl_list(caller_data d)
{
	close_on_exit blk_dev_fd = open_blk_fd(d.card_id, d.cpu_id);
	if (blk_dev_fd.fd == FAIL)
		return false;

	if (d.caller_id == 0) {
		print_blk_dev_header();
	}

	if (d.args.get_arg(BLOCKIO_ID_ARG).empty()) { // print all available block devices for given card and cpu
		for (int i = 0; i < MAX_BLK_DEVS; i++) {
			if (!print_blk_list(d, blk_dev_fd.fd, i))
				return false;
		}
	}
	else {
		unsigned int blk_dev_id = atoi(d.args.get_arg(BLOCKIO_ID_ARG).c_str());
		if (!print_blk_list(d, blk_dev_fd.fd, blk_dev_id))
			return false;
	}

	return true;
}

bool update_blk_config_fields(caller_data d, unsigned int blk_dev_id, const char *mode, const char *ramdisk_size, const char *file_path)
{
	if (!config.blk_dev_exist(d.card_id, d.cpu_id, blk_dev_id)) {
		if (config.add_blk_dev_fields(d.card_id, d.cpu_id, blk_dev_id)) {
			d.LOG_CPU(DEBUG_INFO, "Block device %s added to config file.\n",
				get_block_dev_name_from_id(blk_dev_id).c_str());
		}
		else {
			d.LOG_CPU_ERROR("Cannot add block device %s to config file!\n",
				get_block_dev_name_from_id(blk_dev_id).c_str());
			return false;
		}
	}

	if (!config.save_blk_field(d.card_id, d.cpu_id, blk_dev_id, BLK_CONFIG_MODE_STR, "") ||
		!config.save_blk_field(d.card_id, d.cpu_id, blk_dev_id, BLK_CONFIG_RAMDISK_SIZE_STR, "0") ||
		!config.save_blk_field(d.card_id, d.cpu_id, blk_dev_id, BLK_CONFIG_PATH_STR, ""))
		return false;

	if (mode && (!ramdisk_size != !file_path)) {
		if (ramdisk_size &&
				!config.save_blk_field(d.card_id, d.cpu_id, blk_dev_id, BLK_CONFIG_RAMDISK_SIZE_STR, ramdisk_size)) {
			return false;
		}
		else if (file_path &&
				!config.save_blk_field(d.card_id, d.cpu_id, blk_dev_id, BLK_CONFIG_PATH_STR, file_path)) {
			return false;
		}

		if (!config.save_blk_field(d.card_id, d.cpu_id, blk_dev_id, BLK_CONFIG_MODE_STR, mode))
			return false;
	}
	else {
		LOG_ERROR("Invalid config blockio params!\n");
		return false;
	}

	d.LOG_CPU(DEBUG_INFO, "Block device %s data updated in config.\n", get_block_dev_name_from_id(blk_dev_id).c_str());

	if (!config.get_vca_config_from_file()) {
		LOG_ERROR("could not parse vca configuration file!\n");
		return false;
	}

	return true;
}

bool check_blk_path(caller_data d, int cards_num)
{
	// TODO: refactor handling of command starting in order to simplify
	//       and remove this ugly static
	static std::string previous_file_path;
	unsigned int blk_dev_id = atoi(d.args.get_arg(BLOCKIO_ID_ARG).c_str());
	std::string mode = d.args.get_arg(BLOCKIO_MODE_ARG);
	std::string file_path;
	std::string tmp_path;

	if (!strcmp(d.args.get_cmd_name(), BOOT_CMD) && is_blk_enabled(d.card_id, d.cpu_id, blk_dev_id)) {
		if (blk_dev_id != 0) {
			d.LOG_CPU_ERROR("Currently booting from blockio is supported only for vcablk0.\n");
			return false;
		}
		file_path = config.get_blk_field(d.card_id, d.cpu_id, blk_dev_id, blk_dev_fields::path).get_string();
		mode = config.get_blk_field(d.card_id, d.cpu_id, blk_dev_id, blk_dev_fields::mode).get_string();
	} else if (!strcmp(d.args.get_cmd_name(), BLOCKIO_CMD) && mode == BLOCKIO_MODE_RW) {
		file_path = d.args.get_arg(BLOCKIO_MODE_PARAM_ARG);
	} else {
		return true;
	}

	if (mode == BLOCKIO_MODE_RAMDISK)
		return true;

	if (previous_file_path == file_path) {
		d.LOG_CPU_ERROR("File %s is already opened in other block device. Cannot open the same file on few block devices.\n", file_path.c_str());
		return false;
	}

	for (int i = 0; i < MAX_BLK_DEVS; i++) {
		for (int j = 0; j < cards_num; j++) {
			for (int k = 0; k < MAX_CPU; k++) {
				if (j == d.card_id && k == d.cpu_id) {
					continue; // reusing own device is perfectly ok
				}
				tmp_path = config.get_blk_field(j, k, i, blk_dev_fields::path).get_string();
				mode = config.get_blk_field(j, k, i, blk_dev_fields::mode).get_string();
				if (mode != BLOCKIO_MODE_RAMDISK && file_path == tmp_path && is_blk_enabled(j, k, i)) {
					d.LOG_CPU_ERROR("File %s is already opened in other block device. Cannot open the same file on few block devices.\n", file_path.c_str());
					return false;
				}
			}
		}
	}

	previous_file_path = file_path;
	return true;
}

bool blockio_ctl_open(caller_data d)
{
	close_on_exit blk_dev_fd = open_blk_fd(d.card_id, d.cpu_id);
	if (blk_dev_fd.fd == FAIL)
		return false;

	if (d.args.get_arg(BLOCKIO_ID_ARG).empty())
		return false;

	struct vca_blk_dev_info blk_dev_info;
	unsigned int blk_dev_id = atoi(d.args.get_arg(BLOCKIO_ID_ARG).c_str());

	if (check_blk_disk_exist(blk_dev_fd.fd, blk_dev_id)) {
		d.LOG_CPU_ERROR("Block device %s already exist.\n", get_block_dev_name_from_id(blk_dev_id).c_str());
		return false;
	}

	if (!check_blk_path(d, get_cards_num()))
		return false;

	std::string mode = d.args.get_arg(BLOCKIO_MODE_ARG);
	if (mode.empty()) {
		d.LOG_CPU(DEBUG_INFO, "Using block device described in config\n");
	} else if (!strcmp(mode.c_str(), BLOCKIO_MODE_RAMDISK)) {
		std::string disk_size = d.args.get_arg(BLOCKIO_MODE_PARAM_ARG);
		if (!is_unsigned_number(disk_size.c_str()) || atoi(disk_size.c_str()) == 0) {
			LOG_ERROR("Cannot parse ramdisk size! %s \n", disk_size.c_str());
			return false;
		} else if (!update_blk_config_fields(d, blk_dev_id, mode.c_str(), disk_size.c_str(), NULL)) {
			d.LOG_CPU_ERROR("Cannot update config for block device %s\n",
				get_block_dev_name_from_id(blk_dev_id).c_str());
			return false;
		}
		d.LOG_CPU(DEBUG_INFO, "Using block device described in config\n");
	} else if (!strcmp(mode.c_str(), BLOCKIO_MODE_RO) || !strcmp(mode.c_str(), BLOCKIO_MODE_RW)) {
		std::string file_path = d.args.get_arg(BLOCKIO_MODE_PARAM_ARG);
		char resolved_path[PATH_MAX + 1];
		if (!realpath(file_path.c_str(), resolved_path)) {
			d.LOG_CPU_ERROR("Cannot canonicalize block path (got %s): %s!\n",
					file_path.c_str(), strerror(errno));
			return false;
		} else if (!file_exists(resolved_path)) {
			d.LOG_CPU_ERROR("Wrong blockio type param! File (%s) doesn't exist.\n", resolved_path);
			return false;
		} else if (!update_blk_config_fields(d, blk_dev_id, mode.c_str(), NULL, resolved_path)) {
			d.LOG_CPU_ERROR("Cannot update config for block device %s\n",
				get_block_dev_name_from_id(blk_dev_id).c_str());
			return false;
		}
	} else {
		d.LOG_CPU_ERROR("Unknown block mode %s\n", mode.c_str());
		return false;
	}

	if (config.blk_dev_exist(d.card_id, d.cpu_id, blk_dev_id)) {
		if (!get_blk_config_field(d, blk_dev_id, blk_dev_info)) {
			d.LOG_CPU_ERROR("Cannot get blk_dev_info from config\n");
			return false;
		}
	}
	else {
		d.LOG_CPU_ERROR("Block device %s not defined in config file!\n",
			get_block_dev_name_from_id(blk_dev_id).c_str());
		return false;
	}

	if (!enable_blk_dev(d, blk_dev_id))
		return false;

	if (!is_bios_up_or_ready(d) && (get_cpu_state(d) != get_vca_state_string(VCA_BIOS_DOWN))) {
		if (open_blk_dev(blk_dev_fd.fd, blk_dev_info) != SUCCESS) {
			d.LOG_CPU_ERROR("Open block device %s failed!\n", get_block_dev_name_from_id(blk_dev_id).c_str());
			return false;
		}
	}

	return true;
}

bool blockio_ctl_close(caller_data d)
{
	close_on_exit blk_dev_fd = open_blk_fd(d.card_id, d.cpu_id);
	unsigned int blk_dev_id = atoi(d.args.get_arg(BLOCKIO_ID_ARG).c_str());

	if (blk_dev_fd.fd == FAIL)
		return false;

	if (d.args.get_arg(BLOCKIO_ID_ARG).empty())
		return false;

	if (!is_blk_enabled(d.card_id, d.cpu_id, blk_dev_id)) {
		d.LOG_CPU_WARN("Block device %s is not open."
			" You do not need to close it.\n", get_block_dev_name_from_id(blk_dev_id).c_str());
		return true;
	}

	if (check_if_blkdev_is_not_active(d) || d.args.is_force_cmd_enabled()){
		if (close_blk_dev(blk_dev_fd.fd, blk_dev_id) != SUCCESS) {
			d.LOG_CPU(DEBUG_INFO, "Close block device %s failed!\n", get_block_dev_name_from_id(blk_dev_id).c_str());
			return false;
		}
		return disable_blk_dev(d, blk_dev_id);
	}
	else {
		if (blk_dev_id == 0){
			d.LOG_CPU_ERROR("You're trying to disconnect active block device!" \
				" It may lead to data corruption. Please use os-shutdown command first." \
				" If you're sure that you want to close it anyway, " \
				"and OS is not booted from this device, please use --force flag.\n");
			return false;
		}
		else {
			d.LOG_CPU_ERROR("You're trying to disconnect active block device!" \
				" It may lead to data corruption. Please use os-shutdown command first." \
				" If you're sure that you want to close it anyway, please use --force flag.\n");
			return false;
		}
	}
}

bool blockio_ctl(caller_data d)
{
	d.LOG_CPU(DEBUG_INFO, "Executing " BLOCKIO_CMD " %s command.\n", d.args.get_arg(SUBCMD).c_str());

	if (d.args.get_arg(SUBCMD) == BLOCKIO_SUBCMD_LIST) {
		if (!blockio_ctl_list(d)) {
			d.LOG_CPU_ERROR("Failed to list block devices!\n");
			return false;
		}
	}
	else if (d.args.get_arg(SUBCMD) == BLOCKIO_SUBCMD_OPEN) {
		if (!blockio_ctl_open(d)) {
			d.LOG_CPU_ERROR("Failed to open block device!\n");
			return false;
		}
	}
	else if (d.args.get_arg(SUBCMD) == BLOCKIO_SUBCMD_CLOSE) {
		if (!blockio_ctl_close(d)) {
			d.LOG_CPU_ERROR("Failed to close block device!\n");
			return false;
		}
	}
	else {
		d.LOG_CPU_ERROR("Wrong blockio subcommand!\n");
		return false;
	}

	return true;
}

std::string get_card_gen(const unsigned card_id)
{
	enum vca_card_type card_type = get_card_type(card_id);
	if(card_type == VCA_UNKNOWN)
		return "unknown";

	std::string card_gen =  card_type == VCA_MV ? "VCA GEN 2" : "VCA GEN 1";
	if(card_type == VCA_MV) {
		__u32 boardId = get_board_id(card_id);
		card_gen = card_gen + " FAB " + boost::lexical_cast<std::string>(boardId);
	}
	return card_gen;
}


static std::map<vca::crc_pair, std::string> get_known_eeproms_names()
{
	std::map<vca::crc_pair, std::string> map;

	for (unsigned i = 0; i < vca::num_known_eeproms; i++)
		map[vca::known_eeproms[i].crcs] = vca::known_eeproms[i].version;

	return map;
}

std::string get_eeprom_version(unsigned card_id)
{
	/* get EEPROM CRC for all nodes */
	std::vector<unsigned> eeproms_crc;
	for (unsigned cpu = 0; cpu < MAX_CPU; ++cpu) {
		close_on_exit cpu_fd(open_cpu_fd(card_id, cpu));
		if(!cpu_fd) {
			LOG_ERROR("Could not open card_id, cpu_id: %d %d  "
				"file descriptor!", card_id, cpu);
			return "unknown";
		}
		__u32 crc = 0;
		if(!csm_ioctl(cpu_fd, VCA_READ_EEPROM_CRC, &crc)) {
			LOG_ERROR("Failed to read EEPROM CRC card_id, cpu_id: %d %d", card_id, cpu);
			return "unknown";
		}
		eeproms_crc.push_back(crc);
	}

	/* construct EEPROM CRC pair for given card */
	vca::crc_pair loaded_eeproms_crc = std::make_pair(
				eeproms_crc[0],  // CRC of first plx
				eeproms_crc[2]); // CRC of second plx, as
						 // eeproms_crc[0]==eeproms_crc[1]

	/* get crc_pair to version mapping for known eeproms */
	std::map<vca::crc_pair, std::string> eeprom_version_map =
		get_known_eeproms_names();


	/* if eeprom pair is known, return version string - otherwise "unknown" */
	std::stringstream version;
	if (eeprom_version_map.find(loaded_eeproms_crc) ==
					eeprom_version_map.end())
		version <<  "unknown";
	else
		version << eeprom_version_map[loaded_eeproms_crc];

	/* append CRC info */
	version << " (" + crc_pair_str(loaded_eeproms_crc) + ")";

	return version.str();
}

std::vector<std::string> get_subcmd_list(const char *_cmd)
{
	std::vector<std::string> subcommand_list;
	std::string cmd = char_to_string((char *)_cmd);

	if (cmd == NETWORK_CMD) {
		subcommand_list.push_back(NETWORK_SUBCMD_ALL);
		subcommand_list.push_back(NETWORK_SUBCMD_IP);
		subcommand_list.push_back(NETWORK_SUBCMD_IP6);
		subcommand_list.push_back(NETWORK_SUBCMD_MAC);
		subcommand_list.push_back(NETWORK_SUBCMD_VM_MAC);
		subcommand_list.push_back(NETWORK_SUBCMD_STATS);
		subcommand_list.push_back(NETWORK_SUBCMD_RENEW);
	}
	else if (cmd == BLOCKIO_CMD) {
		subcommand_list.push_back(BLOCKIO_SUBCMD_LIST);
		subcommand_list.push_back(BLOCKIO_SUBCMD_OPEN);
		subcommand_list.push_back(BLOCKIO_SUBCMD_CLOSE);
	}
	else if (cmd == GOLD_CMD) {
		subcommand_list.push_back("on");
		subcommand_list.push_back("off");
	}
	else if (cmd == INFO_CMD) {
		subcommand_list.push_back(INFO_SUBCMD_HW);
		subcommand_list.push_back(INFO_SUBCMD_SYSTEM);
		subcommand_list.push_back(INFO_SUBCMD_NODE_OS);
		subcommand_list.push_back(INFO_SUBCMD_NODE_BIOS);
	}
	else {
		LOG_ERROR("Unknown command: %s\n", cmd.c_str());
	}

	return subcommand_list;
}

std::string print_subcmd_list_content(std::vector<std::string> subcommand_list) {
	std::string output;
	for (size_t i = 0; i < subcommand_list.size(); i++) {
		output += subcommand_list.at(i);
		if (subcommand_list.size() - 1 == i)
			output += ".";
		else
			output += ", ";
	}
	return output;
}

static inline subcmds *get_subcmds(const char *_cmd, const std::vector<std::string> &subcommand_list = std::vector<std::string>())
{
	subcmds *subcmd = new subcmds();
	std::string cmd = char_to_string((char *)_cmd);

	if (cmd == NETWORK_CMD) {
		subcmd->add_subcmd(NETWORK_SUBCMD_ALL,		"print all network information");
		subcmd->add_subcmd(NETWORK_SUBCMD_IP,		"print IP address");
		subcmd->add_subcmd(NETWORK_SUBCMD_IP6,		"print IPv6 address");
		subcmd->add_subcmd(NETWORK_SUBCMD_MAC,		"print MAC address");
		subcmd->add_subcmd(NETWORK_SUBCMD_VM_MAC,	"print VM MAC address");
		subcmd->add_subcmd(NETWORK_SUBCMD_STATS,	"print interface statistics");
		subcmd->add_subcmd(NETWORK_SUBCMD_RENEW,	"renew IP address assigned by DHCP");
	}
	else if (cmd == BLOCKIO_CMD) {
		subcmd->add_subcmd(BLOCKIO_SUBCMD_LIST,		"list all blockio devices");
		subcmd->add_subcmd(BLOCKIO_SUBCMD_OPEN,		"open/create blockio device");
		subcmd->add_subcmd(BLOCKIO_SUBCMD_CLOSE,	"close blockio device");
	}
	else if (cmd == GOLD_CMD) {
		subcmd->add_subcmd("on",			"boot golden bios");
		subcmd->add_subcmd("off",			"boot standard bios");
	}
	else if (cmd == INFO_CMD) {
		subcmd->add_subcmd(INFO_SUBCMD_HW,		"print device information");
		subcmd->add_subcmd(INFO_SUBCMD_SYSTEM,		"print system information");
		subcmd->add_subcmd(INFO_SUBCMD_NODE_OS,		"print OS booted on cpu");
		subcmd->add_subcmd( INFO_SUBCMD_NODE_BIOS,	"print BIOS version on cpu");
	}
	else {
		LOG_ERROR("Unknown command: %s\n", cmd.c_str());
	}

	return subcmd;
}

#if VCACTL_PARSING_FUNCTIONS

parsing_output optional_card_id(const char * arg, args_holder & holder)
{
	if (!arg || !is_unsigned_number(arg))
		return NOT_PARSED;

	int card_id = atoi(arg);
	if (strcmp(holder.get_cmd_name(), "config")) {
		if (!card_exists(card_id)) {
			LOG_ERROR("Card %u not found!\n", card_id);
			return PARSING_FAILED;
		}
	}
	holder.add_arg(CARD_ID_ARG, card_id);
	return PARSED;
}

parsing_output requires_card_id(const char *arg, args_holder & holder)
{
	parsing_output ret = optional_card_id(arg, holder);
	if (ret == NOT_PARSED) {
		LOG_ERROR("This command needs card_id to work!\n");
		return PARSING_FAILED;
	}
	return ret;
}

parsing_output optional_cpu_id(const char *arg, args_holder & holder)
{
	if (!arg || !is_unsigned_number(arg))
		return NOT_PARSED;

	const data_field *card_id = holder(CARD_ID_ARG);
	if (!card_id)
		return NOT_PARSED;

	int cpu_id = atoi(arg);
	if (cpu_id >= MAX_CPU) {
		LOG_ERROR("No cpu %d on card %d found!\n", cpu_id, card_id->get_number());
		return PARSING_FAILED;
	}
	holder.add_arg(CPU_ID_ARG, cpu_id);
	return PARSED;
}

parsing_output requires_cpu_id(const char *arg, args_holder & holder)
{
	parsing_output ret = optional_cpu_id(arg, holder);
	if (ret == NOT_PARSED) {
		LOG_ERROR("This command needs cpu_id to work!\n");
		return PARSING_FAILED;
	}
	return ret;
}

parsing_output optional_blockio_id(const char *arg, args_holder &holder)
{
	int blockio_id;
	if (extract_block_dev_id(arg, blockio_id)) {
		holder.add_arg(BLOCKIO_ID_ARG, blockio_id);
		return PARSED;
	}

	const std::string subcmd = holder.get_arg(SUBCMD);
	if ((subcmd == BLOCKIO_SUBCMD_LIST && arg) || subcmd == BLOCKIO_SUBCMD_OPEN || subcmd == BLOCKIO_SUBCMD_CLOSE) {
		LOG_ERROR("Wrong blockio id! Correct value: %sN, where N >= 0 and N < %d.\n",
			BLK_CONFIG_NAME, MAX_BLK_DEVS);
		return PARSING_FAILED;
	}

	return NOT_PARSED;
}

parsing_output optional_blockio_type(const char *arg, args_holder &holder)
{
	if (holder.get_arg(SUBCMD) == BLOCKIO_SUBCMD_CLOSE && arg) {
		LOG_ERROR("Command \"blockio close\" accept only one parameter (blockio id)!\n");
		return PARSING_FAILED;
	}
	else if (holder.get_arg(SUBCMD) == BLOCKIO_SUBCMD_LIST && arg) {
		LOG_ERROR("Command \"blockio list\" accept only one parameter (blockio id)!\n");
		return PARSING_FAILED;
	}
	else if (holder.get_arg(SUBCMD) == BLOCKIO_SUBCMD_OPEN) {
		if (!arg)
			return PARSED; // parsed is returned to omit next parsing_output function

		if ((!strcmp(arg, BLOCKIO_MODE_RO) || !strcmp(arg, BLOCKIO_MODE_RW) || !strcmp(arg, BLOCKIO_MODE_RAMDISK))) {
			holder.add_arg(BLOCKIO_MODE_ARG, arg);
			return PARSED;
		}
		else {
			LOG_ERROR("Wrong blockio type! Allowed values: RO, RW, ramdisk.\n");
			return PARSING_FAILED;
		}
	}

	return NOT_PARSED;
}

parsing_output optional_blockio_type_param(const char *arg, args_holder &holder)
{
	if (holder.get_arg(SUBCMD) != BLOCKIO_SUBCMD_OPEN)
		return NOT_PARSED;

	if (holder.get_arg(BLOCKIO_MODE_ARG).empty())
		return NOT_PARSED;

	if (!arg) {
		LOG_ERROR("Missing blockio type param!\n");
		return PARSING_FAILED;
	}

	if (holder.get_arg(BLOCKIO_MODE_ARG) == BLOCKIO_MODE_RO ||
	    holder.get_arg(BLOCKIO_MODE_ARG) == BLOCKIO_MODE_RW) {
		if (!file_exists(arg)) {
			LOG_ERROR("Wrong blockio type param! File (%s) doesn't exist.\n", arg);
			return PARSING_FAILED;
		}
	}
	else if (holder.get_arg(BLOCKIO_MODE_ARG) == BLOCKIO_MODE_RAMDISK) {
		if (!is_unsigned_number(arg) || atoi(arg) == 0) {
			LOG_ERROR("Wrong blockio type param!\n"
				  "Invalid block device size (%s). Have to be more than 0.\n", arg);
			return PARSING_FAILED;
		}
	}
	else {
		LOG_ERROR("Wrong blockio type! Allowed values: RO, RW, ramdisk.\n");
		return PARSING_FAILED;
	}

	holder.add_arg(BLOCKIO_MODE_PARAM_ARG, arg);
	return PARSED;
}

parsing_output optional_file(const char *arg, args_holder & holder)
{
	if (!arg)
		return NOT_PARSED;

	if (!strcmp(arg, FORCE_LAST_OS_IMAGE))
		return NOT_PARSED;

	if (!strcmp(BLOCKIO_BOOT_DEV_NAME, arg)) {
		holder.add_arg(FILE_PATH_ARG, arg);
		return PARSED;
	}

	if (file_exists(arg)) {
		holder.add_arg(FILE_PATH_ARG, arg);
		return PARSED;
	}

	LOG_ERROR("file %s does not exist!\n", arg);
	return PARSING_FAILED;
}

parsing_output requires_file(const char *arg, args_holder & holder)
{
	parsing_output ret = optional_file(arg, holder);
	if (ret == NOT_PARSED) {
		LOG_ERROR("missing filepath parameter!\n");
		return PARSING_FAILED;
	}
	return ret;
}

parsing_output requires_mac_addr(const char *arg, args_holder & holder)
{
	const char * error = "please input mac address in canonical form! Example: 00-06-af-7d-66-b8\n";
	if (!arg) {
		LOG_ERROR("Missing MAC address!\n");
		return PARSING_FAILED;
	}

	if (strlen(arg) != 17) {
		LOG_ERROR("%s\n", error);
		return PARSING_FAILED;
	}

	static const char separators[] = { '-', ':' };
	static size_t separators_size = sizeof(separators) / sizeof(separators[0]);
	const char * mac = arg;
	for (int i = 0; i < 17; i += 3)
		if (!is_hex_digit(mac[i]) || !is_hex_digit(mac[i + 1])) {
			LOG_ERROR("%s\n", error);
			return PARSING_FAILED;
		}

	for (int i = 0; i < 14; i += 3) {
		bool found_separator = false;
		for (size_t j = 0; j < separators_size; j++)
			if (mac[i + 2] == separators[j]) {
				found_separator = true;
				break;
			}
		if (!found_separator) {
			LOG_ERROR("%s\n", error);
			return PARSING_FAILED;
		}

	}
	holder.add_arg(MAC_ADDR_ARG, arg);
	return PARSED;
}

parsing_output optional_config_param(const char *arg, args_holder & holder)
{
	int blockio_id;
	if ((arg && config.contains_field(arg)) || extract_block_dev_id(arg, blockio_id)) {
		holder.add_arg(CONFIG_PARAM_NAME, arg);
		return PARSED;
	}
	return NOT_PARSED;
}

parsing_output requires_config_param(const char *arg, args_holder & holder)
{
	if (!arg) {
		LOG_ERROR("Missing parameter name!\n");
		return PARSING_FAILED;
	}
	if (config.contains_field(arg)) {
		/* to be removed in future */
		char * tmp = const_cast<char *>(arg);
		std::replace(tmp, tmp + strlen(tmp), '_', '-');

		holder.add_arg(CONFIG_PARAM_NAME, arg);
		return PARSED;
	}

	LOG_ERROR("Invalid parameter name: %s\n", arg);
	return PARSING_FAILED;
}

parsing_output requires_config_value(const char *arg, args_holder & holder)
{
	if (!arg) {
		LOG_ERROR("Missing parameter value!\n");
		return PARSING_FAILED;
	}

	holder.add_arg(CONFIG_PARAM_VALUE, arg);
	return PARSED;
}

parsing_output optional_value_only_param(const char *arg, args_holder & holder)
{
	if (!arg)
		return NOT_PARSED;

	const data_field * param_name = holder(CONFIG_PARAM_NAME);
	if (!param_name)
		return NOT_PARSED;

	if (config.is_field_cpu(param_name->get_cstring()) && !strcmp(arg, "--value")) {
		holder.add_arg(STRIP_PARAM, arg);
		return PARSED;
	}
	return NOT_PARSED;
}

parsing_output optional_value_force_get_last_os(const char *arg, args_holder & holder)
{
	if (!arg)
		return NOT_PARSED;

	if (!strcmp(arg, FORCE_LAST_OS_IMAGE)) {
		holder.add_arg(FORCE_GET_LAST_OS_IMAGE, arg);
		return PARSED;
	}
	return NOT_PARSED;
}

parsing_output requires_smb_id(const char *arg, args_holder & holder)
{
	if (!arg) {
		LOG_ERROR("missing SMB_ID parameter!\n");
		return PARSING_FAILED;
	}

	if (!is_unsigned_number(arg)) {
		LOG_ERROR("last parameter must be a number between 0-7!\n");
		return PARSING_FAILED;
	}
	int smb_id = atoi(arg);
	if (smb_id > 7 || smb_id < 0) {
		LOG_ERROR("last parameter must be a number between 0-7!\n");
		return PARSING_FAILED;
	}

	holder.add_arg(SMB_ID_ARG, smb_id);
	return PARSED;
}

parsing_output requires_trigger(const char *arg, args_holder & holder)
{
	if (!arg || !is_unsigned_number(arg)) {
		LOG_ERROR("Need trigger value! Only 0 or 1 may be used.\n");
		return PARSING_FAILED;
	}
	int trigger = atoi(arg);
	if (trigger < 0 || trigger > 1) {
		LOG_ERROR("Value %d not allowed! Only 0 or 1 may be used.\n", trigger);
		return PARSING_FAILED;
	}

	holder.add_arg(TRIGGER_ARG, trigger);
	return PARSED;
}

parsing_output optional_subcommand(const char *arg, args_holder &holder)
{
	std::vector<std::string> subcommand_list = get_subcmd_list(holder.get_cmd_name());

	if (subcommand_list.empty())
		return PARSING_FAILED;

	std::vector<std::string>::iterator findIter = std::find(subcommand_list.begin(), subcommand_list.end(), char_to_string((char *)arg));
	if (findIter == subcommand_list.end()) {
		return NOT_PARSED;
	}

	holder.add_arg(SUBCMD, arg);
	return PARSED;
}

parsing_output requires_subcommand(const char *arg, args_holder & holder)
{
	std::vector<std::string> subcommand_list = get_subcmd_list(holder.get_cmd_name());
	if (subcommand_list.empty())
		return PARSING_FAILED;

	if (!arg) {
		LOG_ERROR("Need subcommands: %s\n", print_subcmd_list_content(subcommand_list).c_str());
		return PARSING_FAILED;
	}

	std::vector<std::string>::iterator findIter = std::find(subcommand_list.begin(), subcommand_list.end(), char_to_string((char *)arg));
	if (findIter == subcommand_list.end()) {
		LOG_ERROR("Need subcommands: %s\n", print_subcmd_list_content(subcommand_list).c_str());
		return PARSING_FAILED;
	}

	holder.add_arg(SUBCMD, arg);
	return PARSED;
}

parsing_output optional_ip(const char *arg, args_holder & holder)
{
	if (!arg)
		return NOT_PARSED;

	if (holder(CPU_ID_ARG)) {
		if (!is_ip_address(arg)) {
			LOG_ERROR("Ip address incorrect!\n");
			return PARSING_FAILED;
		}
		holder.add_arg(IP_ADDR_ARG, arg);
	}
	return PARSED;
}

parsing_output optional_bios_cfg_name(const char *arg, args_holder & holder)
{
	if (!arg) {
		return NOT_PARSED;
	}

	if (!is_bios_cfg_option(arg)) {
		LOG_ERROR("BIOS parameter (%s) not allowed! See 'vcactl help' for reference.\n", arg);
		return PARSING_FAILED;
	}

	holder.add_arg(BIOS_CFG_NAME_ARG, arg);
	return PARSED;
}

parsing_output requires_bios_cfg_name(const char *arg, args_holder & holder)
{
	parsing_output ret = optional_bios_cfg_name(arg, holder);
	if (ret == NOT_PARSED) {
		LOG_ERROR("Missing BIOS configuration name! See 'vcactl help' for reference.\n");
		return PARSING_FAILED;
	}
	return ret;
}

parsing_output requires_bios_cfg_value(const char *arg, args_holder & holder)
{
	if (!arg) {
		LOG_ERROR("Missing BIOS configuration value! See 'vcactl help' for reference.\n");
		return PARSING_FAILED;
	}

	std::string arg_s = std::string(arg);
	std::string value = holder.get_arg(BIOS_CFG_NAME_ARG);

	if (value == BIOS_CFG_SGX) {
		if (arg_s != "enable" && arg_s != "disable") {
			LOG_ERROR("Wrong value (%s) for '%s'. See 'vcactl help' for reference.\n",
				  arg, BIOS_CFG_SGX);
			return PARSING_FAILED;
		}
	}
	else if (value == BIOS_CFG_GPU_APERTURE) {
		if (!aperture_from_string(arg, NULL)) {
			LOG_ERROR("Wrong value (%s) for '%s'. See 'vcactl help' for reference.\n",
				  arg, BIOS_CFG_GPU_APERTURE);
			return PARSING_FAILED;
		}
	}
	else if (value == BIOS_CFG_TDP) {
		if (!tdp_from_string(arg, NULL)) {
			LOG_ERROR("Wrong value (%s) for '%s'. See 'vcactl help' for reference.\n",
				arg, BIOS_CFG_TDP);
			return PARSING_FAILED;
		}
	}

	holder.add_arg(BIOS_CFG_VALUE_ARG, arg);
	return PARSED;
}

#endif // VCACTL_PARSING_FUNCTIONS

static inline const cmd_desc **commands_desc_init()
{
	cmd_desc **cmds = new cmd_desc*[commands::SIZE];
	assert (cmds);
	cmds[commands::status] = new cmd_desc("status", "shows status of the cpu",
		new sequential_caller(status), optional_card_id, optional_cpu_id);
	cmds[commands::reset] = new cmd_desc("reset", "resets the cpu",
		new threaded_caller(_reset), optional_card_id, optional_cpu_id);
	cmds[commands::wait] = new cmd_desc("wait", "waits for cpu to boot OS",
		new threaded_caller(wait), optional_card_id, optional_cpu_id);
	cmds[commands::wait_bios] = new cmd_desc("wait-BIOS", "waits for bios to be ready on desired cpu",
		new threaded_caller(wait_bios), optional_card_id, optional_cpu_id);
	cmds[commands::boot] = new cmd_desc("boot", "boot OS for cpu using LBP",
		new sequential_caller(boot), optional_card_id, optional_cpu_id, optional_file, optional_value_force_get_last_os);
	cmds[commands::reboot] = new cmd_desc("reboot", "reboot last used OS",
		new threaded_caller(reboot), optional_card_id, optional_cpu_id, optional_file);
	cmds[commands::update_BIOS] = new cmd_desc("update-BIOS", "update bios for the cpu",
		new threaded_caller(update_bios), optional_card_id, optional_cpu_id, requires_file);
	cmds[commands::recover_BIOS] = new cmd_desc("recover-BIOS", "Recover BIOS",
		new threaded_caller(update_bios), optional_card_id, optional_cpu_id, requires_file);
	cmds[commands::update_MAC_ADDR] = new cmd_desc("update-MAC", "updates mac address of the cpu with desired value (only allowed as root)",
		new sequential_caller(update_mac_addr), requires_card_id, requires_cpu_id, requires_mac_addr);
	cmds[commands::update_EEPROM] = new cmd_desc("update-EEPROM", "update EEPROM for the card",
		new sequential_caller(update_eeprom), optional_card_id, requires_file);
	cmds[commands::clear_SMB_event_log] = new cmd_desc("clear-SMB-event-log", "clear SMB event log for the cpu",
		new threaded_caller(clear_smb_event_log), optional_card_id, optional_cpu_id);
	cmds[commands::script] = new cmd_desc("script", "set script parameter in configuration",
		new sequential_caller(script), optional_card_id, optional_cpu_id, optional_file);
	cmds[commands::config_show] = new cmd_desc("config-show", "shows config for cpu",
		new sequential_caller(config_show), optional_card_id, optional_cpu_id, optional_config_param, optional_value_only_param);
	cmds[commands::config] = new cmd_desc("config", "set parameter in configuration",
		new sequential_caller(config_change), optional_card_id, optional_cpu_id, optional_blockio_id, requires_config_param, requires_config_value);
	cmds[commands::config_use] = new cmd_desc("config-use", "restarts ping daemons with new configuration",
		new sequential_caller(config_use));
	cmds[commands::config_default] = new cmd_desc("config-default", "restore vca configuration to default values",
		new sequential_caller(config_default));
	cmds[commands::temp] = new cmd_desc("temp", "read temp from cpu node",
		new sequential_caller(read_temp), optional_card_id, optional_cpu_id);
	cmds[commands::ICMP_watchdog] = new cmd_desc("ICMP-watchdog", "start/stop ICMP watchdog",
		new threaded_caller(ICMP_watchdog), requires_trigger, optional_card_id, optional_cpu_id, optional_ip);
	cmds[commands::network] = new cmd_desc("network", "get network information",
		new sequential_caller(get_network_info), get_subcmds("network"), requires_subcommand, optional_card_id, optional_cpu_id);
	cmds[commands::info] = new cmd_desc("info", "get VCA information",
		new sequential_caller(print_info_cmd_data), get_subcmds("info"), requires_subcommand, optional_card_id, optional_cpu_id);
	cmds[commands::info_hw] = new cmd_desc("info-hw", "get hardware inforamtion",
		new sequential_caller(print_hw_info), optional_card_id);
	cmds[commands::info_system] = new cmd_desc("info-system", "get general system inforamtion",
		new sequential_caller(print_system_info));
	cmds[commands::pwrbtn_short] = new cmd_desc("pwrbtn-short", "power button toggle",
		new threaded_caller(toggle_power_button), optional_card_id, optional_cpu_id);
	cmds[commands::pwrbtn_long] = new cmd_desc("pwrbtn-long", "power button override 5 sec",
		new threaded_caller(hold_power_button), optional_card_id, optional_cpu_id);
	cmds[commands::help] = new cmd_desc("help", "print usage instruction",
		new sequential_caller(help));
	cmds[commands::blockio] = new cmd_desc("blockio", "controls blockio devices",
		new sequential_caller(blockio_ctl), get_subcmds("blockio"), requires_subcommand, optional_card_id, optional_cpu_id,
						  optional_blockio_id, optional_blockio_type, optional_blockio_type_param);
	cmds[commands::os_shutdown] = new cmd_desc("os-shutdown", "OS shutdown",
		new sequential_caller(os_shutdown), optional_card_id, optional_cpu_id);
	cmds[commands::get_bios_cfg] = new cmd_desc("get-BIOS-cfg", "read BIOS configuration",
		new sequential_caller(get_bios_cfg), optional_card_id, optional_cpu_id, optional_bios_cfg_name);
	cmds[commands::set_bios_cfg] = new cmd_desc("set-BIOS-cfg", "change BIOS configuration",
		new threaded_caller(set_bios_cfg), optional_card_id, optional_cpu_id, requires_bios_cfg_name, requires_bios_cfg_value);

	return (const cmd_desc **)cmds;
}

static const cmd_desc ** const commands_desc = commands_desc_init();

static const cmd_desc debug_cmds_desc[] = {
	cmd_desc("boot-USB", "boot OS for cpu using USB",
		new threaded_caller(boot_USB), optional_card_id, optional_cpu_id),
	cmd_desc("set-SMB-id", "set SMB id for card",
		new sequential_caller(set_SMB_id), requires_card_id, requires_smb_id),
	cmd_desc(GOLD_CMD, "golden BIOS mode control",
		new threaded_caller(gold_control), requires_subcommand, optional_card_id, optional_cpu_id),
};

const cmd_desc * get_command(const char * func_name)
{
	if(config.get_global_field(global_fields::debug_enabled).get_number() == 1)
		for(int i = 0; i < debug_commands::SIZE; i++)
			if(strcmp(func_name, debug_cmds_desc[i].name) == 0)
				return &debug_cmds_desc[i];

	for(int i = 0; i < commands::SIZE; i++)
		if(strcmp(func_name, commands_desc[i]->name) == 0)
			return commands_desc[i];
	return NULL;
}

static void print_help()
{
	printf("Usage: \n"
	"    vcactl [execution_mode] command [subcommand] [card_id [cpu_id]] [command_params] [execution_mode]\n"

	"\nAvailable execution modes:\n"
	"    -v :  increase log verbosity\n"
	"    -vv : increase log verbosity even more\n"
	"    --skip-modprobe-check : turn off checking whether vca device is ready\n"
	"    --force : force command execution (WARNING: you do it at your own risk!)\n"

	"\nAvailable commands are:\n"
	"    status : shows status of the cpu\n"
	"    reset : resets the cpu\n"
	"    wait : waits for cpu to boot OS\n"
	"    wait-BIOS : waits for bios to be ready on desired cpu\n"
	"    boot vcablk<N> : boot OS for cpu using VCA block device vcablk<N>\n"
	"        Currently only booting from vcablk0 is supported.\n"
	"    boot [img_path | --force-last-os-image] : boot OS for cpu using LBP\n"
	"        If no img_path is provided, configuration field 'os-image' is used.\n"
	"        Is possible to use 'boot' command with --force-last-os-image parameter to boot directly from last-os-image field.\n"
	"    reboot [img_path] : reboot last used OS\n"
	"        If no img_path is provided, configuration field 'last-os-image' or 'os-image' is used.\n"
	"    update-BIOS <bios_img_path>: update bios for the cpu.\n"
	"    recover-BIOS <bios_img_path>: updates BIOS via gold bios image. Use in need of recovery. Works only with second generation VCA. May take much longer than standard BIOS update!\n"
	"    get-BIOS-cfg [bios_cfg_name]: read BIOS configuration\n"
	"        Avaiable configurations to read:\n"
	"            - sgx\n"
	"            - gpu-aperture\n"
	"            - tdp\n"
	"    set-BIOS-cfg <bios_cfg_name> <bios_cfg_value>: change BIOS configuration\n"
	"        Available configurations with their allowed values:\n"
	"            - for 'sgx': enable, disable\n"
	"            - for 'gpu-aperture': 128, 256, 512, 1024, 2048, 4096\n"
		"            - for 'tdp': 0 to 11, where 0 mean base value, 1 mean base + 1, etc. \n"
	"    update-MAC <card_id> <cpu_id> <mac_addr>: updates mac address of the cpu with desired value (only allowed as root)\n"
	"        This command requires card id, cpu id and mac address.\n"
	"    update-EEPROM <eeprom_file> : update EEPROM for the card (only allowed as root)\n"
	"        This command does not accept cpu_id parameter - applies to all cpus on card\n"
	"    clear-SMB-event-log : clear SMB event log for the cpu\n"
	"    script : set script parameter in configuration\n"
	"    config-show : shows config for cpu\n"
	"    config <config_param> <config_value>: set parameter in xml configuration file\n"
	"        This command requires xml config parameter and its value.\n"
	"    config-use : restarts ping daemons with new configuration\n"
	"        This command does not take any parameters.\n"
	"    config-default : restore vca configuration to default values\n"
	"        This command does not take any parameters.\n"
	"    temp : read temp from cpu node\n"
	"    ICMP-watchdog <subcmd> : start/stop ICMP watchdog. Subcommands:\n"
	"        1 : enable\n"
	"        0 : disable\n"
	"    network <subcmd> : get network information. Subcommands:\n"
	"        all : print all network information\n"
	"        dhcp-renew : renew IP address assigned by DHCP\n"
	"        ip : print IP address\n"
	"        ip6 : print IPv6 address\n"
	"        mac : print MAC address\n"
	"        stats : print interface statistics\n"
	"        vm-mac : print VM MAC address\n"
	"    blockio <subcmd> : controls block devices. Subcommands:\n"
	"        list [vcablk<N>] : list block devices\n"
	"        open vcablk<N> [[RO|RW <file_path>]|[ramdisk <size_mb>]] : open block device\n"
	"        close vcablk<N> : close block device\n"
	"    info <subcmd> : print VCA information. Subcommands:\n"
	"        hw : print device information\n"
	"        system : print system information\n"
	"        node-os : print OS booted on cpu\n"
	"        BIOS : print BIOS verison information\n"
	"    pwrbtn-short : power button toggle\n"
	"    pwrbtn-long : power button override 5 sec\n"
	"    os-shutdown : shutdown OS\n"
	"    help : print usage instruction\n"
	"\n"

	"\nExample usage:\n"
	"    vcactl reset\n"
	"    vcactl reset 0 2\n"
	"    vcactl config auto-boot 0\n"
	"    vcactl boot 1 2 /home/centOS7.img\n"
	"    vcactl boot 0 1 vcablk0\n"
	"    vcactl boot 1 1 --force-last-os-image\n"
	"    vcactl reboot\n"
	"    vcactl ICMP-watchdog 1 0 2 127.0.0.1\n"
	"    vcactl network ip 0 0\n"
	"    vcactl update-EEPROM /home/Fab234_8713_8733_v23_crc_B7D58879.bin\n"
	"    vcactl update-BIOS 0 0 /home/MonteVistaBios.img\n"
	"    vcactl update-MAC 0 1 00:01:02:03:04:05\n"
	"    vcactl blockio list\n"
	"    vcactl blockio open 0 1 vcablk3 ramdisk 20\n"
	"    vcactl blockio open 0 1 vcablk3 RO ~/disk.img\n"
	"    vcactl blockio list 0 1 vcablk3\n"
	"    vcactl blockio close 0 1  vcablk3\n"
	"    vcactl blockio open vcablk0\n"
	"    vcactl status\n");

}

void single_call(function_caller &f, int card_id, int cpu_id, args_holder &holder)
{
	f.call(caller_data(card_id, cpu_id, holder));
}

void single_card_call(function_caller &f, int card_id, args_holder &holder)
{
	f.call(caller_data(card_id, holder));
}

void all_cpus(function_caller &f, int card_id, args_holder &holder)
{
	for (int j = 0; j < MAX_CPU; j++)
		f.call(caller_data(card_id, j, holder));
}

void all_cards(function_caller &f, args_holder &holder)
{
	int cards_num = get_cards_num();
	for (int i = 0; i < cards_num; i++)
		single_card_call(f, i, holder);
}

void all_cards_all_cpus(function_caller &f, args_holder &holder)
{
	int cards_num = get_cards_num();
	for (int i = 0; i < cards_num; i++)
		all_cpus(f, i, holder);
}

bool parse_and_execute(const cmd_desc *cmd, args_holder &holder, char *argv[])
{
	std::string cmd_name = std::string(cmd->name);

	if (!cmd->parse_args(argv, holder))
		return false;

	int available_nodes = count_available_nodes();
	if (available_nodes == FAIL) {
		LOG_ERROR("Count available devices failed!\n");
		return false;
	}
	else if (available_nodes == 0) {
		LOG_DEBUG("No nodes are available!\n");
		return false;
	}

	if (!holder.is_modprobe_check_skipped() && !devices_ready(holder, available_nodes)) {
		return false;
	}

	const data_field *card_id = holder(CARD_ID_ARG);
	const data_field *cpu_id = holder(CPU_ID_ARG);

	if (cmd_name == UPDATE_EEPROM_CMD || cmd_name == VCA_HW_INFO_CMD ||
	    (cmd_name == INFO_CMD && (holder.get_arg(SUBCMD) == INFO_SUBCMD_HW))) {
		if (!card_id) {
			all_cards(*cmd->caller, holder);
			return true;
		}
		else {
			single_card_call(*cmd->caller, card_id->get_number(), holder);
			return true;
		}
	}
	else if (!card_id || cmd_name == VCA_SYS_INFO_CMD ||
		(cmd_name == INFO_CMD && (holder.get_arg(SUBCMD) == INFO_SUBCMD_SYSTEM))) {
		if (cmd_name == VCA_SYS_INFO_CMD ||
		    (cmd_name == INFO_CMD && (holder.get_arg(SUBCMD) == INFO_SUBCMD_SYSTEM))) {
			single_call(*cmd->caller, 0, 0, holder);
			return true;
		}
		all_cards_all_cpus(*cmd->caller, holder);
		return true;
	}
	else if (cmd_name == CONFIG_DEFAULT_CMD) {
		all_cards_all_cpus(*cmd->caller, holder);
		return true;
	}
	else if (card_id && !cpu_id) {
		all_cpus(*cmd->caller, card_id->get_number(), holder);
		return true;
	}
	else if (card_id && cpu_id) {
		single_call(*cmd->caller, card_id->get_number(), cpu_id->get_number(), holder);
		return true;
	}
	else {
		LOG_ERROR("Cannot parse vcactl command!");
		return false;
	}
}

int main(int argc, char *argv[])
{
	const cmd_desc *cmd;
	args_holder arg_holder;

	if (argc >= 2) {
		log_args(argc, argv);
	}

	if(!vca_named_mutex_create(VCACTL_CONFIG_NAMED_MTX_NAME))
		exit(EBUSY);

	/* start after program name */
	int parsed_args = 1;

	/* execution mode params at begining */
	while (parsed_args < argc) {
		if (arg_holder.process_execution_flag(argv[parsed_args])) {
			++parsed_args;
		} else {
			break;
		}
	}

	/* execution mode params at the end of cmdline */
	while (parsed_args < argc) {
		if (arg_holder.process_execution_flag(argv[argc - 1])) {
			--argc; // drop last one - it was execution mode flag
			argv[argc] = NULL; // and prevent using it as command param
		} else {
			break;
		}
	}

	if (parsed_args == argc) {
		LOG_WARN("Missing command!\n");
		print_help();
		exit(EINVAL);
	}

	/* for command config-default xml config file shouldn't be loaded, because it can be broken */
	if (strcmp(argv[parsed_args], CONFIG_DEFAULT_CMD) && !load_vca_config())
		exit(EIO);

	/* get command name */
	cmd = get_command(argv[parsed_args]);
	++parsed_args;

	thread_manager *thread_mgr = new thread_manager();
	threaded_caller::set_thread_manager(thread_mgr);

	if (!cmd) {
		command_err = EINVAL;
		LOG_ERROR("Wrong command!\n\n");
		print_help();
	}
	else {
		/* success inditaces that flow execution was succesful,
		it does not mean that all commands executed succesfully,
		but that all commands were executed */
		arg_holder.set_cmd_name(cmd->name);
		bool success = parse_and_execute(cmd, arg_holder, argv + parsed_args);
		if (!success)
			command_err = EINVAL;
	}

	/* ~thread_manager waits for all threads to finish before exitting */
	delete thread_mgr;

	return command_err;
}
