/*
* Intel VCA Software Stack (VCASS)
*
* Copyright(c) 2015-2017 Intel Corporation.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License, version 2, as
* published by the Free Software Foundation.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* General Public License for more details.
*
* The full GNU General Public License is included in this distribution in
* the file called "COPYING".
*
* Intel VCA User Space Tools.
*/

#ifndef _VCASSD_VIRTIO_BACKEND_H_
#define _VCASSD_VIRTIO_BACKEND_H_

#include <sys/ioctl.h>

#include <linux/virtio_net.h>
#include <linux/virtio_console.h>
#include <linux/virtio_blk.h>

#include "vcassd_common.h"

#include <vca_dev_common.h>
#include <vca_ioctl.h>

#define ARRAY_SIZE(x) (sizeof(x) / sizeof((x)[0]))

#define min_t(type, x, y) ({			\
	type __min1 = (x);                      \
	type __min2 = (y);                      \
	__min1 < __min2 ? __min1 : __min2; })

/* align addr on a size boundary - adjust address up/down if needed */
#define _ALIGN_DOWN(addr, size)  ((addr)&(~((size)-1)))
#define _ALIGN_UP(addr, size)    _ALIGN_DOWN(addr + size - 1, size)

/* align addr on a size boundary - adjust address up if needed */
#define _ALIGN(addr, size)     _ALIGN_UP(addr, size)

/* to align the pointer to the (next) page boundary */
#define PAGE_ALIGN(addr)        _ALIGN(addr, PAGE_SIZE)

#define ACCESS_ONCE(x) (*(volatile typeof(x) *)&(x))

#define GSO_ENABLED		0
#define MAX_GSO_SIZE		(64 * 1024)
#define ETH_H_LEN		14
#define MAX_NET_PKT_SIZE	(_ALIGN_UP(MAX_GSO_SIZE + ETH_H_LEN, 64))
#define VCA_DEVICE_PAGE_END	0x1000

#ifndef VIRTIO_NET_HDR_F_DATA_VALID
#define VIRTIO_NET_HDR_F_DATA_VALID	2	/* Csum is valid */
#endif

#pragma message "FIXME: move this definition to appropriate header"
/* Publish dma mapped addresses in the desc ring */
#define VIRTIO_RING_F_DMA_MAP          30

#ifndef VIRTIO_NET_F_OFFSET_RXBUF
#pragma message "Kernel header <linux/virtio_net.h> not updated"
#define VIRTIO_NET_F_OFFSET_RXBUF	24	/* Set offset in receive buffer */
#endif

static struct {
	struct vca_device_desc dd;
	struct vca_vqconfig vqconfig[2];
	__u32 host_features, guest_acknowledgements;
	struct virtio_console_config cons_config;
} virtcons_dev_page = {
	.dd = {
	.type = VIRTIO_ID_CONSOLE,
	.num_vq = ARRAY_SIZE(virtcons_dev_page.vqconfig),
	.feature_len = sizeof(virtcons_dev_page.host_features),
	.config_len = sizeof(virtcons_dev_page.cons_config),
},
.vqconfig[0] = {
	.num = htole16(VCA_VRING_ENTRIES),
},
.vqconfig[1] = {
	.num = htole16(VCA_VRING_ENTRIES),
},
};

static struct {
	struct vca_device_desc dd;
	struct vca_vqconfig vqconfig[2];
	__u32 host_features, guest_acknowledgements;
	struct virtio_net_config net_config;
} virtnet_dev_page = {
	.dd = {
	.type = VIRTIO_ID_NET,
	.num_vq = ARRAY_SIZE(virtnet_dev_page.vqconfig),
	.feature_len = sizeof(virtnet_dev_page.host_features),
	.config_len = sizeof(virtnet_dev_page.net_config),
},
.vqconfig[0] = {
	.num = htole16(VCA_VRING_ENTRIES),
},
.vqconfig[1] = {
	.num = htole16(VCA_VRING_ENTRIES),
},
#if GSO_ENABLED
.host_features = htole32(
	1 << VIRTIO_NET_F_CSUM |
	1 << VIRTIO_NET_F_GSO |
	1 << VIRTIO_NET_F_GUEST_TSO4 |
	1 << VIRTIO_NET_F_GUEST_TSO6 |
	1 << VIRTIO_NET_F_GUEST_ECN |
	1 << VIRTIO_RING_F_DMA_MAP |
	1 << VIRTIO_NET_F_GUEST_UFO),
#else
.host_features = htole32(
	1 << VIRTIO_RING_F_DMA_MAP |
	1 << VIRTIO_NET_F_OFFSET_RXBUF
	),
#endif
};

static struct {
	struct vca_device_desc dd;
	struct vca_vqconfig vqconfig[1];
	__u32 host_features, guest_acknowledgements;
	struct virtio_blk_config blk_config;
} virtblk_dev_page = {
	.dd = {
	.type = VIRTIO_ID_BLOCK,
	.num_vq = ARRAY_SIZE(virtblk_dev_page.vqconfig),
	.feature_len = sizeof(virtblk_dev_page.host_features),
	.config_len = sizeof(virtblk_dev_page.blk_config),
},
.vqconfig[0] = {
	.num = htole16(VCA_VRING_ENTRIES),
},
.host_features =
htole32(1 << VIRTIO_BLK_F_SEG_MAX),
.blk_config = {
	.seg_max = htole32(VCA_VRING_ENTRIES - 2),
	.capacity = htole64(0),
}
};

void add_virtio_device(struct vca_info *vca, enum virtio_types type);

#endif