/*
 * This Source is licenced under the NASA OPEN SOURCE AGREEMENT VERSION 1.3
 *
 * Copyright (C) 2012 United States Government as represented by the Administrator of the
 * National Aeronautics and Space Administration.
 * All Rights Reserved.
 *
 * Modifications by MAVinci GmbH, Germany (C) 2009-2016: Using parts of NASA code for building an entirely new class
 *
 */
package eu.mavinci.desktop.gui.doublepanel.mapmanager;

import com.intel.missioncontrol.concurrent.Dispatcher;
import com.intel.missioncontrol.concurrent.Strand;
import eu.mavinci.core.helper.IProperties;
import eu.mavinci.core.helper.ResourceManager;
import eu.mavinci.desktop.gui.doublepanel.mapmanager.wms.MapLayerElevationModel;
import eu.mavinci.desktop.gui.doublepanel.planemain.tree.maplayers.IMapLayerElevationModel;
import eu.mavinci.desktop.gui.doublepanel.planemain.tree.maplayers.IMapLayerWW;
import eu.mavinci.desktop.gui.doublepanel.planemain.tree.maplayers.MapLayerWW;
import eu.mavinci.desktop.gui.widgets.MProgressMonitor;
import eu.mavinci.desktop.gui.wwext.ElevationModelShiftWrapper;
import eu.mavinci.desktop.gui.wwext.ElevationModelShiftWrapper.ShiftType;
import eu.mavinci.desktop.gui.wwext.WWFactory;
import eu.mavinci.desktop.helper.MFile;
import eu.mavinci.desktop.main.core.Application;
import eu.mavinci.desktop.main.debug.Debug;
import eu.mavinci.desktop.main.debug.UserNotificationHubSwing;
import eu.mavinci.desktop.main.i18n.Language;
import eu.mavinci.geo.ISectorReferenced;
import gov.nasa.worldwind.Configuration;
import gov.nasa.worldwind.Factory;
import gov.nasa.worldwind.WorldWind;
import gov.nasa.worldwind.avlist.AVKey;
import gov.nasa.worldwind.avlist.AVList;
import gov.nasa.worldwind.avlist.AVListImpl;
import gov.nasa.worldwind.cache.FileStore;
import gov.nasa.worldwind.data.CachedDataRaster;
import gov.nasa.worldwind.data.DataRaster;
import gov.nasa.worldwind.data.DataStoreProducer;
import gov.nasa.worldwind.data.TiledElevationProducer;
import gov.nasa.worldwind.data.TiledImageProducer;
import gov.nasa.worldwind.data.TiledRasterProducer;
import gov.nasa.worldwind.data.WWDotNetLayerSetConverter;
import gov.nasa.worldwind.exception.WWRuntimeException;
import gov.nasa.worldwind.geom.Sector;
import gov.nasa.worldwind.globes.Earth;
import gov.nasa.worldwind.globes.ElevationModel;
import gov.nasa.worldwind.layers.Layer;
import gov.nasa.worldwind.terrain.BasicElevationModel;
import gov.nasa.worldwind.terrain.BasicElevationModelFactory;
import gov.nasa.worldwind.util.DataConfigurationFilter;
import gov.nasa.worldwind.util.DataConfigurationUtils;
import gov.nasa.worldwind.util.Logging;
import gov.nasa.worldwind.util.WWIO;
import gov.nasa.worldwind.util.WWUtil;
import gov.nasa.worldwind.util.WWXML;
import gov.nasa.worldwindx.applications.worldwindow.features.DataImportUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import javax.xml.xpath.XPath;
import java.awt.Component;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.util.HashSet;
import java.util.OptionalDouble;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;

public class MapLayerGeoTiff extends MapLayerWW
        implements IMapLayerWW,
        IMapLayerResourceRelated,
            IResourceFileReferenced,
            ISectorReferenced,
            IMapLayerElevationModel {

    // Layer layer = null;
    ElevationModel em = null;

    final String geoTiffSource;
    String layerName;
    ResourceManager geoTiffManager;

    public static final String KEY = "eu.mavinci.desktop.gui.doublepanel.mapmanager.MapLayerGeoTiff";
    public static final String KEY_NAME = KEY + ".layerName.";
    public static final String KEY_CREDIT_TXT = KEY + ".screenCredit.txt.";
    public static final String KEY_CREDIT_LINK = KEY + ".screenCredit.link.";
    public static final String KEY_CACHE_NAME = KEY + ".cacheName.";
    public static final String KEY_SHIFT_TYPE = KEY + ".shiftType.";
    public static final String KEY_SHIFT_LAST = KEY + ".shiftLast.";

    String cacheName;

    File file;

    private final Strand strand = new Strand();

    public MapLayerGeoTiff(ResourceManager geoTiffManager, String geoTiffSource) {
        super(null, "GeoTiffMapLayer." + geoTiffSource, true);
        zOrder = -100;
        this.geoTiffSource = geoTiffSource;
        this.geoTiffManager = geoTiffManager;
        String iconPath = "eu/mavinci/icons/16x16/MapLayerGeoTiff.png";
        icon = Application.getImageIconFromResource(iconPath);
        layerName = Application.getProperty(KEY_NAME + geoTiffSource);
        cacheName = Application.getProperty(KEY_CACHE_NAME + geoTiffSource);
        int pos = geoTiffSource.lastIndexOf(ResourceManager.FILENAME_SPLITTER);
        // System.out.println("new geoTiff with resource name:"+ geoTiffSource );
        if (pos > 0) {
            file = new File(geoTiffSource.substring(0, pos));
        } else {
            file = new File(geoTiffSource);
        }

        if (cacheName != null) {
            // load();
        } else if (file.exists()) {
            MFile mFile = new MFile(file);
            if (layerName == null) {
                layerName = "GeoTiff:" + mFile.toString();
                if (pos > 0) {
                    layerName += geoTiffSource.substring(pos);
                }
            }

            Application.setProperty(KEY_NAME + geoTiffSource, layerName);
            // load();
        } else {
            Debug.getLog().log(Level.WARNING, "Closing GeoTiff, since file " + file + " does not exist");
            close();
        }
    }

    protected void close() {
        // (new Exception()).printStackTrace();
        Dispatcher.postToUI(
                new Runnable() {
                    @Override
                    public void run() {
                        geoTiffManager.remove(geoTiffSource);
                        WWFactory.dropDataCache(cacheName);
                    }
                });
    }

    protected boolean startedLoading = false;

    @Override
    public synchronized void setVisible(boolean isVisible) {
        super.setVisible(isVisible);
        if (!startedLoading && isVisible) {
            load();
        }
    }

    @Override
    public synchronized void loadState(IProperties session, String keyPrefix) {
        super.loadState(session, keyPrefix);
        if (!startedLoading) {
            load();
        }
    }

    protected synchronized void load() {
        if (!isVisible()) {
            return;
        }

        startedLoading = true;

        // The load crashes when too many Tiffs are loaded. Hence we use only one thread.
        // the issue is NOT cpu bottle neck, its a too little MEMORY issue... until
        // we have a magic to know in advance how many are fitting to ram, do only one at a time
        // ThreadingHelper.getThreadingHelper().invokeConcurrentQueue(r) or Application.runTask(r)

        strand.post(
                new Runnable() {
                    @Override
                    public void run() {
                        try {

                            // File sourceFile =
                            // downloadAndUnzipToTempFile(WWIO.makeURL(IMAGE_URL),
                            // ".tif");
                            FileStore fileStore = WorldWind.getDataFileStore();

                            // Import the image into the FileStore by converting it to
                            // the World Wind Java cache format.
                            startImportFile(file.getName(), file, fileStore);
                            // Debug.getUserNotifier().handleInformation(Language.getFormat(KEY+".msg.readyLoaded",
                            // file),
                            // Language.getFormat(KEY+".msg.readyLoaded.title"));
                            Debug.getLog().fine(Language.getFormat(KEY + ".msg.readyLoaded", file));
                        } catch (Throwable e) {
                            Debug.getLog()
                                .log(Debug.MINOR_WARNING, "Could not open GeoTiff Resource" + geoTiffSource, e);
                            close();
                        }
                    }
                });
    }

    protected static final String BASE_CACHE_PATH = "UserGeoTiff/";

    @Override
    public String getTitle() {
        return layerName;
    }

    @Override
    public ElevationModel getWWElevationModel() {
        if (em == null) {
            return null;
        }

        if (em instanceof ElevationModelShiftWrapper) {
            ElevationModelShiftWrapper emw = (ElevationModelShiftWrapper)em;
            emw.setSector(getSector());
        }

        return em;
    }

    public String getResourceString() {
        return geoTiffSource;
    }

    @Override
    public File getResourceFile() {
        return file;
    }

    @Override
    public OptionalDouble getMaxElev() {
        return OptionalDouble.empty();
    }

    @Override
    public OptionalDouble getMinElev() {
        return OptionalDouble.empty();
    }

    Sector sector = null;

    @Override
    public Sector getSector() {
        if (sector == null && dataConfig != null) {
            sector = WWXML.getSector(dataConfig.getDocumentElement(), "Sector", null);
        }

        return sector;
    }

    Document dataConfig = null;

    protected void addImportedData(final Document dataConfig, final AVList params) {
        this.dataConfig = dataConfig;

        Dispatcher.postToUI(
                new Runnable() {

                    @Override
                    public void run() {
                        // System.out.println("importing data (from cache Or File) " + dataConfig + " params"+params);
                        Element domElement = dataConfig.getDocumentElement();
                        String type = DataConfigurationUtils.getDataConfigType(domElement);

                        if (type == null) {
                            return;
                        }

                        if (type.equalsIgnoreCase("Layer")) {
                            addLayerToWorldWindow(domElement, params);
                        } else if (type.equalsIgnoreCase("ElevationModel")) {
                            addElevationModelToWorldWindow(domElement, params);
                        }
                    }
                });
    }

    protected void addLayerToWorldWindow(Element domElement, AVList params) {
        // Layer layer = null;
        try {
            Factory factory = (Factory)WorldWind.createConfigurationComponent(AVKey.LAYER_FACTORY);
            wwLayer = (Layer)factory.createFromConfigSource(domElement, params);
        } catch (Exception e) {
            String message =
                Logging.getMessage(
                    "generic.CreationFromConfigurationFailed",
                    DataConfigurationUtils.getDataConfigDisplayName(domElement));
            Logging.logger().log(java.util.logging.Level.SEVERE, message, e);
        }

        if (wwLayer == null) {
            return;
        }

        wwLayer.setEnabled(true); // BasicLayerFactory creates layer which is intially disabled
        wwLayer.setValue(WWFactory.KEY_AV_Sector, getSector());

        try {
            // System.out.println("geoTiff layer set:" + getInternalName() + " opacity: " + lastOpacitySet);
            wwLayer.setOpacity(lastOpacitySet);
        } catch (Exception e) {
            Debug.getLog().log(Level.WARNING, "Colud not set geoTiff layer opacity", e);
        }

        mapLayerVisibilityChanged(this, isVisible);
    }

    protected void addElevationModelToWorldWindow(Element domElement, AVList params) {
        icon = Application.getImageIconFromResource(MapLayerElevationModel.iconPathElevation);
        try {
            Factory factory = new BasicElevationModelFactory();
            BasicElevationModel slave = (BasicElevationModel)factory.createFromConfigSource(domElement, params);
            // double shift;
            // try{
            // shift= EarthElevationModel.getElevation(getSector().getCentroid());
            // } catch (ElevationModelRequestException e){
            // shift = e.achievedAltitude;
            // }
            //// System.out.println("shift by " + shift);
            slave.setDetailHint(Double.parseDouble(Application.getProperty("_EXPERT.DetailHint", "0.4")));

            boolean useSHIFT_DEM_WRAPPER =
                Boolean.parseBoolean(Application.getProperty("_EXPERT.useSHIFT_DEM_WRAPPER", "true"));
            // useEGM_SRTM_SHIFT = false;
            Debug.getLog().info("useSHIFT_DEM_WRAPPER=" + useSHIFT_DEM_WRAPPER);
            if (useSHIFT_DEM_WRAPPER) {
                ElevationModelShiftWrapper emw = new ElevationModelShiftWrapper(slave, getSector());
                em = emw;
                try {
                    ShiftType shiftType =
                        ShiftType.valueOf(
                            Application.getProperty(KEY_SHIFT_TYPE + geoTiffSource, emw.getShiftType() + ""));
                    emw.setShiftType(shiftType);
                } catch (Exception e) {
                    emw.setShiftType(ShiftType.MANUAL);
                }

                if (Application.containsKey(KEY_SHIFT_LAST + geoTiffSource)) {
                    emw.setShift(
                        Double.parseDouble(
                            Application.getProperty(KEY_SHIFT_LAST + geoTiffSource, emw.getShift() + "")));
                    // System.out.println("restore:"+em.getShift());
                } else {
                    // System.out.println("auto");
                    emw.autoAdjustShift();
                }
            } else {
                em = slave;
            }

        } catch (Exception e) {
            String message =
                Logging.getMessage(
                    "generic.CreationFromConfigurationFailed",
                    DataConfigurationUtils.getDataConfigDisplayName(domElement));
            Logging.logger().log(java.util.logging.Level.SEVERE, message, e);
        }

        // ElevationModel defaultElevationModel = WWFactory.getGlobe().getElevationModel();
        // if (defaultElevationModel instanceof CompoundElevationModel)
        // {
        // if (!((CompoundElevationModel) defaultElevationModel).containsElevationModel(em)){
        //// System.out.println("adding it to globe");
        // ((CompoundElevationModel) defaultElevationModel).addElevationModel(em);
        // }
        // }
        // else
        // {
        // CompoundElevationModel cm = new CompoundElevationModel();
        // cm.addElevationModel(defaultElevationModel);
        // cm.addElevationModel(em);
        // WWFactory.getGlobe().setElevationModel(cm);
        // throw new RuntimeException();
        // }

        mapLayerVisibilityChanged(this, isVisible);
    }

    protected void startImportFile(String displayName, File file, FileStore fileStore) throws Throwable {
        // Use the FileStore's install location as the destination for the
        // imported image tiles. The install location
        // is an area in the data file store for permanantly resident data.

        Configuration.setValue(AVKey.PRODUCER_ENABLE_FULL_PYRAMID, true);
        // System.out.println("cacheName:"+cacheName);

        // this tries to load existing data and fills the list of known caches.
        for (File f : fileStore.getLocations()) {
            // System.out.println("fileStoreLoc" + f);
            if (!f.exists()) {
                continue;
            }

            // if (!fileStore.isInstallLocation(file.getPath()))
            // continue;

            loadImportedDataFromDirectory(f);
        }

        if (dataConfig == null) {
            Debug.getLog().config("start import geoTiff " + file + " from the scratch");
            Document localDataConfig = null;
            // Import the file into a form usable by World Wind components.

            // create name

            if (file.getParentFile() != null) {
                cacheName = file.getParentFile().getName() + "." + file.getName();
            } else {
                cacheName = file.getName();
            }

            int i = 0;
            while (existingCaches.contains(cacheName)) {
                i++;
                cacheName = file.getName() + "-" + i;
            }

            cacheName +=
                file.getAbsolutePath()
                    .hashCode(); // add some stuff, since simultaniously loading will otherwise cause problems
            // System.out.println("newCacheName" + cacheName);
            Application.setProperty(KEY_CACHE_NAME + geoTiffSource, cacheName);

            localDataConfig =
                importDataFromFile(UserNotificationHubSwing.getCurrentFocusWindow(), file, fileStore, cacheName);

            if (localDataConfig != null) {
                AVList params = new AVListImpl();
                addImportedData(localDataConfig, params);
            }
        }
    }

    HashSet<String> existingCaches = new HashSet<String>();

    // **************************************************************//
    // ******************** Loading Previously Imported Data ******//
    // **************************************************************//

    protected void loadImportedDataFromDirectory(File dir) {
        String[] names = WWIO.listDescendantFilenames(dir, new DataConfigurationFilter(), false);
        if (names == null || names.length == 0) {
            return;
        }

        XPath path = WWXML.makeXPath();

        for (String filename : names) {
            Document doc = null;

            try {
                File dataConfigFile = new File(dir, filename);
                doc = WWXML.openDocument(dataConfigFile);
                doc = DataConfigurationUtils.convertToStandardDataConfigDocument(doc);
            } catch (WWRuntimeException e) {
                e.printStackTrace();
            }

            if (doc == null) {
                continue;
            }

            String dataCacheName = WWXML.getText(doc.getDocumentElement(), "DataCacheName", path);
            if (dataCacheName == null) {
                continue;
            }

            existingCaches.add(dataCacheName);
            // System.out.println("foundCacheName:"+dataCacheName);
            if (dataCacheName.equals(cacheName)) {
                // This data configuration came from an existing file from disk,
                // therefore we cannot guarantee that the
                // current version of World Wind's data importers produced it. This
                // data configuration file may have been
                // created by a previous version of World Wind, or by another
                // program. Set fallback values for any missing
                // parameters that World Wind needs to construct a Layer or
                // ElevationModel from this data configuration.
                AVList params = new AVListImpl();
                setFallbackParams(doc, filename, params);
                // System.out.println(getResourceFile().getName()+ " <-> "+ WWXML.getText(doc.getDocumentElement(),
                // "DataCacheName",
                // WWXML.makeXPath()));

                // Add the data configuraiton to the ImportedDataPanel.
                Debug.getLog().config("load geoTiff " + cacheName + " from cache");
                addImportedData(doc, params);
                return;
            }
        }
    }

    protected static void setFallbackParams(Document dataConfig, String filename, AVList params) {
        XPath xpath = WWXML.makeXPath();
        Element domElement = dataConfig.getDocumentElement();

        // If the data configuration document doesn't define a cache name, then
        // compute one using the file's path
        // relative to its file cache directory.
        String s = WWXML.getText(domElement, "DataCacheName", xpath);
        if (s == null || s.length() == 0) {
            DataConfigurationUtils.getDataConfigCacheName(filename, params);
        }

        // If the data configuration document doesn't define the data's extreme
        // elevations, provide default values using
        // the minimum and maximum elevations of Earth.
        String type = DataConfigurationUtils.getDataConfigType(domElement);
        if (type.equalsIgnoreCase("ElevationModel")) {
            if (WWXML.getDouble(domElement, "ExtremeElevations/@min", xpath) == null) {
                params.setValue(AVKey.ELEVATION_MIN, Earth.ELEVATION_MIN);
            }

            if (WWXML.getDouble(domElement, "ExtremeElevations/@max", xpath) == null) {
                params.setValue(AVKey.ELEVATION_MAX, Earth.ELEVATION_MAX);
            }
        }
    }
    // **************************************************************//
    // ******************** Importing Data From File **************//
    // **************************************************************//

    protected static Document importDataFromFile(
            Component parentComponent, File file, FileStore fileStore, String cacheName) throws Exception {
        // Create a DataStoreProducer which is capable of processing the file.
        final DataStoreProducer producer = createDataStoreProducerFromFile(file);
        if (producer == null) {
            throw new IllegalArgumentException("Unrecognized file type");
        }

        // Create a ProgressMonitor that will provide feedback on how
        final MProgressMonitor progressMonitor =
            new MProgressMonitor(
                parentComponent,
                Language.getFormat(KEY + ".importing", file.getName()),
                Language.getString(KEY + ".importingStart"),
                0,
                100);

        final AtomicInteger progress = new AtomicInteger(0);

        // Configure the ProgressMonitor to receive progress events from the DataStoreProducer. This stops sending
        // progress events when the user clicks the "Cancel" button, ensuring that the ProgressMonitor does not
        PropertyChangeListener progressListener =
            new PropertyChangeListener() {
                public void propertyChange(PropertyChangeEvent evt) {
                    if (progressMonitor.isCanceled()) {
                        return;
                    }

                    if (evt.getPropertyName().equals(AVKey.PROGRESS)) {
                        progress.set((int)(100 * (Double)evt.getNewValue()));
                    }
                }
            };
        producer.addPropertyChangeListener(progressListener);
        progressMonitor.setProgress(0);

        // Configure a timer to check if the user has clicked the ProgressMonitor's "Cancel" button. If so, stop
        // production as soon as possible. This just stops the production from completing; it doesn't clean up any state
        // changes made during production,
        java.util.Timer progressTimer = new java.util.Timer();
        progressTimer.schedule(
            new TimerTask() {
                public void run() {
                    progressMonitor.setProgress(progress.get());

                    if (progressMonitor.isCanceled()) {
                        producer.stopProduction();
                        this.cancel();
                    }
                }
            },
            progressMonitor.getMillisToDecideToPopup(),
            100L);

        Document doc = null;
        try {
            // Import the file into the specified FileStore.
            doc = createDataStoreFromFile(file, fileStore, producer, cacheName);
            if (null != doc) {
                createRasterServerConfigDoc(fileStore, producer);
            }

            // The user clicked the ProgressMonitor's "Cancel" button. Revert any change made during production, and
            // discard the returned DataConfiguration reference.
            if (progressMonitor.isCanceled()) {
                doc = null;
                producer.removeProductionState();
            }
        } finally {
            // Remove the progress event listener from the DataStoreProducer. stop the progress timer, and signify to
            // the
            // ProgressMonitor that we're done.
            producer.removePropertyChangeListener(progressListener);
            producer.removeAllDataSources();
            progressMonitor.close();
            progressTimer.cancel();
        }

        return doc;
    }

    @Override
    public void storeState(IProperties session, String keyPrefix) {
        super.storeState(session, keyPrefix);
        if (em instanceof ElevationModelShiftWrapper) {
            ElevationModelShiftWrapper emw = (ElevationModelShiftWrapper)em;
            if (emw.isInit()) {
                // Debug.printStackTrace("storing state:"+em.getShift());
                Application.setProperty(KEY_SHIFT_TYPE + geoTiffSource, emw.getShiftType() + "");
                Application.setProperty(KEY_SHIFT_LAST + geoTiffSource, emw.getShift() + "");
            }
        }
    }

    protected static Document createDataStoreFromFile(
            File file, FileStore fileStore, DataStoreProducer producer, String cacheName) throws Exception {
        File importLocation = DataImportUtil.getDefaultImportLocation(fileStore);
        if (importLocation == null) {
            String message = Logging.getMessage("generic.NoDefaultImportLocation");
            Logging.logger().severe(message);
            return null;
        }

        // Create the production parameters. These parameters instruct the DataStoreProducer where to import the cached
        // data, and what name to put in the data configuration document.
        AVList params = new AVListImpl();
        params.setValue(AVKey.DATASET_NAME, cacheName);
        params.setValue(AVKey.DATA_CACHE_NAME, cacheName);
        params.setValue(AVKey.FILE_STORE_LOCATION, importLocation.getAbsolutePath());

        // These parameters define producer's behavior:
        // create a full tile cache OR generate only first two low resolution levels
        boolean enableFullPyramid = Configuration.getBooleanValue(AVKey.PRODUCER_ENABLE_FULL_PYRAMID, false);
        if (!enableFullPyramid) {
            params.setValue(AVKey.SERVICE_NAME, AVKey.SERVICE_NAME_LOCAL_RASTER_SERVER);
            params.setValue(AVKey.TILED_RASTER_PRODUCER_LIMIT_MAX_LEVEL, 2);
        }

        producer.setStoreParameters(params);

        // Use the specified file as the the production data source.
        producer.offerDataSource(file, null);

        try {
            // Convert the file to a form usable by World Wind components, according to the specified DataStoreProducer.
            // This throws an exception if production fails for any reason.
            producer.startProduction();
        } catch (Exception e) {
            // Exception attempting to convert the file. Revert any change made during production.
            producer.removeProductionState();
            throw e;
        }

        // Return the DataConfiguration from the production results. Since production successfully completed, the
        // DataStoreProducer should contain a DataConfiguration in the production results. We test the production
        // results anyway.
        Iterable<?> results = producer.getProductionResults();
        if (results != null && results.iterator() != null && results.iterator().hasNext()) {
            Object o = results.iterator().next();
            if (o != null && o instanceof Document) {
                return (Document)o;
            }
        }

        return null;
    }

    protected static void createRasterServerConfigDoc(FileStore fileStore, DataStoreProducer producer) {
        File importLocation = DataImportUtil.getDefaultImportLocation(fileStore);
        if (importLocation == null) {
            String message = Logging.getMessage("generic.NoDefaultImportLocation");
            Logging.logger().severe(message);
            return;
        }

        Document doc = WWXML.createDocumentBuilder(true).newDocument();

        Element root = WWXML.setDocumentElement(doc, "RasterServer");
        WWXML.setTextAttribute(root, "version", "1.0");

        StringBuffer sb = new StringBuffer();
        sb.append(importLocation.getAbsolutePath()).append(File.separator);

        AVList rasterServerParams = new AVListImpl();

        rasterServerParams.setValue(AVKey.BANDS_ORDER, "Auto");
        rasterServerParams.setValue(AVKey.BLACK_GAPS_DETECTION, "enable");

        AVList productionParams = producer.getProductionParameters();
        productionParams = (null == productionParams) ? new AVListImpl() : productionParams;

        if (productionParams.hasKey(AVKey.DATA_CACHE_NAME)) {
            String value = productionParams.getStringValue(AVKey.DATA_CACHE_NAME);
            rasterServerParams.setValue(AVKey.DATA_CACHE_NAME, value);
            sb.append(value).append(File.separator);
        } else {
            String message = Logging.getMessage("generic.MissingRequiredParameter", AVKey.DATA_CACHE_NAME);
            Logging.logger().severe(message);
            throw new WWRuntimeException(message);
        }

        if (productionParams.hasKey(AVKey.DATASET_NAME)) {
            String value = productionParams.getStringValue(AVKey.DATASET_NAME);
            rasterServerParams.setValue(AVKey.DATASET_NAME, value);
            sb.append(value).append(".RasterServer.xml");
        } else {
            String message = Logging.getMessage("generic.MissingRequiredParameter", AVKey.DATASET_NAME);
            Logging.logger().severe(message);
            throw new WWRuntimeException(message);
        }

        Object o = productionParams.getValue(AVKey.DISPLAY_NAME);
        if (WWUtil.isEmpty(o)) {
            o = productionParams.getValue(AVKey.DATASET_NAME);
        }

        rasterServerParams.setValue(AVKey.DISPLAY_NAME, o);

        String rasterServerConfigFilePath = sb.toString();

        Sector extent = null;
        if (productionParams.hasKey(AVKey.SECTOR)) {
            o = productionParams.getValue(AVKey.SECTOR);
            if (null != o && o instanceof Sector) {
                extent = (Sector)o;
            }
        }

        if (null != extent) {
            WWXML.appendSector(root, "Sector", extent);
        } else {
            String message = Logging.getMessage("generic.MissingRequiredParameter", AVKey.SECTOR);
            Logging.logger().severe(message);
            throw new WWRuntimeException(message);
        }

        Element sources = doc.createElementNS(null, "Sources");
        if (producer instanceof TiledRasterProducer) {
            TiledRasterProducer tiledRasterProducer = (TiledRasterProducer)producer;

            for (DataRaster raster : tiledRasterProducer.getDataRasters()) {
                if (raster instanceof CachedDataRaster) {
                    CachedDataRaster readRaster = (CachedDataRaster)raster;
                    o = readRaster.getDataSource();
                    if (WWUtil.isEmpty(o)) {
                        Logging.logger().finest(Logging.getMessage("nullValue.DataSourceIsNull"));
                        continue;
                    }

                    File f = WWIO.getFileForLocalAddress(o);
                    if (WWUtil.isEmpty(f)) {
                        String message = Logging.getMessage("TiledRasterProducer.UnrecognizedDataSource", o);
                        Logging.logger().finest(message);
                        continue;
                    }

                    Element source = WWXML.appendElement(sources, "Source");
                    WWXML.setTextAttribute(source, "type", "file");
                    WWXML.setTextAttribute(source, "path", f.getAbsolutePath());

                    AVList params = readRaster.getParams();
                    if (null == params) {
                        Logging.logger().warning(Logging.getMessage("nullValue.ParamsIsNull"));
                        continue;
                    }

                    Sector sector = raster.getSector();
                    if (null == sector && params.hasKey(AVKey.SECTOR)) {
                        o = params.getValue(AVKey.SECTOR);
                        if (null != o && o instanceof Sector) {
                            sector = (Sector)o;
                        }
                    }

                    if (null != sector) {
                        WWXML.appendSector(source, "Sector", sector);
                    }

                    String[] keysToCopy =
                        new String[] {
                            AVKey.PIXEL_FORMAT,
                            AVKey.DATA_TYPE,
                            AVKey.PIXEL_WIDTH,
                            AVKey.PIXEL_HEIGHT,
                            AVKey.COORDINATE_SYSTEM,
                            AVKey.PROJECTION_NAME
                        };

                    WWUtil.copyValues(params, rasterServerParams, keysToCopy, false);
                } else {
                    String message =
                        Logging.getMessage(
                            "TiledRasterProducer.UnrecognizedRasterType",
                            raster.getClass().getName(),
                            raster.getStringValue(AVKey.DATASET_NAME));
                    Logging.logger().severe(message);
                    throw new WWRuntimeException(message);
                }
            }
        }

        // add sources
        root.appendChild(sources);

        WWXML.saveDocumentToFile(doc, rasterServerConfigFilePath);
    }

    // **************************************************************//
    // ******************** Utility Methods ***********************//
    // **************************************************************//

    protected static DataStoreProducer createDataStoreProducerFromFile(File file) {
        if (file == null) {
            return null;
        }

        DataStoreProducer producer = null;

        AVList params = new AVListImpl();
        if (DataImportUtil.isDataRaster(file, params)) {
            if (AVKey.ELEVATION.equals(params.getStringValue(AVKey.PIXEL_FORMAT))) {
                producer = new TiledElevationProducer();
            } else if (AVKey.IMAGE.equals(params.getStringValue(AVKey.PIXEL_FORMAT))) {
                producer = new TiledImageProducer();
            }
        } else if (DataImportUtil.isWWDotNetLayerSet(file)) {
            producer = new WWDotNetLayerSetConverter();
        }

        return producer;
    }

    public static ImportableDataFilter importableDataFilter = new ImportableDataFilter();

    public static class ImportableDataFilter extends javax.swing.filechooser.FileFilter {
        public ImportableDataFilter() {}

        public boolean accept(File file) {
            if (file == null || file.isDirectory()) {
                return true;
            }

            String nameLower = file.getName().toLowerCase();
            if (nameLower.endsWith(".jpg")
                    || nameLower.endsWith(".jpeg")
                    || nameLower.endsWith(".png")
                    || nameLower.endsWith(".gif")
                    || nameLower.endsWith(".ovr")
                    || nameLower.endsWith(".aux.xml")
                    || nameLower.endsWith(".jpg.xml")
                    || nameLower.endsWith(".jpeg.xml")
                    || nameLower.endsWith(".bmp")) {
                return false;
            }

            if (DataImportUtil.isDataRaster(file, null)) {
                return true;
            } else if (DataImportUtil.isWWDotNetLayerSet(file)) {
                return true;
            }

            return false;
        }

        public String getDescription() {
            return Language.getString(KEY + ".supportetDataDesc"); // "Supported Images/Elevations";
        }
    }

    public void uninstall() {
        if (getWWElevationModel() != null) {
            WWFactory.updateElevationModel(getWWElevationModel(), false);
        }

        close();
    }
}
