/*
 * This Source is licenced under the NASA OPEN SOURCE AGREEMENT VERSION 1.3
 *
 * Copyright (C) 2012 United States Government as represented by the Administrator of the
 * National Aeronautics and Space Administration.
 * All Rights Reserved.
 *
 * Modifications by MAVinci GmbH, Germany (C) 2009-2016: Using parts of NASA code for building an entirely new class
 *
 */
package eu.mavinci.desktop.gui.doublepanel.mapmanager.wms;

import eu.mavinci.core.helper.IStoreable;
import eu.mavinci.core.helper.ResourceManager;
import eu.mavinci.desktop.gui.doublepanel.mapmanager.IMapLayerResourceRelated;
import eu.mavinci.desktop.gui.doublepanel.planemain.tree.maplayers.IMapLayer;
import eu.mavinci.desktop.gui.doublepanel.planemain.tree.maplayers.IMapLayerDescription;
import eu.mavinci.desktop.gui.doublepanel.planemain.tree.maplayers.IMapLayerListener;
import eu.mavinci.desktop.gui.doublepanel.planemain.tree.maplayers.IMapLayerWW;
import eu.mavinci.desktop.gui.doublepanel.planemain.tree.maplayers.MapLayer;
import eu.mavinci.desktop.gui.doublepanel.planemain.tree.maplayers.MapLayerMap;
import eu.mavinci.desktop.gui.doublepanel.planemain.tree.maplayers.MapLayerSectorReferenced;
import eu.mavinci.desktop.gui.wwext.ElevationModelShiftWrapper;
import eu.mavinci.desktop.gui.wwext.WWFactory;
import eu.mavinci.desktop.helper.FileHelper;
import eu.mavinci.desktop.main.core.Application;
import eu.mavinci.desktop.main.debug.Debug;
import eu.mavinci.desktop.main.i18n.Language;
import gov.nasa.worldwind.Factory;
import gov.nasa.worldwind.WorldWind;
import gov.nasa.worldwind.avlist.AVKey;
import gov.nasa.worldwind.avlist.AVList;
import gov.nasa.worldwind.avlist.AVListImpl;
import gov.nasa.worldwind.exception.WWRuntimeException;
import gov.nasa.worldwind.globes.ElevationModel;
import gov.nasa.worldwind.layers.Layer;
import gov.nasa.worldwind.ogc.OGCBoundingBox;
import gov.nasa.worldwind.ogc.wms.WMSCapabilities;
import gov.nasa.worldwind.ogc.wms.WMSLayerCapabilities;
import gov.nasa.worldwind.ogc.wms.WMSLayerStyle;
import gov.nasa.worldwind.retrieve.RetrievalPostProcessor;
import gov.nasa.worldwind.retrieve.Retriever;
import gov.nasa.worldwind.retrieve.URLRetriever;
import gov.nasa.worldwind.util.Logging;
import gov.nasa.worldwind.util.WWIO;
import gov.nasa.worldwind.util.WWUtil;
import gov.nasa.worldwind.util.WWXML;
import gov.nasa.worldwind.wms.CapabilitiesRequest;
import javafx.application.Platform;

import javax.swing.Icon;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class MapLayerWMS extends MapLayerSectorReferenced implements IMapLayerResourceRelated, IMapLayerDescription {

    protected final String wmsSource;
    protected URI serverURI;
    public File capabilitiesFile;
    File capabilitiesFileBackup;

    protected String layerName;
    ResourceManager wmsManager;
    String description;

    boolean onlyOnceChildVisible = false;

    boolean hasBrokenProjectionHighAltitudes = false;
    boolean isBingMaps = false;

    public static final String KEY = "eu.mavinci.desktop.gui.doublepanel.mapmanager.wms.MapLayerWMS";

    public static int BING_MAPS_EXPIRE_SEC = 24 * 60 * 60;
    public static String BING_MAPS_BASE_ADDR = "http://wms.onterrasystems.com";

    IStoreable controller;

    public MapLayerWMS(ResourceManager wmsManager, String wmsSoure, IStoreable controller, boolean defVis) {
        this(wmsManager, wmsSoure, defVis);
        this.controller = controller;
    }

    public MapLayerWMS(ResourceManager wmsManager, String wmsSoure) {
        this(wmsManager, wmsSoure, true);
    }

    static ExecutorService executor = Executors.newSingleThreadExecutor();

    public MapLayerWMS(ResourceManager wmsManager, String wmsSoure, boolean defVis) {
        super("WMSmapLayer." + wmsSoure, defVis);
        this.wmsSource = wmsSoure;
        this.wmsManager = wmsManager;
        String iconPath = "eu/mavinci/icons/16x16/MapLayerWMS.png";
        icon = Application.getImageIconFromResource(iconPath);
        layerName = "WMS:" + wmsSoure;
        // See if the server name is a valid URI. Throw an exception if not.
        try {
            this.serverURI = new URI(wmsSource.trim());
        } catch (URISyntaxException e1) {
            Debug.getLog().log(Level.FINE, "WMS Server \"" + wmsSource + "\" has not wellformed URL", e1);
            close();
            return;
        }
        // server name is not a
        // valid uri.
        capabilitiesFile =
            new File(
                Application.getAppSettingsDir().toString()
                    + File.separatorChar
                    + "wmsCapabilities"
                    + File.separatorChar
                    + FileHelper.urlToFileName(serverURI.toString())
                    + ".xml");
        capabilitiesFileBackup = new File(capabilitiesFile.getAbsolutePath() + "~");

        if (!capabilitiesFile.exists()) {
            capabilitiesFile.getParentFile().mkdirs();
        }

        backupCache();
    }

    public Future<?> retrieveServerCapabilities() {
        return executor.submit(
            () -> {
                try {
                    cacheCapabilities();
                } catch (Exception e) {
                    Debug.getLog().log(Level.FINE, "WMS Server \"" + wmsSource + "\" is not reachable", e);
                }

                if (FileHelper.canRead(capabilitiesFile, null)) {
                    // layers manipulation must be done on UI thread to avoid sync problems
                    Platform.runLater(this::doLoad);
                } else {
                    Debug.getLog()
                        .log(Debug.MINOR_WARNING, "WMS Server \"" + wmsSource + "\" is not cached. Deactivating Layer");
                    close();
                }
            });
    }

    protected void doLoad() {
        try {
            load();
            loadingReady();
        } catch (Exception e) {
            if (capabilitiesFileBackup == null || !capabilitiesFileBackup.exists()) {
                Debug.getLog().log(Level.SEVERE, "Could not open WMS Layer from:" + serverURI, e);
                close();
            } else {
                try {
                    restoreBackupCache();
                    load();
                    Debug.getLog()
                        .log(
                            Level.FINE,
                            "Could not open WMS Layer from:"
                                + serverURI
                                + " using old Capabilities instead:"
                                + capabilitiesFileBackup,
                            e);
                    loadingReady();
                } catch (Exception e1) {
                    Debug.getLog().log(Level.FINE, "Could not open WMS Layer from:" + serverURI, e);
                    Debug.getLog()
                        .log(
                            Level.SEVERE,
                            "Could also not open WMS Layer from backuped Capabilities File:" + capabilitiesFileBackup,
                            e1);
                    close();
                }
            }
        }
    }

    @Override
    public Icon getIcon() {
        if (isBingMaps) {
            return Application.getImageIconFromResource("eu/mavinci/icons/16x16/ActLoadWMSonTerra.png");
        }

        return super.getIcon();
    }

    protected void backupCache() {
        try {
            FileHelper.copyFile(capabilitiesFile, capabilitiesFileBackup);
        } catch (IOException e) {
        }
    }

    protected void restoreBackupCache() throws IOException {
        FileHelper.copyFile(capabilitiesFileBackup, capabilitiesFile);
    }

    // private Thread loadingThread = null;
    private final TreeSet<LayerInfo> layerInfos =
        new TreeSet<LayerInfo>(
            new Comparator<LayerInfo>() {
                public int compare(LayerInfo infoA, LayerInfo infoB) {
                    String nameA = infoA.getTitle();
                    String nameB = infoB.getTitle();
                    return nameA.compareTo(nameB);
                }
            });

    boolean isClosed = false;

    protected void close() {
        isClosed = true;
        Platform.runLater(
            () -> {
                if (wmsManager != null) {
                    wmsManager.remove(wmsSource);
                } else {
                    getParentLayer().removeMapLayer(MapLayerWMS.this);
                }
            });
    }

    protected void load() throws Exception {
        final WMSCapabilities caps;

        setMute(true);

        try (InputStream is = new FileInputStream(capabilitiesFile)) {
            caps = new WMSCapabilities(is);
            caps.parse();

            // System.out.println(caps);
            if (wmsSource.startsWith(MapLayerWMSRoot.SENTINEL_URL)) {
                description = Language.getString(KEY + ".liveMapsDesc");
                layerName = "MAVinci Hyperspectral Live Maps";
                onlyOnceChildVisible = true;
            } else if (caps.getServiceInformation() == null) {
                description = "";
                Debug.getLog().config("WMS Server service Informations are NULL:" + serverURI);
            } else {
                description = caps.getServiceInformation().getServiceAbstract();

                if (caps.getServiceInformation().getServiceTitle() != null) {
                    layerName = "WMS:" + caps.getServiceInformation().getServiceTitle();

                    // System.out.println("layer title: "+caps.getServiceInformation().getServiceTitle());
                    isBingMaps = caps.getServiceInformation().getServiceTitle().startsWith("OnTerra");
                    hasBrokenProjectionHighAltitudes = isBingMaps;
                    if (isBingMaps) {
                        description = Language.getString(KEY + ".cacheWillDeletedAfter24h");
                    }
                }
            }

            // Gather up all the named layers and make a world wind layer for each.
            final List<WMSLayerCapabilities> namedLayerCaps = caps.getNamedLayers();

            if (namedLayerCaps == null) {
                throw new Exception("WMS Server has no fitting layers " + serverURI);
            }

            for (WMSLayerCapabilities lc : namedLayerCaps) {
                Set<WMSLayerStyle> styles = lc.getStyles();
                if (styles == null || styles.size() == 0) {
                    LayerInfo layerInfo = createLayerInfo(caps, lc, null);
                    MapLayerWMS.this.layerInfos.add(layerInfo);
                } else {
                    for (WMSLayerStyle style : styles) {
                        LayerInfo layerInfo = createLayerInfo(caps, lc, style);
                        MapLayerWMS.this.layerInfos.add(layerInfo);
                    }
                }
            }

            if (MapLayerWMS.this.layerInfos.size() == 0) {
                throw new Exception("WMS Server contains no layers: " + serverURI);
            }

            setMute(false);
        }
    }

    protected void loadingReady() {
        if (isClosed) {
            return;
        }
        // Fill the sublayers with the layer titles.
        try {
            reconstructChildren();
            loadState();
        } catch (Throwable t) {
            Debug.getLog().log(Level.SEVERE, "Could not load layers from WMS server " + serverURI, t);
            close();
        }
    }

    protected void loadState() {
        if (controller == null) {
            controller = (IStoreable)parent;
        }

        if (controller != null) {
            controller.loadState();
        }
    }

    protected void reconstructChildren() {
        if (isClosed) {
            return;
        }

        setMute(true);
        try {
            mapLayerValuesChanged(MapLayerWMS.this);
            while (sizeMapLayer() > 0) {
                removeMapLayer(0);
            }

            for (LayerInfo layerInfo : layerInfos) {
                // System.out.println("WMSsourcE: " + wmsSource + " "+layerInfo.getName() + " " + layerInfo.getTitle());
                if (wmsSource.startsWith("http://services.sentinel-hub.com")) {
                    if (layerInfo.getName().equals("ID")
                            || layerInfo.getName().equals("DATE")
                            || layerInfo.getName().equals("FILL")
                            || layerInfo.getName().equals("OUTLINE")
                            || layerInfo.getName().equals("CLOUDS")
                            || layerInfo.getTitle().endsWith(" : Index")
                            || layerInfo.getTitle().endsWith(" : Reflectance")
                            || layerInfo.getTitle().endsWith(" : Sentinel DN")) {
                        continue; // skip boring meta layers
                    }
                    // System.out.println("layerInfo.getName():"+layerInfo.getName() + " "+layerInfo.getTitle());
                    layerInfo.params.setValue(
                        AVKey.NUM_EMPTY_LEVELS, 5); // level 0..4=5Levels are NOT provided by sentinel-hub!
                    layerInfo.params.setValue(
                        AVKey.NUM_LEVELS, 12); // it seem that 0..10=11 level are perfect fot 10m resolution data, but
                    // to be on the save side, lets go up to 12 here
                    long expireTime = 1 * 24 * 60 * 60 * 1000L; // 1 day
                    layerInfo.params.setValue(
                        AVKey.EXPIRY_TIME, System.currentTimeMillis() - expireTime); // all files older than this will
                    // be redownloaded
                    layerInfo.params.setValue(AVKey.IMAGE_FORMAT, "image/jpeg");
                }

                annotateComponent(layerInfo.params);
                // System.out.println("WMS layer info "+layerInfo +" "+ layerInfo.params.getValues());

                Object component = createComponent(layerInfo);
                MapLayer mapLayer = null;
                annotateComponent((AVList)component);
                if (component instanceof Layer) {
                    Layer layer = (Layer)component;
                    layer.setValue(WWFactory.KEY_AV_Description, layerInfo.getAbstract());
                    if (hasBrokenProjectionHighAltitudes) {
                        layer.setMaxActiveAltitude(100000); // 100km
                        layer.setMinActiveAltitude(0);
                        // System.out.println("Layer : min ActAlt="+ layer.getMinActiveAltitude());
                    }

                    if (isBingMaps) {
                        WWFactory.dropDataCache(layer, BING_MAPS_EXPIRE_SEC, BING_MAPS_BASE_ADDR);
                    }

                    boolean defVis = true;
                    if (isBingMaps && layer.getName() != null) {
                        defVis = layer.getName().contains("Default");
                        // System.out.println("name:" + layer.getName() + " vis:" + defVis);
                    }

                    // nicing layer names e.g. for sentinel stuff
                    if (layer.getName().endsWith(" : Default")) {
                        layer.setName(layer.getName().substring(0, layer.getName().length() - " : Default".length()));
                    }

                    if (wmsSource.startsWith("http://services.sentinel-hub.com")) {
                        defVis = layer.getName().endsWith("no clouds (MAIN) : RGB");
                    }

                    // if (layer instanceof WMSTiledImageLayer) {
                    // WMSTiledImageLayer wmsLayer = (WMSTiledImageLayer) layer;
                    // wmsLayer.getTilesInSector(sector, levelNumber)
                    // }

                    mapLayer = new MapLayerMap(layer, -200 + MapLayerWMSRoot.wmsLoadingsCount++, defVis, true, true);
                } else if (component instanceof ElevationModel) {
                    ElevationModel model = (ElevationModel)component;
                    boolean useSHIFT_DEM_WRAPPER =
                        Boolean.parseBoolean(Application.getProperty("_EXPERT.useSHIFT_DEM_WRAPPER", "true"));
                    // useEGM_SRTM_SHIFT = false;
                    Debug.getLog().fine("useSHIFT_DEM_WRAPPER=" + useSHIFT_DEM_WRAPPER);
                    if (useSHIFT_DEM_WRAPPER) {
                        model = new ElevationModelShiftWrapper(model, getSector());
                    }

                    mapLayer =
                        new MapLayerElevationModel(
                            layerInfo.params.getStringValue(AVKey.DISPLAY_NAME), model, layerInfo.getAbstract());
                    WWFactory.updateElevationModel(model, true);

                    if (isBingMaps) {
                        WWFactory.dropDataCache(model, BING_MAPS_EXPIRE_SEC, BING_MAPS_BASE_ADDR);
                    }
                }

                if (mapLayer != null) {
                    addMapLayer(mapLayer);
                    mapLayer.addMapListener(childVisibilityListener);
                }
            }

            if (layerInfos.size() > 0 && sizeMapLayer() == 0) {
                throw new RuntimeException("all layers from this WMS are not loadable!");
            }
        } finally {
            setMute(false);
            mapLayerStructureChanged(this);
        }
    }

    IMapLayerListener childVisibilityListener =
        new IMapLayerListener() {

            @Override
            public void mapLayerVisibilityChanged(IMapLayer layer, boolean newVisibility) {
                if (onlyOnceChildVisible && newVisibility) {
                    for (IMapLayer l : subLayers) {
                        if (l != layer) {
                            l.setVisible(false);
                        }
                    }
                }
            }

            @Override
            public void mapLayerValuesChanged(IMapLayer layer) {}

            @Override
            public void mapLayerStructureChanged(IMapLayer layer) {}

            @Override
            public void childMapLayerRemoved(int i, IMapLayer layer) {}

            @Override
            public void childMapLayerInserted(int i, IMapLayer layer) {}
        };

    protected void annotateComponent(AVList component) {}

    protected static String getFactoryKeyForCapabilities(WMSCapabilities caps) {
        boolean hasApplicationBilFormat = false;

        Set<String> formats = caps.getImageFormats();
        for (String s : formats) {
            if (s.contains("application/bil")) {
                hasApplicationBilFormat = true;
                break;
            }
        }

        return hasApplicationBilFormat ? AVKey.ELEVATION_MODEL_FACTORY : AVKey.LAYER_FACTORY;
    }

    protected Object createComponent(LayerInfo layerInfo) {
        WMSCapabilities caps = layerInfo.caps;
        AVList params = layerInfo.params;
        AVList configParams = params.copy(); // Copy to insulate changes from the caller.

        // Some wms servers are slow, so increase the timeouts and limits used by world wind's retrievers.
        configParams.setValue(AVKey.URL_CONNECT_TIMEOUT, 30000);
        configParams.setValue(AVKey.URL_READ_TIMEOUT, 30000);
        configParams.setValue(AVKey.RETRIEVAL_QUEUE_STALE_REQUEST_LIMIT, 60000);

        try {
            String factoryKey = getFactoryKeyForCapabilities(caps);
            Factory factory = (Factory)WorldWind.createConfigurationComponent(factoryKey);
           
            if (layerInfo.capsLayer.getCRS().size() > 0) {
                for (String epsg : layerInfo.capsLayer.getCRS()) {
                    if (epsg.equalsIgnoreCase("EPSG:4326") || epsg.equalsIgnoreCase("CRS:84")) {
                        configParams.setValue(AVKey.PROJECTION_EPSG_CODE, epsg);
                        break;
                    }
                }

                if (!configParams.hasKey(AVKey.PROJECTION_EPSG_CODE)) {
                    configParams.setValue(AVKey.PROJECTION_EPSG_CODE, layerInfo.capsLayer.getCRS().iterator().next());
                }
            } else if (layerInfo.capsLayer.getBoundingBoxes().iterator().hasNext()) {
                for (OGCBoundingBox bb : layerInfo.capsLayer.getBoundingBoxes()) {
                    if (bb.getCRS() != null
                            && (bb.getCRS().equalsIgnoreCase("EPSG:4326") || bb.getCRS().equalsIgnoreCase("CRS:84"))) {
                        configParams.setValue(AVKey.PROJECTION_EPSG_CODE, bb.getCRS());
                        break;
                    }
                }

                if (!configParams.hasKey(AVKey.PROJECTION_EPSG_CODE)) {
                    configParams.setValue(
                        AVKey.PROJECTION_EPSG_CODE, layerInfo.capsLayer.getBoundingBoxes().iterator().next().getCRS());
                }
            }

            Object o = factory.createFromConfigSource(caps, configParams);
            if (o instanceof AVList) {
                AVList component = (AVList)o;
                component.setValue(WWFactory.KEY_AV_Sector, configParams.getValue(AVKey.SECTOR));
            }

            return o;
        } catch (Exception e) {
            // e.printStackTrace();
            // for( Entry<String, Object> entry: configParams.getEntries()){
            // System.out.println("e:" + entry.getKey() + "->"+entry.getValue());
            // }
            Debug.getLog().log(Level.WARNING, "Problems creating WMS Layer", e);
        }

        return null;
    }

    private LayerInfo createLayerInfo(WMSCapabilities caps, WMSLayerCapabilities layerCaps, WMSLayerStyle style) {
        // Create the layer info specified by the layer's capabilities entry and the selected style.

        LayerInfo linfo = new LayerInfo();
        linfo.caps = caps;
        linfo.capsLayer = layerCaps;
        linfo.params = new AVListImpl();
        linfo.params.setValue(AVKey.LAYER_NAMES, layerCaps.getName());
        if (style != null) {
            linfo.params.setValue(AVKey.STYLE_NAMES, style.getName());
        }

        String abs = layerCaps.getLayerAbstract();
        if (!WWUtil.isEmpty(abs)) {
            linfo.params.setValue(AVKey.LAYER_ABSTRACT, abs);
        }

        linfo.params.setValue(AVKey.DISPLAY_NAME, makeTitle(caps, linfo));
        return linfo;
    }

    private static String makeTitle(WMSCapabilities caps, LayerInfo layerInfo) {
        String layerNames = layerInfo.params.getStringValue(AVKey.LAYER_NAMES);
        String styleNames = layerInfo.params.getStringValue(AVKey.STYLE_NAMES);
        String[] lNames = layerNames.split(",");
        String[] sNames = styleNames != null ? styleNames.split(",") : null;

        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < lNames.length; i++) {
            if (sb.length() > 0) {
                sb.append(", ");
            }

            String layerName = lNames[i];
            WMSLayerCapabilities lc = caps.getLayerByName(layerName);
            String layerTitle = lc.getTitle();
            sb.append(layerTitle != null ? layerTitle : layerName);

            if (sNames == null || sNames.length <= i) {
                continue;
            }

            String styleName = sNames[i];
            WMSLayerStyle style = lc.getStyleByName(styleName);
            if (style == null) {
                continue;
            }

            sb.append(" : ");
            String styleTitle = style.getTitle();
            sb.append(styleTitle != null ? styleTitle : styleName);
        }

        return sb.toString();
    }

    protected void cacheCapabilities() throws Exception {
        if (WorldWind.isOfflineMode()) {
            return;
        }

        // marco: this code is long an ugly, and copyied from the WW Class:

        // Capabilities
        if (this.serverURI == null) {
            String message = Logging.getMessage("nullValue.URIIsNull");
            Debug.getLog().severe(message);
            throw new IllegalArgumentException(message);
        }

        // Request the capabilities document from the server.
        CapabilitiesRequest req = new CapabilitiesRequest(this.serverURI, "WMS");
        URL capsURL = req.getUri().toURL();

        URLRetriever retriever =
            URLRetriever.createRetriever(
                capsURL,
                new RetrievalPostProcessor() {
                    public ByteBuffer run(Retriever retriever) {
                        return retriever.getBuffer();
                    }
                });

        if (retriever == null) {
            String message = Logging.getMessage("generic.UnrecognizedProtocol");
            Debug.getLog().fine(message);
            throw new WWRuntimeException(message);
        }

        retriever.call();

        if (!retriever.getState().equals(URLRetriever.RETRIEVER_STATE_SUCCESSFUL)) {
            String message = Logging.getMessage("generic.RetrievalFailed", serverURI.toString());
            Debug.getLog().fine(message);
            throw new WWRuntimeException(message);
        }

        if (retriever.getBuffer() == null || retriever.getBuffer().limit() == 0) {
            String message = Logging.getMessage("generic.RetrievalReturnedNoContent", serverURI.toString());
            Debug.getLog().fine(message);
            throw new WWRuntimeException(message);
        }

        if (retriever.getContentType().equalsIgnoreCase("application/vnd.ogc.se_xml")) {
            String exceptionMessage = WWXML.extractOGCServiceException(retriever.getBuffer());
            String message =
                Logging.getMessage(
                    "OGC.ServiceException",
                    serverURI.toString() + ": " + exceptionMessage != null ? exceptionMessage : "");
            Debug.getLog().fine(message);
            throw new WWRuntimeException(message);
        }

        // Parse the DOM as a capabilities document.
        // is = WWIO.getInputStreamFromByteBuffer(retriever.getBuffer());
        String xml = WWIO.byteBufferToString(retriever.getBuffer(), "iso-8859-1");
        String serverURL = serverURI.toString();
        int pos = serverURL.indexOf('@');
        if (pos >= 0) {
            int pos1 = serverURL.indexOf("//");
            int pos2 = serverURL.indexOf('/', pos1 + 2);
            String domainWithPW = serverURL.substring(0, pos2);
            String userPw = serverURL.substring(pos1 + 2, pos);
            String domainWithoutPW = domainWithPW.replaceFirst(Pattern.quote(userPw + '@'), "");
            // System.out.println("domainWithPW" + domainWithPW);
            // System.out.println("domainWithoutPW" + domainWithoutPW);

            // System.out.println("before" + xml);
            xml = xml.replaceAll(Pattern.quote(domainWithoutPW), Matcher.quoteReplacement(domainWithPW));
            // System.out.println("after" + xml);
        }

        // System.out.println(xml);
        if (xml.indexOf("OnTerra WMS") > 0) {
            // Fixing MS Bing WMS server, since it is not really WMS compatibel
            xml = xml.replaceAll(Pattern.quote("version=\"1.3.0\""), Matcher.quoteReplacement("version=\"1.1.1\""));
            xml = xml.replaceAll(Pattern.quote("WMT_MS_Capabilities"), Matcher.quoteReplacement("WMS_Capabilities"));
            xml =
                xml.replaceAll(Pattern.quote("<Name>OnTerra WMS</Name>"), Matcher.quoteReplacement("<Name>WMS</Name>"));
        }

        if (xml.indexOf("MAVinci") > 0) {
            // Fixing WMS-Proxy WMS server, since it is not really WMS compatibel
            xml = xml.replaceAll(Pattern.quote("version=\"1.3.0\""), Matcher.quoteReplacement("version=\"1.1.1\""));
            // System.out.println("---->:"+xml);
        }
        // System.out.println("----");
        // System.out.println(xml);

        // System.out.println("write file to: " + capabilitiesFile);
        try (Writer out = new OutputStreamWriter(new FileOutputStream(capabilitiesFile), "iso-8859-1")) {
            out.write(xml);
            out.close();
        }
        // WWIO.writeTextFile(xml, capabilitiesFile);

    }

    @Override
    public String getTitle() {
        return layerName;
    }

    public String getResourceString() {
        return wmsSource;
    }

    private static class LayerInfo {
        private WMSCapabilities caps;
        private WMSLayerCapabilities capsLayer;
        private AVListImpl params = new AVListImpl();

        private String getTitle() {
            return params.getStringValue(AVKey.DISPLAY_NAME);
        }

        @SuppressWarnings("unused")
        private String getName() {
            return params.getStringValue(AVKey.LAYER_NAMES);
        }

        private String getAbstract() {
            return params.getStringValue(AVKey.LAYER_ABSTRACT);
        }
    }

    @Override
    public String getDescription() {
        return description;
    }

    public void dropCache() {
        setMute(true);
        try {
            capabilitiesFile.delete();
            capabilitiesFileBackup.delete();
        } catch (Exception e) {
            Debug.getLog().log(Level.WARNING, "could not delete cpability file of Layer: " + this, e);
        }

        for (IMapLayer sub : subLayers) {
            try {
                if (sub instanceof IMapLayerWW) {
                    IMapLayerWW subWW = (IMapLayerWW)sub;
                    WWFactory.dropDataCache(subWW.getWWLayer());
                }

                if (sub instanceof MapLayerElevationModel) {
                    MapLayerElevationModel elev = (MapLayerElevationModel)sub;
                    WWFactory.dropDataCache(elev.getWWElevationModel());
                }
            } catch (Exception e) {
                Debug.getLog().log(Level.WARNING, "could not drop cache of Layer: " + sub, e);
            }
        }

        setMute(false);
    }

    @Override
    public String getTooltip() {
        return serverURI.toString();
    }

    public boolean isBingMaps() {
        return isBingMaps;
    }
}
