/*
 * This Source is licenced under the NASA OPEN SOURCE AGREEMENT VERSION 1.3
 *
 * Copyright (C) 2012 United States Government as represented by the Administrator of the
 * National Aeronautics and Space Administration.
 * All Rights Reserved.
 *
 * Modifications by MAVinci GmbH, Germany (C) 2009-2016: Using parts of NASA code to build a contour lines layer
 *
 */

package eu.mavinci.desktop.gui.doublepanel.planemain.wwd;

import eu.mavinci.desktop.gui.doublepanel.planemain.tree.maplayers.DPanPlaneMainController;
import eu.mavinci.desktop.gui.wwext.WWFactory;
import eu.mavinci.desktop.main.core.Application;
import gov.nasa.worldwind.geom.Intersection;
import gov.nasa.worldwind.geom.Sector;
import gov.nasa.worldwind.layers.RenderableLayer;
import gov.nasa.worldwind.render.ContourLine;
import gov.nasa.worldwind.render.DrawContext;
import gov.nasa.worldwind.render.Renderable;
import gov.nasa.worldwind.util.Logging;
import gov.nasa.worldwind.util.TimedExpirySupport;
import gov.nasa.worldwind.util.WWMath;
import java.awt.Color;
import java.util.ArrayList;
import java.util.Arrays;

public class ContourLinesLayer extends RenderableLayer {

    class MContourLine extends ContourLine {

        public MContourLine(double elevation) {
            super(elevation);
            setViewClippingEnabled(true);
            maxConnectingDistance = 5;
        }

        TimedExpirySupport expirySupportM = new TimedExpirySupport(2000, 3000);

        public void render(DrawContext dc) {
            if (dc == null) {
                String message = Logging.getMessage("nullValue.DrawContextIsNull");
                Logging.logger().severe(message);
                throw new IllegalArgumentException(message);
            }

            if (!this.isEnabled()) {
                return;
            }

            if (!this.getSector().intersects(dc.getVisibleSector())) {
                return;
            }

            if (!this.isValid(dc)) {
                makeContourLine(dc);
                this.expirySupportM.restart(dc);
                this.globeStateKey = dc.getGlobe().getGlobeStateKey(dc);
            }

            for (Renderable r : this.getRenderables()) {
                r.render(dc);
            }
        }

        protected boolean isValid(DrawContext dc) {
            if (this.expirySupportM.isExpired(dc)) {
                return false;
            }

            return this.globeStateKey != null && this.globeStateKey.equals(dc.getGlobe().getStateKey(dc));
        }

        /** Update the contour line according to the current terrain geometry. */
        public void update() {
            this.expirySupportM.setExpired(true);
        }

        /**
         * Update the renderable list with appropriate renderables to display the contour line.
         *
         * @param dc the current <code>DrawContext</code>.
         */
        protected void makeContourLine(DrawContext dc) {
            if (dc == null) {
                String message = Logging.getMessage("nullValue.DrawContextIsNull");
                Logging.logger().severe(message);
                throw new IllegalArgumentException(message);
            }

            this.getRenderables().clear();

            // the following test is key to speeding up this enough, to make almost 1000 line levels work!
            Sector s = dc.getVisibleSector();

            if (lastSector == null || !s.equals(lastSector)) {
                lastMinMax = dc.getGlobe().getMinAndMaxElevations(dc.getVisibleSector());
                lastSector = s;
            }

            double elev = getElevation();
            if (elev <= lastMinMax[0] || elev >= lastMinMax[1]) {
                return;
            }

            double min = Math.ceil(lastMinMax[0] / step) * step;
            double max = Math.floor(lastMinMax[1] / step) * step;
            int c = (int)Math.round(((max - min) / step)) + 1;
            int i = (int)Math.round(elev / step);
            int sparsing = WWMath.powerOfTwoCeiling(c / maxVisLines); // in the Netherlands this value can become 0 !!
            // System.out.println("min:"+min + " max:"+max + " sparsing:"+sparsing);
            if (sparsing < 1 || i % sparsing != 0) {
                return;
            }

            // Get intersection points with terrain
            double ve = dc.getVerticalExaggeration();
            Intersection[] interArray = dc.getSurfaceGeometry().intersect(this.getElevation() * ve, this.getSector());

            if (interArray != null) {
                ArrayList<Intersection> inter = new ArrayList<Intersection>(Arrays.asList(interArray));

                // Filter intersection segment list
                if (isViewClippingEnabled()) {
                    inter = filterIntersectionsOnViewFrustum(dc, inter);
                }

                inter = filterIntersections(dc, inter);

                // Create polyline segments
                makePolylinesConnected(dc, inter, this.maxConnectingDistance);
            }
        }
    }

    double[] lastMinMax;
    Sector lastSector;

    int maxVisLines = Integer.parseInt(Application.getProperty("_EXPERT.contourLines.maxVisLines", "50"));
    int step = Integer.parseInt(Application.getProperty("_EXPERT.contourLines.step", "10"));
    int stepBold = Integer.parseInt(Application.getProperty("_EXPERT.contourLines.stepBold", "40"));
    int stepBoldBlue = Integer.parseInt(Application.getProperty("_EXPERT.contourLines.stepBoldBlue", "80"));
    int width = Integer.parseInt(Application.getProperty("_EXPERT.contourLines.width", "1"));
    int widthBold = Integer.parseInt(Application.getProperty("_EXPERT.contourLines.widthBold", "2"));

    public ContourLinesLayer() {
        setName(DPanPlaneMainController.TOOL_LAYER_PREFIX + "contourLines");
        setPickEnabled(false);

        for (int x = 0; x < WWFactory.getGlobe().getMaxElevation(); x += step) {
            // for (int x =200; x < 201; x+=step){
            // System.out.println("addLine:" + x);
            ContourLine contourLine = new MContourLine(x);
            contourLine.setLineWidth(width);
            if (x % stepBoldBlue == 0) {
                contourLine.setLineWidth(widthBold);
                contourLine.setColor(new Color(0.0f, 0.1f, 0.6f));
            }

            if (x % stepBold == 0) {
                contourLine.setLineWidth(widthBold);
            }

            addRenderable(contourLine);
        }
    }

}
