/*
 * This Source is licenced under the NASA OPEN SOURCE AGREEMENT VERSION 1.3
 *
 * Copyright (C) 2001, 2006 United States Government as represented by the Administrator of the
 * National Aeronautics and Space Administration.
 * All Rights Reserved.
 *
 * Modifications by MAVinci GmbH, Germany (C) 2009-2016: dragging on various different reference levels
 *
 */
package eu.mavinci.desktop.gui.wwext;

import eu.mavinci.core.flightplan.CFlightplan;
import eu.mavinci.core.flightplan.CPicAreaCorners;
import eu.mavinci.core.flightplan.FlightplanContainerFullException;
import eu.mavinci.core.flightplan.IFlightplanLatLonReferenced;
import eu.mavinci.core.flightplan.IFlightplanPositionReferenced;
import eu.mavinci.core.flightplan.IFlightplanRelatedObject;
import eu.mavinci.core.flightplan.IMuteable;
import eu.mavinci.core.flightplan.visitors.AFlightplanVisitor;
import eu.mavinci.desktop.gui.doublepanel.mapmanager.kml.KMLAltitudeMode;
import eu.mavinci.desktop.gui.doublepanel.mapmanager.kml.KMLWrapperPosition;
import eu.mavinci.desktop.gui.doublepanel.planemain.tagging.MapLayerPicArea;
import eu.mavinci.desktop.gui.doublepanel.planemain.tree.maplayers.MapLayerStartingPosition;
import eu.mavinci.desktop.gui.doublepanel.planemain.wwd.AltitudeModes;
import eu.mavinci.desktop.gui.widgets.delegatedtree.IPlaneTreeController;
import eu.mavinci.desktop.main.core.Application;
import eu.mavinci.desktop.main.debug.Debug;
import eu.mavinci.flightplan.ISplittingLine;
import eu.mavinci.flightplan.Origin;
import eu.mavinci.flightplan.PhantomCorner;
import eu.mavinci.flightplan.PicArea;
import eu.mavinci.flightplan.StartProcedure;
import eu.mavinci.geo.ISectorReferenced;
import gov.nasa.worldwind.Movable;
import gov.nasa.worldwind.SceneController;
import gov.nasa.worldwind.View;
import gov.nasa.worldwind.WorldWindow;
import gov.nasa.worldwind.geom.Angle;
import gov.nasa.worldwind.geom.Intersection;
import gov.nasa.worldwind.geom.Line;
import gov.nasa.worldwind.geom.Matrix;
import gov.nasa.worldwind.geom.Position;
import gov.nasa.worldwind.geom.Vec4;
import gov.nasa.worldwind.globes.Globe;
import gov.nasa.worldwind.terrain.SectorGeometryList;
import gov.nasa.worldwind.util.Logging;
import gov.nasa.worldwind.util.RayCastingSupport;
import java.awt.Cursor;
import java.awt.Point;
import java.awt.Toolkit;
import java.util.OptionalDouble;
import java.util.logging.Level;
import javafx.application.Platform;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.SimpleBooleanProperty;

public class MDragger {

    protected final WorldWindow wwd;
    // protected final Component wwdComponent;
    protected boolean dragging = false;
    private final BooleanProperty draggingProperty = new SimpleBooleanProperty();

    private Cursor lastCursor;
    protected final Cursor emptyCursor =
        Toolkit.getDefaultToolkit()
            .createCustomCursor(
                Application.getImageIconFromResource("com/intel/missioncontrol/gfx/cursor_drag.svg").getImage(),
                new Point(16, 16),
                "EmptyCursor");

    protected Point dragRefCursorPoint;
    protected Vec4 dragRefObjectPoint;
    protected double dragRefAltitude;
    protected Movable dragObject;

    public Movable getDragObject() {
        return dragObject;
    }

    protected Object userData = null;
    protected AltitudeModes altMode = AltitudeModes.absolute;
    protected boolean sticksToGround = false;

    View view;
    Globe globe;
    Vec4 refPoint;
    Position refPos;
    Matrix m;
    IPlaneTreeController controller;

    public MDragger(WorldWindow wwd, IPlaneTreeController controller) {
        // super(wwd);
        if (wwd == null) {
            String msg = Logging.getMessage("nullValue.WorldWindow");
            Logging.logger().severe(msg);
            throw new IllegalArgumentException(msg);
        }

        this.wwd = wwd;
        // wwdComponent = (Component)wwd;
        this.controller = controller;
    }

    public BooleanProperty isDraggingProperty() {
        return draggingProperty;
    }

    public Object getDragObjectUserData() {
        return userData;
    }

    double planeRefElevation;

    public synchronized boolean isDragging() {
        return this.dragging;
    }

    synchronized boolean tryStartDragging() {
        if (!isDragging()) {
            return false;
        }

        setDragging(true);
        return true;
    }

    synchronized boolean tryStopDragging() {
        if (!isDragging()) {
            return false;
        }

        setDragging(false);
        return true;
    }

    protected Cursor getLastCursor() {
        return lastCursor;
    }

    protected void setLastCursor(Cursor lastCursor) {
        if (lastCursor == emptyCursor) {
            Logging.logger().log(Level.WARNING, "Concurrent modification of dragging cursor", new Exception());
        } else {
            this.lastCursor = lastCursor;
        }
    }

    /**
     * returns true if dragging was started
     *
     * @param dragedRenderable
     * @param cursorPointStart
     * @param planeRefElevation
     * @param pickedPosition
     * @return
     */
    public boolean startDragging(
            Object dragedRenderable, Point cursorPointStart, double planeRefElevation, Position pickedPosition) {
        synchronized (this) {
            if (isDragging()) {
                return false;
            }

            if (dragedRenderable == null) {
                return false;
            }

            if (!(dragedRenderable instanceof IWWRenderableWithUserData)) {
                return false;
            }

            if (cursorPointStart == null) {
                return false;
            }

            sticksToGround = false;
            altMode = AltitudeModes.absolute;
            dragObject = null;
            this.planeRefElevation = planeRefElevation;

            view = wwd.getView();
            globe = wwd.getModel().getGlobe();

            IWWRenderableWithUserData userDatContainer = (IWWRenderableWithUserData)dragedRenderable;
            if (!userDatContainer.isDraggable()) {
                return false;
            }

            userData = userDatContainer.getUserData();
            if (userData == null) {
                return false;
            }
            // System.out.println("Start Dragging user Data " + userData + " userDataClass:" + userData.getClass());
            //
            if (userData instanceof PhantomCorner) {
                PhantomCorner corner = (PhantomCorner)userData;
                try {
                    userData = corner.makeReal();
                } catch (FlightplanContainerFullException e) {
                    Debug.getLog().log(Debug.MINOR_WARNING, "cant add point by drag&drop", e);
                }
            }

            if (userData instanceof PicArea) {
                PicArea picArea = (PicArea)userData;
                dragObject = (Movable)dragedRenderable;
                sticksToGround = true;
                altMode = AltitudeModes.clampToGround;
                refPos = new Position(picArea.getCenterShifted(), 0);
            } else if (userData instanceof MapLayerPicArea) {
                MapLayerPicArea picArea = (MapLayerPicArea)userData;
                dragObject = (Movable)dragedRenderable;
                sticksToGround = true;
                altMode = AltitudeModes.clampToGround;
                refPos = new Position(picArea.getCenter(), 0);
                System.out.println("start dragging picArea");
            } else if (userData instanceof MapLayerStartingPosition) {
                MapLayerStartingPosition startPos = (MapLayerStartingPosition)userData;
                if (!startPos.isChangeable()) {
                    return false;
                }

                sticksToGround = true;
                altMode = AltitudeModes.relativeToGround;
                dragObject = (Movable)dragedRenderable;
                refPos = startPos.getPosition();
            } else if (userData instanceof KMLWrapperPosition) {
                KMLWrapperPosition posWrapper = (KMLWrapperPosition)userData;
                sticksToGround = posWrapper.getParent().getAltitudeMode() != KMLAltitudeMode.absolute;
                altMode = AltitudeModes.fromKMLAltMode(posWrapper.getParent().getAltitudeMode());
                refPos = posWrapper.getPosition();
            } else if (userData instanceof IFlightplanLatLonReferenced) {
                IFlightplanLatLonReferenced posRef = (IFlightplanLatLonReferenced)userData;
                sticksToGround = posRef.isStickingToGround();
                double elev = planeRefElevation;
                if (sticksToGround) {
                    altMode = AltitudeModes.relativeToGround;
                    elev =
                        EarthElevationModel.getElevationAsGoodAsPossible(
                            Angle.fromDegreesLatitude(posRef.getLat()), Angle.fromDegreesLongitude(posRef.getLon()));
                } else {
                    altMode = AltitudeModes.relativeToStart;
                    if (userData instanceof IFlightplanPositionReferenced) {
                        IFlightplanPositionReferenced wp = (IFlightplanPositionReferenced)userData;
                        elev += wp.getAltWithinM();
                    }
                }

                refPos =
                    new Position(
                        Angle.fromDegreesLatitude(posRef.getLat()), Angle.fromDegreesLongitude(posRef.getLon()), elev);
            } else if (userData instanceof ISectorReferenced) {
                ISectorReferenced sectorRef = (ISectorReferenced)userData;
                OptionalDouble minElev = sectorRef.getMinElev();
                OptionalDouble maxElev = sectorRef.getMaxElev();
                if (minElev.isPresent() && maxElev.isPresent()) {
                    double offset = (minElev.getAsDouble() + maxElev.getAsDouble()) * 0.5;
                    planeRefElevation += offset;
                }

                sticksToGround = false;
                altMode = AltitudeModes.absolute;
                refPos = pickedPosition;
                if (refPos == null) {
                    Intersection[] intersection =
                        globe.intersect(
                            view.computeRayFromScreenPoint(cursorPointStart.x, cursorPointStart.y), planeRefElevation);
                    if (intersection == null) {
                        return false;
                    }

                    refPos = globe.computePositionFromPoint(intersection[0].getIntersectionPoint());
                }

                if (refPos == null) {
                    return false;
                    // if (refPos == null){
                    // ISectorReferenced sectorRef = (ISectorReferenced) userData;
                    // Sector sec = sectorRef.getSector();
                    // if (!sec.equals(Sector.EMPTY_SECTOR)){
                    // refPos = new Position(sec.getCentroid(), sectorRef.getMinElev() + planeRefElevation);
                    // } else {
                    // return;
                    // }
                    // }
                }
            } else if (dragedRenderable instanceof Movable) {
                dragObject = (Movable)dragedRenderable;
                refPos = dragObject.getReferencePosition();
            } else {
                return false;
            }

            if (refPos == null) {
                return false;
            }

            setDragging(true);
        }

        if (sticksToGround) {
            refPos = EarthElevationModel.setOnGround(refPos);
        }

        // mm: wir haben std. meerespiegel als bezug, daher die nutzen
        refPoint = globe.computePointFromPosition(refPos);
        m = globe.computeModelCoordinateOriginTransform(refPos).getInverse();

        // setLastCursor(wwdComponent.getCursor());
        // wwdComponent.setCursor(emptyCursor);

        // Save initial reference points for object and cursor in screen coordinates
        // Note: y is inverted for the object point.
        this.dragRefObjectPoint = view.project(refPoint);
        // Save cursor position
        this.dragRefCursorPoint = cursorPointStart;
        // System.out.println();
        // System.out.println("refPosition:" + refPos);
        // System.out.println("dragRefObjectPoint"+dragRefObjectPoint);
        // System.out.println("dragRefCursorPoint"+dragRefCursorPoint);
        // System.out.println("dragRefObjectPointCorrected x:"+ this.dragRefObjectPoint.x + " y:"
        // +(wwdComponent.getHeight() -
        // this.dragRefObjectPoint.y) );
        // System.out.println("dragOffset x:"+ (this.dragRefObjectPoint.x-this.dragRefCursorPoint.x) + " y:"
        // +(wwdComponent.getHeight() -
        // this.dragRefObjectPoint.y-this.dragRefCursorPoint.y) );
        // System.out.println("sticksToGround:"+sticksToGround);
        if (controller.isFlatEarthMode()) {
            sticksToGround = true;
        }

        switch (altMode) {
        case clampToGround:
            this.dragRefAltitude = globe.computePositionFromPoint(refPoint).getElevation();
            break;
        default:
            this.dragRefAltitude = refPos.getElevation();
        }
        // System.out.println("dragRefAltitude:"+dragRefAltitude);
        lastShift = Vec4.ZERO;
        return true;
    }

    synchronized void setDragging(boolean isDragging) {
        this.dragging = isDragging;
        draggingProperty.set(isDragging);
        if (userData instanceof IFlightplanRelatedObject) {
            IFlightplanRelatedObject fpObj = (IFlightplanRelatedObject)userData;
            CFlightplan fp = fpObj.getFlightplan();
            if (fp != null && isDragging) {
                fp.setMuteAutoRecalc(true);
            }
        }
    }

    public void stopDragging() {
        Platform.runLater(
            () -> {
                if (!tryStopDragging()) return;

                // wwdComponent.setCursor(getLastCursor());
                if (userData instanceof MapLayerStartingPosition) {
                    // making the starting position of the simulator
                    // moveable
                    if (lastPos != null) {
                        MapLayerStartingPosition startPoint = (MapLayerStartingPosition)userData;
                        startPoint.setStartingPoint(lastPos);
                    }
                }

                if (userData instanceof IFlightplanRelatedObject) {
                    IFlightplanRelatedObject fpObj = (IFlightplanRelatedObject)userData;
                    CFlightplan fp = fpObj.getFlightplan();
                    if (fp != null) {
                        if (!fp.setMuteAutoRecalc(false)) {
                            // only trigger rerendering in case of computation isnt happening
                            fp.flightplanStatementChanged(
                                fpObj); // to trigger rendering update since dragging is now over
                        }
                    }
                }
            });
    }

    public void move(Point curserPoint) {
        Platform.runLater(
            () -> {
                if (curserPoint == null) {
                    return;
                }

                if (!isDragging()) {
                    return;
                }

                Line ray = view.computeRayFromScreenPoint(curserPoint.x, curserPoint.y);
                Position pickPos = null;
                if (sticksToGround) {
                    // i didn't understand the following if (maybe we can get rid of it?)
                    // globe.getMaxElevation() == 1 in 2D mode
                    if (view.getEyePosition().getElevation() < globe.getMaxElevation() * 10
                            || globe.getMaxElevation() == 1) {
                        // Use ray casting below some altitude
                        // Try ray intersection with current terrain geometry
                        SceneController sceneController = wwd.getSceneController();
                        SectorGeometryList terrain = sceneController != null ? sceneController.getTerrain() : null;
                        Intersection[] intersections = terrain != null ? terrain.intersect(ray) : null;
                        if (intersections != null && intersections.length > 0) {
                            pickPos = globe.computePositionFromPoint(intersections[0].getIntersectionPoint());
                        } else {
                            // Fallback on raycasting using elevation data
                            pickPos =
                                RayCastingSupport.intersectRayWithTerrain(
                                    globe, ray.getOrigin(), ray.getDirection(), 1000, 2);
                        }
                    }
                }
                // System.out.println("pickpos0: " +pickPos);
                // fallback for groundsticker, or directly for fix altitude dragger
                if (pickPos == null) {
                    // Use intersection with sphere at reference altitude.
                    Intersection inters[] = globe.intersect(ray, this.dragRefAltitude);
                    if (inters != null) {
                        pickPos = globe.computePositionFromPoint(inters[0].getIntersectionPoint());
                    }
                }
                // System.out.println("pickpos1: " +pickPos);

                if (pickPos != null) {
                    // Intersection with globe. Move reference point to the intersection point,
                    // but maintain current altitude.
                    Position p;
                    double ground = 0;
                    try {
                        ground = EarthElevationModel.getElevation(pickPos);
                    } catch (ElevationModelRequestException e) {
                        if (e.isWorst) {
                            ground = this.dragRefAltitude;
                        } else {
                            ground = e.achievedAltitude;
                        }
                    }

                    if (sticksToGround) {
                        p = new Position(pickPos, ground);
                    } else {
                        p = new Position(pickPos, this.dragRefAltitude);
                    }

                    switch (altMode) {
                    case absolute:
                        break;
                    case relativeToGround:
                        p = new Position(p, p.elevation - ground);
                        break;
                    case clampToGround:
                        p = new Position(p, 0);
                        break;
                    case relativeToStart:
                        p = new Position(p, p.elevation - planeRefElevation);
                        break;
                    }

                    moveObject(p);
                }
            });
    }

    Vec4 lastShift;
    Position lastPos;

    MuteVisitor muteVis = new MuteVisitor();

    protected void moveObject(Position p) {
        lastPos = p;

        if (userData instanceof Origin) {
            Origin origin = (Origin)userData;
            origin.setIsAuto(false);
            origin.setDefined(true);
        }

        if (userData instanceof IFlightplanLatLonReferenced) {
            IFlightplanLatLonReferenced latLonRef = (IFlightplanLatLonReferenced)userData;
            latLonRef.setLatLon(p.latitude.degrees, p.longitude.degrees);
        } else if (userData instanceof IFlightplanRelatedObject) {
            IFlightplanRelatedObject fpObj = (IFlightplanRelatedObject)userData;
            Vec4 v = globe.computePointFromPosition(p);
            // System.out.println("new Pos: " + p + " v="+v);
            // System.out.println(" new Pos localCoord: " + v.transformBy4(m));

            v = v.transformBy4(m);

            Vec4 delta = v.subtract3(lastShift);
            // System.out.println(" lastShift" + lastShift);
            // System.out.println(" delta" + delta);
            muteVis.setMute(true);
            lastShift = v;
            AllObjectShifterVisitor vis = new AllObjectShifterVisitor(delta);
            muteVis.startVisit(fpObj);
            vis.startVisit(fpObj);
            muteVis.setMute(false);
            muteVis.startVisit(fpObj);
            if (!(fpObj instanceof CFlightplan)) {
                CFlightplan fp = fpObj.getFlightplan();
                if (fp != null) {
                    fp.flightplanStatementChanged(fpObj);
                }
            }
        } else if (userData instanceof KMLWrapperPosition) {
            KMLWrapperPosition posWrapper = (KMLWrapperPosition)userData;
            posWrapper.setPosition(new Position(p, posWrapper.getPosition().elevation));
        } else if (userData instanceof MapLayerStartingPosition) {
            dragObject.moveTo(p);
        } else if (dragObject != null) {
            dragObject.moveTo(p);
        }

        // Debug.profiler.requestFinished(draggingRequest);
    }

    public Position getLastDraggingPosition() {
        return lastPos;
    }

    private class AllObjectShifterVisitor extends AFlightplanVisitor {

        Vec4 shift;
        // Globe globe;

        public AllObjectShifterVisitor(Vec4 shift) {
            this.shift = shift;
            // this.globe = WWFactory.getGlobe();
            // System.out.println("planeRefElevation:"+ planeRefElevation);
        }

        @Override
        public boolean visit(IFlightplanRelatedObject fpObj) {
            if (fpObj instanceof IFlightplanLatLonReferenced) {
                if (fpObj instanceof ISplittingLine) {
                    return false;
                }

                if (fpObj instanceof StartProcedure) {
                    return false;
                }

                IFlightplanLatLonReferenced latLonRef = (IFlightplanLatLonReferenced)fpObj;
                Angle lat = Angle.fromDegrees(latLonRef.getLat());
                Angle lon = Angle.fromDegrees(latLonRef.getLon());

                Matrix m = globe.computeModelCoordinateOriginTransform(lat, lon, planeRefElevation);
                Vec4 v = shift.transformBy4(m);
                Position pNew = globe.computePositionFromPoint(v);
                // System.out.println("fpObj: " + fpObj +" p: " + (new LatLon(lat,lon)) + " -> " + v + " ---> " + pNew)
                // ;
                latLonRef.setLatLon(pNew.getLatitude().degrees, pNew.getLongitude().degrees);
            }

            return false;
        }

    }

    private static class MuteVisitor extends AFlightplanVisitor implements IMuteable {

        boolean mute;

        public MuteVisitor() {}

        @Override
        public boolean visit(IFlightplanRelatedObject fpObj) {
            IMuteable muteable = null;
            if (fpObj instanceof IMuteable) {
                muteable = (IMuteable)fpObj;
                if (mute) {
                    muteable.setMute(mute);
                } else {
                    if (fpObj instanceof CPicAreaCorners
                            || fpObj instanceof MapLayerPicArea.MapLayerPicAreaFpContainer) {
                        // this way others are informed about the change deep inside
                        muteable.setMute(false);
                    } else {
                        muteable.setSilentUnmute();
                    }
                }
            }

            return false;
        }

        @Override
        public void setMute(boolean mute) {
            this.mute = mute;
        }

        @Override
        public boolean isMute() {
            return mute;
        }

        @Override
        public void setSilentUnmute() {
            this.mute = false;
        }
    }
}
