/*
 * This Source is licenced under the NASA OPEN SOURCE AGREEMENT VERSION 1.3
 *
 * Copyright (C) 2012 United States Government as represented by the Administrator of the
 * National Aeronautics and Space Administration.
 * All Rights Reserved.
 *
 * Modifications by MAVinci GmbH, Germany (C) 2009-2016:
 * adapted from nasa sources to support different view modes
 */
package eu.mavinci.desktop.gui.wwext;

import com.intel.missioncontrol.hardware.IHardwareConfiguration;
import com.intel.missioncontrol.helper.Ensure;
import com.intel.missioncontrol.settings.MapInteractionStyle;
import eu.mavinci.core.helper.MinMaxPair;
import eu.mavinci.core.plane.AirplaneCacheEmptyException;
import eu.mavinci.core.plane.listeners.IAirplaneCacheChangedListener;
import eu.mavinci.core.plane.listeners.IAirplaneListenerGuiClose;
import eu.mavinci.core.plane.listeners.IAirplaneListenerOrientation;
import eu.mavinci.core.plane.listeners.IAirplaneListenerPosition;
import eu.mavinci.core.plane.listeners.IAirplaneListenerPowerOn;
import eu.mavinci.core.plane.listeners.IAirplaneListenerStartPos;
import eu.mavinci.core.plane.sendableobjects.OrientationData;
import eu.mavinci.core.plane.sendableobjects.PositionData;
import eu.mavinci.desktop.gui.doublepanel.camerasettings.CameraHelper;
import eu.mavinci.desktop.gui.doublepanel.planemain.ViewModes;
import eu.mavinci.desktop.gui.widgets.delegatedtree.IPlaneTreeController;
import eu.mavinci.desktop.main.debug.Debug;
import eu.mavinci.plane.IAirplane;
import gov.nasa.worldwind.avlist.AVKey;
import gov.nasa.worldwind.geom.Angle;
import gov.nasa.worldwind.geom.LatLon;
import gov.nasa.worldwind.geom.Position;
import gov.nasa.worldwind.geom.Vec4;
import gov.nasa.worldwind.view.ViewUtil;
import java.lang.ref.WeakReference;
import java.util.logging.Level;
import javafx.beans.value.ObservableValue;

/**
 * This view can handle the different modes from {@link ViewModes} therefore it prevents different kinds of user
 * interaction in certain modes
 *
 * @author caller
 */
public class MView extends MView2d
        implements IAirplaneListenerOrientation,
            IAirplaneListenerPosition,
            IAirplaneCacheChangedListener,
            IAirplaneListenerGuiClose,
            IAirplaneListenerStartPos,
            IAirplaneListenerPowerOn {

    private double lastZoom = getZoom();
    private Angle lastHeading = getHeading();
    private Angle lastPitch = getPitch();

    public static final int MAX_COMMON_ZOOM = 15000;

    private WeakReference<IAirplane> plane;

    Angle fieldOfViewDef;

    private final IPlaneTreeController controller;

    public MView(
            IAirplane plane,
            IPlaneTreeController controller,
            ObservableValue<MapInteractionStyle> mapInteractionStyleProperty) {
        super(controller, plane != null ? plane.getHardwareConfiguration() : null, mapInteractionStyleProperty);

        this.controller = controller;

        fieldOfViewDef = getFieldOfView();
        setPlane(plane);
    }

    public void setPlane(IAirplane plane) {
        IAirplane p = this.plane == null ? null : this.plane.get();
        if (p != null) {
            p.getAirplaneCache().removeChangeListener(this);
            p.removeListener(this);
            setHardwareConfiguration(null);
        }

        this.plane = new WeakReference<IAirplane>(plane);
        if (plane != null) {
            plane.getAirplaneCache().addChangeListener(this);
            plane.addListener(this);
            setHardwareConfiguration(plane.getHardwareConfiguration());
        }
    }

    public IPlaneTreeController getController() {
        return controller;
    }

    OrientationData lastOrientation;

    @Override
    public void recv_orientation(OrientationData o) {
        lastOrientation = o;
        if (viewMode.isPlaneCentered() && !isFlatEarth) {
            IAirplane airplane = plane.get();
            if (airplane != null && airplane.getHardwareConfiguration().getPlatformDescription().isInCopterMode()) {
                setRollInt(Angle.fromDegrees(o.cameraRoll));
                setPitchInt(
                    Angle.fromDegrees(
                        o.cameraPitch
                            - 90)); // in falcon mode we will look into the camera direction, since view has another 90
                // deg orientation offset make minus
                setHeadingInt(Angle.fromDegrees(o.cameraYaw));
            } else {
                // for fixwing we like to look to the flying direction... so even the view has an
                setRollInt(Angle.fromDegrees(o.roll));
                setPitchInt(Angle.fromDegrees(o.pitch)); // in fixwing mode look into nose direction
                setHeadingInt(Angle.fromDegrees(o.yaw));
            }
            //			System.out.println("orient:"+o);
        }
    }

    @Override
    public void setFlatEarth(boolean enabled) {
        super.setFlatEarth(enabled);
        if (!enabled && viewMode.isPlaneCentered()) {
            recv_orientation(lastOrientation);
        }
    }

    @Override
    public void recv_position(PositionData p) {
        double elevationStartPoint;
        Position pos;
        try {
            elevationStartPoint = plane.get().getAirplaneCache().getStartElevOverWGS84();
            pos = plane.get().getAirplaneCache().getCurPos();
        } catch (AirplaneCacheEmptyException e) {
            return;
        }
        //		Position pos = new Position(Angle.fromDegreesLatitude(p.lat), Angle
        //				.fromDegreesLongitude(p.lon), p.altitude / 100
        //				+ elevationStartPoint);

        if (viewMode == ViewModes.STAY && shouldAutoCenter) {
            recenterNow();
        } else if (viewMode == ViewModes.FOLLOW) {
            setCenterPositionInt(pos);

            // maybe adjust zoom
            if (shouldAutoCenter && getZoom() > MAX_COMMON_ZOOM) {
                setZoom(MAX_COMMON_ZOOM);
                //				System.out.println("auto zoom" + shouldAutoCenter);
            }

            shouldAutoCenter = false;
        } else if (viewMode.isPlaneCentered()) {
            // set eye to this position
            double d;
            try {
                d =
                    Math.max(
                        plane.get().getAirplaneCache().getCurAlt(), plane.get().getAirplaneCache().getCurGroundElev());
            } catch (AirplaneCacheEmptyException e) {
                d = p.altitude / 100. + elevationStartPoint;
                Debug.getLog().log(Level.FINER, "cant calculate airplane altitude");
            }

            Angle fieldOfView = fieldOfViewDef;

            IHardwareConfiguration hardwareConfiguration = getHardwareConfiguration();
            Ensure.notNull(hardwareConfiguration);

            if (viewMode == ViewModes.CAMERA) {
                IAirplane planeR = plane.get();
                if (planeR != null) {
                    Vec4[] corners = CameraHelper.getCornerDirections(hardwareConfiguration);
                    MinMaxPair minMax = new MinMaxPair();
                    for (Vec4 v : corners) {
                        minMax.update(v.x);
                        minMax.update(v.y);
                    }

                    double halfWidth = minMax.absMax();

                    halfWidth *= 1.5; // add some margin

                    fieldOfView =
                        Angle.fromRadians(
                            2 * Math.atan(halfWidth / CameraHelper.getFocalLength35mm(hardwareConfiguration)));
                }
            }

            setFieldOfView(fieldOfView);

            Position pos2 = new Position(pos, d);
            setEyePositionInt(pos2);
            shouldAutoCenter = false;
        }
    }

    @Override
    public void recv_startPos(Double lon, Double lat, Integer pressureZero) {
        if (pressureZero != 0) return;
        try {
            if (plane.get().getAirplaneCache().wasLastRecvStartPosMajorChange()) shouldAutoCenter = true;
            doCentering(
                plane.get().getAirplaneCache().getStartPosBaro(),
                plane.get().getAirplaneCache().getStartElevOverWGS84());
        } catch (AirplaneCacheEmptyException e) {
            Debug.getLog().log(Level.SEVERE, "cache shouldn't be empty!", e);
            return;
        }
    }

    public void doCentering(LatLon pos, double elev) {
        if (!shouldAutoCenter) return;

        setCenterPositionInt(new Position(pos, elev));

        // maybe adjust zoom
        if (getZoom() > MAX_COMMON_ZOOM) {
            setZoom(MAX_COMMON_ZOOM);
            //			System.out.println("auto zoom" + shouldAutoCenter);
        }

        shouldAutoCenter = false;
    }

    @Override
    public void viewModeChanges(ViewModes newViewMode) {
        if (this.viewMode.equals(newViewMode)) return;

        ViewModes lastMode = this.viewMode;
        this.viewMode = newViewMode;

        // always a good idea ;-) if you change your view
        stopAnimations();
        stopMovement();
        stopMovementOnCenter();
        if (viewMode != ViewModes.CAMERA) {
            setFieldOfView(fieldOfViewDef);
        }

        if (viewMode.isPlaneCentered()) {
            // so last view wasn't coockpit

            // save this for being able to resore it later on switch back to non
            // cockpit viewmode
            lastZoom = getZoom();
            lastHeading = getHeading();
            lastPitch = getPitch();
            lastPosition = getCenterPosition();

            // zoom = 0, because eye positioin is already in cockpit
            setZoomInt(0);
        } else {
            switch (lastMode) {
            case COCKPIT:
                // System.out.println("recentering");
                if (canFocusOnTerrainCenter()) focusOnTerrainCenter();

                if (!isFlatEarth) {
                    setRollInt(Angle.ZERO); // normal view modes cant roll
                    setHeadingInt(lastHeading);
                    setPitchInt(lastPitch);
                }

                // restore last zoom and camera orientation
                setZoomInt(lastZoom);
                setCenterPositionInt(lastPosition);

                break;
            case FOLLOW:
                if (!isFlatEarth && canFocusOnTerrainCenter()) focusOnTerrainCenter();
                break;

            case STAY:
                // problem with zooming on stay -> follow
                // zoomlevel jumps because of the altitude jump
                shouldInitFollowZoom = true;
                break;

            default:
                break;
            }
        }

        try {
            recv_orientation(plane.get().getAirplaneCache().getOrientation());
        } catch (AirplaneCacheEmptyException e) {
        }

        try {
            recv_position(plane.get().getAirplaneCache().getPosition());
        } catch (AirplaneCacheEmptyException e) {
        }

        firePropertyChange(AVKey.VIEW, null, this);
    }

    protected boolean shouldAutoCenter = true;

    @Override
    public void clearTrackCache() {
        recenterNow();
    }

    private void recenterNow() {
        shouldAutoCenter = true;
        try {
            doCentering(plane.get().getAirplaneCache().getCurLatLon(), plane.get().getAirplaneCache().getCurAlt());
        } catch (AirplaneCacheEmptyException e) {
            try {
                doCentering(
                    plane.get().getAirplaneCache().getStartPosBaro(),
                    plane.get().getAirplaneCache().getStartElevOverWGS84());
            } catch (AirplaneCacheEmptyException e1) {
            }
        }
    }

    @Override
    public void clearAltitudeLogCache() {}

    @Override
    public void clearOrientationCache() {}

    @Override
    public void clearPhotoFootprintCache() {}

    @Override
    public void guiClose() {}

    @Override
    public boolean guiCloseRequest() {
        return true;
    }

    public static final String KEY = "airplaneView";

    @Override
    public void storeToSessionNow() {
        storeState(plane.get().getSession(), KEY);
    }

    public Position getCenterPositionToStore() {
        if (viewMode.isPlaneCentered()) {
            return lastPosition;
        } else if (viewMode == ViewModes.FOLLOW && canFocusOnTerrainCenter()) {
            Position p = getCenterPosition();
            double z = getZoom();
            focusOnTerrainCenter();
            Position center = getCenterPosition();
            setCenterPositionInt(p);
            setZoomInt(z);
            return center;
        } else {
            return super.getCenterPositionToStore();
        }
    }

    public double getZoomToStore() {
        if (viewMode.isPlaneCentered()) {
            return lastZoom;
        } else if (viewMode == ViewModes.FOLLOW && canFocusOnTerrainCenter()) {
            Position p = getCenterPosition();
            double z = getZoom();
            focusOnTerrainCenter();
            double zoom = getZoom();
            setCenterPositionInt(p);
            setZoomInt(z);
            return zoom;
        } else {
            return super.getZoomToStore();
        }
    }

    public Angle getHeadingToStore() {
        if (viewMode.isPlaneCentered()) return lastHeading;
        else return super.getHeadingToStore();
    }

    public Angle getPitchToStore() {
        if (viewMode.isPlaneCentered()) return lastPitch;
        else return super.getPitchToStore();
    }

    @Override
    public void clearArbitratyPlotCache() {}

    @Override
    public void recv_powerOn() {
        shouldAutoCenter = true;
    }

    @Override
    public void clearTrackCacheOld() {}

    // TODO Marco: 2018-04-27 keep this code for later when we like to see zoom to 2D interaction on dataset selection
    /*MapLayerMatch flyToMatch;
    Angle flyToMatchFinalPitch;
    Angle flyToMatchFinalHeading;

    @Override
    public void selectionChange(Object userData) {
        super.selectionChange(userData);
        if (userData instanceof MapLayerMatch) {
            flyToMatch = (MapLayerMatch)userData;

            if (!viewMode.equals(ViewModes.STAY)) return;
            super.stopAnimations();

            try {
                if (wwWidget != null) {
                    wwWidget.preload2D(flyToMatch);
                }
            } catch (IOException e) {
                flyToMatch = null;
                Debug.getLog().log(Level.WARNING, "cant load image", e);
            }

            Ensure.notNull(flyToMatch, "flyToMatch");
            CPhotoLogLine line = flyToMatch.getPhotoLogLine();
            IAirplane strongPlaneRef = plane.get();
            OrientationData o =
                CameraHelper.getCorrectedOrientation(line, 90, strongPlaneRef.getHardwareConfiguration());
            Position p = flyToMatch.getShiftedPositionExport(strongPlaneRef.getHardwareConfiguration());
            flyToMatchFinalHeading = Angle.fromDegrees(o.yaw);
            flyToMatchFinalPitch = Angle.fromDegrees(o.pitch);
            addPanToAnimator(p, flyToMatchFinalHeading, flyToMatchFinalPitch, 0.5, false);

            final MapLayerMatching matching = flyToMatch.getMatching();
            DEPRECATED_ThreadingHelper.getThreadingHelper()
                .invokeLaterOnUiThread(
                    new Runnable() {
                        @Override
                        public void run() {
                            controller.setSelection(matching);
                        }
                    });
        }
    }

    @Override
    public void setZoom(double zoom) {
        super.setZoom(zoom);

        //		System.err.println("zoom:"+zoom + " flyTo:"+flyToMatch);
        MapLayerMatch tempFlyToMatch = flyToMatch;
        if (zoom < 1 && tempFlyToMatch != null) {
            // switch to 2d view
            try {
                if (wwWidget != null) {
                    wwWidget.switchTo2D();
                }

                // make sure that after zooming out, the view isnt too flat, so we are not scrolling infinitily
                // backwards
                super.stopAnimations();
                DEPRECATED_ThreadingHelper.getThreadingHelper()
                    .invokeLaterOnUiThread(
                        new Runnable() {

                            @Override
                            public void run() {
                                setPitch(flyToMatchFinalPitch);
                                setHeading(flyToMatchFinalHeading);
                                if (getPitch().degrees > 80) setPitch(Angle.fromDegrees(80));
                            }
                        });
                // to make sure after zooming out, we are not accidentially 2d-ing again into it
                flyToMatch = null;
            } catch (Exception e) {
                Debug.getLog().log(Level.WARNING, "could not jump into 2d view of matching:" + tempFlyToMatch);
            }
        }
    }*/

    @Override
    protected double computeNearDistance(Position eyePosition) {
        // dont clip objects even when they are on altitude after coming close to them.
        // most likely ;-) we have no images higher than 1000m, so this should be safe!
        // making it always small will lead to z-fighting artefacts on large zoom number in the rendering
        double distanceToSurface = ViewUtil.computeElevationAboveSurface(this.dc, eyePosition);
        if (distanceToSurface < 1000) return 0.5;
        return super.computeNearDistance(eyePosition);
        //		return val;
    }

}
