/*
 * This Source is licenced under the NASA OPEN SOURCE AGREEMENT VERSION 1.3
 *
 * Copyright (C) 2012 United States Government as represented by the Administrator of the
 * National Aeronautics and Space Administration.
 * All Rights Reserved.
 *
 * Modifications by MAVinci GmbH, Germany (C) 2009-2016:
 * support different screen credit types
 */
package eu.mavinci.desktop.gui.wwext;

import eu.mavinci.desktop.helper.SwingHelper;
import gov.nasa.worldwind.avlist.AVKey;
import gov.nasa.worldwind.geom.Position;
import gov.nasa.worldwind.render.AnnotationAttributes;
import gov.nasa.worldwind.render.DrawContext;
import gov.nasa.worldwind.render.ScreenAnnotation;
import gov.nasa.worldwind.render.ScreenCredit;
import gov.nasa.worldwind.util.Logging;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;

public class ScreenCreditAnnotation extends ScreenAnnotation implements ScreenCredit {

    private String link;
    private Rectangle viewport;

    public static int WIDTH = 500;
    public static int HEIGHT = 18;

    boolean isLocalized;

    public void setIsLocalized(boolean isLocalized) {
        this.isLocalized = isLocalized;
    }

    public boolean isLocalized() {
        return isLocalized;
    }

    public ScreenCreditAnnotation(String text, Point position, String link, boolean isLocalized) {
        super(text, position);
        this.isLocalized = isLocalized;
        this.link = link;
        AnnotationAttributes a = getAttributes();
        a.setCornerRadius(0);
        // a.setBackgroundColor(null);
        a.setFrameShape(AVKey.SHAPE_NONE);

        a.setTextColor(Color.WHITE);
        Font f = Font.decode("Arial-PLAIN-12").deriveFont(SwingHelper.scaleFloat(13));
        a.setEffect(AVKey.TEXT_EFFECT_OUTLINE);
        a.setFont(f);
        a.setAdjustWidthToText(AVKey.SIZE_FIXED);
        a.setDrawOffset(new Point(0, 0));
        a.setBorderWidth(0);
        a.setInsets(new Insets(0, 0, 0, 0));
        a.setSize(SwingHelper.scaleDimension(new Dimension(WIDTH, HEIGHT)));
        setAttributes(a);
    }

    public ScreenCreditAnnotation(
            String text, Point position, AnnotationAttributes defaults, String link, boolean isLocalized) {
        super(text, position, defaults);
        this.link = link;
        this.isLocalized = isLocalized;
    }

    public ScreenCreditAnnotation(
            String text, Point position, Font font, Color textColor, String link, boolean isLocalized) {
        super(text, position, font, textColor);
        this.link = link;
        this.isLocalized = isLocalized;
    }

    public ScreenCreditAnnotation(String text, Point position, Font font, String link, boolean isLocalized) {
        super(text, position, font);
        this.link = link;
        this.isLocalized = isLocalized;
    }

    @Override
    public void setViewport(Rectangle viewport) {
        // System.out.println("setViewport" + viewport);
        if (viewport == null) {
            String msg = Logging.getMessage("nullValue.ViewportIsNull");
            Logging.logger().severe(msg);
            throw new IllegalArgumentException(msg);
        }

        this.viewport = viewport;
        // this.setScreenPoint(new Point(viewport.x+50, viewport.y-110));
        this.setScreenPoint(new Point(viewport.x + (int)(WIDTH / 2), viewport.y - (int)(HEIGHT / 2)));
    }

    @Override
    public void draw(DrawContext dc, int width, int height, double opacity, Position pickPosition) {
        super.draw(dc, width, height, opacity, pickPosition);
        // System.out.println("draw");
    }

    @Override
    public Rectangle getViewport() {
        return this.viewport;
    }

    @Override
    public void setLink(String link) {
        this.link = link;
    }

    @Override
    public String getLink() {
        return this.link;
    }

    @Override
    public double getOpacity() {
        return super.attributes.getOpacity();
    }

    @Override
    public void setOpacity(double opacity) {
        super.attributes.setOpacity(opacity);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ScreenCreditAnnotation that = (ScreenCreditAnnotation)o;

        if (link != null ? !link.equals(that.link) : that.link != null) return false;
        return text != null ? text.equals(that.text) : that.text == null;
    }

    @Override
    public int hashCode() {
        int result = link != null ? link.hashCode() : 0;
        result = 31 * result + (text != null ? text.hashCode() : 0);
        return result;
    }
}
