#
# Blacklists "ahci" or "isci" driver depending on input parameter.
#
# blacklist_script.sh <ahci|isci>
#
# The driver provided in the input will be added to 
# blacklist.conf/blacklist file and mkinitrd will be executed.
#

SYSIMAGE_ROOT=/
KERNEL_LIST=
MODULE=megasr
OBJFILE=${MODULE}.ko
GREP=/bin/grep
RHEL_FILE=/etc/redhat-release
SLES_FILE=/etc/SuSE-release
BLACKLIST_FILE=

#
# Find all kernels under /lib/modules that have megasr
#
get_kernel_list()
{
	local MODULES_ROOT=${SYSIMAGE_ROOT}/lib/modules/`uname -r`
	cd $MODULES_ROOT

	KERNEL_LIST=`find . -name $OBJFILE | cut -d / -f 2`

	cd - > /dev/null
}


#
# Blacklist ahci driver so that hotplug mechanism won't try to load the driver
#
blacklist_ahci()
{
	local TEMP_STR=`$GREP ahci $BLACKLIST_FILE`
	
	if [ ! -f $BLACKLIST_FILE ]; then
		echo Error $BLACKLIST_FILE not found. 
		exit 1
	fi

	if [ -z "$TEMP_STR" ]; then
		echo "Blacklisting ahci module"

		cp -f $BLACKLIST_FILE ${BLACKLIST_FILE}.lsi.backup
		echo Copied  file $BLACKLIST_FILE to ${BLACKLIST_FILE}.lsi.backup 

		echo "" >> $BLACKLIST_FILE
		echo "# Blacklist ahci to let megasr load" >> $BLACKLIST_FILE
		echo "blacklist ahci" >> $BLACKLIST_FILE
	fi
}


#
# Blacklist isci driver so that hotplug mechanism won't try to load the driver
#
blacklist_isci()
{
	local TEMP_STR=`$GREP isci $BLACKLIST_FILE`
	
	if [ ! -f $BLACKLIST_FILE ]; then
		echo Error $BLACKLIST_FILE not found. 
		exit 1
	fi

	if [ -z "$TEMP_STR" ]; then
		echo "Blacklisting isci module"

		cp -f $BLACKLIST_FILE ${BLACKLIST_FILE}.lsi.backup
		echo Copied  file $BLACKLIST_FILE to ${BLACKLIST_FILE}.lsi.backup 

		echo "" >> $BLACKLIST_FILE
		echo "# Blacklist isci to let megasr load" >> $BLACKLIST_FILE
		echo "blacklist isci" >> $BLACKLIST_FILE
	fi
}
#
# Make new initrd for RHEL
#
make_new_initrd_for_RHEL()
{
	local KERNEL=`uname -r`
	local BOOT_DIR=${SYSIMAGE_ROOT}/boot
	local BASE_DIR=${SYSIMAGE_ROOT}
	local IMAGE=

	cd $BOOT_DIR

	if [ -f initramfs-${KERNEL}.img ]; then
		IMAGE=initramfs-${KERNEL}.img	
	elif [ -f initrd-${KERNEL}.img ]; then
		IMAGE=initrd-${KERNEL}.img
	else
		echo "Error: Image file not found"
		exit 1
	fi

	echo Found image file $IMAGE

	cp -f $IMAGE ${IMAGE}.lsi.backup

	echo Copied  file $IMAGE to ${IMAGE}.lsi.backup
	echo "Creating initrd for RHEL.."
	/sbin/mkinitrd -f /boot/${IMAGE} $KERNEL
}


#
# Make new initrd for SLES
#
make_new_initrd_for_SLES()
{
	local KERNEL=`uname -r`
	local BOOT_DIR=${SYSIMAGE_ROOT}/boot
	local IMAGE=initrd-${KERNEL}

	cd $BOOT_DIR

	if [ ! -f $IMAGE ]; then
		echo $IMAGE not found
		exit 1
	fi

	cp -f $IMAGE ${IMAGE}.lsi.backup
	echo Copied  file $IMAGE to ${IMAGE}.lsi.backup
	echo "Creating initrd for SLES.."
	/sbin/mkinitrd
}

echo "CAUTION: This script modifies initrd image."
echo "Original image is saved as <imagename>.lsi.backup".
echo "---"
echo ""

get_kernel_list

if [ -z "$KERNEL_LIST" ]; then
	echo "The megasr is not found in /lib/modules/`uname -r`. Nothing to do .."
	exit
fi

# Locate blacklist file
if [ -f ${SYSIMAGE_ROOT}/etc/modprobe.d/blacklist.conf ]; then
	echo Found blacklist file ${SYSIMAGE_ROOT}/etc/modprobe.d/blacklist.conf
	BLACKLIST_FILE=${SYSIMAGE_ROOT}/etc/modprobe.d/blacklist.conf
elif [ -f ${SYSIMAGE_ROOT}/etc/modprobe.d/blacklist ]; then
	echo Found blacklist file ${SYSIMAGE_ROOT}/etc/modprobe.d/blacklist
	BLACKLIST_FILE=${SYSIMAGE_ROOT}/etc/modprobe.d/blacklist
elif [ -f ${SYSIMAGE_ROOT}/etc/modprobe.conf ]; then
	echo Found blacklist file ${SYSIMAGE_ROOT}/etc/modprobe.conf
	BLACKLIST_FILE=${SYSIMAGE_ROOT}/etc/modprobe.conf
else
	echo "Error: Blacklist file not found"
	exit 1
fi

if [ $1 == "ahci" ]; then
	blacklist_ahci
elif [ $1 == "isci" ]; then
	blacklist_isci
else
	echo "Error: Invalid input parameter. Should be [ahci|isci]"
	exit 1
fi

echo ""
if [ -f $RHEL_FILE ]; then
	make_new_initrd_for_RHEL
elif [ -f $SLES_FILE ]; then
	make_new_initrd_for_SLES 
else
	echo "Error: Unsupported OS"
	exit 1
fi

