#
# Blacklists "ahci" or "isci" driver depending on input parameter.
#
# Input parameter "ahci_g" changes the module load order in case of Grantley Servers
#
# install_rpm.sh <ahci|isci|ahci_g>
#
# The driver provided in the input will be added to 
# megasr.conf file and mkinitrd will be executed.
#

SYSIMAGE_ROOT=/
MODULE=megasr
OBJFILE=${MODULE}.ko
GREP=/bin/grep
RHEL_FILE=/etc/redhat-release
SLES_FILE=/etc/SuSE-release
MEGASR_FILE=



#
# Blacklist ahci driver so that hotplug mechanism won't try to load the driver
#
blacklist_ahci()
{
	local TEMP_STR=`$GREP ahci $MEGASR_FILE`
	
	if [ ! -f $MEGASR_FILE ]; then
		echo Error $MEGASR_FILE not found. 
		exit 1
	fi

	if [ -z "$TEMP_STR" ]; then
		echo "Blacklisting ahci module"

		cp -f $MEGASR_FILE ${MEGASR_FILE}.lsi.backup
		echo Copied  file $MEGASR_FILE to ${MEGASR_FILE}.lsi.backup 

		echo "" >> $MEGASR_FILE
		echo "# Blacklist ahci to let megasr load" >> $MEGASR_FILE
		echo "blacklist ahci" >> $MEGASR_FILE
	fi
}


#
# Blacklist isci driver so that hotplug mechanism won't try to load the driver
#
blacklist_isci()
{
	local TEMP_STR=`$GREP isci $MEGASR_FILE`
	
	if [ ! -f $MEGASR_FILE ]; then
		echo Error $MEGASR_FILE not found. 
		exit 1
	fi

	if [ -z "$TEMP_STR" ]; then
		echo "Blacklisting isci module"

		cp -f $MEGASR_FILE ${MEGASR_FILE}.lsi.backup
		echo Copied  file $MEGASR_FILE to ${MEGASR_FILE}.lsi.backup 

		echo "" >> $MEGASR_FILE
		echo "# Blacklist isci to let megasr load" >> $MEGASR_FILE
		echo "blacklist isci" >> $MEGASR_FILE
	fi
}

#
# Changes the module load order to let the megasr driver to load first (Grantley Servers only)
#
Change_Module_Load_Order()
{
	local TEMP_STR=`$GREP ahci $MEGASR_FILE`
	
	# insert module
	insmod /lib/modules/$(uname -r)/extra/megasr/megasr.ko	
	
	if [ ! -f $MEGASR_FILE ]; then
		echo Error $MEGASR_FILE not found. 
		exit 1
	fi

	if [ -z "$TEMP_STR" ]; then
		echo "Changing Module Load order"

		cp -f $MEGASR_FILE ${MEGASR_FILE}.lsi.backup
		echo Copied  file $MEGASR_FILE to ${MEGASR_FILE}.lsi.backup 

		echo "" >> $MEGASR_FILE
		echo "# Change Module Load order to let megasr driver load first" >> $MEGASR_FILE
		echo "install ahci /sbin/modprobe megasr 2>&1 |:; /sbin/modprobe --ignore-install ahci" >> $MEGASR_FILE
	fi
}


#
# Make new initrd for RHEL
#
make_new_initrd_for_RHEL()
{
	local KERNEL=`uname -r`
	local BOOT_DIR=${SYSIMAGE_ROOT}/boot
	local BASE_DIR=${SYSIMAGE_ROOT}
	local IMAGE=

	cd $BOOT_DIR

	if [ -f initramfs-${KERNEL}.img ]; then
		IMAGE=initramfs-${KERNEL}.img	
	elif [ -f initrd-${KERNEL}.img ]; then
		IMAGE=initrd-${KERNEL}.img
	else
		echo "Error: Image file not found"
		exit 1
	fi

	echo Found image file $IMAGE

	cp -f $IMAGE ${IMAGE}.lsi.backup

	echo Copied  file $IMAGE to ${IMAGE}.lsi.backup
	echo "Creating initrd for RHEL.."
	/bin/mkinitrd -f /boot/${IMAGE} $KERNEL
}


echo "CAUTION: This script modifies initrd image."
echo "Original image is saved as <imagename>.lsi.backup".
echo "---"
echo ""

# Install kmod RPM
rpm -ivh kmod-megasr-RHEL7-6.el7.x86_64.rpm 

# Locate megasr conf file
if [ -f ${SYSIMAGE_ROOT}/etc/modprobe.d/megasr.conf ]; then
	echo Found megasr configuration file ${SYSIMAGE_ROOT}/etc/modprobe.d/megasr.conf
	MEGASR_FILE=${SYSIMAGE_ROOT}/etc/modprobe.d/megasr.conf
else
	echo Created megasr configuration file ${SYSIMAGE_ROOT}/etc/modprobe.d/megasr.conf
	touch ${SYSIMAGE_ROOT}/etc/modprobe.d/megasr.conf
	MEGASR_FILE=${SYSIMAGE_ROOT}/etc/modprobe.d/megasr.conf
fi

if [ $1 == "ahci" ]; then
	blacklist_ahci
elif [ $1 == "isci" ]; then
	blacklist_isci
elif [ $1 == "ahci_g" ]; then
	Change_Module_Load_Order	
else
	echo "Error: Invalid input parameter. Should be [ahci|isci|ahci_g]"
	exit 1
fi

echo ""
if [ -f $RHEL_FILE ]; then
	make_new_initrd_for_RHEL
else
	echo "Error: Unsupported OS"
	exit 1
fi

