/*
*        NASA OPEN SOURCE AGREEMENT VERSION 1.3
*
*        THIS OPEN SOURCE AGREEMENT ("AGREEMENT") DEFINES THE RIGHTS OF USE,
*        REPRODUCTION, DISTRIBUTION, MODIFICATION AND REDISTRIBUTION OF CERTAIN
*        COMPUTER SOFTWARE ORIGINALLY RELEASED BY THE UNITED STATES GOVERNMENT
*        AS REPRESENTED BY THE GOVERNMENT AGENCY LISTED BELOW ("GOVERNMENT
*        AGENCY").  THE UNITED STATES GOVERNMENT, AS REPRESENTED BY GOVERNMENT
*        AGENCY, IS AN INTENDED THIRD-PARTY BENEFICIARY OF ALL SUBSEQUENT
*        DISTRIBUTIONS OR REDISTRIBUTIONS OF THE SUBJECT SOFTWARE.  ANYONE WHO
*        USES, REPRODUCES, DISTRIBUTES, MODIFIES OR REDISTRIBUTES THE SUBJECT
*        SOFTWARE, AS DEFINED HEREIN, OR ANY PART THEREOF, IS, BY THAT ACTION,
*        ACCEPTING IN FULL THE RESPONSIBILITIES AND OBLIGATIONS CONTAINED IN
*        THIS AGREEMENT.
*
*        Government Agency: National Aeronautics and Space Administration (NASA)
*        Government Agency Original Software Designation: ARC-15166-1
*        Government Agency Original Software Title: WorldWind Version 1.3
*        User Registration Requested.  Please Visit https://opensource.arc.nasa.gov/
*        Government Agency Point of Contact for Original Software: Patrick.Hogan@nasa.gov
*        ________________________________________________
*
*
*        1. DEFINITIONS
*
*        A. "Contributor" means Government Agency, as the developer of the
*        Original Software, and any entity that makes a Modification.
*        B. "Covered Patents" mean patent claims licensable by a Contributor
*        that are necessarily infringed by the use or sale of its Modification
*        alone or when combined with the Subject Software.
*        C. "Display" means the showing of a copy of the Subject Software,
*        either directly or by means of an image, or any other device.
*        D. "Distribution" means conveyance or transfer of the Subject
*        Software, regardless of means, to another.
*        E. "Larger Work" means computer software that combines Subject
*        Software, or portions thereof, with software separate from the Subject
*        Software that is not governed by the terms of this Agreement.
*        F.  "Modification" means any alteration of, including addition to or
*        deletion from, the substance or structure of either the Original
*        Software or Subject Software, and includes derivative works, as that
*        term is defined in the Copyright Statute, 17 USC 101.  However, the
*        act of including Subject Software as part of a Larger Work does not in
*        and of itself constitute a Modification.
*        G. "Original Software" means the computer software first released
*        under this Agreement by Government Agency with Government Agency
*        designation ARC-15166-1 and entitled WorldWind, including source code,
*        object code and accompanying documentation, if any.
*        H. "Recipient" means anyone who acquires the Subject Software under
*        this Agreement, including all Contributors.
*        I. "Redistribution" means Distribution of the Subject Software after a
*        Modification has been made.
*        J. "Reproduction" means the making of a counterpart, image or copy of
*        the Subject Software.
*        K. "Sale" means the exchange of the Subject Software for money or
*        equivalent value.
*        L. "Subject Software" means the Original Software, Modifications, or
*        any respective parts thereof.
*        M. "Use" means the application or employment of the Subject Software
*        for any purpose.
*
*        2. GRANT OF RIGHTS
*
*        A. Under Non-Patent Rights: Subject to the terms and conditions of
*        this Agreement, each Contributor, with respect to its own contribution
*        to the Subject Software, hereby grants to each Recipient a
*        non-exclusive, world-wide, royalty-free license to engage in the
*        following activities pertaining to the Subject Software:
*
*        1. Use
*        2. Distribution
*        3. Reproduction
*        4. Modification
*        5. Redistribution
*        6. Display
*
*        B. Under Patent Rights: Subject to the terms and conditions of this
*        Agreement, each Contributor, with respect to its own contribution to
*        the Subject Software, hereby grants to each Recipient under Covered
*        Patents a non-exclusive, world-wide, royalty-free license to engage in
*        the following activities pertaining to the Subject Software:
*
*        1. Use
*        2. Distribution
*        3. Reproduction
*        4. Sale
*        5. Offer for Sale
*
*        C. The rights granted under Paragraph B. also apply to the combination
*        of a Contributor's Modification and the Subject Software if, at the
*        time the Modification is added by the Contributor, the addition of
*        such Modification causes the combination to be covered by the Covered
*        Patents.  It does not apply to any other combinations that include a
*        Modification.
*
*        D. The rights granted in Paragraphs A. and B. allow the Recipient to
*        sublicense those same rights.  Such sublicense must be under the same
*        terms and conditions of this Agreement.
*
*        3. OBLIGATIONS OF RECIPIENT
*
*        A. Distribution or Redistribution of the Subject Software must be made
*        under this Agreement except for additions covered under paragraph 3H.
*
*        1. Whenever a Recipient distributes or redistributes the Subject
*        Software, a copy of this Agreement must be included with each copy
*        of the Subject Software; and
*        2. If Recipient distributes or redistributes the Subject Software in
*        any form other than source code, Recipient must also make the
*        source code freely available, and must provide with each copy of
*        the Subject Software information on how to obtain the source code
*        in a reasonable manner on or through a medium customarily used for
*        software exchange.
*
*        B. Each Recipient must ensure that the following copyright notice
*        appears prominently in the Subject Software:
*
*        Copyright (C) 2001 United States Government
*        as represented by the Administrator of the
*        National Aeronautics and Space Administration.
*        All Rights Reserved.
*
*        C. Each Contributor must characterize its alteration of the Subject
*        Software as a Modification and must identify itself as the originator
*        of its Modification in a manner that reasonably allows subsequent
*        Recipients to identify the originator of the Modification.  In
*        fulfillment of these requirements, Contributor must include a file
*        (e.g., a change log file) that describes the alterations made and the
*        date of the alterations, identifies Contributor as originator of the
*        alterations, and consents to characterization of the alterations as a
*        Modification, for example, by including a statement that the
*        Modification is derived, directly or indirectly, from Original
*        Software provided by Government Agency. Once consent is granted, it
*        may not thereafter be revoked.
*
*        D. A Contributor may add its own copyright notice to the Subject
*        Software.  Once a copyright notice has been added to the Subject
*        Software, a Recipient may not remove it without the express permission
*        of the Contributor who added the notice.
*
*        E. A Recipient may not make any representation in the Subject Software
*        or in any promotional, advertising or other material that may be
*        construed as an endorsement by Government Agency or by any prior
*        Recipient of any product or service provided by Recipient, or that may
*        seek to obtain commercial advantage by the fact of Government Agency's
*        or a prior Recipient's participation in this Agreement.
*
*        F. In an effort to track usage and maintain accurate records of the
*        Subject Software, each Recipient, upon receipt of the Subject
*        Software, is requested to register with Government Agency by visiting
*        the following website: https://opensource.arc.nasa.gov.  Recipient's
*        name and personal information shall be used for statistical purposes
*        only. Once a Recipient makes a Modification available, it is requested
*        that the Recipient inform Government Agency at the web site provided
*        above how to access the Modification.
*
*        G. Each Contributor represents that that its Modification is believed
*        to be Contributor's original creation and does not violate any
*        existing agreements, regulations, statutes or rules, and further that
*        Contributor has sufficient rights to grant the rights conveyed by this
*        Agreement.
*
*        H. A Recipient may choose to offer, and to charge a fee for, warranty,
*        support, indemnity and/or liability obligations to one or more other
*        Recipients of the Subject Software.  A Recipient may do so, however,
*        only on its own behalf and not on behalf of Government Agency or any
*        other Recipient.  Such a Recipient must make it absolutely clear that
*        any such warranty, support, indemnity and/or liability obligation is
*        offered by that Recipient alone.  Further, such Recipient agrees to
*        indemnify Government Agency and every other Recipient for any
*        liability incurred by them as a result of warranty, support, indemnity
*        and/or liability offered by such Recipient.
*
*        I. A Recipient may create a Larger Work by combining Subject Software
*        with separate software not governed by the terms of this agreement and
*        distribute the Larger Work as a single product. In such case, the
*        Recipient must make sure Subject Software, or portions thereof,
*        included in the Larger Work is subject to this Agreement.
*
*        J. Notwithstanding any provisions contained herein, Recipient is
*        hereby put on notice that export of any goods or technical data from
*        the United States may require some form of export license from the
*        U.S. Government.  Failure to obtain necessary export licenses may
*        result in criminal liability under U.S. laws.  Government Agency
*        neither represents that a license shall not be required nor that, if
*        required, it shall be issued.  Nothing granted herein provides any
*        such export license.
*
*        4. DISCLAIMER OF WARRANTIES AND LIABILITIES; WAIVER AND INDEMNIFICATION
*
*        A. No Warranty: THE SUBJECT SOFTWARE IS PROVIDED "AS IS" WITHOUT ANY
*        WARRANTY OF ANY KIND, EITHER EXPRESSED, IMPLIED, OR STATUTORY,
*        INCLUDING, BUT NOT LIMITED TO, ANY WARRANTY THAT THE SUBJECT SOFTWARE
*        WILL CONFORM TO SPECIFICATIONS, ANY IMPLIED WARRANTIES OF
*        MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR FREEDOM FROM
*        INFRINGEMENT, ANY WARRANTY THAT THE SUBJECT SOFTWARE WILL BE ERROR
*        FREE, OR ANY WARRANTY THAT DOCUMENTATION, IF PROVIDED, WILL CONFORM TO
*        THE SUBJECT SOFTWARE. THIS AGREEMENT DOES NOT, IN ANY MANNER,
*        CONSTITUTE AN ENDORSEMENT BY GOVERNMENT AGENCY OR ANY PRIOR RECIPIENT
*        OF ANY RESULTS, RESULTING DESIGNS, HARDWARE, SOFTWARE PRODUCTS OR ANY
*        OTHER APPLICATIONS RESULTING FROM USE OF THE SUBJECT SOFTWARE.
*        FURTHER, GOVERNMENT AGENCY DISCLAIMS ALL WARRANTIES AND LIABILITIES
*        REGARDING THIRD-PARTY SOFTWARE, IF PRESENT IN THE ORIGINAL SOFTWARE,
*        AND DISTRIBUTES IT "AS IS."
*
*        B. Waiver and Indemnity: RECIPIENT AGREES TO WAIVE ANY AND ALL CLAIMS
*        AGAINST THE UNITED STATES GOVERNMENT, ITS CONTRACTORS AND
*        SUBCONTRACTORS, AS WELL AS ANY PRIOR RECIPIENT.  IF RECIPIENT'S USE OF
*        THE SUBJECT SOFTWARE RESULTS IN ANY LIABILITIES, DEMANDS, DAMAGES,
*        EXPENSES OR LOSSES ARISING FROM SUCH USE, INCLUDING ANY DAMAGES FROM
*        PRODUCTS BASED ON, OR RESULTING FROM, RECIPIENT'S USE OF THE SUBJECT
*        SOFTWARE, RECIPIENT SHALL INDEMNIFY AND HOLD HARMLESS THE UNITED
*        STATES GOVERNMENT, ITS CONTRACTORS AND SUBCONTRACTORS, AS WELL AS ANY
*        PRIOR RECIPIENT, TO THE EXTENT PERMITTED BY LAW.  RECIPIENT'S SOLE
*        REMEDY FOR ANY SUCH MATTER SHALL BE THE IMMEDIATE, UNILATERAL
*        TERMINATION OF THIS AGREEMENT.
*
*
*        5. GENERAL TERMS
*
*        A. Termination: This Agreement and the rights granted hereunder will
*        terminate automatically if a Recipient fails to comply with these
*        terms and conditions, and fails to cure such noncompliance within
*        thirty (30) days of becoming aware of such noncompliance.  Upon
*        termination, a Recipient agrees to immediately cease use and
*        distribution of the Subject Software.  All sublicenses to the Subject
*        Software properly granted by the breaching Recipient shall survive any
*        such termination of this Agreement.
*
*        B. Severability: If any provision of this Agreement is invalid or
*        unenforceable under applicable law, it shall not affect the validity
*        or enforceability of the remainder of the terms of this Agreement.
*
*        C. Applicable Law: This Agreement shall be subject to United States
*        federal law only for all purposes, including, but not limited to,
*        determining the validity of this Agreement, the meaning of its
*        provisions and the rights, obligations and remedies of the parties.
*
*        D. Entire Understanding: This Agreement constitutes the entire
*        understanding and agreement of the parties relating to release of the
*        Subject Software and may not be superseded, modified or amended except
*        by further written agreement duly executed by the parties.
*
*        E. Binding Authority: By accepting and using the Subject Software
*        under this Agreement, a Recipient affirms its authority to bind the
*        Recipient to all terms and conditions of this Agreement and that that
*        Recipient hereby agrees to all terms and conditions herein.
*
*        F. Point of Contact: Any Recipient contact with Government Agency is
*        to be directed to the designated representative as follows:
*        Patrick.Hogan@nasa.gov.
*    */

public class MView extends MView2d
        implements IAirplaneListenerOrientation,
            IAirplaneListenerPosition,
            IAirplaneListenerStartPos,
            IAirplaneListenerPowerOn {

    private double lastZoom = getZoom();
    private Angle lastHeading = getHeading();
    private Angle lastPitch = getPitch();

    public static final int MAX_COMMON_ZOOM = 15000;

    ReadOnlyObjectProperty<Drone> uav;
    private WeakReference<IAirplane> plane;

    Angle fieldOfViewDef;

    private static final int MIN_ZOOM_LIMIT = 0;
    private static final int MAX_ZOOM_LIMIT = 20000000; // 2000 km

    public MView() {
        super(null);

        fieldOfViewDef = getFieldOfView();

        setPlane(null);
        getOrbitViewLimits().setZoomLimits(MIN_ZOOM_LIMIT, MAX_ZOOM_LIMIT);
    }

    public void setViewPort(int width, int height) {
        this.viewport = new java.awt.Rectangle(0, 0, width, height);
    }

    public MView(ReadOnlyObjectProperty<Drone> uav, SynchronizationRoot syncRoot) {
        super(null);
        this.uav = uav; // if this is just a property path selection, it would get garbage collected.... so lets store a
        // reference
        fieldOfViewDef = getFieldOfView();
        uav.addListener(
            (observable, oldValue, newValue) -> syncRoot.dispatch(() -> setPlane(newValue.getLegacyPlane())));
        Drone uavIns = uav.get();
        setPlane(uavIns != null ? uavIns.getLegacyPlane() : null);
        getOrbitViewLimits().setZoomLimits(MIN_ZOOM_LIMIT, MAX_ZOOM_LIMIT);
    }

    public void setPlane(IAirplane plane) {
        IAirplane p = this.plane == null ? null : this.plane.get();
        if (p != null) {
            p.removeListener(this);
            setHardwareConfiguration(null);
        }

        this.plane = new WeakReference<IAirplane>(plane);
        if (plane != null) {
            plane.addListener(this);
            setHardwareConfiguration(plane.getHardwareConfiguration());
        }
    }

    private OrientationData lastOrientation;

    @Override
    public void recv_orientation(OrientationData o) {
        lastOrientation = o;
        if (viewMode.isPlaneCentered() && !isFlatEarth) {
            IAirplane airplane = plane.get();
            if (airplane != null && airplane.getHardwareConfiguration().getPlatformDescription().isInCopterMode()) {
                setRollInt(Angle.fromDegrees(o.cameraRoll));
                setPitchInt(
                    Angle.fromDegrees(
                        o.cameraPitch
                            - 90)); // in falcon mode we will look into the camera direction, since view has another 90
                // deg orientation offset make minus
                setHeadingInt(Angle.fromDegrees(o.cameraYaw));
            } else {
                // for fixwing we like to look to the flying direction... so even the view has an
                setRollInt(Angle.fromDegrees(o.roll));
                setPitchInt(Angle.fromDegrees(o.pitch)); // in fixwing mode look into nose direction
                setHeadingInt(Angle.fromDegrees(o.yaw));
            }
            //			System.out.println("orient:"+o);
        }
    }

    public void setFlatEarth(boolean enabled) {
        super.setFlatEarth(enabled);
        if (!enabled && viewMode.isPlaneCentered()) {
            recv_orientation(lastOrientation);
        }
    }

    @Override
    public void recv_position(PositionData p) {
        double elevationStartPoint;
        Position pos;
        try {
            elevationStartPoint = plane.get().getAirplaneCache().getStartElevOverWGS84();
            pos = plane.get().getAirplaneCache().getCurPos();
        } catch (AirplaneCacheEmptyException e) {
            return;
        }
        //		Position pos = new Position(Angle.fromDegreesLatitude(p.lat), Angle
        //				.fromDegreesLongitude(p.lon), p.altitude / 100
        //				+ elevationStartPoint);

        if (viewMode == ViewMode.DEFAULT && shouldAutoCenter) {
            recenterNow();
        } else if (viewMode == ViewMode.FOLLOW) {
            setCenterPositionInt(pos);

            // maybe adjust zoom
            if (shouldAutoCenter && getZoom() > MAX_COMMON_ZOOM) {
                setZoom(MAX_COMMON_ZOOM);
                //				System.out.println("auto zoom" + shouldAutoCenter);
            }

            shouldAutoCenter = false;
        } else if (viewMode.isPlaneCentered()) {
            // set eye to this position
            double d;
            try {
                d =
                    Math.max(
                        plane.get().getAirplaneCache().getCurAlt(), plane.get().getAirplaneCache().getCurGroundElev());
            } catch (AirplaneCacheEmptyException e) {
                d = p.altitude / 100. + elevationStartPoint;
                Debug.getLog().log(Level.FINER, "cant calculate airplane altitude");
            }

            Angle fieldOfView = fieldOfViewDef;

            IHardwareConfiguration hardwareConfiguration = getHardwareConfiguration();
            Ensure.notNull(hardwareConfiguration);

            if (viewMode == ViewMode.PAYLOAD) {
                IAirplane planeR = plane.get();
                if (planeR != null) {
                    Vec4[] corners = CameraHelper.getCornerDirections(hardwareConfiguration);
                    MinMaxPair minMax = new MinMaxPair();
                    for (Vec4 v : corners) {
                        minMax.update(v.x);
                        minMax.update(v.y);
                    }

                    double halfWidth = minMax.absMax();

                    halfWidth *= 1.5; // add some margin

                    fieldOfView =
                        Angle.fromRadians(
                            2 * Math.atan(halfWidth / CameraHelper.getFocalLength35mm(hardwareConfiguration)));
                }
            }

            setFieldOfView(fieldOfView);

            Position pos2 = new Position(pos, d);
            setEyePositionInt(pos2);
            shouldAutoCenter = false;
        }
    }

    @Override
    public void recv_startPos(Double lon, Double lat, Integer pressureZero) {
        if (pressureZero != 0) return;
        try {
            if (plane.get().getAirplaneCache().wasLastRecvStartPosMajorChange()) shouldAutoCenter = true;
            doCentering(
                plane.get().getAirplaneCache().getStartPosBaro(),
                plane.get().getAirplaneCache().getStartElevOverWGS84());
        } catch (AirplaneCacheEmptyException e) {
            Debug.getLog().log(Level.SEVERE, "cache shouldn't be empty!", e);
            return;
        }
    }

    public void doCentering(LatLon pos, double elev) {
        if (!shouldAutoCenter) return;

        setCenterPositionInt(new Position(pos, elev));

        // maybe adjust zoom
        if (getZoom() > MAX_COMMON_ZOOM) {
            setZoom(MAX_COMMON_ZOOM);
            //			System.out.println("auto zoom" + shouldAutoCenter);
        }

        shouldAutoCenter = false;
    }

    public void setViewMode(ViewMode newViewMode) {
        if (this.viewMode.equals(newViewMode)) return;

        ViewMode lastMode = this.viewMode;
        this.viewMode = newViewMode;

        // always a good idea ;-) if you change your view
        stopAnimations();
        stopMovement();
        stopMovementOnCenter();
        if (viewMode != ViewMode.PAYLOAD) {
            setFieldOfView(fieldOfViewDef);
        }

        if (viewMode.isPlaneCentered() && !lastMode.isPlaneCentered()) {
            // so last view wasn't coockpit

            // save this for being able to resore it later on switch back to non
            // cockpit viewmode
            lastZoom = getZoom();
            lastHeading = getHeading();
            lastPitch = getPitch();
            lastPosition = getCenterPosition();

            // zoom = 0, because eye position is already in cockpit
            setZoomInt(0);
        } else {
            switch (lastMode) {
            case COCKPIT:
            case PAYLOAD:
                if (canFocusOnTerrainCenter()) focusOnTerrainCenter();

                if (!isFlatEarth) {
                    setRollInt(Angle.ZERO); // normal view modes cant roll
                    setHeadingInt(lastHeading);
                    setPitchInt(lastPitch);
                }

                // restore last zoom and camera orientation
                setZoomInt(lastZoom);
                setCenterPositionInt(lastPosition);

                break;
            case FOLLOW:
                if (!isFlatEarth && canFocusOnTerrainCenter()) focusOnTerrainCenter();
                break;

            case DEFAULT:
                // problem with zooming on stay -> follow
                // zoomlevel jumps because of the altitude jump
                shouldInitFollowZoom = true;
                break;

            default:
                break;
            }
        }

        try {
            recv_orientation(plane.get().getAirplaneCache().getOrientation());
        } catch (AirplaneCacheEmptyException e) {
        }

        try {
            recv_position(plane.get().getAirplaneCache().getPosition());
        } catch (AirplaneCacheEmptyException e) {
        }

        firePropertyChange(AVKey.VIEW, null, this);
    }

    protected boolean shouldAutoCenter = true;

    private void recenterNow() {
        shouldAutoCenter = true;
        try {
            doCentering(plane.get().getAirplaneCache().getCurLatLon(), plane.get().getAirplaneCache().getCurAlt());
        } catch (AirplaneCacheEmptyException e) {
            try {
                doCentering(
                    plane.get().getAirplaneCache().getStartPosBaro(),
                    plane.get().getAirplaneCache().getStartElevOverWGS84());
            } catch (AirplaneCacheEmptyException e1) {
            }
        }
    }

    public static final String KEY = "airplaneView";

    @Override
    public void recv_powerOn() {
        shouldAutoCenter = true;
    }

    @Override
    protected double computeNearDistance(Position eyePosition) {
        // dont clip objects even when they are on altitude after coming close to them.
        // most likely ;-) we have no images higher than 1000m, so this should be safe!
        // making it always small will lead to z-fighting artefacts on large zoom number in the rendering
        double distanceToSurface = ViewUtil.computeElevationAboveSurface(this.dc, eyePosition);
        if (distanceToSurface < 1000) return 0.5;
        return super.computeNearDistance(eyePosition);
    }

}
