/*
*        NASA OPEN SOURCE AGREEMENT VERSION 1.3
*
*        THIS OPEN SOURCE AGREEMENT ("AGREEMENT") DEFINES THE RIGHTS OF USE,
*        REPRODUCTION, DISTRIBUTION, MODIFICATION AND REDISTRIBUTION OF CERTAIN
*        COMPUTER SOFTWARE ORIGINALLY RELEASED BY THE UNITED STATES GOVERNMENT
*        AS REPRESENTED BY THE GOVERNMENT AGENCY LISTED BELOW ("GOVERNMENT
*        AGENCY").  THE UNITED STATES GOVERNMENT, AS REPRESENTED BY GOVERNMENT
*        AGENCY, IS AN INTENDED THIRD-PARTY BENEFICIARY OF ALL SUBSEQUENT
*        DISTRIBUTIONS OR REDISTRIBUTIONS OF THE SUBJECT SOFTWARE.  ANYONE WHO
*        USES, REPRODUCES, DISTRIBUTES, MODIFIES OR REDISTRIBUTES THE SUBJECT
*        SOFTWARE, AS DEFINED HEREIN, OR ANY PART THEREOF, IS, BY THAT ACTION,
*        ACCEPTING IN FULL THE RESPONSIBILITIES AND OBLIGATIONS CONTAINED IN
*        THIS AGREEMENT.
*
*        Government Agency: National Aeronautics and Space Administration (NASA)
*        Government Agency Original Software Designation: ARC-15166-1
*        Government Agency Original Software Title: WorldWind Version 1.3
*        User Registration Requested.  Please Visit https://opensource.arc.nasa.gov/
*        Government Agency Point of Contact for Original Software: Patrick.Hogan@nasa.gov
*        ________________________________________________
*
*
*        1. DEFINITIONS
*
*        A. "Contributor" means Government Agency, as the developer of the
*        Original Software, and any entity that makes a Modification.
*        B. "Covered Patents" mean patent claims licensable by a Contributor
*        that are necessarily infringed by the use or sale of its Modification
*        alone or when combined with the Subject Software.
*        C. "Display" means the showing of a copy of the Subject Software,
*        either directly or by means of an image, or any other device.
*        D. "Distribution" means conveyance or transfer of the Subject
*        Software, regardless of means, to another.
*        E. "Larger Work" means computer software that combines Subject
*        Software, or portions thereof, with software separate from the Subject
*        Software that is not governed by the terms of this Agreement.
*        F.  "Modification" means any alteration of, including addition to or
*        deletion from, the substance or structure of either the Original
*        Software or Subject Software, and includes derivative works, as that
*        term is defined in the Copyright Statute, 17 USC 101.  However, the
*        act of including Subject Software as part of a Larger Work does not in
*        and of itself constitute a Modification.
*        G. "Original Software" means the computer software first released
*        under this Agreement by Government Agency with Government Agency
*        designation ARC-15166-1 and entitled WorldWind, including source code,
*        object code and accompanying documentation, if any.
*        H. "Recipient" means anyone who acquires the Subject Software under
*        this Agreement, including all Contributors.
*        I. "Redistribution" means Distribution of the Subject Software after a
*        Modification has been made.
*        J. "Reproduction" means the making of a counterpart, image or copy of
*        the Subject Software.
*        K. "Sale" means the exchange of the Subject Software for money or
*        equivalent value.
*        L. "Subject Software" means the Original Software, Modifications, or
*        any respective parts thereof.
*        M. "Use" means the application or employment of the Subject Software
*        for any purpose.
*
*        2. GRANT OF RIGHTS
*
*        A. Under Non-Patent Rights: Subject to the terms and conditions of
*        this Agreement, each Contributor, with respect to its own contribution
*        to the Subject Software, hereby grants to each Recipient a
*        non-exclusive, world-wide, royalty-free license to engage in the
*        following activities pertaining to the Subject Software:
*
*        1. Use
*        2. Distribution
*        3. Reproduction
*        4. Modification
*        5. Redistribution
*        6. Display
*
*        B. Under Patent Rights: Subject to the terms and conditions of this
*        Agreement, each Contributor, with respect to its own contribution to
*        the Subject Software, hereby grants to each Recipient under Covered
*        Patents a non-exclusive, world-wide, royalty-free license to engage in
*        the following activities pertaining to the Subject Software:
*
*        1. Use
*        2. Distribution
*        3. Reproduction
*        4. Sale
*        5. Offer for Sale
*
*        C. The rights granted under Paragraph B. also apply to the combination
*        of a Contributor's Modification and the Subject Software if, at the
*        time the Modification is added by the Contributor, the addition of
*        such Modification causes the combination to be covered by the Covered
*        Patents.  It does not apply to any other combinations that include a
*        Modification.
*
*        D. The rights granted in Paragraphs A. and B. allow the Recipient to
*        sublicense those same rights.  Such sublicense must be under the same
*        terms and conditions of this Agreement.
*
*        3. OBLIGATIONS OF RECIPIENT
*
*        A. Distribution or Redistribution of the Subject Software must be made
*        under this Agreement except for additions covered under paragraph 3H.
*
*        1. Whenever a Recipient distributes or redistributes the Subject
*        Software, a copy of this Agreement must be included with each copy
*        of the Subject Software; and
*        2. If Recipient distributes or redistributes the Subject Software in
*        any form other than source code, Recipient must also make the
*        source code freely available, and must provide with each copy of
*        the Subject Software information on how to obtain the source code
*        in a reasonable manner on or through a medium customarily used for
*        software exchange.
*
*        B. Each Recipient must ensure that the following copyright notice
*        appears prominently in the Subject Software:
*
*        Copyright (C) 2001 United States Government
*        as represented by the Administrator of the
*        National Aeronautics and Space Administration.
*        All Rights Reserved.
*
*        C. Each Contributor must characterize its alteration of the Subject
*        Software as a Modification and must identify itself as the originator
*        of its Modification in a manner that reasonably allows subsequent
*        Recipients to identify the originator of the Modification.  In
*        fulfillment of these requirements, Contributor must include a file
*        (e.g., a change log file) that describes the alterations made and the
*        date of the alterations, identifies Contributor as originator of the
*        alterations, and consents to characterization of the alterations as a
*        Modification, for example, by including a statement that the
*        Modification is derived, directly or indirectly, from Original
*        Software provided by Government Agency. Once consent is granted, it
*        may not thereafter be revoked.
*
*        D. A Contributor may add its own copyright notice to the Subject
*        Software.  Once a copyright notice has been added to the Subject
*        Software, a Recipient may not remove it without the express permission
*        of the Contributor who added the notice.
*
*        E. A Recipient may not make any representation in the Subject Software
*        or in any promotional, advertising or other material that may be
*        construed as an endorsement by Government Agency or by any prior
*        Recipient of any product or service provided by Recipient, or that may
*        seek to obtain commercial advantage by the fact of Government Agency's
*        or a prior Recipient's participation in this Agreement.
*
*        F. In an effort to track usage and maintain accurate records of the
*        Subject Software, each Recipient, upon receipt of the Subject
*        Software, is requested to register with Government Agency by visiting
*        the following website: https://opensource.arc.nasa.gov.  Recipient's
*        name and personal information shall be used for statistical purposes
*        only. Once a Recipient makes a Modification available, it is requested
*        that the Recipient inform Government Agency at the web site provided
*        above how to access the Modification.
*
*        G. Each Contributor represents that that its Modification is believed
*        to be Contributor's original creation and does not violate any
*        existing agreements, regulations, statutes or rules, and further that
*        Contributor has sufficient rights to grant the rights conveyed by this
*        Agreement.
*
*        H. A Recipient may choose to offer, and to charge a fee for, warranty,
*        support, indemnity and/or liability obligations to one or more other
*        Recipients of the Subject Software.  A Recipient may do so, however,
*        only on its own behalf and not on behalf of Government Agency or any
*        other Recipient.  Such a Recipient must make it absolutely clear that
*        any such warranty, support, indemnity and/or liability obligation is
*        offered by that Recipient alone.  Further, such Recipient agrees to
*        indemnify Government Agency and every other Recipient for any
*        liability incurred by them as a result of warranty, support, indemnity
*        and/or liability offered by such Recipient.
*
*        I. A Recipient may create a Larger Work by combining Subject Software
*        with separate software not governed by the terms of this agreement and
*        distribute the Larger Work as a single product. In such case, the
*        Recipient must make sure Subject Software, or portions thereof,
*        included in the Larger Work is subject to this Agreement.
*
*        J. Notwithstanding any provisions contained herein, Recipient is
*        hereby put on notice that export of any goods or technical data from
*        the United States may require some form of export license from the
*        U.S. Government.  Failure to obtain necessary export licenses may
*        result in criminal liability under U.S. laws.  Government Agency
*        neither represents that a license shall not be required nor that, if
*        required, it shall be issued.  Nothing granted herein provides any
*        such export license.
*
*        4. DISCLAIMER OF WARRANTIES AND LIABILITIES; WAIVER AND INDEMNIFICATION
*
*        A. No Warranty: THE SUBJECT SOFTWARE IS PROVIDED "AS IS" WITHOUT ANY
*        WARRANTY OF ANY KIND, EITHER EXPRESSED, IMPLIED, OR STATUTORY,
*        INCLUDING, BUT NOT LIMITED TO, ANY WARRANTY THAT THE SUBJECT SOFTWARE
*        WILL CONFORM TO SPECIFICATIONS, ANY IMPLIED WARRANTIES OF
*        MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR FREEDOM FROM
*        INFRINGEMENT, ANY WARRANTY THAT THE SUBJECT SOFTWARE WILL BE ERROR
*        FREE, OR ANY WARRANTY THAT DOCUMENTATION, IF PROVIDED, WILL CONFORM TO
*        THE SUBJECT SOFTWARE. THIS AGREEMENT DOES NOT, IN ANY MANNER,
*        CONSTITUTE AN ENDORSEMENT BY GOVERNMENT AGENCY OR ANY PRIOR RECIPIENT
*        OF ANY RESULTS, RESULTING DESIGNS, HARDWARE, SOFTWARE PRODUCTS OR ANY
*        OTHER APPLICATIONS RESULTING FROM USE OF THE SUBJECT SOFTWARE.
*        FURTHER, GOVERNMENT AGENCY DISCLAIMS ALL WARRANTIES AND LIABILITIES
*        REGARDING THIRD-PARTY SOFTWARE, IF PRESENT IN THE ORIGINAL SOFTWARE,
*        AND DISTRIBUTES IT "AS IS."
*
*        B. Waiver and Indemnity: RECIPIENT AGREES TO WAIVE ANY AND ALL CLAIMS
*        AGAINST THE UNITED STATES GOVERNMENT, ITS CONTRACTORS AND
*        SUBCONTRACTORS, AS WELL AS ANY PRIOR RECIPIENT.  IF RECIPIENT'S USE OF
*        THE SUBJECT SOFTWARE RESULTS IN ANY LIABILITIES, DEMANDS, DAMAGES,
*        EXPENSES OR LOSSES ARISING FROM SUCH USE, INCLUDING ANY DAMAGES FROM
*        PRODUCTS BASED ON, OR RESULTING FROM, RECIPIENT'S USE OF THE SUBJECT
*        SOFTWARE, RECIPIENT SHALL INDEMNIFY AND HOLD HARMLESS THE UNITED
*        STATES GOVERNMENT, ITS CONTRACTORS AND SUBCONTRACTORS, AS WELL AS ANY
*        PRIOR RECIPIENT, TO THE EXTENT PERMITTED BY LAW.  RECIPIENT'S SOLE
*        REMEDY FOR ANY SUCH MATTER SHALL BE THE IMMEDIATE, UNILATERAL
*        TERMINATION OF THIS AGREEMENT.
*
*
*        5. GENERAL TERMS
*
*        A. Termination: This Agreement and the rights granted hereunder will
*        terminate automatically if a Recipient fails to comply with these
*        terms and conditions, and fails to cure such noncompliance within
*        thirty (30) days of becoming aware of such noncompliance.  Upon
*        termination, a Recipient agrees to immediately cease use and
*        distribution of the Subject Software.  All sublicenses to the Subject
*        Software properly granted by the breaching Recipient shall survive any
*        such termination of this Agreement.
*
*        B. Severability: If any provision of this Agreement is invalid or
*        unenforceable under applicable law, it shall not affect the validity
*        or enforceability of the remainder of the terms of this Agreement.
*
*        C. Applicable Law: This Agreement shall be subject to United States
*        federal law only for all purposes, including, but not limited to,
*        determining the validity of this Agreement, the meaning of its
*        provisions and the rights, obligations and remedies of the parties.
*
*        D. Entire Understanding: This Agreement constitutes the entire
*        understanding and agreement of the parties relating to release of the
*        Subject Software and may not be superseded, modified or amended except
*        by further written agreement duly executed by the parties.
*
*        E. Binding Authority: By accepting and using the Subject Software
*        under this Agreement, a Recipient affirms its authority to bind the
*        Recipient to all terms and conditions of this Agreement and that that
*        Recipient hereby agrees to all terms and conditions herein.
*
*        F. Point of Contact: Any Recipient contact with Government Agency is
*        to be directed to the designated representative as follows:
*        Patrick.Hogan@nasa.gov.
*    *//*
 * This Source is licenced under the NASA OPEN SOURCE AGREEMENT VERSION 1.3
 *
 * Copyright (C) 2011 United States Government as represented by the Administrator of the
 * National Aeronautics and Space Administration.
 * All Rights Reserved.
 *
 * Modifications by MAVinci GmbH, Germany (C) 2009-2016: Adding foreground forcing ability to renderer
 *
 */
package eu.mavinci.desktop.gui.wwext;

import com.intel.missioncontrol.PublishSource;
import gov.nasa.worldwind.View;
import gov.nasa.worldwind.avlist.AVKey;
import gov.nasa.worldwind.exception.WWRuntimeException;
import gov.nasa.worldwind.geom.Angle;
import gov.nasa.worldwind.geom.Frustum;
import gov.nasa.worldwind.geom.Position;
import gov.nasa.worldwind.geom.Vec4;
import gov.nasa.worldwind.render.DrawContext;
import gov.nasa.worldwind.render.GeographicText;
import gov.nasa.worldwind.render.OrderedRenderable;
import gov.nasa.worldwind.terrain.SectorGeometryList;
import gov.nasa.worldwind.util.Logging;
import gov.nasa.worldwind.util.OGLTextRenderer;
import gov.nasa.worldwind.util.WWMath;

import com.jogamp.opengl.GL2;
import com.jogamp.opengl.glu.GLU;
import java.awt.Color;
import java.awt.Font;
import java.awt.Point;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;

/**
 * @author dcollins
 * @version $Id: GeographicTextRenderer.java 1 2011-07-16 23:22:47Z dcollins $
 */

public class GeographicTextRendererForgroundAble {
    private gov.nasa.worldwind.render.TextRenderer lastTextRenderer = null;
    private final GLU glu = new GLU();

    private static final Font DEFAULT_FONT = Font.decode("Arial-PLAIN-12");
    private static final Color DEFAULT_COLOR = Color.white;
    private boolean cullText = false;
    private int cullTextMargin = 0;
    private String effect = AVKey.TEXT_EFFECT_SHADOW;

    // Distance scaling and fading
    private double distanceMinScale = 1d;
    private double distanceMaxScale = 1d;
    private double distanceMinOpacity = 1d;
    private boolean isDistanceScaling = false;
    private double lookAtDistance = 0;

    private boolean hasJOGLv111Bug = false;

    public GeographicTextRendererForgroundAble() {}

    boolean alwaysOnTop = false;
    double onTopEyeDistance = 10;

    public double getOnTopEyeDistance() {
        return onTopEyeDistance;
    }

    public void setOnTopEyeDistance(double onTopEyeDistance) {
        this.onTopEyeDistance = onTopEyeDistance;
    }

    public void setAlwaysOnTop(boolean alwaysOnTop) {
        if (this.alwaysOnTop == alwaysOnTop) {
            return;
        }

        this.alwaysOnTop = alwaysOnTop;
    }

    public boolean isAlwaysOnTop() {
        return alwaysOnTop;
    }

    /**
     * Determines whether overlapping text are culled. If <code>true</code> text items are sorted front to back
     * according to their respective priority rather then back to front, and will be drawn only if they do not overlap
     * an already drawn text. If <code>false</code> all text will be drawn back to front whether they overlap or not.
     *
     * @return <code>true</code> if overlapping text are culled.
     */
    public boolean isCullTextEnabled() {
        return cullText;
    }

    /**
     * Set whether overlapping text should be culled. If <code>true</code> text items will be sorted front to back
     * according to their respective priority rather then back to front, and will be drawn only if they do not overlap
     * an already drawn text. If <code>false</code> all text will be drawn back to front whether they overlap or not.
     *
     * @param cullText <code>true</code> if overlapping text should be culled.
     */
    public void setCullTextEnabled(boolean cullText) {
        this.cullText = cullText;
    }

    /**
     * Get the empty margin that surrounds a text item when considering it's bounding rectangle during text culling -
     * see {@link #setCullTextEnabled(boolean)}. The smaller the margin, the closer text items can get before being
     * considered as overlapping.
     *
     * @return the empty margin that surrounds a text item - in pixels.
     */
    public int getCullTextMargin() {
        return this.cullTextMargin;
    }

    /**
     * Set the empty margin that surrounds a text item when considering it's bounding rectangle during text culling -
     * see {@link #setCullTextEnabled(boolean)}. The smaller the margin, the closer text items can get before being
     * considered as overlapping.
     *
     * @param margin the empty margin that surrounds a text item - in pixels.
     */
    public void setCullTextMargin(int margin) {
        this.cullTextMargin = margin;
    }

    /**
     * Get the effect used to decorate the text. Can be one of {@link AVKey#TEXT_EFFECT_SHADOW} (default), {@link
     * AVKey#TEXT_EFFECT_OUTLINE} or {@link AVKey#TEXT_EFFECT_NONE}.
     *
     * @return the effect used for text rendering.
     */
    public String getEffect() {
        return this.effect;
    }

    /**
     * Set the effect used to decorate the text. Can be one of {@link AVKey#TEXT_EFFECT_SHADOW} (default), {@link
     * AVKey#TEXT_EFFECT_OUTLINE} or {@link AVKey#TEXT_EFFECT_NONE}.
     *
     * @param effect the effect to use for text rendering.
     */
    public void setEffect(String effect) {
        if (effect == null) {
            String msg = Logging.getMessage("nullValue.StringIsNull");
            Logging.logger().fine(msg);
            throw new IllegalArgumentException(msg);
        }

        this.effect = effect;
    }

    /**
     * Get the minimum scale that can be applied to a text item when it gets farther away from the eye than the view
     * lookat point - screen center.
     *
     * @return the minimum scale that can be applied to a text item when it gets away from the eye.
     */
    public double getDistanceMinScale() {
        return this.distanceMinScale;
    }

    /**
     * Set the minimum scale that can be applied to a text item when it gets farther away from the eye than the view
     * lookat point - screen center. Use a value less then one to have the text 'fade' away. A value of one will have no
     * effect.
     *
     * @param scale the minimum scale that can be applied to a text item when it gets away from the eye.
     */
    public void setDistanceMinScale(double scale) {
        this.distanceMinScale = scale;
    }

    /**
     * Get the maximum scale that can be applied to a text item when it gets closer to the eye than the view lookat
     * point - screen center.
     *
     * @return the maximum scale that can be applied to a text item when it closer to the eye.
     */
    public double getDistanceMaxScale() {
        return this.distanceMaxScale;
    }

    /**
     * Set the maximum scale that can be applied to a text item when it gets closer to the eye than the view lookat
     * point - screen center. Use a value greater then one to have the text magnified in the foreground. A value of one
     * will have no effect.
     *
     * @param scale the maximum scale that can be applied to a text item when it closer to the eye.
     */
    public void setDistanceMaxScale(double scale) {
        this.distanceMaxScale = scale;
    }

    /**
     * Get the minimum opacity that can be applied to a text item when it gets farther away from the eye than the view
     * lookat point - screen center.
     *
     * @return the minimum opacity that can be applied to a text item when it gets away from the eye.
     */
    public double getDistanceMinOpacity() {
        return this.distanceMinOpacity;
    }

    /**
     * Set the minimum opacity that can be applied to a text item when it gets farther away from the eye than the view
     * lookat point - screen center. Use a value less then one to have the text 'fade' away. A value of one will have no
     * effect.
     *
     * @param opacity the minimum opacity that can be applied to a text item when it gets away from the eye.
     */
    public void setDistanceMinOpacity(double opacity) {
        this.distanceMinOpacity = opacity;
    }

    public void render(DrawContext dc, Iterable<GeographicText> text) {
        this.drawMany(dc, text);
    }

    public void render(DrawContext dc, GeographicText text, Vec4 textPoint) {
        if (!isTextValid(text, false)) {
            return;
        }

        this.drawOne(dc, text, textPoint);
    }

    private void drawMany(DrawContext dc, Iterable<GeographicText> textIterable) {
        if (dc == null) {
            String msg = Logging.getMessage("nullValue.DrawContextIsNull");
            Logging.logger().fine(msg);
            throw new IllegalArgumentException(msg);
        }

        if (textIterable == null) {
            String msg = Logging.getMessage("nullValue.IterableIsNull");
            Logging.logger().fine(msg);
            throw new IllegalArgumentException(msg);
        }

        if (dc.getVisibleSector() == null) {
            return;
        }

        SectorGeometryList geos = dc.getSurfaceGeometry();
        if (geos == null) {
            return;
        }

        Iterator<GeographicText> iterator = textIterable.iterator();
        if (!iterator.hasNext()) {
            return;
        }

        Frustum frustumInModelCoords = dc.getView().getFrustumInModelCoordinates();
        double horizon = dc.getView().getHorizonDistance();

        while (iterator.hasNext()) {
            GeographicText text = iterator.next();
            if (!isTextValid(text, true)) {
                continue;
            }

            if (!text.isVisible()) {
                continue;
            }

            Angle lat = text.getPosition().getLatitude();
            Angle lon = text.getPosition().getLongitude();

            if (!dc.getVisibleSector().contains(lat, lon)) {
                continue;
            }

            Vec4 textPoint =
                geos.getSurfacePoint(lat, lon, text.getPosition().getElevation() * dc.getVerticalExaggeration());
            if (textPoint == null) {
                continue;
            }

            double eyeDistance = isAlwaysOnTop() ? onTopEyeDistance : dc.getView().getEyePoint().distanceTo3(textPoint);
            if (eyeDistance > horizon) {
                continue;
            }

            if (!frustumInModelCoords.contains(textPoint)) {
                continue;
            }

            dc.addOrderedRenderable(new OrderedText(text, textPoint, eyeDistance));
        }
    }

    private void drawOne(DrawContext dc, GeographicText text, Vec4 textPoint) {
        if (dc == null) {
            String msg = Logging.getMessage("nullValue.DrawContextIsNull");
            Logging.logger().fine(msg);
            throw new IllegalArgumentException(msg);
        }

        if (dc.getView() == null) {
            String msg = Logging.getMessage("nullValue.ViewIsNull");
            Logging.logger().fine(msg);
            throw new IllegalArgumentException(msg);
        }

        if (dc.getVisibleSector() == null) {
            return;
        }

        SectorGeometryList geos = dc.getSurfaceGeometry();
        if (geos == null) {
            return;
        }

        if (!text.isVisible()) {
            return;
        }

        if (textPoint == null) {
            if (text.getPosition() == null) {
                return;
            }

            Angle lat = text.getPosition().getLatitude();
            Angle lon = text.getPosition().getLongitude();

            if (!dc.getVisibleSector().contains(lat, lon)) {
                return;
            }

            textPoint =
                geos.getSurfacePoint(lat, lon, text.getPosition().getElevation() * dc.getVerticalExaggeration());
            if (textPoint == null) {
                return;
            }
        }

        double horizon = dc.getView().getHorizonDistance();
        double eyeDistance = isAlwaysOnTop() ? onTopEyeDistance : dc.getView().getEyePoint().distanceTo3(textPoint);
        if (eyeDistance > horizon) {
            return;
        }

        if (!dc.getView().getFrustumInModelCoordinates().contains(textPoint)) {
            return;
        }

        dc.addOrderedRenderable(new OrderedText(text, textPoint, eyeDistance));
    }

    protected static boolean isTextValid(GeographicText text, boolean checkPosition) {
        if (text == null || text.getText() == null) {
            return false;
        }

        // noinspection RedundantIfStatement
        if (checkPosition && text.getPosition() == null) {
            return false;
        }

        return true;
    }

    protected class OrderedText implements OrderedRenderable, Comparable<OrderedText> {
        GeographicText text;
        Vec4 point;
        double eyeDistance;

        OrderedText(GeographicText text, Vec4 point, double eyeDistance) {
            this.text = text;
            this.point = point;
            this.eyeDistance = eyeDistance;
        }

        // When overlapping text are culled we want to sort them front to back by priority.
        public int compareTo(OrderedText t) {
            if (t.text.getPriority() - this.text.getPriority() == 0) {
                return (int)(this.eyeDistance - t.eyeDistance);
            } else {
                return (int)(t.text.getPriority() - this.text.getPriority());
            }
        }

        public double getDistanceFromEye() {
            return this.eyeDistance;
        }

        private GeographicTextRendererForgroundAble getRenderer() {
            return GeographicTextRendererForgroundAble.this;
        }

        public void render(DrawContext dc) {
            GeographicTextRendererForgroundAble.this.beginRendering(dc);
            try {
                if (cullText) {
                    ArrayList<OrderedText> textList = new ArrayList<OrderedText>();
                    textList.add(this);

                    // Draw as many as we can in a batch to save ogl state switching.
                    Object nextItem = dc.peekOrderedRenderables();
                    while (nextItem != null && nextItem instanceof OrderedText) {
                        OrderedText ot = (OrderedText)nextItem;
                        if (ot.getRenderer() != GeographicTextRendererForgroundAble.this) {
                            break;
                        }

                        textList.add(ot);
                        dc.pollOrderedRenderables(); // take it off the queue
                        nextItem = dc.peekOrderedRenderables();
                    }

                    Collections.sort(textList); // sort for rendering priority then front to back

                    ArrayList<Rectangle2D> textBounds = new ArrayList<Rectangle2D>();
                    for (OrderedText ot : textList) {
                        double[] scaleAndOpacity =
                            GeographicTextRendererForgroundAble.this.computeDistanceScaleAndOpacity(dc, ot);
                        Rectangle2D newBounds =
                            GeographicTextRendererForgroundAble.this.computeTextBounds(dc, ot, scaleAndOpacity[0]);
                        if (newBounds == null) {
                            continue;
                        }

                        boolean overlap = false;
                        newBounds =
                            GeographicTextRendererForgroundAble.this.computeExpandedBounds(newBounds, cullTextMargin);
                        for (Rectangle2D rect : textBounds) {
                            if (rect.intersects(newBounds)) {
                                overlap = true;
                            }
                        }

                        if (!overlap) {
                            textBounds.add(newBounds);
                            GeographicTextRendererForgroundAble.this.drawText(
                                dc, ot, scaleAndOpacity[0], scaleAndOpacity[1]);
                        }
                    }
                } else // just draw each label
                {
                    double[] scaleAndOpacity =
                        GeographicTextRendererForgroundAble.this.computeDistanceScaleAndOpacity(dc, this);
                    GeographicTextRendererForgroundAble.this.drawText(dc, this, scaleAndOpacity[0], scaleAndOpacity[1]);
                    // Draw as many as we can in a batch to save ogl state switching.
                    Object nextItem = dc.peekOrderedRenderables();
                    while (nextItem != null && nextItem instanceof OrderedText) {
                        OrderedText ot = (OrderedText)nextItem;
                        if (ot.getRenderer() != GeographicTextRendererForgroundAble.this) {
                            break;
                        }

                        scaleAndOpacity =
                            GeographicTextRendererForgroundAble.this.computeDistanceScaleAndOpacity(dc, ot);
                        GeographicTextRendererForgroundAble.this.drawText(
                            dc, ot, scaleAndOpacity[0], scaleAndOpacity[1]);
                        dc.pollOrderedRenderables(); // take it off the queue
                        nextItem = dc.peekOrderedRenderables();
                    }
                }
            } catch (WWRuntimeException e) {
                Logging.logger().log(java.util.logging.Level.SEVERE, "generic.ExceptionWhileRenderingText", e);
            } catch (Exception e) {
                Logging.logger().log(java.util.logging.Level.SEVERE, "generic.ExceptionWhileRenderingText", e);
            } finally {
                GeographicTextRendererForgroundAble.this.endRendering(dc);
            }
        }

        public void pick(DrawContext dc, java.awt.Point pickPoint) {}
    }

    protected Rectangle2D computeTextBounds(DrawContext dc, OrderedText uText, double scale) throws Exception {
        GeographicText geographicText = uText.text;

        final CharSequence charSequence = geographicText.getText();
        if (charSequence == null) {
            return null;
        }

        final Vec4 screenPoint = dc.getView().project(uText.point);
        if (screenPoint == null) {
            return null;
        }

        Font font = geographicText.getFont();
        if (font == null) {
            font = DEFAULT_FONT;
        }

        try {
            gov.nasa.worldwind.render.TextRenderer textRenderer =
                OGLTextRenderer.getOrCreateTextRenderer(dc.getTextRendererCache(), font);
            if (textRenderer != this.lastTextRenderer) {
                if (this.lastTextRenderer != null) {
                    this.lastTextRenderer.end3DRendering();
                }

                textRenderer.begin3DRendering();
                this.lastTextRenderer = textRenderer;
            }

            Rectangle2D textBound = textRenderer.getBounds(charSequence);
            double x = screenPoint.x - textBound.getWidth() / 2d;
            double y = screenPoint.y + textBound.getHeight() / 2d;
            Rectangle2D bounds = new Rectangle2D.Float();
            bounds.setRect(x, y, textBound.getWidth(), textBound.getHeight());

            return computeScaledBounds(bounds, scale);
        } catch (Exception e) {
            handleTextRendererExceptions(e);
            return null;
        }
    }

    protected Rectangle2D computeScaledBounds(Rectangle2D bounds, double scale) {
        if (scale == 1) {
            return bounds;
        }

        // Scale rectangle from bottom center
        double halfWidth = bounds.getWidth() / 2;
        double halfHeight = bounds.getHeight() / 2;
        bounds.setRect(
            bounds.getX() + halfWidth - halfWidth * scale,
            bounds.getY() - halfHeight + halfHeight * scale,
            bounds.getWidth() * scale,
            bounds.getHeight() * scale);
        return bounds;
    }

    protected Rectangle2D computeExpandedBounds(Rectangle2D bounds, int margin) {
        if (margin == 0) {
            return bounds;
        }

        // Add margin around rectangle
        bounds.setRect(
            bounds.getX() - margin,
            bounds.getY() - margin,
            bounds.getWidth() + margin * 2,
            bounds.getHeight() + margin * 2);
        return bounds;
    }

    protected double[] computeDistanceScaleAndOpacity(DrawContext dc, OrderedText ot) {
        if (!this.isDistanceScaling) {
            return new double[] {1, 1};
        }

        // Determine scale and opacity factors based on distance from eye vs the distance to the look at point.
        double lookAtDistance = this.lookAtDistance;
        double eyeDistance = ot.getDistanceFromEye();
        double distanceFactor = Math.sqrt(lookAtDistance / eyeDistance);
        double scale = WWMath.clamp(distanceFactor, this.getDistanceMinScale(), this.getDistanceMaxScale());
        double opacity = WWMath.clamp(distanceFactor, this.getDistanceMinOpacity(), 1);

        return new double[] {scale, opacity};
    }

    protected double computeLookAtDistance(DrawContext dc) {
        View view = dc.getView();

        // Get point in the middle of the screen
        // TODO: Get a point on the surface rather then the geoid
        Position groundPos =
            view.computePositionFromScreenPoint(view.getViewport().getCenterX(), view.getViewport().getCenterY());

        // Update look at distance if center point found
        if (groundPos != null) {
            // Compute distance from eye to the position in the middle of the screen
            this.lookAtDistance = view.getEyePoint().distanceTo3(dc.getGlobe().computePointFromPosition(groundPos));
        }

        return this.lookAtDistance;
    }

    protected void beginRendering(DrawContext dc) {
        GL2 gl = dc.getGL().getGL2();
        int attribBits =
            GL2.GL_ENABLE_BIT // for enable/disable changes
                | GL2.GL_COLOR_BUFFER_BIT // for alpha test func and ref, and blend
                | GL2.GL_CURRENT_BIT // for current color
                | GL2.GL_DEPTH_BUFFER_BIT // for depth test, depth func, and depth mask
                | GL2.GL_TRANSFORM_BIT // for modelview and perspective
                | GL2.GL_VIEWPORT_BIT; // for depth range
        gl.glPushAttrib(attribBits);

        gl.glMatrixMode(GL2.GL_PROJECTION);
        gl.glPushMatrix();
        gl.glLoadIdentity();
        glu.gluOrtho2D(0, dc.getView().getViewport().width, 0, dc.getView().getViewport().height);
        gl.glMatrixMode(GL2.GL_MODELVIEW);
        gl.glPushMatrix();
        gl.glLoadIdentity();
        gl.glMatrixMode(GL2.GL_TEXTURE);
        gl.glPushMatrix();
        gl.glLoadIdentity();
        // Set model view as current matrix mode
        gl.glMatrixMode(GL2.GL_MODELVIEW);

        // Enable the depth test but don't write to the depth buffer.
        gl.glEnable(GL2.GL_DEPTH_TEST);
        gl.glDepthMask(false);

        // Suppress polygon culling.
        gl.glDisable(GL2.GL_CULL_FACE);

        // Suppress any fully transparent image pixels
        gl.glEnable(GL2.GL_ALPHA_TEST);
        gl.glAlphaFunc(GL2.GL_GREATER, 0.001f);

        // Cache distance scaling values
        this.isDistanceScaling =
            this.getDistanceMinScale() != 1 || this.getDistanceMaxScale() != 1 || this.distanceMinOpacity != 1;
        this.computeLookAtDistance(dc);
    }

    protected void endRendering(DrawContext dc) {
        if (this.lastTextRenderer != null) {
            this.lastTextRenderer.end3DRendering();
            this.lastTextRenderer = null;
        }

        GL2 gl = dc.getGL().getGL2();

        gl.glMatrixMode(GL2.GL_PROJECTION);
        gl.glPopMatrix();
        gl.glMatrixMode(GL2.GL_MODELVIEW);
        gl.glPopMatrix();
        gl.glMatrixMode(GL2.GL_TEXTURE);
        gl.glPopMatrix();

        gl.glPopAttrib();
    }

    protected Vec4 drawText(DrawContext dc, OrderedText uText, double scale, double opacity) throws Exception {
        if (uText.point == null) {
            String msg = Logging.getMessage("nullValue.PointIsNull");
            Logging.logger().fine(msg);
            return null;
        }

        GeographicText geographicText = uText.text;
        GL2 gl = dc.getGL().getGL2();

        final CharSequence charSequence = geographicText.getText();
        if (charSequence == null) {
            return null;
        }

        final Vec4 screenPoint = dc.getView().project(uText.point);
        if (screenPoint == null) {
            return null;
        }

        Font font = geographicText.getFont();
        if (font == null) {
            font = DEFAULT_FONT;
        }

        try {
            gov.nasa.worldwind.render.TextRenderer textRenderer =
                OGLTextRenderer.getOrCreateTextRenderer(dc.getTextRendererCache(), font);
            if (textRenderer != this.lastTextRenderer) {
                if (this.lastTextRenderer != null) {
                    this.lastTextRenderer.end3DRendering();
                }

                textRenderer.begin3DRendering();
                this.lastTextRenderer = textRenderer;
            }

            this.setDepthFunc(dc, uText, screenPoint);

            Rectangle2D textBounds =
                textRenderer.getBounds(charSequence); // note:may already be calculated during culling
            textBounds = this.computeScaledBounds(textBounds, scale);
            Point.Float drawPoint = computeDrawPoint(dc, textBounds, screenPoint);

            if (drawPoint != null) {
                if (scale != 1d) {
                    gl.glScaled(scale, scale, 1d);
                    drawPoint.setLocation(drawPoint.x / (float)scale, drawPoint.y / (float)scale);
                }

                Color color = geographicText.getColor();
                if (color == null) {
                    color = DEFAULT_COLOR;
                }

                color = this.applyOpacity(color, opacity);

                Color background = geographicText.getBackgroundColor();
                if (background != null) {
                    background = this.applyOpacity(background, opacity);
                    textRenderer.setColor(background);
                    if (this.effect.equals(AVKey.TEXT_EFFECT_SHADOW)) {
                        textRenderer.draw3D(charSequence, drawPoint.x + 1, drawPoint.y - 1, 0, 1);
                    } else if (this.effect.equals(AVKey.TEXT_EFFECT_OUTLINE)) {
                        textRenderer.draw3D(charSequence, drawPoint.x + 1, drawPoint.y - 1, 0, 1);
                        textRenderer.draw3D(charSequence, drawPoint.x + 1, drawPoint.y + 1, 0, 1);
                        textRenderer.draw3D(charSequence, drawPoint.x - 1, drawPoint.y - 1, 0, 1);
                        textRenderer.draw3D(charSequence, drawPoint.x - 1, drawPoint.y + 1, 0, 1);
                    }
                }

                textRenderer.setColor(color);
                textRenderer.draw3D(charSequence, drawPoint.x, drawPoint.y, 0, 1);
                textRenderer.flush();

                if (scale != 1d) {
                    gl.glLoadIdentity();
                }
            }
        } catch (Exception e) {
            handleTextRendererExceptions(e);
        }

        return screenPoint;
    }

    protected Color applyOpacity(Color color, double opacity) {
        if (opacity >= 1) {
            return color;
        }

        float[] compArray = color.getRGBComponents(null);
        return new Color(compArray[0], compArray[1], compArray[2], compArray[3] * (float)opacity);
    }

    private void handleTextRendererExceptions(Exception e) throws Exception {
        if (e instanceof IOException) {
            if (!this.hasJOGLv111Bug) {
                // This is likely a known JOGL 1.1.1 bug - see AMZN-287 or 343
                // Log once and then ignore.
                Logging.logger().log(java.util.logging.Level.SEVERE, "generic.ExceptionWhileRenderingText", e);
                this.hasJOGLv111Bug = true;
            }
        } else {
            throw e;
        }
    }

    /**
     * Computes the final draw point for the given rectangle lower left corner and target screen point. If the returned
     * point is <code>null</code> the text will not be drawn.
     *
     * @param dc the current {@link DrawContext}
     * @param rect the text rectangle to draw.
     * @param screenPoint the projected screen point the text relates to.
     * @return the final draw point for the given rectangle lower left corner or <code>null</code>.
     */
    protected Point.Float computeDrawPoint(DrawContext dc, Rectangle2D rect, Vec4 screenPoint) {
        return new Point.Float(
            (float)(screenPoint.x - rect.getWidth() / 2d), (float)(screenPoint.y - rect.getHeight() / 2d));
    }

    protected void setDepthFunc(DrawContext dc, OrderedText uText, Vec4 screenPoint) {
        GL2 gl = dc.getGL().getGL2();

        if (isAlwaysOnTop()) {
            gl.glDepthFunc(GL2.GL_ALWAYS);
            return;
        }

        Position eyePos = dc.getView().getEyePosition();
        if (eyePos == null) {
            gl.glDepthFunc(GL2.GL_ALWAYS);
            return;
        }

        double altitude = eyePos.getElevation();
        if (altitude < (dc.getGlobe().getMaxElevation() * dc.getVerticalExaggeration())) {
            double depth = screenPoint.z - (8d * 0.00048875809d);
            depth = depth < 0d ? 0d : (depth > 1d ? 1d : depth);
            gl.glDepthFunc(GL2.GL_LESS);
            gl.glDepthRange(depth, depth);
        }
        // else if (screenPoint.z >= 1d)
        // {
        // gl.glDepthFunc(GL2.GL_EQUAL);
        // gl.glDepthRange(1d, 1d);
        // }
        else {
            gl.glDepthFunc(GL2.GL_ALWAYS);
        }
    }
}
