/*
*        NASA OPEN SOURCE AGREEMENT VERSION 1.3
*
*        THIS OPEN SOURCE AGREEMENT ("AGREEMENT") DEFINES THE RIGHTS OF USE,
*        REPRODUCTION, DISTRIBUTION, MODIFICATION AND REDISTRIBUTION OF CERTAIN
*        COMPUTER SOFTWARE ORIGINALLY RELEASED BY THE UNITED STATES GOVERNMENT
*        AS REPRESENTED BY THE GOVERNMENT AGENCY LISTED BELOW ("GOVERNMENT
*        AGENCY").  THE UNITED STATES GOVERNMENT, AS REPRESENTED BY GOVERNMENT
*        AGENCY, IS AN INTENDED THIRD-PARTY BENEFICIARY OF ALL SUBSEQUENT
*        DISTRIBUTIONS OR REDISTRIBUTIONS OF THE SUBJECT SOFTWARE.  ANYONE WHO
*        USES, REPRODUCES, DISTRIBUTES, MODIFIES OR REDISTRIBUTES THE SUBJECT
*        SOFTWARE, AS DEFINED HEREIN, OR ANY PART THEREOF, IS, BY THAT ACTION,
*        ACCEPTING IN FULL THE RESPONSIBILITIES AND OBLIGATIONS CONTAINED IN
*        THIS AGREEMENT.
*
*        Government Agency: National Aeronautics and Space Administration (NASA)
*        Government Agency Original Software Designation: ARC-15166-1
*        Government Agency Original Software Title: WorldWind Version 1.3
*        User Registration Requested.  Please Visit https://opensource.arc.nasa.gov/
*        Government Agency Point of Contact for Original Software: Patrick.Hogan@nasa.gov
*        ________________________________________________
*
*
*        1. DEFINITIONS
*
*        A. "Contributor" means Government Agency, as the developer of the
*        Original Software, and any entity that makes a Modification.
*        B. "Covered Patents" mean patent claims licensable by a Contributor
*        that are necessarily infringed by the use or sale of its Modification
*        alone or when combined with the Subject Software.
*        C. "Display" means the showing of a copy of the Subject Software,
*        either directly or by means of an image, or any other device.
*        D. "Distribution" means conveyance or transfer of the Subject
*        Software, regardless of means, to another.
*        E. "Larger Work" means computer software that combines Subject
*        Software, or portions thereof, with software separate from the Subject
*        Software that is not governed by the terms of this Agreement.
*        F.  "Modification" means any alteration of, including addition to or
*        deletion from, the substance or structure of either the Original
*        Software or Subject Software, and includes derivative works, as that
*        term is defined in the Copyright Statute, 17 USC 101.  However, the
*        act of including Subject Software as part of a Larger Work does not in
*        and of itself constitute a Modification.
*        G. "Original Software" means the computer software first released
*        under this Agreement by Government Agency with Government Agency
*        designation ARC-15166-1 and entitled WorldWind, including source code,
*        object code and accompanying documentation, if any.
*        H. "Recipient" means anyone who acquires the Subject Software under
*        this Agreement, including all Contributors.
*        I. "Redistribution" means Distribution of the Subject Software after a
*        Modification has been made.
*        J. "Reproduction" means the making of a counterpart, image or copy of
*        the Subject Software.
*        K. "Sale" means the exchange of the Subject Software for money or
*        equivalent value.
*        L. "Subject Software" means the Original Software, Modifications, or
*        any respective parts thereof.
*        M. "Use" means the application or employment of the Subject Software
*        for any purpose.
*
*        2. GRANT OF RIGHTS
*
*        A. Under Non-Patent Rights: Subject to the terms and conditions of
*        this Agreement, each Contributor, with respect to its own contribution
*        to the Subject Software, hereby grants to each Recipient a
*        non-exclusive, world-wide, royalty-free license to engage in the
*        following activities pertaining to the Subject Software:
*
*        1. Use
*        2. Distribution
*        3. Reproduction
*        4. Modification
*        5. Redistribution
*        6. Display
*
*        B. Under Patent Rights: Subject to the terms and conditions of this
*        Agreement, each Contributor, with respect to its own contribution to
*        the Subject Software, hereby grants to each Recipient under Covered
*        Patents a non-exclusive, world-wide, royalty-free license to engage in
*        the following activities pertaining to the Subject Software:
*
*        1. Use
*        2. Distribution
*        3. Reproduction
*        4. Sale
*        5. Offer for Sale
*
*        C. The rights granted under Paragraph B. also apply to the combination
*        of a Contributor's Modification and the Subject Software if, at the
*        time the Modification is added by the Contributor, the addition of
*        such Modification causes the combination to be covered by the Covered
*        Patents.  It does not apply to any other combinations that include a
*        Modification.
*
*        D. The rights granted in Paragraphs A. and B. allow the Recipient to
*        sublicense those same rights.  Such sublicense must be under the same
*        terms and conditions of this Agreement.
*
*        3. OBLIGATIONS OF RECIPIENT
*
*        A. Distribution or Redistribution of the Subject Software must be made
*        under this Agreement except for additions covered under paragraph 3H.
*
*        1. Whenever a Recipient distributes or redistributes the Subject
*        Software, a copy of this Agreement must be included with each copy
*        of the Subject Software; and
*        2. If Recipient distributes or redistributes the Subject Software in
*        any form other than source code, Recipient must also make the
*        source code freely available, and must provide with each copy of
*        the Subject Software information on how to obtain the source code
*        in a reasonable manner on or through a medium customarily used for
*        software exchange.
*
*        B. Each Recipient must ensure that the following copyright notice
*        appears prominently in the Subject Software:
*
*        Copyright (C) 2001 United States Government
*        as represented by the Administrator of the
*        National Aeronautics and Space Administration.
*        All Rights Reserved.
*
*        C. Each Contributor must characterize its alteration of the Subject
*        Software as a Modification and must identify itself as the originator
*        of its Modification in a manner that reasonably allows subsequent
*        Recipients to identify the originator of the Modification.  In
*        fulfillment of these requirements, Contributor must include a file
*        (e.g., a change log file) that describes the alterations made and the
*        date of the alterations, identifies Contributor as originator of the
*        alterations, and consents to characterization of the alterations as a
*        Modification, for example, by including a statement that the
*        Modification is derived, directly or indirectly, from Original
*        Software provided by Government Agency. Once consent is granted, it
*        may not thereafter be revoked.
*
*        D. A Contributor may add its own copyright notice to the Subject
*        Software.  Once a copyright notice has been added to the Subject
*        Software, a Recipient may not remove it without the express permission
*        of the Contributor who added the notice.
*
*        E. A Recipient may not make any representation in the Subject Software
*        or in any promotional, advertising or other material that may be
*        construed as an endorsement by Government Agency or by any prior
*        Recipient of any product or service provided by Recipient, or that may
*        seek to obtain commercial advantage by the fact of Government Agency's
*        or a prior Recipient's participation in this Agreement.
*
*        F. In an effort to track usage and maintain accurate records of the
*        Subject Software, each Recipient, upon receipt of the Subject
*        Software, is requested to register with Government Agency by visiting
*        the following website: https://opensource.arc.nasa.gov.  Recipient's
*        name and personal information shall be used for statistical purposes
*        only. Once a Recipient makes a Modification available, it is requested
*        that the Recipient inform Government Agency at the web site provided
*        above how to access the Modification.
*
*        G. Each Contributor represents that that its Modification is believed
*        to be Contributor's original creation and does not violate any
*        existing agreements, regulations, statutes or rules, and further that
*        Contributor has sufficient rights to grant the rights conveyed by this
*        Agreement.
*
*        H. A Recipient may choose to offer, and to charge a fee for, warranty,
*        support, indemnity and/or liability obligations to one or more other
*        Recipients of the Subject Software.  A Recipient may do so, however,
*        only on its own behalf and not on behalf of Government Agency or any
*        other Recipient.  Such a Recipient must make it absolutely clear that
*        any such warranty, support, indemnity and/or liability obligation is
*        offered by that Recipient alone.  Further, such Recipient agrees to
*        indemnify Government Agency and every other Recipient for any
*        liability incurred by them as a result of warranty, support, indemnity
*        and/or liability offered by such Recipient.
*
*        I. A Recipient may create a Larger Work by combining Subject Software
*        with separate software not governed by the terms of this agreement and
*        distribute the Larger Work as a single product. In such case, the
*        Recipient must make sure Subject Software, or portions thereof,
*        included in the Larger Work is subject to this Agreement.
*
*        J. Notwithstanding any provisions contained herein, Recipient is
*        hereby put on notice that export of any goods or technical data from
*        the United States may require some form of export license from the
*        U.S. Government.  Failure to obtain necessary export licenses may
*        result in criminal liability under U.S. laws.  Government Agency
*        neither represents that a license shall not be required nor that, if
*        required, it shall be issued.  Nothing granted herein provides any
*        such export license.
*
*        4. DISCLAIMER OF WARRANTIES AND LIABILITIES; WAIVER AND INDEMNIFICATION
*
*        A. No Warranty: THE SUBJECT SOFTWARE IS PROVIDED "AS IS" WITHOUT ANY
*        WARRANTY OF ANY KIND, EITHER EXPRESSED, IMPLIED, OR STATUTORY,
*        INCLUDING, BUT NOT LIMITED TO, ANY WARRANTY THAT THE SUBJECT SOFTWARE
*        WILL CONFORM TO SPECIFICATIONS, ANY IMPLIED WARRANTIES OF
*        MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR FREEDOM FROM
*        INFRINGEMENT, ANY WARRANTY THAT THE SUBJECT SOFTWARE WILL BE ERROR
*        FREE, OR ANY WARRANTY THAT DOCUMENTATION, IF PROVIDED, WILL CONFORM TO
*        THE SUBJECT SOFTWARE. THIS AGREEMENT DOES NOT, IN ANY MANNER,
*        CONSTITUTE AN ENDORSEMENT BY GOVERNMENT AGENCY OR ANY PRIOR RECIPIENT
*        OF ANY RESULTS, RESULTING DESIGNS, HARDWARE, SOFTWARE PRODUCTS OR ANY
*        OTHER APPLICATIONS RESULTING FROM USE OF THE SUBJECT SOFTWARE.
*        FURTHER, GOVERNMENT AGENCY DISCLAIMS ALL WARRANTIES AND LIABILITIES
*        REGARDING THIRD-PARTY SOFTWARE, IF PRESENT IN THE ORIGINAL SOFTWARE,
*        AND DISTRIBUTES IT "AS IS."
*
*        B. Waiver and Indemnity: RECIPIENT AGREES TO WAIVE ANY AND ALL CLAIMS
*        AGAINST THE UNITED STATES GOVERNMENT, ITS CONTRACTORS AND
*        SUBCONTRACTORS, AS WELL AS ANY PRIOR RECIPIENT.  IF RECIPIENT'S USE OF
*        THE SUBJECT SOFTWARE RESULTS IN ANY LIABILITIES, DEMANDS, DAMAGES,
*        EXPENSES OR LOSSES ARISING FROM SUCH USE, INCLUDING ANY DAMAGES FROM
*        PRODUCTS BASED ON, OR RESULTING FROM, RECIPIENT'S USE OF THE SUBJECT
*        SOFTWARE, RECIPIENT SHALL INDEMNIFY AND HOLD HARMLESS THE UNITED
*        STATES GOVERNMENT, ITS CONTRACTORS AND SUBCONTRACTORS, AS WELL AS ANY
*        PRIOR RECIPIENT, TO THE EXTENT PERMITTED BY LAW.  RECIPIENT'S SOLE
*        REMEDY FOR ANY SUCH MATTER SHALL BE THE IMMEDIATE, UNILATERAL
*        TERMINATION OF THIS AGREEMENT.
*
*
*        5. GENERAL TERMS
*
*        A. Termination: This Agreement and the rights granted hereunder will
*        terminate automatically if a Recipient fails to comply with these
*        terms and conditions, and fails to cure such noncompliance within
*        thirty (30) days of becoming aware of such noncompliance.  Upon
*        termination, a Recipient agrees to immediately cease use and
*        distribution of the Subject Software.  All sublicenses to the Subject
*        Software properly granted by the breaching Recipient shall survive any
*        such termination of this Agreement.
*
*        B. Severability: If any provision of this Agreement is invalid or
*        unenforceable under applicable law, it shall not affect the validity
*        or enforceability of the remainder of the terms of this Agreement.
*
*        C. Applicable Law: This Agreement shall be subject to United States
*        federal law only for all purposes, including, but not limited to,
*        determining the validity of this Agreement, the meaning of its
*        provisions and the rights, obligations and remedies of the parties.
*
*        D. Entire Understanding: This Agreement constitutes the entire
*        understanding and agreement of the parties relating to release of the
*        Subject Software and may not be superseded, modified or amended except
*        by further written agreement duly executed by the parties.
*
*        E. Binding Authority: By accepting and using the Subject Software
*        under this Agreement, a Recipient affirms its authority to bind the
*        Recipient to all terms and conditions of this Agreement and that that
*        Recipient hereby agrees to all terms and conditions herein.
*
*        F. Point of Contact: Any Recipient contact with Government Agency is
*        to be directed to the designated representative as follows:
*        Patrick.Hogan@nasa.gov.
*    *//*
 * This Source is licenced under the NASA OPEN SOURCE AGREEMENT VERSION 1.3
 *
 * Copyright (C) 2001,2009 United States Government as represented by the Administrator of the
 * National Aeronautics and Space Administration.
 * All Rights Reserved.
 *
 * Modifications by MAVinci GmbH, Germany (C) 2009-2016: make layer and renderable avaliable in picking
 *
 */
package eu.mavinci.desktop.gui.wwext;

import com.intel.missioncontrol.PublishSource;
import com.jogamp.common.nio.Buffers;
import eu.mavinci.desktop.gui.doublepanel.planemain.tagging.AMapLayerCoverage.GridPoint;
import eu.mavinci.desktop.main.debug.Debug;
import gov.nasa.worldwind.WorldWind;
import gov.nasa.worldwind.geom.Angle;
import gov.nasa.worldwind.geom.Extent;
import gov.nasa.worldwind.geom.Matrix;
import gov.nasa.worldwind.geom.Position;
import gov.nasa.worldwind.geom.Sector;
import gov.nasa.worldwind.geom.Vec4;
import gov.nasa.worldwind.layers.Layer;
import gov.nasa.worldwind.pick.PickSupport;
import gov.nasa.worldwind.render.AbstractSurfaceObject;
import gov.nasa.worldwind.render.DrawContext;
import gov.nasa.worldwind.render.OrderedRenderable;
import gov.nasa.worldwind.render.PreRenderable;
import gov.nasa.worldwind.render.Renderable;
import gov.nasa.worldwind.util.BufferWrapper;
import gov.nasa.worldwind.util.Logging;
import gov.nasa.worldwind.util.OGLUtil;
import gov.nasa.worldwind.util.SurfaceTileDrawContext;
import gov.nasa.worldwind.util.WWMath;
import gov.nasa.worldwindx.examples.analytics.AnalyticSurfaceAttributes;
import eu.mavinci.desktop.gui.doublepanel.planemain.tagging.AMapLayerCoverage;
import eu.mavinci.desktop.main.debug.Debug;

import com.jogamp.opengl.GL;
import com.jogamp.opengl.GL2;
import java.awt.Color;
import java.awt.Point;
import java.nio.ByteBuffer;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

/**
 * AnalyticSurface represents a connected grid of geographic locations, covering a specified {@link Sector} at a
 * specified base altitude in meters. The number of grid locations is defined by the AnalyticSurface's dimensions. The
 * default dimensions are <code>(10, 10)</code>. Callers specify the dimensions by using one of the constructors
 * accepting <code>width</code> and <code>height</code>, or by invoking {@link #setDimensions(int, int)}. Each grid
 * point has the following set of attributes:
 *
 * <ul>
 *   <li>Scalar value : the grid point's height relative to the surface's base altitude, both in meters
 *   <li>Color : the grid point's RGBA color components
 * </ul>
 *
 * Callers specify the attributes at each grid point by invoking {@link #setValues(Iterable)} with an {@link Iterable}
 * of {@link GridPointAttributes}. Grid points are assigned attributes from this iterable staring at the upper left hand
 * corner, and proceeding in row-first order across the grid. The iterable should contain at least <code>width * height
 * </code> values, where width and height are the AnalyticSurface's grid dimensions. If the caller does not specify any
 * GridPointAttributes, or the caller specified iterable contains too few values, the unassigned grid points are given
 * default attributes: the default scalar value is 0, and the default color is {@link java.awt.Color#BLACK}.
 *
 * @author dcollins
 * @version $Id: AnalyticSurface.java 1 2011-07-16 23:22:47Z dcollins $
 */

public class MAnalyticSurface implements Renderable, PreRenderable {
    /** GridPointAttributes defines the properties associated with a single grid point of an AnalyticSurface. */
    public interface GridPointAttributes {
        /**
         * Returns the scalar value associated with a grid point. By default, AnalyticSurface interprets this value as
         * the grid point's height relative to the AnalyticSurface's base altitude, both in meters.
         *
         * @return the grid point's scalar value.
         */
        double getValue();

        /**
         * Returns the {@link java.awt.Color} associated with a grid point. By default, AnalyticSurface interprets this
         * Color as the RGBA components of a grid point's RGBA color.
         *
         * @return the grid point's RGB color components.
         */
        java.awt.Color getColor();
    }

    protected static final double DEFAULT_ALTITUDE = 0d;
    /** The default altitude mode. */
    protected static final int DEFAULT_ALTITUDE_MODE = WorldWind.ABSOLUTE;

    protected static final int DEFAULT_DIMENSION = 10;
    protected static final double DEFAULT_VALUE = 0d;
    protected static final Color DEFAULT_COLOR = Color.BLACK;
    protected static final GridPointAttributes DEFAULT_GRID_POINT_ATTRIBUTES =
        createGridPointAttributes(DEFAULT_VALUE, DEFAULT_COLOR);
    /** The time period between surface regeneration when altitude mode is relative-to-ground. */
    protected static final long RELATIVE_TO_GROUND_REGEN_PERIOD = 2000;

    protected boolean visible = true;
    protected Sector sector;
    protected double altitude;
    protected int altitudeMode = DEFAULT_ALTITUDE_MODE;
    protected int width;
    protected int height;
    protected Iterable<? extends GridPointAttributes> values;
    protected double verticalScale = 1d;
    protected AnalyticSurfaceAttributes surfaceAttributes = new AnalyticSurfaceAttributes();
    protected Object pickObject;
    protected Layer clientLayer;
    // Runtime rendering state.
    protected double[] extremeValues;
    protected boolean expired = true;
    protected boolean updateFailed;
    protected Object globeStateKey;
    protected long regenTime;
    // Computed surface rendering properties.
    protected Position referencePos;
    protected Vec4 referencePoint;
    protected RenderInfo surfaceRenderInfo;
    protected AnalyticSurfaceObject clampToGroundSurface;
    protected AnalyticSurfaceObject shadowSurface;
    protected final PickSupport pickSupport = new PickSupport();

    /**
     * Constructs a new AnalyticSurface with the specified {@link Sector}, base altitude in meters, grid dimensions, and
     * iterable of GridPointAttributes. The iterable should contain at least <code>with * height</code> non-null
     * GridPointAttributes.
     *
     * @param sector the Sector which defines the surface's geographic region.
     * @param altitude the base altitude to place the surface at, in meters.
     * @param width the surface grid width, in number of grid points.
     * @param height the surface grid height, in number of grid points.
     * @param iterable the attributes associated with each grid point.
     * @throws IllegalArgumentException if the sector is null, if the width is less than 1, if the height is less than
     *     1, or if the iterable is null.
     */
    public MAnalyticSurface(
            Sector sector, double altitude, int width, int height, Iterable<? extends GridPointAttributes> iterable) {
        if (sector == null) {
            String message = Logging.getMessage("nullValue.SectorIsNull");
            Logging.logger().severe(message);
            throw new IllegalArgumentException(message);
        }

        if (width <= 0) {
            String message = Logging.getMessage("Geom.WidthInvalid", width);
            Logging.logger().severe(message);
            throw new IllegalArgumentException(message);
        }

        if (height <= 0) {
            String message = Logging.getMessage("Geom.HeightInvalid", width);
            Logging.logger().severe(message);
            throw new IllegalArgumentException(message);
        }

        if (iterable == null) {
            String message = Logging.getMessage("nullValue.IterableIsNull");
            Logging.logger().severe(message);
            throw new IllegalArgumentException(message);
        }

        this.sector = sector;
        this.altitude = altitude;
        this.width = width;
        this.height = height;
        this.values = iterable;
        this.setExpired(true);
    }

    /**
     * Constructs a new AnalyticSurface with the specified {@link Sector}, base altitude in meters, and grid dimensions.
     * The new AnalyticSurface has the default {@link GridPointAttributes}.
     *
     * @param sector the Sector which defines the surface's geographic region.
     * @param altitude the base altitude to place the surface at, in meters.
     * @param width the surface grid width, in number of grid points.
     * @param height the surface grid height, in number of grid points.
     * @throws IllegalArgumentException if the sector is null, if the width is less than 1, or if the height is less
     *     than 1.
     */
    public MAnalyticSurface(Sector sector, double altitude, int width, int height) {
        this(sector, altitude, width, height, createDefaultValues(width * height));
    }

    /**
     * Constructs a new AnalyticSurface with the specified {@link Sector} and base altitude in meters. The new
     * AnalyticSurface has default dimensions of <code>(10, 10)</code>, and default {@link GridPointAttributes}.
     *
     * @param sector the Sector which defines the surface's geographic region.
     * @param altitude the base altitude to place the surface at, in meters.
     * @throws IllegalArgumentException if the sector is null.
     */
    public MAnalyticSurface(Sector sector, double altitude) {
        this(sector, altitude, DEFAULT_DIMENSION, DEFAULT_DIMENSION);
    }

    /**
     * Constructs a new AnalyticSurface with the specified grid dimensions. The new AnalyticSurface is has the default
     * Sector {@link Sector#EMPTY_SECTOR}, the default altitude of 0 meters, and default {@link GridPointAttributes}.
     *
     * @param width the surface grid width, in number of grid points.
     * @param height the surface grid height, in number of grid points.
     * @throws IllegalArgumentException if the sector is null.
     */
    public MAnalyticSurface(int width, int height) {
        this(Sector.EMPTY_SECTOR, DEFAULT_ALTITUDE, width, height);
    }

    /**
     * Constructs a new AnalyticSurface with the default Sector {@link Sector#EMPTY_SECTOR}, the default altitude of 0
     * meters, default dimensions of <code>(10, 10)</code>, and default {@link GridPointAttributes}.
     */
    public MAnalyticSurface() {
        this(DEFAULT_DIMENSION, DEFAULT_DIMENSION);
    }

    /**
     * Returns true if the surface is visible in the scene, and false otherwise.
     *
     * @return true if the surface is visible in the scene, and false otherwise
     */
    public synchronized boolean isVisible() {
        return this.visible;
    }

    /**
     * Sets whether or not the surface is visible in the scene.
     *
     * @param visible true to make the surface visible, and false to make it hidden.
     */
    public synchronized void setVisible(boolean visible) {
        this.visible = visible;
    }

    public synchronized void disable() {
        setVisible(false);
        setValues(new Vector<GridPoint>());
    }

    /**
     * Returns the {@link Sector} defining the geographic boundary of this surface.
     *
     * @return this surface's geographic boundary, as a Sector.
     */
    public synchronized Sector getSector() {
        return this.sector;
    }

    /**
     * interface to set multiple parameters simultaniously, to make in atomic by a single synchronization
     *
     * @param sector
     * @param values
     * @param width
     * @param height
     */
    public synchronized void setAll(
            Sector sector, Iterable<? extends GridPointAttributes> values, int width, int height) {
        setSector(sector);
        setValues(values);
        setDimensions(width, height);
        setVisible(true);
    }

    /**
     * Sets this surface's geographic boundary, as a {@link Sector}.
     *
     * @param sector this surface's new geographic boundary, as a Sector.
     * @throws IllegalArgumentException if the surface is null.
     */
    public synchronized void setSector(Sector sector) {
        if (sector == null) {
            String message = Logging.getMessage("nullValue.SectorIsNull");
            Logging.logger().severe(message);
            throw new IllegalArgumentException(message);
        }

        this.sector = sector;
        this.setExpired(true);
    }

    /**
     * Returns this surface's base altitude, in meters.
     *
     * @return this surface's base altitude, in meters.
     */
    public synchronized double getAltitude() {
        return altitude;
    }

    /**
     * Sets this surface's base altitude, in meters.
     *
     * @param altitude the new base altitude, in meters.
     */
    public synchronized void setAltitude(double altitude) {
        this.altitude = altitude;
        this.setExpired(true);
    }

    /**
     * Returns the surface's altitude mode, one of {@link gov.nasa.worldwind.WorldWind#CLAMP_TO_GROUND}, {@link
     * gov.nasa.worldwind.WorldWind#RELATIVE_TO_GROUND}, or {@link gov.nasa.worldwind.WorldWind#ABSOLUTE}. The altitude
     * mode {@link gov.nasa.worldwind.WorldWind#CONSTANT} is not supported and {@link
     * gov.nasa.worldwind.WorldWind#ABSOLUTE} is used if specified.
     *
     * @return the surface's altitude mode.
     */
    public synchronized int getAltitudeMode() {
        return altitudeMode;
    }

    /**
     * Specifies the surface's altitude mode, one of {@link gov.nasa.worldwind.WorldWind#CLAMP_TO_GROUND}, {@link
     * gov.nasa.worldwind.WorldWind#RELATIVE_TO_GROUND}, or {@link gov.nasa.worldwind.WorldWind#ABSOLUTE}. The altitude
     * mode {@link gov.nasa.worldwind.WorldWind#CONSTANT} is not supported and {@link
     * gov.nasa.worldwind.WorldWind#ABSOLUTE} is used if specified.
     *
     * @param altitudeMode the surface's altitude mode.
     */
    public synchronized void setAltitudeMode(int altitudeMode) {
        this.altitudeMode = altitudeMode;
        this.setExpired(true);
    }

    /**
     * Returns the number of horizontal and vertical points composing this surface as an array with two values. The
     * value at index 0 indicates the grid width, and the value at index 1 indicates the grid height.
     *
     * @return the dimensions of this surface's grid.
     */
    public synchronized int[] getDimensions() {
        return new int[] {this.width, this.height};
    }

    /**
     * Sets the number of horizontal and vertical points composing this surface.
     *
     * @param width the new grid width.
     * @param height the new grid height.
     * @throws IllegalArgumentException if either width or heigth are less than 1.
     */
    public synchronized void setDimensions(int width, int height) {
        if (width <= 0) {
            String message = Logging.getMessage("Geom.WidthInvalid", width);
            Logging.logger().severe(message);
            throw new IllegalArgumentException(message);
        }

        if (height <= 0) {
            String message = Logging.getMessage("Geom.HeightInvalid", height);
            Logging.logger().severe(message);
            throw new IllegalArgumentException(message);
        }

        this.width = width;
        this.height = height;
        this.setExpired(true);
    }

    /**
     * Returns the surface's iterable of {@link GridPointAttributes}. See {@link #setValues(Iterable)} for details on
     * how this iterable is interpreted by AnalyticSurface.
     *
     * @return this surface's GridPointAttributes.
     */
    public synchronized Iterable<? extends GridPointAttributes> getValues() {
        return this.values;
    }

    /**
     * Sets this surface's iterable of {@link GridPointAttributes}. Grid points are assigned attributes from this
     * iterable staring at the upper left hand corner, and proceeding in row-first order across the grid. The iterable
     * should contain at least <code>width *
     * height</code> values, where width and height are the AnalyticSurface's grid dimensions. If the iterable contains
     * too few values, the unassigned grid points are given default attributes: the default scalar value is 0, and the
     * default color is {@link java.awt.Color#BLACK}. (totally opaque).
     *
     * @param iterable the new grid point attributes.
     * @throws IllegalArgumentException if the iterable is null.
     */
    public synchronized void setValues(Iterable<? extends GridPointAttributes> iterable) {
        if (iterable == null) {
            String message = Logging.getMessage("nullValue.IterableIsNull");
            Logging.logger().severe(message);
            throw new IllegalArgumentException(message);
        }

        this.values = iterable;
        this.extremeValues = computeExtremeValues(iterable);
        this.setExpired(true);
    }

    /**
     * Returns the scale applied to the value at each grid point.
     *
     * @return the surface's vertical scale coefficient.
     */
    public synchronized double getVerticalScale() {
        return this.verticalScale;
    }

    /**
     * Sets the scale applied to the value at each grid point. Before rendering, this value is applied to each grid
     * points scalar value, thus increasing or decreasing it's height relative to the surface's base altitude, both in
     * meters.
     *
     * @param scale the surface's vertical scale coefficient.
     */
    public synchronized void setVerticalScale(double scale) {
        this.verticalScale = scale;
        this.setExpired(true);
    }

    /**
     * Returns a copy of the rendering attributes associated with this surface. Modifying the contents of the returned
     * reference has no effect on this surface. In order to make an attribute change take effect, invoke {@link
     * #setSurfaceAttributes(AnalyticSurfaceAttributes)} with the modified attributes.
     *
     * @return a copy of this surface's rendering attributes.
     */
    public synchronized AnalyticSurfaceAttributes getSurfaceAttributes() {
        return this.surfaceAttributes.copy();
    }

    /**
     * Sets the rendering attributes associated with this surface. The caller cannot assume that modifying the attribute
     * reference after calling setSurfaceAttributes() will have any effect, as the implementation may defensively copy
     * the attribute reference. In order to make an attribute change take effect, invoke
     * setSurfaceAttributes(AnalyticSurfaceAttributes) again with the modified attributes.
     *
     * @param attributes this surface's new rendering attributes.
     * @throws IllegalArgumentException if attributes is null.
     */
    public synchronized void setSurfaceAttributes(AnalyticSurfaceAttributes attributes) {
        if (attributes == null) {
            String message = Logging.getMessage("nullValue.AttributesIsNull");
            Logging.logger().severe(message);
            throw new IllegalArgumentException(message);
        }

        this.surfaceAttributes = attributes.copy();
        this.setExpired(true);
    }

    /**
     * Returns the object which is associated with this surface during picking. A null value is permitted and indicates
     * that the surface itself will be the object returned during picking.
     *
     * @return this surface's pick object.
     */
    public synchronized Object getPickObject() {
        return this.pickObject;
    }

    /**
     * Sets the object associated with this surface during picking. A null value is permitted and indicates that the
     * surface itself will be the object returned during picking.
     *
     * @param pickObject the object to associated with this surface during picking. A null value is permitted and
     *     indicates that the surface itself will be the object returned during picking.
     */
    public synchronized void setPickObject(Object pickObject) {
        this.pickObject = pickObject;
    }

    /**
     * Returns the layer associated with this surface during picking.
     *
     * @return this surface's pick layer.
     */
    public synchronized Layer getClientLayer() {
        return this.clientLayer;
    }

    /**
     * Sets the layer associated with this surface during picking. A null value is permitted, and indicates that no
     * layer is associated with this surface.
     *
     * @param layer this surface's pick layer.
     */
    public synchronized void setClientLayer(Layer layer) {
        this.clientLayer = layer;
    }

    /**
     * {@inheritDoc}
     *
     * @param dc
     */
    public synchronized void preRender(DrawContext dc) {
        if (dc == null) {
            String message = Logging.getMessage("nullValue.DrawContextIsNull");
            Logging.logger().severe(message);
            throw new IllegalArgumentException(message);
        }

        if (!this.isVisible()) {
            return;
        }

        if (!this.intersectsFrustum(dc)) {
            return;
        }

        if (this.isExpired(dc)) {
            this.update(dc);
        }

        if (this.isExpired(dc)) {
            return;
        }

        this.preRenderSurfaceObjects(dc);
    }

    /**
     * {@inheritDoc}
     *
     * @param dc the <code>DrawContext</code> to be used
     */
    public synchronized void render(DrawContext dc) {
        if (dc == null) {
            String message = Logging.getMessage("nullValue.DrawContextIsNull");
            Logging.logger().severe(message);
            throw new IllegalArgumentException(message);
        }

        if (!this.isVisible()) {
            return;
        }

        if (!intersectsFrustum(dc)) {
            return;
        }

        if (this.isExpired(dc)) {
            this.update(dc);
        }

        if (this.isExpired(dc)) {
            return;
        }

        this.makeOrderedRenderable(dc);
        this.drawSurfaceObjects(dc);
    }

    /**
     * Returns this surface's extent in model coordinates.
     *
     * @param dc the current DrawContext.
     * @return this surface's extent in the specified DrawContext.
     * @throws IllegalArgumentException if the DrawContext is null.
     */
    public synchronized Extent getExtent(DrawContext dc) {
        if (dc == null) {
            String message = Logging.getMessage("nullValue.DrawContextIsNull");
            Logging.logger().severe(message);
            throw new IllegalArgumentException(message);
        }

        if (this.getAltitudeMode() == WorldWind.CLAMP_TO_GROUND) {
            return Sector.computeBoundingBox(dc.getGlobe(), dc.getVerticalExaggeration(), this.getSector());
        } else {
            double minAltitude = this.getAltitude();
            double maxAltitude = this.getAltitude();
            double[] minAndMaxElevations = dc.getGlobe().getMinAndMaxElevations(this.getSector());

            if (this.extremeValues != null) {
                minAltitude = this.getAltitude() + this.getVerticalScale() * this.extremeValues[0];
                maxAltitude = this.getAltitude() + this.getVerticalScale() * this.extremeValues[1];
            }

            if (minAndMaxElevations != null) {
                if (this.getAltitudeMode() == WorldWind.RELATIVE_TO_GROUND) {
                    minAltitude -= minAndMaxElevations[0];
                    maxAltitude += minAndMaxElevations[1];
                }

                if (minAltitude > minAndMaxElevations[0]) {
                    minAltitude = minAndMaxElevations[0];
                }

                if (maxAltitude < minAndMaxElevations[1]) {
                    maxAltitude = minAndMaxElevations[1];
                }
            }

            return Sector.computeBoundingBox(
                dc.getGlobe(), dc.getVerticalExaggeration(), this.getSector(), minAltitude, maxAltitude);
        }
    }

    /**
     * Test if this AnalyticSurface intersects the specified draw context's frustum. During picking mode, this tests
     * intersection against all of the draw context's pick frustums. During rendering mode, this tests intersection
     * against the draw context's viewing frustum.
     *
     * @param dc the current draw context.
     * @return true if this AnalyticSurface intersects the draw context's frustum; false otherwise.
     */
    protected boolean intersectsFrustum(DrawContext dc) {
        // A null extent indicates an object which has no location.
        Extent extent = this.getExtent(dc);
        if (extent == null) {
            return false;
        }

        // Test this object's extent against the pick frustum list
        if (dc.isPickingMode()) {
            return dc.getPickFrustums().intersectsAny(extent);
        }

        // Test this object's extent against the viewing frustum.
        return dc.getView().getFrustumInModelCoordinates().intersects(extent);
    }

    // **************************************************************//
    // ******************** Attribute Construction ****************//
    // **************************************************************//

    /**
     * Returns the minimum and maximum values in the specified iterable of {@link GridPointAttributes}. Values
     * equivalent to the specified <code>missingDataSignal</code> are ignored. This returns null if the iterable is
     * empty or contains only missing values.
     *
     * @param iterable the GridPointAttributes to search for the minimum and maximum value.
     * @param missingDataSignal the number indicating a specific value to ignore.
     * @return an array containing the minimum value in index 0 and the maximum value in index 1, or null if the
     *     iterable is empty or contains only missing values.
     * @throws IllegalArgumentException if the iterable is null.
     */
    public static synchronized double[] computeExtremeValues(
            Iterable<? extends GridPointAttributes> iterable, double missingDataSignal) {
        if (iterable == null) {
            String message = Logging.getMessage("nullValue.IterableIsNull");
            Logging.logger().severe(message);
            throw new IllegalArgumentException(message);
        }

        double minValue = Double.MAX_VALUE;
        double maxValue = -Double.MAX_VALUE;

        for (GridPointAttributes attr : iterable) {
            double value = attr.getValue();
            if (Double.compare(value, missingDataSignal) == 0) {
                continue;
            }

            if (minValue > value) {
                minValue = value;
            }

            if (maxValue < value) {
                maxValue = value;
            }
        }

        if (minValue == Double.MAX_VALUE || minValue == -Double.MIN_VALUE) {
            return null;
        }

        return new double[] {minValue, maxValue};
    }

    /**
     * Returns the minimum and maximum values in the specified iterable of {@link GridPointAttributes}. Values
     * equivalent to <code>Double.NaN</code> are ignored. This returns null if the buffer is empty or contains only NaN
     * values.
     *
     * @param iterable the GridPointAttributes to search for the minimum and maximum value.
     * @return an array containing the minimum value in index 0 and the maximum value in index 1, or null if the
     *     iterable is empty or contains only NaN values.
     * @throws IllegalArgumentException if the iterable is null.
     */
    public static synchronized double[] computeExtremeValues(Iterable<? extends GridPointAttributes> iterable) {
        if (iterable == null) {
            String message = Logging.getMessage("nullValue.IterableIsNull");
            Logging.logger().severe(message);
            throw new IllegalArgumentException(message);
        }

        return computeExtremeValues(iterable, Double.NaN);
    }

    /**
     * Returns a new instance of {@link GridPointAttributes} with the specified value and color.
     *
     * @param value the new GridPointAttributes' value.
     * @param color the new GridPointAttributes' color.
     * @return a new GridPointAttributes defined by the specified value and color.
     */
    public static synchronized GridPointAttributes createGridPointAttributes(
            final double value, final java.awt.Color color) {
        return new MAnalyticSurface.GridPointAttributes() {
            public synchronized double getValue() {
                return value;
            }

            public synchronized Color getColor() {
                return color;
            }
        };
    }

    /**
     * Returns a new instance of {@link GridPointAttributes} with a Color computed from the specified value and value
     * range. The color's RGB components are computed by mapping value's relative position in the range <code>
     * [minValue, maxValue]</code> to the range of color hues <code>[minHue, maxHue]</code>. The color's Alpha component
     * is computed by mapping the values's relative position in the range <code>
     * [minValue, minValue + (maxValue - minValue) / 10]</code> to the range <code>[0, 1]</code>. This has the effect of
     * interpolating hue and alpha based on the grid point value.
     *
     * @param value the new GridPointAttributes' value.
     * @param minValue the minimum value.
     * @param maxValue the maximum value.
     * @param minHue the mimimum color hue, corresponding to the minimum value.
     * @param maxHue the maximum color hue, corresponding to the maximum value.
     * @return a new GridPointAttributes defined by the specified value, value range, and color hue range.
     */
    public static synchronized MAnalyticSurface.GridPointAttributes createColorGradientAttributes(
            final double value, double minValue, double maxValue, double minHue, double maxHue) {
        double hueFactor = WWMath.computeInterpolationFactor(value, minValue, maxValue);
        Color color = Color.getHSBColor((float)WWMath.mixSmooth(hueFactor, minHue, maxHue), 1f, 1f);
        double opacity = WWMath.computeInterpolationFactor(value, minValue, minValue + (maxValue - minValue) * 0.1);
        Color rgbaColor = new Color(color.getRed(), color.getGreen(), color.getBlue(), (int)(255 * opacity));

        return createGridPointAttributes(value, rgbaColor);
    }

    /**
     * Returns a new iterable populated with the default {@link GridPointAttributes}. The default GridPointAttributes
     * have a value of 0, and the color {@link java.awt.Color#BLACK}.
     *
     * @param count the desired number of GridPointAttributes to return.
     * @return an iterable containing <code>count</code> default GridPointAttributes.
     */
    public static synchronized Iterable<? extends GridPointAttributes> createDefaultValues(int count) {
        ArrayList<GridPointAttributes> list = new ArrayList<GridPointAttributes>(count);
        Collections.fill(list, DEFAULT_GRID_POINT_ATTRIBUTES);
        return list;
    }

    /**
     * Returns a new iterable populated with {@link GridPointAttributes} computed by invoking {@link
     * #createColorGradientAttributes(double, double, double, double, double)} for each double value in the speicfied
     * {@link BufferWrapper}. Values equivalent to the specified <code>missingDataSignal</code> are replaced with the
     * specified <code>minValue</code>.
     *
     * @param values the buffer of values.
     * @param missingDataSignal the number indicating a specific value to ignore.
     * @param minValue the minimum value.
     * @param maxValue the maximum value.
     * @param minHue the mimimum color hue, corresponding to the minimum value.
     * @param maxHue the maximum color hue, corresponding to the maximum value.
     * @return an iiterable GridPointAttributes defined by the specified buffer of values.
     */
    public static synchronized Iterable<? extends MAnalyticSurface.GridPointAttributes> createColorGradientValues(
            BufferWrapper values,
            double missingDataSignal,
            double minValue,
            double maxValue,
            double minHue,
            double maxHue) {
        if (values == null) {
            String message = Logging.getMessage("nullValue.BufferIsNull");
            Logging.logger().severe(message);
            throw new IllegalArgumentException(message);
        }

        ArrayList<MAnalyticSurface.GridPointAttributes> attributesList =
            new ArrayList<MAnalyticSurface.GridPointAttributes>();

        for (int i = 0; i < values.length(); i++) {
            double value = values.getDouble(i);
            if (Double.compare(value, missingDataSignal) == 0) {
                value = minValue;
            }

            attributesList.add(createColorGradientAttributes(value, minValue, maxValue, minHue, maxHue));
        }

        return attributesList;
    }

    // **************************************************************//
    // ******************** Surface Rendering *********************//
    // **************************************************************//

    protected void makeOrderedRenderable(DrawContext dc) {
        // Clamp-to-ground analytic surface is drawn entirely by surface objects prepared during
        // preRenderSurfaceObjects().
        if (this.getAltitudeMode() == WorldWind.CLAMP_TO_GROUND) {
            return;
        }

        Extent extent = this.getExtent(dc);
        double eyeDistance = dc.getView().getEyePoint().distanceTo3(extent.getCenter()) - extent.getRadius();
        if (eyeDistance < 1) {
            eyeDistance = 1;
        }

        dc.addOrderedRenderable(new OrderedSurface(this, eyeDistance));
    }

    protected void drawOrderedRenderable(DrawContext dc) {
        this.beginDrawing(dc);
        try {
            this.doDrawOrderedRenderable(dc);
        } finally {
            this.endDrawing(dc);
        }
    }

    protected void doDrawOrderedRenderable(DrawContext dc) {
        this.bind(dc);

        // If the outline and interior will be drawn, then draw the outline color, but do not affect the depth
        // buffer. When the interior is drawn, it will draw on top of these colors, and the outline will be visible
        // behind the potentially transparent interior.
        if (this.surfaceAttributes.isDrawOutline() && this.surfaceAttributes.isDrawInterior()) {
            dc.getGL().glDepthMask(false);
            this.drawOutline(dc);
            dc.getGL().glDepthMask(true);
        }

        if (this.surfaceAttributes.isDrawInterior()) {
            this.drawInterior(dc);
        }

        if (this.surfaceAttributes.isDrawOutline()) {
            this.drawOutline(dc);
        }
    }

    protected void bind(DrawContext dc) {
        GL2 gl = dc.getGL().getGL2();
        gl.glVertexPointer(3, GL2.GL_FLOAT, 0, this.surfaceRenderInfo.cartesianVertexBuffer);
        gl.glNormalPointer(GL2.GL_FLOAT, 0, this.surfaceRenderInfo.cartesianNormalBuffer);
        gl.glColorPointer(4, GL2.GL_UNSIGNED_BYTE, 0, this.surfaceRenderInfo.colorBuffer);
    }

    protected void drawInterior(DrawContext dc) {
        GL2 gl = dc.getGL().getGL2();

        if (!dc.isPickingMode()) {
            // Bind the shapes vertex colors as the diffuse material parameter.
            gl.glEnable(GL2.GL_COLOR_MATERIAL);
            gl.glColorMaterial(GL2.GL_FRONT_AND_BACK, GL2.GL_DIFFUSE);
            gl.glEnableClientState(GL2.GL_COLOR_ARRAY);
            this.surfaceAttributes
                .getInteriorMaterial()
                .apply(gl, GL2.GL_FRONT_AND_BACK, (float)this.surfaceAttributes.getInteriorOpacity());
        }

        gl.glCullFace(GL2.GL_FRONT);
        this.surfaceRenderInfo.drawInterior(dc);

        gl.glCullFace(GL2.GL_BACK);
        this.surfaceRenderInfo.drawInterior(dc);
    }

    protected void drawOutline(DrawContext dc) {
        GL2 gl = dc.getGL().getGL2();

        if (!dc.isPickingMode()) {
            gl.glEnable(GL2.GL_LINE_SMOOTH);
            // Unbind the shapes vertex colors as the diffuse material parameter.
            gl.glDisable(GL2.GL_LIGHTING);
            gl.glDisable(GL2.GL_COLOR_MATERIAL);
            // Set the outline color.
            Color color = this.surfaceAttributes.getOutlineMaterial().getDiffuse();
            // Convert the floating point opacity from the range [0, 1] to the unsigned byte range [0, 255].
            int alpha = (int)(255 * this.surfaceAttributes.getOutlineOpacity() + 0.5);
            gl.glColor4ub((byte)color.getRed(), (byte)color.getGreen(), (byte)color.getBlue(), (byte)alpha);
        }

        gl.glDisableClientState(GL2.GL_COLOR_ARRAY);
        gl.glLineWidth((float)this.surfaceAttributes.getOutlineWidth());
        this.surfaceRenderInfo.drawOutline(dc);

        if (!dc.isPickingMode()) {
            gl.glEnable(GL2.GL_LIGHTING);
            gl.glDisable(GL2.GL_LINE_SMOOTH);
            gl.glDisable(GL2.GL_LINE_STIPPLE);
        }
    }

    protected void beginDrawing(DrawContext dc) {
        GL2 gl = dc.getGL().getGL2();

        gl.glPushAttrib(
            GL2.GL_COLOR_BUFFER_BIT // for alpha test func and ref, blend func
                | GL2.GL_CURRENT_BIT
                | GL2.GL_DEPTH_BUFFER_BIT
                | GL2.GL_LINE_BIT // for line width
                | GL2.GL_POLYGON_BIT // for cull face
                | (!dc.isPickingMode() ? GL2.GL_LIGHTING_BIT : 0) // for lighting.
                | (!dc.isPickingMode() ? GL2.GL_TRANSFORM_BIT : 0)); // for normalize state.
        gl.glPushClientAttrib(GL2.GL_CLIENT_VERTEX_ARRAY_BIT);

        // Enable the alpha test.
        gl.glEnable(GL2.GL_ALPHA_TEST);
        gl.glAlphaFunc(GL2.GL_GREATER, 0.0f);

        gl.glEnable(GL2.GL_CULL_FACE);
        gl.glEnableClientState(GL2.GL_VERTEX_ARRAY);
        gl.glEnableClientState(GL2.GL_NORMAL_ARRAY);

        if (dc.isPickingMode()) {
            Color color = dc.getUniquePickColor();

            this.pickSupport.addPickableObject(
                color.getRGB(),
                (this.getPickObject() != null) ? this.getPickObject() : this,
                new Position(this.sector.getCentroid(), this.altitude),
                false);

            gl.glColor3ub((byte)color.getRed(), (byte)color.getGreen(), (byte)color.getBlue());
        } else {
            // Enable blending in non-premultiplied color mode. Premultiplied colors don't work with GL fixed
            // functionality lighting.
            gl.glEnable(GL2.GL_BLEND);
            OGLUtil.applyBlending(gl, false);

            // Enable lighting with GL_LIGHT1.
            gl.glDisable(GL2.GL_COLOR_MATERIAL);
            gl.glDisable(GL2.GL_LIGHT0);
            gl.glEnable(GL2.GL_LIGHTING);
            gl.glEnable(GL2.GL_LIGHT1);
            gl.glEnable(GL2.GL_NORMALIZE);
            // Configure the lighting model for two-sided smooth shading.
            gl.glLightModeli(GL2.GL_LIGHT_MODEL_LOCAL_VIEWER, GL2.GL_TRUE);
            gl.glLightModeli(GL2.GL_LIGHT_MODEL_TWO_SIDE, GL2.GL_TRUE);
            gl.glShadeModel(GL2.GL_SMOOTH);
            // Configure GL_LIGHT1 as a white light eminating from the viewer's eye point.
            OGLUtil.applyLightingDirectionalFromViewer(gl, GL2.GL_LIGHT1, new Vec4(1.0, 0.5, 1.0).normalize3());
        }

        dc.getView().pushReferenceCenter(dc, this.referencePoint);
    }

    protected void endDrawing(DrawContext dc) {
        dc.getView().popReferenceCenter(dc);
        dc.getGL().getGL2().glPopAttrib();
        dc.getGL().getGL2().glPopClientAttrib();
    }

    // **************************************************************//
    // ******************** Surface Construction ******************//
    // **************************************************************//

    protected boolean isExpired(DrawContext dc) {
        if (this.expired) {
            return true;
        }

        if (this.getAltitudeMode() == WorldWind.RELATIVE_TO_GROUND) {
            if (dc.getFrameTimeStamp() - this.regenTime > RELATIVE_TO_GROUND_REGEN_PERIOD) {
                return true;
            }
        }

        if (this.getAltitudeMode() == WorldWind.RELATIVE_TO_GROUND || this.getAltitudeMode() == WorldWind.ABSOLUTE) {
            Object gsk = dc.getGlobe().getStateKey(dc);
            if (this.globeStateKey != null ? !this.globeStateKey.equals(gsk) : gsk != null) {
                return true;
            }
        }

        return false;
    }

    protected void setExpired(boolean expired) {
        this.expired = expired;

        if (this.expired) {
            if (this.clampToGroundSurface != null) {
                this.clampToGroundSurface.markAsModified();
            }

            if (this.shadowSurface != null) {
                this.shadowSurface.markAsModified();
            }
        }
    }

    protected void update(DrawContext dc) {
        if (this.updateFailed) {
            return;
        }

        try {
            this.doUpdate(dc);
            this.setExpired(false);
            this.globeStateKey = dc.getGlobe().getStateKey(dc);
            this.regenTime = dc.getFrameTimeStamp();
        } catch (Exception e) {
            String message = Logging.getMessage("generic.ExceptionWhileUpdating", this);
            Logging.logger().log(java.util.logging.Level.SEVERE, message, e);
            this.updateFailed = true;
        }
    }

    protected void doUpdate(DrawContext dc) {
        this.referencePos = new Position(this.sector.getCentroid(), this.altitude);
        this.referencePoint = dc.getGlobe().computePointFromPosition(this.referencePos);

        if (this.surfaceRenderInfo == null
                || this.surfaceRenderInfo.getGridWidth() != this.width
                || this.surfaceRenderInfo.getGridHeight() != this.height) {
            this.surfaceRenderInfo = new RenderInfo(this.width, this.height);
        }

        this.updateSurfacePoints(dc, this.surfaceRenderInfo);
        this.updateSurfaceNormals(this.surfaceRenderInfo);
    }

    protected void updateSurfacePoints(DrawContext dc, RenderInfo outRenderInfo) {
        Iterator<? extends GridPointAttributes> iter = this.values.iterator();

        double latStep = -this.sector.getDeltaLatDegrees() / (double)(this.height - 1);
        double lonStep = this.sector.getDeltaLonDegrees() / (double)(this.width - 1);

        double lat = this.sector.getMaxLatitude().degrees;
        for (int y = 0; y < this.height; y++) {
            double lon = this.sector.getMinLongitude().degrees;
            for (int x = 0; x < this.width; x++) {
                GridPointAttributes attr = iter.hasNext() ? iter.next() : null;
                this.updateNextSurfacePoint(dc, Angle.fromDegrees(lat), Angle.fromDegrees(lon), attr, outRenderInfo);

                lon += lonStep;
            }

            lat += latStep;
        }

        outRenderInfo.cartesianVertexBuffer.rewind();
        outRenderInfo.geographicVertexBuffer.rewind();
        outRenderInfo.colorBuffer.rewind();
        outRenderInfo.shadowColorBuffer.rewind();
    }

    protected void updateNextSurfacePoint(
            DrawContext dc, Angle lat, Angle lon, GridPointAttributes attr, RenderInfo outRenderInfo) {
        Color color = (attr != null) ? attr.getColor() : DEFAULT_COLOR;
        // Convert the floating point opacity from the range [0, 1] to the unsigned byte range [0, 255].
        int alpha = (int)(color.getAlpha() * this.surfaceAttributes.getInteriorOpacity() + 0.5);
        outRenderInfo.colorBuffer.put((byte)color.getRed());
        outRenderInfo.colorBuffer.put((byte)color.getGreen());
        outRenderInfo.colorBuffer.put((byte)color.getBlue());
        outRenderInfo.colorBuffer.put((byte)alpha);

        // We need geographic vertices if the surface's altitude mode is clamp-to-ground, or if we're drawing the
        // surface's shadow.
        if (this.getAltitudeMode() == WorldWind.CLAMP_TO_GROUND || this.surfaceAttributes.isDrawShadow()) {
            outRenderInfo.geographicVertexBuffer.put((float)(lon.degrees - this.referencePos.getLongitude().degrees));
            outRenderInfo.geographicVertexBuffer.put((float)(lat.degrees - this.referencePos.getLatitude().degrees));
            outRenderInfo.geographicVertexBuffer.put(1);
        }

        // We need cartesian vertices if the surface's altitude mode is absolute or relative-to-ground.
        if (this.getAltitudeMode() != WorldWind.CLAMP_TO_GROUND) // WorldWind.ABSOLUTE or WorldWind.RELATIVE_TO_GROUND
        {
            Vec4 point = this.computeSurfacePoint(dc, lat, lon, (attr != null ? attr.getValue() : DEFAULT_VALUE));
            outRenderInfo.cartesianVertexBuffer.put((float)(point.x - this.referencePoint.x));
            outRenderInfo.cartesianVertexBuffer.put((float)(point.y - this.referencePoint.y));
            outRenderInfo.cartesianVertexBuffer.put((float)(point.z - this.referencePoint.z));
        }

        // We need shadow colors if the surface's shadow is enabled.
        if (this.surfaceAttributes.isDrawShadow()) {
            // Convert the floating point opacity from the range [0, 1] to the unsigned byte range [0, 255].
            int shadowAlpha = (int)(alpha * this.surfaceAttributes.getShadowOpacity() + 0.5);
            outRenderInfo.shadowColorBuffer.put((byte)0);
            outRenderInfo.shadowColorBuffer.put((byte)0);
            outRenderInfo.shadowColorBuffer.put((byte)0);
            outRenderInfo.shadowColorBuffer.put((byte)shadowAlpha);
        }
    }

    protected Vec4 computeSurfacePoint(DrawContext dc, Angle lat, Angle lon, double value) {
        double offset = this.altitude + this.verticalScale * value;

        if (this.getAltitudeMode() == WorldWind.RELATIVE_TO_GROUND) {
            return dc.computeTerrainPoint(lat, lon, dc.getVerticalExaggeration() * offset);
        } else // WorldWind.ABSOLUTE
        {
            return dc.getGlobe().computePointFromPosition(lat, lon, offset);
        }
    }

    protected void updateSurfaceNormals(RenderInfo outRenderInfo) {
        WWMath.computeNormalsForIndexedTriangleStrip(
            outRenderInfo.interiorIndexBuffer,
            outRenderInfo.cartesianVertexBuffer,
            outRenderInfo.cartesianNormalBuffer);
    }

    // **************************************************************//
    // ******************** Internal Support Classes **************//
    // **************************************************************//

    protected static class OrderedSurface implements OrderedRenderable {
        protected final MAnalyticSurface surface;
        protected final double eyeDistance;

        public OrderedSurface(MAnalyticSurface surface, double eyeDistance) {
            this.surface = surface;
            this.eyeDistance = eyeDistance;
        }

        public synchronized double getDistanceFromEye() {
            return this.eyeDistance;
        }

        public synchronized void pick(DrawContext dc, Point pickPoint) {
            this.surface.pickSupport.beginPicking(dc);
            try {
                this.render(dc);
            } finally {
                this.surface.pickSupport.endPicking(dc);
                this.surface.pickSupport.resolvePick(dc, dc.getPickPoint(), this.surface.getClientLayer());
            }
        }

        public synchronized void render(DrawContext dc) {
            this.surface.drawOrderedRenderable(dc);
        }
    }

    protected static class RenderInfo {
        protected final int gridWidth;
        protected final int gridHeight;
        protected final IntBuffer interiorIndexBuffer;
        protected final IntBuffer outlineIndexBuffer;
        protected final FloatBuffer cartesianVertexBuffer;
        protected final FloatBuffer cartesianNormalBuffer;
        protected final FloatBuffer geographicVertexBuffer;
        protected final ByteBuffer colorBuffer;
        protected final ByteBuffer shadowColorBuffer;

        public RenderInfo(int gridWidth, int gridHeight) {
            int numVertices = gridWidth * gridHeight;
            Debug.getLog()
                .fine(
                    "Trying to create RenderInfo gridWidth:"
                        + gridWidth
                        + " gridHeight:"
                        + gridHeight
                        + " numVertices:"
                        + numVertices);
            this.gridWidth = gridWidth;
            this.gridHeight = gridHeight;
            this.interiorIndexBuffer = WWMath.computeIndicesForGridInterior(gridWidth, gridHeight);
            this.outlineIndexBuffer = WWMath.computeIndicesForGridOutline(gridWidth, gridHeight);
            this.cartesianVertexBuffer = Buffers.newDirectFloatBuffer(3 * numVertices);
            this.cartesianNormalBuffer = Buffers.newDirectFloatBuffer(3 * numVertices);
            this.geographicVertexBuffer = Buffers.newDirectFloatBuffer(3 * numVertices);
            this.colorBuffer = Buffers.newDirectByteBuffer(4 * numVertices);
            this.shadowColorBuffer = Buffers.newDirectByteBuffer(4 * numVertices);
        }

        public synchronized int getGridWidth() {
            return this.gridWidth;
        }

        public synchronized int getGridHeight() {
            return this.gridHeight;
        }

        public synchronized int getNumVertices() {
            return this.gridWidth * this.gridHeight;
        }

        public synchronized void drawInterior(DrawContext dc) {
            GL2 gl = dc.getGL().getGL2(); // GL initialization checks for GL2 compatibility.
            gl.glDrawElements(
                GL.GL_TRIANGLE_STRIP,
                this.interiorIndexBuffer.remaining(),
                GL.GL_UNSIGNED_INT,
                this.interiorIndexBuffer);
        }

        public synchronized void drawOutline(DrawContext dc) {
            GL2 gl = dc.getGL().getGL2(); // GL initialization checks for GL2 compatibility.
            gl.glDrawElements(
                GL.GL_LINE_LOOP, this.outlineIndexBuffer.remaining(), GL.GL_UNSIGNED_INT, this.outlineIndexBuffer);
        }
    }

    // **************************************************************//
    // ******************** Surface Objects ***********************//
    // **************************************************************//

    protected void preRenderSurfaceObjects(DrawContext dc) {
        if (this.surfaceAttributes.isDrawShadow()) {
            if (this.shadowSurface == null) {
                this.shadowSurface = this.createShadowSurface();
            }

            this.shadowSurface.preRender(dc);
        }

        if (this.getAltitudeMode() == WorldWind.CLAMP_TO_GROUND) {
            if (this.clampToGroundSurface == null) {
                this.clampToGroundSurface = this.createClampToGroundSurface();
            }

            this.clampToGroundSurface.preRender(dc);
        }
    }

    protected void drawSurfaceObjects(DrawContext dc) {
        if (this.surfaceAttributes.isDrawShadow()) {
            if (!dc.isPickingMode() && this.shadowSurface != null) {
                this.shadowSurface.render(dc);
            }
        }

        if (this.getAltitudeMode() == WorldWind.CLAMP_TO_GROUND && this.clampToGroundSurface != null) {
            this.clampToGroundSurface.render(dc);
        }
    }

    protected AnalyticSurfaceObject createClampToGroundSurface() {
        return new ClampToGroundSurface(this);
    }

    protected AnalyticSurfaceObject createShadowSurface() {
        return new ShadowSurface(this);
    }

    public static class AnalyticSurfaceObject extends AbstractSurfaceObject {
        protected MAnalyticSurface analyticSurface;

        public MAnalyticSurface getAnalyticSurface() {
            return analyticSurface;
        }

        public AnalyticSurfaceObject(MAnalyticSurface analyticSurface) {
            this.analyticSurface = analyticSurface;
        }

        public synchronized void markAsModified() {
            super.updateModifiedTime();
            super.clearCaches();
        }

        public synchronized List<Sector> getSectors(DrawContext dc) {
            if (dc == null) {
                String message = Logging.getMessage("nullValue.DrawContextIsNull");
                Logging.logger().severe(message);
                throw new IllegalArgumentException(message);
            }

            return Arrays.asList(this.analyticSurface.sector);
        }

        protected void drawGeographic(DrawContext dc, SurfaceTileDrawContext sdc) {
            this.beginDrawing(dc);
            try {
                this.doDrawGeographic(dc, sdc);
            } finally {
                this.endDrawing(dc);
            }
        }

        protected void beginDrawing(DrawContext dc) {
            GL2 gl = dc.getGL().getGL2();

            gl.glPushAttrib(
                GL2.GL_COLOR_BUFFER_BIT // for alpha func and ref, blend func
                    | GL2.GL_CURRENT_BIT // for current RGBA color
                    | GL2.GL_DEPTH_BUFFER_BIT // for depth test disable
                    | GL2.GL_LINE_BIT); // for line width);
            gl.glPushClientAttrib(GL2.GL_CLIENT_VERTEX_ARRAY_BIT);

            gl.glMatrixMode(GL2.GL_MODELVIEW);
            gl.glPushMatrix();

            gl.glEnable(GL2.GL_BLEND);
            OGLUtil.applyBlending(gl, false);
            gl.glDisable(GL2.GL_DEPTH_TEST);
            gl.glEnableClientState(GL2.GL_VERTEX_ARRAY);
        }

        protected void endDrawing(DrawContext dc) {
            GL2 gl = dc.getGL().getGL2();
            gl.glPopMatrix();
            gl.glPopClientAttrib();
            gl.glPopAttrib();
        }

        protected void doDrawGeographic(DrawContext dc, SurfaceTileDrawContext sdc) {
            Matrix modelview = sdc.getModelviewMatrix(this.analyticSurface.referencePos);
            dc.getGL().getGL2().glMultMatrixd(modelview.toArray(new double[16], 0, false), 0);

            this.bind(dc);

            if (this.analyticSurface.surfaceAttributes.isDrawInterior()) {
                this.drawInterior(dc);
            }

            if (this.analyticSurface.surfaceAttributes.isDrawOutline()) {
                this.drawOutline(dc);
            }
        }

        protected void bind(DrawContext dc) {}

        protected void drawInterior(DrawContext dc) {
            GL2 gl = dc.getGL().getGL2();

            if (!dc.isPickingMode()) {
                gl.glEnableClientState(GL2.GL_COLOR_ARRAY);
            }

            this.analyticSurface.surfaceRenderInfo.drawInterior(dc);
        }

        protected void drawOutline(DrawContext dc) {
            GL2 gl = dc.getGL().getGL2();

            if (!dc.isPickingMode()) {
                gl.glEnable(GL2.GL_LINE_SMOOTH);
                gl.glDisableClientState(GL2.GL_COLOR_ARRAY);
            }

            gl.glLineWidth((float)this.analyticSurface.surfaceAttributes.getOutlineWidth());
            this.analyticSurface.surfaceRenderInfo.drawOutline(dc);
        }
    }

    protected static class ClampToGroundSurface extends AnalyticSurfaceObject {
        public ClampToGroundSurface(MAnalyticSurface analyticSurface) {
            super(analyticSurface);
        }

        @Override
        protected void bind(DrawContext dc) {
            GL2 gl = dc.getGL().getGL2();
            gl.glVertexPointer(3, GL2.GL_FLOAT, 0, this.analyticSurface.surfaceRenderInfo.geographicVertexBuffer);
            gl.glColorPointer(4, GL2.GL_UNSIGNED_BYTE, 0, this.analyticSurface.surfaceRenderInfo.colorBuffer);
        }

        protected void drawOutline(DrawContext dc) {
            if (!dc.isPickingMode()) {
                // Set the outline color.
                Color color = this.analyticSurface.surfaceAttributes.getOutlineMaterial().getDiffuse();
                // Convert the floating point opacity from the range [0, 1] to the unsigned byte range [0, 255].
                int alpha = (int)(255 * this.analyticSurface.surfaceAttributes.getOutlineOpacity() + 0.5);
                dc.getGL()
                    .getGL2()
                    .glColor4ub((byte)color.getRed(), (byte)color.getGreen(), (byte)color.getBlue(), (byte)alpha);
            }

            super.drawOutline(dc);
        }
    }

    protected static class ShadowSurface extends AnalyticSurfaceObject {
        public ShadowSurface(MAnalyticSurface analyticSurface) {
            super(analyticSurface);
        }

        @Override
        protected void buildPickRepresentation(DrawContext dc) {
            // The analytic surface's shadow is not drawn during picking. Suppress any attempt to create a pick
            // representation for the shadow to eliminate unnecessary overhead.
        }

        @Override
        protected void bind(DrawContext dc) {
            GL2 gl = dc.getGL().getGL2();
            gl.glVertexPointer(3, GL2.GL_FLOAT, 0, this.analyticSurface.surfaceRenderInfo.geographicVertexBuffer);
            gl.glColorPointer(4, GL2.GL_UNSIGNED_BYTE, 0, this.analyticSurface.surfaceRenderInfo.shadowColorBuffer);
        }

        protected void drawOutline(DrawContext dc) {
            if (!dc.isPickingMode()) {
                // Convert the floating point opacity from the range [0, 1] to the unsigned byte range [0, 255].
                int alpha =
                    (int)
                        (255
                                * this.analyticSurface.surfaceAttributes.getOutlineOpacity()
                                * this.analyticSurface.surfaceAttributes.getShadowOpacity()
                            + 0.5);
                dc.getGL().getGL2().glColor4ub((byte)0, (byte)0, (byte)0, (byte)alpha);
            }

            super.drawOutline(dc);
        }
    }
}
