/*
*        NASA OPEN SOURCE AGREEMENT VERSION 1.3
*
*        THIS OPEN SOURCE AGREEMENT ("AGREEMENT") DEFINES THE RIGHTS OF USE,
*        REPRODUCTION, DISTRIBUTION, MODIFICATION AND REDISTRIBUTION OF CERTAIN
*        COMPUTER SOFTWARE ORIGINALLY RELEASED BY THE UNITED STATES GOVERNMENT
*        AS REPRESENTED BY THE GOVERNMENT AGENCY LISTED BELOW ("GOVERNMENT
*        AGENCY").  THE UNITED STATES GOVERNMENT, AS REPRESENTED BY GOVERNMENT
*        AGENCY, IS AN INTENDED THIRD-PARTY BENEFICIARY OF ALL SUBSEQUENT
*        DISTRIBUTIONS OR REDISTRIBUTIONS OF THE SUBJECT SOFTWARE.  ANYONE WHO
*        USES, REPRODUCES, DISTRIBUTES, MODIFIES OR REDISTRIBUTES THE SUBJECT
*        SOFTWARE, AS DEFINED HEREIN, OR ANY PART THEREOF, IS, BY THAT ACTION,
*        ACCEPTING IN FULL THE RESPONSIBILITIES AND OBLIGATIONS CONTAINED IN
*        THIS AGREEMENT.
*
*        Government Agency: National Aeronautics and Space Administration (NASA)
*        Government Agency Original Software Designation: ARC-15166-1
*        Government Agency Original Software Title: WorldWind Version 1.3
*        User Registration Requested.  Please Visit https://opensource.arc.nasa.gov/
*        Government Agency Point of Contact for Original Software: Patrick.Hogan@nasa.gov
*        ________________________________________________
*
*
*        1. DEFINITIONS
*
*        A. "Contributor" means Government Agency, as the developer of the
*        Original Software, and any entity that makes a Modification.
*        B. "Covered Patents" mean patent claims licensable by a Contributor
*        that are necessarily infringed by the use or sale of its Modification
*        alone or when combined with the Subject Software.
*        C. "Display" means the showing of a copy of the Subject Software,
*        either directly or by means of an image, or any other device.
*        D. "Distribution" means conveyance or transfer of the Subject
*        Software, regardless of means, to another.
*        E. "Larger Work" means computer software that combines Subject
*        Software, or portions thereof, with software separate from the Subject
*        Software that is not governed by the terms of this Agreement.
*        F.  "Modification" means any alteration of, including addition to or
*        deletion from, the substance or structure of either the Original
*        Software or Subject Software, and includes derivative works, as that
*        term is defined in the Copyright Statute, 17 USC 101.  However, the
*        act of including Subject Software as part of a Larger Work does not in
*        and of itself constitute a Modification.
*        G. "Original Software" means the computer software first released
*        under this Agreement by Government Agency with Government Agency
*        designation ARC-15166-1 and entitled WorldWind, including source code,
*        object code and accompanying documentation, if any.
*        H. "Recipient" means anyone who acquires the Subject Software under
*        this Agreement, including all Contributors.
*        I. "Redistribution" means Distribution of the Subject Software after a
*        Modification has been made.
*        J. "Reproduction" means the making of a counterpart, image or copy of
*        the Subject Software.
*        K. "Sale" means the exchange of the Subject Software for money or
*        equivalent value.
*        L. "Subject Software" means the Original Software, Modifications, or
*        any respective parts thereof.
*        M. "Use" means the application or employment of the Subject Software
*        for any purpose.
*
*        2. GRANT OF RIGHTS
*
*        A. Under Non-Patent Rights: Subject to the terms and conditions of
*        this Agreement, each Contributor, with respect to its own contribution
*        to the Subject Software, hereby grants to each Recipient a
*        non-exclusive, world-wide, royalty-free license to engage in the
*        following activities pertaining to the Subject Software:
*
*        1. Use
*        2. Distribution
*        3. Reproduction
*        4. Modification
*        5. Redistribution
*        6. Display
*
*        B. Under Patent Rights: Subject to the terms and conditions of this
*        Agreement, each Contributor, with respect to its own contribution to
*        the Subject Software, hereby grants to each Recipient under Covered
*        Patents a non-exclusive, world-wide, royalty-free license to engage in
*        the following activities pertaining to the Subject Software:
*
*        1. Use
*        2. Distribution
*        3. Reproduction
*        4. Sale
*        5. Offer for Sale
*
*        C. The rights granted under Paragraph B. also apply to the combination
*        of a Contributor's Modification and the Subject Software if, at the
*        time the Modification is added by the Contributor, the addition of
*        such Modification causes the combination to be covered by the Covered
*        Patents.  It does not apply to any other combinations that include a
*        Modification.
*
*        D. The rights granted in Paragraphs A. and B. allow the Recipient to
*        sublicense those same rights.  Such sublicense must be under the same
*        terms and conditions of this Agreement.
*
*        3. OBLIGATIONS OF RECIPIENT
*
*        A. Distribution or Redistribution of the Subject Software must be made
*        under this Agreement except for additions covered under paragraph 3H.
*
*        1. Whenever a Recipient distributes or redistributes the Subject
*        Software, a copy of this Agreement must be included with each copy
*        of the Subject Software; and
*        2. If Recipient distributes or redistributes the Subject Software in
*        any form other than source code, Recipient must also make the
*        source code freely available, and must provide with each copy of
*        the Subject Software information on how to obtain the source code
*        in a reasonable manner on or through a medium customarily used for
*        software exchange.
*
*        B. Each Recipient must ensure that the following copyright notice
*        appears prominently in the Subject Software:
*
*        Copyright (C) 2001 United States Government
*        as represented by the Administrator of the
*        National Aeronautics and Space Administration.
*        All Rights Reserved.
*
*        C. Each Contributor must characterize its alteration of the Subject
*        Software as a Modification and must identify itself as the originator
*        of its Modification in a manner that reasonably allows subsequent
*        Recipients to identify the originator of the Modification.  In
*        fulfillment of these requirements, Contributor must include a file
*        (e.g., a change log file) that describes the alterations made and the
*        date of the alterations, identifies Contributor as originator of the
*        alterations, and consents to characterization of the alterations as a
*        Modification, for example, by including a statement that the
*        Modification is derived, directly or indirectly, from Original
*        Software provided by Government Agency. Once consent is granted, it
*        may not thereafter be revoked.
*
*        D. A Contributor may add its own copyright notice to the Subject
*        Software.  Once a copyright notice has been added to the Subject
*        Software, a Recipient may not remove it without the express permission
*        of the Contributor who added the notice.
*
*        E. A Recipient may not make any representation in the Subject Software
*        or in any promotional, advertising or other material that may be
*        construed as an endorsement by Government Agency or by any prior
*        Recipient of any product or service provided by Recipient, or that may
*        seek to obtain commercial advantage by the fact of Government Agency's
*        or a prior Recipient's participation in this Agreement.
*
*        F. In an effort to track usage and maintain accurate records of the
*        Subject Software, each Recipient, upon receipt of the Subject
*        Software, is requested to register with Government Agency by visiting
*        the following website: https://opensource.arc.nasa.gov.  Recipient's
*        name and personal information shall be used for statistical purposes
*        only. Once a Recipient makes a Modification available, it is requested
*        that the Recipient inform Government Agency at the web site provided
*        above how to access the Modification.
*
*        G. Each Contributor represents that that its Modification is believed
*        to be Contributor's original creation and does not violate any
*        existing agreements, regulations, statutes or rules, and further that
*        Contributor has sufficient rights to grant the rights conveyed by this
*        Agreement.
*
*        H. A Recipient may choose to offer, and to charge a fee for, warranty,
*        support, indemnity and/or liability obligations to one or more other
*        Recipients of the Subject Software.  A Recipient may do so, however,
*        only on its own behalf and not on behalf of Government Agency or any
*        other Recipient.  Such a Recipient must make it absolutely clear that
*        any such warranty, support, indemnity and/or liability obligation is
*        offered by that Recipient alone.  Further, such Recipient agrees to
*        indemnify Government Agency and every other Recipient for any
*        liability incurred by them as a result of warranty, support, indemnity
*        and/or liability offered by such Recipient.
*
*        I. A Recipient may create a Larger Work by combining Subject Software
*        with separate software not governed by the terms of this agreement and
*        distribute the Larger Work as a single product. In such case, the
*        Recipient must make sure Subject Software, or portions thereof,
*        included in the Larger Work is subject to this Agreement.
*
*        J. Notwithstanding any provisions contained herein, Recipient is
*        hereby put on notice that export of any goods or technical data from
*        the United States may require some form of export license from the
*        U.S. Government.  Failure to obtain necessary export licenses may
*        result in criminal liability under U.S. laws.  Government Agency
*        neither represents that a license shall not be required nor that, if
*        required, it shall be issued.  Nothing granted herein provides any
*        such export license.
*
*        4. DISCLAIMER OF WARRANTIES AND LIABILITIES; WAIVER AND INDEMNIFICATION
*
*        A. No Warranty: THE SUBJECT SOFTWARE IS PROVIDED "AS IS" WITHOUT ANY
*        WARRANTY OF ANY KIND, EITHER EXPRESSED, IMPLIED, OR STATUTORY,
*        INCLUDING, BUT NOT LIMITED TO, ANY WARRANTY THAT THE SUBJECT SOFTWARE
*        WILL CONFORM TO SPECIFICATIONS, ANY IMPLIED WARRANTIES OF
*        MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR FREEDOM FROM
*        INFRINGEMENT, ANY WARRANTY THAT THE SUBJECT SOFTWARE WILL BE ERROR
*        FREE, OR ANY WARRANTY THAT DOCUMENTATION, IF PROVIDED, WILL CONFORM TO
*        THE SUBJECT SOFTWARE. THIS AGREEMENT DOES NOT, IN ANY MANNER,
*        CONSTITUTE AN ENDORSEMENT BY GOVERNMENT AGENCY OR ANY PRIOR RECIPIENT
*        OF ANY RESULTS, RESULTING DESIGNS, HARDWARE, SOFTWARE PRODUCTS OR ANY
*        OTHER APPLICATIONS RESULTING FROM USE OF THE SUBJECT SOFTWARE.
*        FURTHER, GOVERNMENT AGENCY DISCLAIMS ALL WARRANTIES AND LIABILITIES
*        REGARDING THIRD-PARTY SOFTWARE, IF PRESENT IN THE ORIGINAL SOFTWARE,
*        AND DISTRIBUTES IT "AS IS."
*
*        B. Waiver and Indemnity: RECIPIENT AGREES TO WAIVE ANY AND ALL CLAIMS
*        AGAINST THE UNITED STATES GOVERNMENT, ITS CONTRACTORS AND
*        SUBCONTRACTORS, AS WELL AS ANY PRIOR RECIPIENT.  IF RECIPIENT'S USE OF
*        THE SUBJECT SOFTWARE RESULTS IN ANY LIABILITIES, DEMANDS, DAMAGES,
*        EXPENSES OR LOSSES ARISING FROM SUCH USE, INCLUDING ANY DAMAGES FROM
*        PRODUCTS BASED ON, OR RESULTING FROM, RECIPIENT'S USE OF THE SUBJECT
*        SOFTWARE, RECIPIENT SHALL INDEMNIFY AND HOLD HARMLESS THE UNITED
*        STATES GOVERNMENT, ITS CONTRACTORS AND SUBCONTRACTORS, AS WELL AS ANY
*        PRIOR RECIPIENT, TO THE EXTENT PERMITTED BY LAW.  RECIPIENT'S SOLE
*        REMEDY FOR ANY SUCH MATTER SHALL BE THE IMMEDIATE, UNILATERAL
*        TERMINATION OF THIS AGREEMENT.
*
*
*        5. GENERAL TERMS
*
*        A. Termination: This Agreement and the rights granted hereunder will
*        terminate automatically if a Recipient fails to comply with these
*        terms and conditions, and fails to cure such noncompliance within
*        thirty (30) days of becoming aware of such noncompliance.  Upon
*        termination, a Recipient agrees to immediately cease use and
*        distribution of the Subject Software.  All sublicenses to the Subject
*        Software properly granted by the breaching Recipient shall survive any
*        such termination of this Agreement.
*
*        B. Severability: If any provision of this Agreement is invalid or
*        unenforceable under applicable law, it shall not affect the validity
*        or enforceability of the remainder of the terms of this Agreement.
*
*        C. Applicable Law: This Agreement shall be subject to United States
*        federal law only for all purposes, including, but not limited to,
*        determining the validity of this Agreement, the meaning of its
*        provisions and the rights, obligations and remedies of the parties.
*
*        D. Entire Understanding: This Agreement constitutes the entire
*        understanding and agreement of the parties relating to release of the
*        Subject Software and may not be superseded, modified or amended except
*        by further written agreement duly executed by the parties.
*
*        E. Binding Authority: By accepting and using the Subject Software
*        under this Agreement, a Recipient affirms its authority to bind the
*        Recipient to all terms and conditions of this Agreement and that that
*        Recipient hereby agrees to all terms and conditions herein.
*
*        F. Point of Contact: Any Recipient contact with Government Agency is
*        to be directed to the designated representative as follows:
*        Patrick.Hogan@nasa.gov.
*    *//*
 * This Source is licenced under the NASA OPEN SOURCE AGREEMENT VERSION 1.3
 *
 * Copyright (C) 2012 United States Government as represented by the Administrator of the
 * National Aeronautics and Space Administration.
 * All Rights Reserved.
 *
 * Modifications by MAVinci GmbH, Germany (C) 2009-2016: add hiding of empty tiles
 *
 */
package eu.mavinci.desktop.gui.wwext;

import com.intel.missioncontrol.PublishSource;
import com.jogamp.opengl.util.texture.TextureData;
import eu.mavinci.desktop.main.debug.Debug;
import gov.nasa.worldwind.Configuration;
import gov.nasa.worldwind.WorldWind;
import gov.nasa.worldwind.avlist.AVKey;
import gov.nasa.worldwind.avlist.AVList;
import gov.nasa.worldwind.cache.BasicMemoryCache;
import gov.nasa.worldwind.cache.MemoryCache;
import gov.nasa.worldwind.geom.Angle;
import gov.nasa.worldwind.geom.Vec4;
import gov.nasa.worldwind.layers.mercator.MercatorSector;
import gov.nasa.worldwind.layers.mercator.MercatorTextureTile;
import gov.nasa.worldwind.render.DrawContext;
import gov.nasa.worldwind.retrieve.HTTPRetriever;
import gov.nasa.worldwind.retrieve.RetrievalPostProcessor;
import gov.nasa.worldwind.retrieve.Retriever;
import gov.nasa.worldwind.retrieve.URLRetriever;
import gov.nasa.worldwind.util.LevelSet;
import gov.nasa.worldwind.util.Logging;
import gov.nasa.worldwind.util.OGLUtil;
import gov.nasa.worldwind.util.WWIO;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import static com.google.common.math.DoubleMath.log2;


public class MBasicMercatorTiledImageLayer extends MMercatorTiledImageLayer {
    private static final int MAX_BLACK_TILE_SIZE = 2000;
    private static final int MAX_BLACK_TILE_SIZE_SUSPICIOUS = 4500;
    private static final double THRESHOLD_BLACK_IMG = 2.0;
    private final Object fileLock = new Object();

    public MBasicMercatorTiledImageLayer(LevelSet levelSet) {
        super(levelSet);
        if (!WorldWind.getMemoryCacheSet().containsCache(MercatorTextureTile.class.getName())) {
            long size = Configuration.getLongValue(AVKey.TEXTURE_IMAGE_CACHE_SIZE, 3000000L);
            MemoryCache cache = new BasicMemoryCache((long)(0.85 * size), size);
            cache.setName("Texture Tiles");
            WorldWind.getMemoryCacheSet().addCache(MercatorTextureTile.class.getName(), cache);
        }
    }

    public MBasicMercatorTiledImageLayer(AVList params) {
        this(new LevelSet(params));
        this.setValue(AVKey.CONSTRUCTION_PARAMETERS, params);
    }

    protected void forceTextureLoad(MercatorTextureTile tile) {
        final URL textureURL = this.getDataFileStore().findFile(tile.getPath(), true);

        if (textureURL != null && !this.isTextureExpired(tile, textureURL)) {
            this.loadTexture(tile, textureURL);
        }
    }

    protected void requestTexture(DrawContext dc, MercatorTextureTile tile) {
        Vec4 centroid = tile.getCentroidPoint(dc.getGlobe());
        if (this.getReferencePoint() != null) {
            tile.setPriority(centroid.distanceTo3(this.getReferencePoint()));
        }

        RequestTask task = new RequestTask(tile, this);
        if (!WorldWind.getTaskService().isFull()) {
            WorldWind.getTaskService().addTask(task);
        }else{
            this.getRequestQ().add(task);
        }
    }

    private static class RequestTask implements Runnable, Comparable<RequestTask> {
        private final MBasicMercatorTiledImageLayer layer;
        private final MercatorTextureTile tile;

        private RequestTask(MercatorTextureTile tile, MBasicMercatorTiledImageLayer layer) {
            this.layer = layer;
            this.tile = tile;
        }

        public void run() {
            // TODO: check to ensure load is still needed

            final java.net.URL textureURL = this.layer.getDataFileStore().findFile(tile.getPath(), false);
            if (textureURL != null && !this.layer.isTextureExpired(tile, textureURL)) {
                int aval = this.layer.loadTexture(tile, textureURL);
                if (aval == 1) {
                    layer.getLevels().unmarkResourceAbsent(tile);
                    this.layer.firePropertyChange(AVKey.LAYER, null, this);
                    return;
                } else if (aval == 0) {
                    // Assume that something's wrong with the file and delete
                    // it.
                    this.layer.getDataFileStore().removeFile(textureURL);
                    layer.getLevels().markResourceAbsent(tile);
                    String message = Logging.getMessage("generic.DeletedCorruptDataFile", textureURL);
                    Logging.logger().info(message);
                    // restart download
                } else { // aval==-1
                    layer.getLevels().markResourceAbsent(tile);
                    return; // no redownload
                }
            }

            this.layer.downloadTexture(this.tile);
        }

        /**
         * @param that the task to compare
         * @return -1 if <code>this</code> less than <code>that</code>, 1 if greater than, 0 if equal
         * @throws IllegalArgumentException if <code>that</code> is null
         */
        public int compareTo(RequestTask that) {
            if (that == null) {
                String msg = Logging.getMessage("nullValue.RequestTaskIsNull");
                Logging.logger().severe(msg);
                throw new IllegalArgumentException(msg);
            }

            return this.tile.getPriority() == that.tile.getPriority()
                ? 0
                : this.tile.getPriority() < that.tile.getPriority() ? -1 : 1;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }

            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            final RequestTask that = (RequestTask)o;

            // Don't include layer in comparison so that requests are shared
            // among layers
            return !(tile != null ? !tile.equals(that.tile) : that.tile != null);
        }

        public int hashCode() {
            return (tile != null ? tile.hashCode() : 0);
        }

        public String toString() {
            return this.tile.toString();
        }
    }

    private boolean isTextureExpired(MercatorTextureTile tile, java.net.URL textureURL) {
        if (!WWIO.isFileOutOfDate(textureURL, tile.getLevel().getExpiryTime())) {
            return false;
        }

        // The file has expired. Delete it.
        this.getDataFileStore().removeFile(textureURL);
        String message = Logging.getMessage("generic.DataFileExpired", textureURL);
        Logging.logger().fine(message);
        return true;
    }

    protected TextureData absentTextureData = null;
    protected TextureData absentTextureData2 = null;

    HashMap<URL, Boolean> hideTile = new HashMap<>();

    /**
     * try to load a texture
     *
     * @param tile
     * @param textureURL
     * @return 0 == no data avaliable. 1 == all right, its loaded. -1 == data avaliable, but its black, dont show it!
     */
    protected int loadTexture(MercatorTextureTile tile, java.net.URL textureURL) {
        TextureData textureData;

        synchronized (this.fileLock) {
            textureData = readTexture(textureURL, this.isUseMipMaps());
        }

        if (textureData == null) {
            return 0;
        }

        try {
            Boolean hide = hideTile.get(textureURL);
            if (hide != null) {
                return -1;
            } else {
                long length = new File(textureURL.toURI()).length();
                if (length <= MAX_BLACK_TILE_SIZE) {
                    hideTile.put(textureURL, true);
                    return -1;
                }

                if (length <= MAX_BLACK_TILE_SIZE_SUSPICIOUS) {
                    if (isTileBlack(textureData)) {
                        hideTile.put(textureURL, true);
                        return -1;
                    }
                }
            }
        } catch (URISyntaxException e) {
            Debug.getLog().severe(e.getMessage());
        }

        tile.setTextureData(textureData);
        if (tile.getLevelNumber() != 0 || !this.isRetainLevelZeroTiles()) {
            this.addTileToCache(tile);
        }

        return 1;
    }

    private boolean isTileBlack(TextureData textureData) {
        Buffer buffer = textureData.getBuffer();

        if (buffer instanceof ByteBuffer) {
            ByteBuffer byteBuffer = ((ByteBuffer)buffer);
            byte[] bufferArray = byteBuffer.array();
            final int length = bufferArray.length / 3;

            double sum =
                -IntStream.range(0, length)
                    .boxed()
                    .map(
                        idx ->
                            String.valueOf(
                                (int)
                                    Math.round(
                                        0.1140 * (bufferArray[idx * 3] & 0xFF)
                                            + 0.5870 * (bufferArray[idx * 3 + 1] & 0xFF)
                                            + 0.2989 * (bufferArray[idx * 3 + 2] & 0xFF))))
                    .collect(Collectors.groupingBy(Function.identity(), Collectors.counting()))
                    .entrySet()
                    .stream()
                    .map(
                        e -> {
                            return e.getValue() * (log2(e.getValue() / (double)length)) / (double)length;
                        })
                    .mapToDouble(Double::doubleValue)
                    .sum();

            if (sum < THRESHOLD_BLACK_IMG) {
                return true;
            }
        }

        return false;
    }

    private static TextureData readTexture(java.net.URL url, boolean useMipMaps) {
        try {
            return OGLUtil.newTextureData(Configuration.getMaxCompatibleGLProfile(), url, useMipMaps);
        } catch (Exception e) {
            String msg = Logging.getMessage("layers.TextureLayer.ExceptionAttemptingToReadTextureFile", url.toString());
            Logging.logger().log(java.util.logging.Level.SEVERE, msg, e);
            return null;
        }
    }

    private void addTileToCache(MercatorTextureTile tile) {
        WorldWind.getMemoryCache(MercatorTextureTile.class.getName()).add(tile.getTileKey(), tile);
    }

    protected void downloadTexture(final MercatorTextureTile tile) {
        if (!WorldWind.getRetrievalService().isAvailable()) {
            return;
        }

        java.net.URL url;
        try {
            url = tile.getResourceURL();
            if (url == null) {
                return;
            }

            if (WorldWind.getNetworkStatus().isHostUnavailable(url)) {
                return;
            }
        } catch (java.net.MalformedURLException e) {
            Logging.logger()
                .log(
                    java.util.logging.Level.SEVERE,
                    Logging.getMessage("layers.TextureLayer.ExceptionCreatingTextureUrl", tile),
                    e);
            return;
        }

        Retriever retriever;

        if ("http".equalsIgnoreCase(url.getProtocol()) || "https".equalsIgnoreCase(url.getProtocol())) {
            retriever = new HTTPRetriever(url, new DownloadPostProcessor(tile, this));
            retriever.setValue(URLRetriever.EXTRACT_ZIP_ENTRY, "true"); // supports
            // legacy
            // layers
        } else {
            Logging.logger().severe(Logging.getMessage("layers.TextureLayer.UnknownRetrievalProtocol", url.toString()));
            return;
        }

        // Apply any overridden timeouts.
        Integer cto = getIntegerValue(this, AVKey.URL_CONNECT_TIMEOUT);
        if (cto != null && cto > 0) {
            retriever.setConnectTimeout(cto);
        }

        Integer cro = getIntegerValue(this, AVKey.URL_READ_TIMEOUT);
        if (cro != null && cro > 0) {
            retriever.setReadTimeout(cro);
        }

        Integer srl = getIntegerValue(this, AVKey.RETRIEVAL_QUEUE_STALE_REQUEST_LIMIT);
        if (srl != null && srl > 0) {
            retriever.setStaleRequestLimit(srl);
        }

        WorldWind.getRetrievalService().runRetriever(retriever, tile.getPriority());
    }

    private void saveBuffer(java.nio.ByteBuffer buffer, java.io.File outFile) throws java.io.IOException {
        synchronized (this.fileLock) // synchronized with read of file in
        {
            WWIO.saveBuffer(buffer, outFile);
        }
    }

    protected void dataDownloadSucceded(MercatorTextureTile tile, Retriever retriever) {}

    private static class DownloadPostProcessor implements RetrievalPostProcessor {
        // TODO: Rewrite this inner class, factoring out the generic parts.
        private final MercatorTextureTile tile;
        private final MBasicMercatorTiledImageLayer layer;

        public DownloadPostProcessor(MercatorTextureTile tile, MBasicMercatorTiledImageLayer layer) {
            this.tile = tile;
            this.layer = layer;
        }

        public ByteBuffer run(Retriever retriever) {
            if (retriever == null) {
                String msg = Logging.getMessage("nullValue.RetrieverIsNull");
                Logging.logger().severe(msg);
                throw new IllegalArgumentException(msg);
            }

            try {
                if (!retriever.getState().equals(Retriever.RETRIEVER_STATE_SUCCESSFUL)) {
                    return null;
                }

                URLRetriever r = (URLRetriever)retriever;
                ByteBuffer buffer = r.getBuffer();

                if (retriever instanceof HTTPRetriever) {
                    HTTPRetriever htr = (HTTPRetriever)retriever;
                    if (htr.getResponseCode() == HttpURLConnection.HTTP_NO_CONTENT) {
                        // Mark tile as missing to avoid excessive attempts
                        this.layer.getLevels().markResourceAbsent(this.tile);
                        return null;
                    } else if (htr.getResponseCode() != HttpURLConnection.HTTP_OK) {
                        // Also mark tile as missing, but for an unknown reason.
                        this.layer.getLevels().markResourceAbsent(this.tile);
                        return null;
                    }
                }

                this.layer.dataDownloadSucceded(tile, retriever);

                final File outFile = this.layer.getDataFileStore().newFile(this.tile.getPath());
                if (outFile == null) {
                    return null;
                }

                if (outFile.exists()) {
                    return buffer;
                }

                // TODO: Better, more generic and flexible handling of
                // file-format type
                if (buffer != null) {
                    String contentType = r.getContentType();
                    if (contentType == null) {
                        // TODO: logger message
                        return null;
                    }

                    if (contentType.contains("xml") || contentType.contains("html") || contentType.contains("text")) {
                        this.layer.getLevels().markResourceAbsent(this.tile);

                        StringBuffer sb = new StringBuffer();
                        while (buffer.hasRemaining()) {
                            sb.append((char)buffer.get());
                        }
                        // TODO: parse out the message if the content is xml or
                        // html.
                        Logging.logger().severe(sb.toString());

                        return null;
                    } else if (contentType.contains("dds")) {
                        this.layer.saveBuffer(buffer, outFile);
                    } else if (contentType.contains("zip")) {
                        // Assume it's zipped DDS, which the retriever would
                        // have unzipped into the buffer.
                        this.layer.saveBuffer(buffer, outFile);
                    } else if (contentType.contains("image")) {
                        BufferedImage image = this.layer.convertBufferToImage(buffer);
                        if (image != null) {
                            image = this.layer.modifyImage(image);
                            if (this.layer.isTileValid(image)) {
                                if (!this.layer.transformAndSave(image, tile.getMercatorSector(), outFile)) {
                                    image = null;
                                }
                            } else {
                                this.layer.getLevels().markResourceAbsent(this.tile);
                                return null;
                            }
                        }

                        if (image == null) {
                            // Just save whatever it is to the cache.
                            this.layer.saveBuffer(buffer, outFile);
                        }
                    }

                    if (buffer != null) {
                        this.layer.firePropertyChange(AVKey.LAYER, null, this);
                    }

                    return buffer;
                }
            } catch (java.io.IOException e) {
                this.layer.getLevels().markResourceAbsent(this.tile);
                Logging.logger()
                    .log(
                        java.util.logging.Level.SEVERE,
                        Logging.getMessage("layers.TextureLayer.ExceptionSavingRetrievedTextureFile", tile.getPath()),
                        e);
            }

            return null;
        }
    }

    protected boolean isTileValid(BufferedImage image) {
        // override in subclass to check image tile
        // if false is returned, then tile is marked absent
        return true;
    }

    protected BufferedImage modifyImage(BufferedImage image) {
        // override in subclass to modify image tile
        return image;
    }

    private BufferedImage convertBufferToImage(ByteBuffer buffer) {
        try {
            InputStream is = new ByteArrayInputStream(buffer.array());
            return ImageIO.read(is);
        } catch (IOException e) {
            return null;
        }
    }

    private boolean transformAndSave(BufferedImage image, MercatorSector sector, File outFile) {
        try {
            image = transform(image, sector);
            String extension = outFile.getName().substring(outFile.getName().lastIndexOf('.') + 1);
            synchronized (this.fileLock) // synchronized with read of file in
            {
                return ImageIO.write(image, extension, outFile);
            }
        } catch (IOException e) {
            return false;
        }
    }

    private BufferedImage transform(BufferedImage image, MercatorSector sector) {
        int type = image.getType();
        if (type == 0) {
            type = BufferedImage.TYPE_INT_RGB;
        }

        BufferedImage trans = new BufferedImage(image.getWidth(), image.getHeight(), type);
        double miny = sector.getMinLatPercent();
        double maxy = sector.getMaxLatPercent();
        for (int y = 0; y < image.getHeight(); y++) {
            double sy = 1.0 - y / (double)(image.getHeight() - 1);
            Angle lat = Angle.fromRadians(sy * sector.getDeltaLatRadians() + sector.getMinLatitude().radians);
            double dy = 1.0 - (MercatorSector.gudermannianInverse(lat) - miny) / (maxy - miny);
            dy = Math.max(0.0, Math.min(1.0, dy));
            int iy = (int)(dy * (image.getHeight() - 1));

            for (int x = 0; x < image.getWidth(); x++) {
                trans.setRGB(x, y, image.getRGB(x, iy));
            }
        }

        return trans;
    }
}
